/*
 * smc-milter.c
 *
 * Description:	 Sendmail Confirmation System (SMC).
 * Developed by: Alexander Djourik <sasha@iszf.irk.ru>
 *		 Anton Gorbunov <anton@iszf.irk.ru>
 *		 Pavel Zhilin <pzh@iszf.irk.ru>
 *
 * Copyright (c) 2003,2004,2005 Alexander Djourik. All rights reserved.
 *
 * Based on the Sendmail distribution's "A Sample Filter",
 * Copyright (c) 2000 Sendmail, Inc. and its suppliers. All rights reserved.
 *
 * Some ideas were borrowed from:
 *
 * Milter-Sender from Snert. Copyright (C) 2002, 2003 by Anthony Howe
 * <achowe@snert.com>
 *
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Please see the file COPYING in this directory for full copyright
 * information.
 */

#include <pwd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <unistd.h>
#include <ctype.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>
#include <netdb.h>
#include <pthread.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <openssl/md5.h>
#include "regex.h"
#include "libmilter/mfapi.h"
#include "smc-milter.h"
#include "utils.h"
#include "relay.h"
#include "smtp.h"

#if !defined O_SYNC && defined O_FSYNC
#define O_SYNC O_FSYNC
#endif

/* Static variables for config defaults, etc. */
static int  runmode   = 0;
static char *smfisock = NULL;
static char *smfiuser = NULL;
static char *domain   = NULL;
static int  timeout   = 0;
static int  extreplace = 1;
static int  cleanhtml = 1;
static int  lifetime  = ACCESS_LIFETIME;
static int  cachetime = CACHE_LIFETIME;
static int  maxcount  = MQUEUE_LIMIT;
static int  maxdelay  = MQUEUE_LIFETIME;
static int  maxrecs   = MAXREC_LIMIT;
static int  terminate = 0;
static int  nochild   = 0;

struct mlfiPriv {
	unsigned long status;
	char helo_host[MAXLINE];
	char conn_addr[MAXLINE];
	char from_host[MAXLINE];
	char from_addr[MAXLINE];
	char dnsbl_report[MAXLINE];
	char check_report[MAXLINE];
	char boundary[MAXLINE];
	char temp_file[MAXLINE];
};

#ifndef bool
#define bool   int
#define TRUE   1
#define FALSE  0
#endif /* ! bool */

#define MLFIPRIV ((struct mlfiPriv *) smfi_getpriv(ctx))

sfsistat mlfi_cleanup (SMFICTX *ctx, sfsistat rc, bool ok);

static char *exts[] = {
/*
 * Microsoft considers these extensions dangerous:
 * http://support.microsoft.com/support/kb/articles/Q262/6/17.ASP
 *
 * File extension  File type
 * --------------------------------------------------- */
 ".ade",	// Microsoft Access project extension 
 ".adp",	// Microsoft Access project 
 ".bas",	// Microsoft Visual Basic class module 
 ".bat",	// Batch file 
 ".chm",	// Compiled HTML Help file 
 ".cmd",	// Microsoft Windows NT Command script 
 ".com",	// Microsoft MS-DOS program 
 ".cpl",	// Control Panel extension 
 ".crt",	// Security certificate 
 ".exe",	// Program 
 ".hlp",	// Help file 
 ".hta",	// HTML program
 ".inf",	// Setup Information 
 ".ins",	// Internet Naming Service 
 ".isp",	// Internet Communication settings 
 ".js",		// JScript file 
 ".jse",	// Jscript Encoded Script file 
 ".lnk",	// Shortcut 
 ".mdb",	// Microsoft Access program 
 ".mde",	// Microsoft Access MDE database 
 ".msc",	// Microsoft Common Console document 
 ".msi",	// Microsoft Windows Installer package 
 ".msp",	// Microsoft Windows Installer patch 
 ".mst",	// Microsoft Visual Test source files 
 ".pcd",	// Photo CD image, Microsoft Visual compiled script 
 ".pif",	// Shortcut to MS-DOS program 
 ".reg",	// Registration entries 
 ".scr",	// Screen saver 
 ".sct",	// Windows Script Component 
 ".shs",	// Shell Scrap object 
 ".shb",	// Shell Scrap object
 ".url",	// Internet shortcut 
 ".vb",		// VBScript file 
 ".vbe",	// VBScript Encoded script file 
 ".vbs",	// VBScript file 
 ".wsc",	// Windows Script Component 
 ".wsf",	// Windows Script file 
 ".wsh",	// Windows Script Host Settings file
NULL };

/* Known dangerous HTML tags */
static char *tags[] = {
	"<(meta|base)[^>]*>",
	"<frame[^>]*>.*?</frame[^>]*>",
	"<iframe[^>]*>.*?</iframe[^>]*>",
	"<script[^>]*>.*?</script[^>]*>",
	"<object[^>]*>.*?</object[^>]*>",
	"<embed[^>]*>.*?</embed[^>]*>",
	"<applet[^>]*>.*?</applet[^>]*>",
	"<(i?frame|object|embed|applet)[^>]*>",
	"\\son(\\w+)=[^> ]+", NULL
};

#ifndef strndup
char *strndup(char const* s, size_t n)
{
	void* voided = (void*)s;
	void* match;
	char *ret;
	size_t len;

	if (!s || !n) return NULL;
	match = memchr(voided, 0, n);
	len = match
		? (char*)match - s
		: n+1;
	ret = (char*)malloc(len);
	memcpy(ret, s, len-1);
	ret[len] = '\0';
	return ret;
}
#endif /* strndup */

void 
signal_handler (int sig) {
	switch (sig) {
	case SIGTERM:
	    terminate = 1;
	    break;
	case SIGCHLD:
	    nochild = 1;
	    break;
	}
}

void
signal_setup (void) {
	signal(SIGPIPE, SIG_IGN);
	signal(SIGCHLD, signal_handler);
	signal(SIGTERM, signal_handler);
	nochild = 0;
}

void
strtolower (char *str) {

	/* check for required data presented */
	if (!str) return;

	for (; *str; str++) *str = tolower (*str);
}

char *
find_boundary (char *pstart, char *boundary, int *term) {
	int len;

	/* check for required data presented */
	if (!boundary || *term) return NULL;

	for (len = strlen(boundary); *pstart; pstart++) {
	    if (*pstart == '-' && *(pstart + 1) == '-')
	    if (strncmp(pstart + 2, boundary, len) == 0) {
		if (*(pstart + len + 3) == '-' &&
		    *(pstart + len + 4) == '-') *term = 1;
		return (pstart + len + 2);
	    }
	}

	return NULL;
}

char *
get_header (char *buffer, char *hname, char *hvalue) {
	char *s;
	char *p = buffer;
	int maxsize = MAXLINE - 1;
	int size;

	/* strip header on front */
	while (isspace(*p)) p++;

	/* find content header */
	if (strncasecmp(p, "Content-", 8)) return NULL;

	/* get header name */
	for (s = p; !isspace(*p) && *p != ':';) p++;
	if (*p++ != ':') return NULL;

	size = p - s - 1;
	if (size > maxsize) size = maxsize;
	memcpy(hname, s, size);
	hname[size] = '\0';

	/* skip white spaces */
	while (isspace(*p)) p++;

	/* get header value */
	for (s = p; (p - s) < maxsize;) {

	    /* strip field value on front */
	    while (isspace(*p)) p++;

	    while (!isspace(*p) && *p != '=' && *p != ';') p++;
	    if (*p != '=' && *p != ';') break;
	    p++;

	    if (*p++ == '"') {
		while (*p != '\n' && *p != '"' && *p != ';') p++;
		if (*p == '"') p++;
		if (*p++ != ';') break;
	    }
	}

	size = p - s - 1;
	if (size > maxsize) size = maxsize;
	memcpy(hvalue, s, size);
	hvalue[size] = '\0';

	return p;
}

int
get_param (char *buffer, char *param, char *value) {
	char *p = buffer;
	int size = strlen(param);

	while ((p = strchr(p, ';'))) {
	    p++;

            /* skip white spaces */
            while (isspace(*p)) p++;

	    if (strncasecmp(p, param, size) == 0) {
		if (sscanf(p + size, "=\"%1023[^\"]\"", value) > 0 ||
		    sscanf(p + size, "=%1023s", value) > 0) return 1;
		return 0;
	    }
	}

	return 0;
}

int
regex_match (char *regex, char *buffer) {
	regex_t preg;
	int status;

	/* check for required data presented */
	if (!regex || !buffer) return 0;

	if (regcomp(&preg, regex, REG_EXTENDED|REG_ICASE) != REG_NOERROR) {
	    printf("(match_regex) Error compiling regex \"%s\": %m", regex);
	    return 0;
	}

	status = regexec(&preg, buffer, (size_t) 0, NULL, 0);
	regfree(&preg);

	if (status == REG_NOERROR) return 1;

	return 0;
}

int
regex_remove (char *regex, char *buffer) {
	regmatch_t pmatch;
	regex_t preg;
	int status, rsize;

	/* check for required data presented */
	if (!regex || !buffer) return 0;

	if (regcomp(&preg, regex, REG_EXTENDED|REG_ICASE) != REG_NOERROR) {
	    printf("(remove_regex) Error compiling regex \"%s\": %m", regex);
	    return 0;
	}

	status = regexec(&preg, buffer, 1, &pmatch, 0);
	regfree(&preg);
	
	if (status == REG_NOERROR) {
	    rsize = strlen(buffer) - pmatch.rm_eo + 1;
	    memmove(buffer + pmatch.rm_so, buffer + pmatch.rm_eo, rsize);
	    return (pmatch.rm_eo - pmatch.rm_so);
	}

	return 0;
}

int
remove_tags (char *pstart, char *pend) {
	char **tag;
	char *buffer;
	int rsize = 0;
	int len;

	/* check for required data presented */
	if (!pstart || !pend) return 0;

	if (!(buffer = strndup(pstart, pend - pstart))) return 0;

	for (tag = tags; *tag; tag++)
	    while ((len = regex_remove(*tag, buffer)))
		rsize += len;

	len = pend - pstart - rsize;
	memcpy(pstart, buffer, len);
	memmove(pstart + len, pend, strlen(pend) + 1);

	free(buffer);

	return rsize;
}

int
replace_ext (char *pstart, char *pend) {
	char *buffer, *pfile, *pname;
	char name[MAXLINE];
	char **ext;
	int found = 0;
	int len;

	/* check for required data presented */
	if (!pstart || !pend) return 0;

	if (!(buffer = strndup(pstart, pend - pstart))) return 0;

	if ((pname = strcasestr(buffer, "name="))) {
	    if (sscanf(pname + 5, "\"%1023[^\n\"]\"", name) == 1 ||
		sscanf(pname + 5, "%1023[^\n]", name) == 1) {
		for (ext = exts, len = strlen(name); *ext; ext++)
		if (strcasecmp(name + len - strlen(*ext), *ext) == 0) {
		    found++;
		    while ((pfile = strcasestr(pname, name))) {
			pfile[len-1] = '_';
			pname = pfile;
		    }
		}
		memcpy(pstart, buffer, pend - pstart);
	    }
	}

	free(buffer);

	return found;
}

int
dnsbl_check_connect (SMFICTX *ctx, char *id, char *conn) {
	struct mlfiPriv *priv = MLFIPRIV;
	char buffer[MAXLINE], flag[MAXLINE];
	char address[MAXLINE], info[MAXLINE];
	int status;
	FILE *fh;

	/* check for private data presented */
        if (!priv) return -1;

	/* check for required data presented */
	if (!conn) return 0;

	/* do dnsbl policy checks */
	if ((fh = fopen(POLICY_FILE, "r"))) {
	    while (fgets(buffer, MAXLINE, fh)) {
		if (*buffer == '#') continue;
		if (strncasecmp(id, buffer, strlen(id))) continue;
		if (sscanf(buffer, "%*s %s %s %[^\n]", flag, address, info) == 3) {
		    if (id == RGX_ID)
			status = regex_match(address, conn);
		    else status = check_dnsbl(address, conn, cachetime);
		    if (status < 0) {
			fclose(fh);
			return -1;
		    } else if (status > 0) {
			if (strncasecmp("NO", flag, 2) == 0) {
			    priv->status |= STAT_WLIST;
			    sprintf(priv->dnsbl_report, "Host %s accepted: %s", conn, info);
			} else if (strncasecmp("YES", flag, 3) == 0) {
			    priv->status |= STAT_BLOCK;
			    sprintf(priv->dnsbl_report, "Host %s rejected: %s", conn, info);
			}
			fclose(fh);
			return 1;
		    }
		}
	    }
	    fclose(fh);
	}

	return 0;
}

sfsistat
mlfi_connect (SMFICTX *ctx, char *hostname, _SOCK_ADDR *hostaddr) {
	struct mlfiPriv *priv = MLFIPRIV;

	/* check for private data presented */
        if (priv) return SMFIS_TEMPFAIL;

	/* allocate memory for private data */
	if (!(priv = calloc(1, sizeof(struct mlfiPriv)))) {
	    syslog(LOG_ERR, "%s", strerror(errno));
	    return SMFIS_TEMPFAIL;
	}

	/* set private data pointer */
	if (smfi_setpriv(ctx, priv) != MI_SUCCESS)
	    return SMFIS_TEMPFAIL;

	return SMFIS_CONTINUE;
}

sfsistat
mlfi_helo (SMFICTX *ctx, char *helohost) {
	struct mlfiPriv *priv = MLFIPRIV;

	/* check for required data presented */
	if (!helohost || *helohost == '\0') {
	    smfi_setreply(ctx, "501", "5.5.2",
		"HELO requires domain address");
	    return SMFIS_REJECT;
	}

	/* store helo hostname */
	strncpy(priv->helo_host, helohost, MAXLINE - 15);

	return SMFIS_CONTINUE;
}

sfsistat
mlfi_envfrom (SMFICTX *ctx, char **envfrom) {
	struct mlfiPriv *priv = MLFIPRIV;
	char *conn_host, *from_conn, *from_host, *from_addr;
	int result = 0;

	/* check for private data presented */
        if (!priv) return SMFIS_TEMPFAIL;

	/* check for HELO data presented */
	if (*priv->helo_host == '\0') {
	    smfi_setreply(ctx, "503", "5.0.0", "Need HELO before MAIL");
	    return SMFIS_REJECT;
	}

	/* get macro data */
	from_conn = smfi_getsymval(ctx, "{client_addr}");
	conn_host = smfi_getsymval(ctx, "{client_name}");
	from_addr = smfi_getsymval(ctx, "{mail_addr}");
	from_host = smfi_getsymval(ctx, "{mail_host}");

	/* check for required data presented */
	if (!from_conn || !from_host || !from_addr)
	    return SMFIS_TEMPFAIL;

	if (!conn_host || *conn_host == '\0')
	    conn_host = priv->helo_host;

	/* store data */
	if (*from_addr == '\0') {
	    /*	RFC-821 Section 4.1.1: A host that supports a receiver-SMTP
		MUST support the reserved mailbox "Postmaster" */
	    sprintf(priv->from_addr, "postmaster@%s", conn_host);
	    from_host = conn_host;
	} else {
	    strtolower(from_addr);
	    strncpy(priv->from_addr, from_addr, MAXLINE - 1);
	    if (*from_host == '\0') from_host = domain;
	}

	strncpy(priv->from_host, from_host, MAXLINE - 1);
	strncpy(priv->conn_addr, from_conn, MAXLINE - 1);

	/* accept local relay connections */
	if (check_relay(from_conn, SM_ACCESS_FILE) > 0) {
	    priv->status |= STAT_RELAY;
	    return SMFIS_CONTINUE;
	}

	/* load average at which filter will sleep for one
	   second before accepting incoming connections. */
	usleep(10000);

	/* do default check_connect policy */
	switch(check_connect(from_conn, from_host, cachetime)) {
	case ERROR_INT:
	    return SMFIS_TEMPFAIL;
	case NOREC_MX:
	    sprintf(priv->check_report,
		"No MX hosts for a site %s", from_host);
	    priv->status |= STAT_CHECK;
	    break;
	case NOREC_DOMAIN:
	    sprintf(priv->check_report,
		"No NS hosts for a site %s", from_host);
	    priv->status |= STAT_CHECK;
	    break;
	case NOREC_CLIENT:
	    sprintf(priv->check_report,
		"No NS hosts for connection address %s", from_conn);
	    priv->status |= STAT_CHECK;
	    break;
	case HOST_UNKNOWN:
	    sprintf(priv->check_report,
		"Host %s is not related to %s", from_conn, from_host);
	    priv->status |= STAT_CHECK;
	    break;
	default:
	    sprintf(priv->check_report,
		"Host %s is related to %s", from_conn, from_host);
	    break;
	}

	/* do dnsbl policy checks */
	if (!(result |= dnsbl_check_connect(ctx, RGX_ID, conn_host)))
	if (!(result |= dnsbl_check_connect(ctx, RBL_ID, from_conn)))
	if (!(result |= dnsbl_check_connect(ctx, DUL_ID, from_conn)))
	if (result < 0) return SMFIS_TEMPFAIL;

	return SMFIS_CONTINUE;
}

sfsistat
mlfi_envrcpt (SMFICTX *ctx, char **envrcpt) {
	struct mlfiPriv *priv = MLFIPRIV;
	char buffer[MAXLINE];
	char md_sign[MD5_STRING_LENGTH + 1];
	char *msg_id, *rcpt_addr, *rcpt_host;
	int ret, now = (int) time(NULL);
	rec value;

	/* check for private data presented */
        if (!priv) return SMFIS_TEMPFAIL;

	/* get macro data */
	msg_id = smfi_getsymval(ctx, "{i}");
	rcpt_addr = smfi_getsymval(ctx, "{rcpt_addr}");
	rcpt_host = smfi_getsymval(ctx, "{rcpt_host}");

	/* check for required data presented */
	if (!rcpt_addr || !mailer) return SMFIS_TEMPFAIL;

	/* check local user address */
	if (*rcpt_addr == '\0') {
	    smfi_setreply(ctx, "550", "5.1.1", "User unknown");
	    return SMFIS_REJECT;
	}

	/* check for recipient is local */
	if (!strchr(rcpt_addr, '@') || *rcpt_host == '\0')
	    priv->status |= STAT_LOCAL;
	
	/* check remote recipient */
	if (!(priv->status & (STAT_RELAY | STAT_LOCAL))) {
	    if (!verify(rcpt_addr, rcpt_host, msg_id, domain, cachetime)) {
		smfi_setreply(ctx, "550", "5.1.1", "Unknown recipient");
		return SMFIS_REJECT;
	    }
	}

	/* accept trust recipients mail */
	if (find_user(USERS_FILE, rcpt_addr)) {
	    priv->status |= STAT_ULIST;
	    return SMFIS_CONTINUE;
	}

	/* accept mail from whitelisted senders */
	if ((priv->status & STAT_WLIST)) return SMFIS_CONTINUE;

	/* reject mail from blackholes */
	if ((priv->status & STAT_BLOCK)) {
	    smfi_setreply(ctx, "550", "5.7.1", priv->dnsbl_report);
	    return SMFIS_REJECT;
	}

	/* lowercase recipient address */
	strtolower(rcpt_addr);

	/* accept local and outgoing mail now */
	if ((priv->status & STAT_RELAY)) {
	    if ((priv->status & STAT_LOCAL)) return SMFIS_CONTINUE;

	    /* Autoconfirmation algorithm.
	    Please see the files COPYING and README in this directory
	    for more information about this algorithm and for full copyright
	    information. */

	    /* calculate backward md5 digits for this contact */
	    md5sign(rcpt_addr, priv->from_addr, md_sign);

	    /* check access database for this contact */
	    if ((ret = get_record(HASH_ACCESS_DB, md_sign, &value)) < 0)
		return SMFIS_TEMPFAIL;

	    if (ret == 1) {
		value.time1 = now;
		value.data = 0;
	    }

	    value.time2 = now;

	    /* update contact statistics */
	    add_record(HASH_ACCESS_DB, md_sign, &value, lifetime);

	    return SMFIS_CONTINUE;
	}

	/* calculate forward md5 digits for this contact */
	md5sign(priv->from_addr, rcpt_addr, md_sign);

	/* check access database for this contact */
	if (get_record(HASH_ACCESS_DB, md_sign, &value) == 0) {

	    /* mark this contact as authenticated */
	    if (value.time1) value.time1 = 0;

	    value.time2 = now;

	    /* update this record */
	    add_record(HASH_ACCESS_DB, md_sign, &value, lifetime);
		
	    sprintf(priv->dnsbl_report, "Accepted as authenticated");
	    priv->status |= STAT_WLIST;

	    return SMFIS_CONTINUE;
	}

	/* accept mail from trusted senders */
	if (!(priv->status & STAT_CHECK)) return SMFIS_CONTINUE;

	/* calculate connection md5 digits for this contact */
	md5sign(priv->conn_addr, priv->from_addr, md_sign);

	/* check mqueue database for this contact */
	if ((ret = get_record(HASH_MQUEUE_DB, md_sign, &value)) < 0)
	    return SMFIS_TEMPFAIL;

	if (ret == 1) {

	    /* try to verify sender address */
	    switch (verify(priv->from_addr, priv->from_host, msg_id, domain, cachetime)) {
	    case 0:
		smfi_setreply(ctx, "550", "5.7.1",
		    "Sender address verification failed");
		return SMFIS_REJECT;
	    case 1: 
		value.time1 = now;
		value.time2 = now;
		value.data = 0;

		/* add new contact to queue */
		add_record(HASH_MQUEUE_DB, md_sign, &value, lifetime);
		break;
	    default:
		smfi_setreply(ctx, "451", "4.7.0",
		    "Sender address verification in progress");
		return SMFIS_TEMPFAIL;
	    } 

	} else if (ret == 0) {

	    /* Check delays algorithm:

	    Statistical check of delivery delays, generated by a [tempfail]
	    return code, that allows to exclude a direct receiving of a mail,
	    which not passed through the standard mail server.

	    Copyright (c) 2003 Alexander Djourik. All rights reserved.
	    Please see the file COPYING in this directory for full copyright
	    information. */

	    if ((now - value.time2) >= MQUEUE_COST) {

		del_record(HASH_MQUEUE_DB, md_sign);
		if ((now - value.time1) <= maxdelay) {
		    if (++value.data >= maxcount) {

			/* accept delayed mail */
			return SMFIS_CONTINUE;
		    } else {
			value.time2 = now;
			add_record(HASH_MQUEUE_DB, md_sign, &value, lifetime);
		    }
		}
	    }
	}

	sprintf(buffer, "Transaction delayed, reason: %s", priv->check_report);
	smfi_setreply(ctx, "451", "4.7.0", buffer);

	return SMFIS_TEMPFAIL;
}

sfsistat
mlfi_header (SMFICTX *ctx, char *headerf, char *headerv) {
	struct mlfiPriv *priv = MLFIPRIV;

	/* check for private data presented */
        if (!priv) return SMFIS_TEMPFAIL;

	/* check for required data presented */
	if (!headerf || !headerv) return SMFIS_TEMPFAIL;

	/* skip null headers */
	if (*headerf == '\0' || *headerv == '\0')
	    return SMFIS_CONTINUE;

	/* get boundary if found */
	if (strcasecmp(headerf, "Content-Type") == 0)
	    get_param(headerv, "boundary", priv->boundary);

	return SMFIS_CONTINUE;
}

sfsistat
mlfi_eoh (SMFICTX *ctx) {
	struct mlfiPriv *priv = MLFIPRIV;

	/* check for private data presented */
        if (!priv) return SMFIS_TEMPFAIL;

	return SMFIS_CONTINUE;
}

sfsistat
mlfi_body (SMFICTX * ctx, u_char * bodyp, size_t len) {
	struct mlfiPriv *priv = MLFIPRIV;
	int fd;

	/* check for private data presented */
        if (!priv) return SMFIS_TEMPFAIL;

	/* check for required data presented */
	if (!bodyp) return SMFIS_TEMPFAIL;

	if (extreplace || cleanhtml) {

	    /* open message body tempfile */
	    if (*priv->temp_file == '\0') {
		strcpy(priv->temp_file, TMP_FILE);
		if ((fd = mkstemp(priv->temp_file)) < 0) {
		    syslog(LOG_ERR, "Can't make message tempfile: %s", strerror(errno));
		    return SMFIS_TEMPFAIL;
		}
	    } else {
		if ((fd = open(priv->temp_file, O_WRONLY|O_APPEND|O_CREAT|O_SYNC)) < 0) {
		    syslog(LOG_ERR, "Can't open message tempfile: %s", strerror(errno));
		    return SMFIS_TEMPFAIL;
		}
	    }

	    /* append body chunk */
	    write(fd, bodyp, len);
	    close(fd);
	}

        return SMFIS_CONTINUE;
}

int
content_filter (char *bodyp, char *boundary) {
	char *pstart = NULL, *pend = NULL, *q;
	char mime[MAXLINE], mboundary[MAXLINE];
	char name[MAXLINE], value[MAXLINE];
	int  mod_stat = 0;
	int  term = 0;

	/* check for required data presented */
	if (!bodyp) return 0;

	for (pstart = bodyp, pend = bodyp + strlen(bodyp),
	    *mboundary = '\0'; !term;) {

	    if (*boundary) {
		if (!(pstart = find_boundary(pstart, boundary, &term))) break;
		if (!(pend = find_boundary(pstart, boundary, &term))) break;
	    } else term = 1;

	    for (q = pstart, *mime = '\0';
		(q = get_header(q, name, value));)
		if (strcasecmp(name, "Content-Type") == 0) {
		    sscanf(value, "%255s", mime);
		    break;
		}

	    if (*mime == '\0' || // undefined type means text/plain 
		strncasecmp(mime, "text/plain", 10) == 0 ||
		strncasecmp(mime, "text/html", 9) == 0) {

	        // remove dangerous html tags 
		if (cleanhtml) mod_stat |= remove_tags(pstart, pend);

	    } else if (strncasecmp(mime, "multipart", 9) == 0) {
		if (get_param(value, "boundary", mboundary))
		    mod_stat |= content_filter(pstart, mboundary);
	    } else {
		// replace attached files extensions 
		if (extreplace) {
		    if (!get_param(value, "name", name)) {
			for (q = pstart; (q = get_header(q, name, value));)
			    if (strcasecmp(name, "Content-Disposition") == 0 &&
				get_param(value, "filename", name))
				mod_stat |= replace_ext(pstart, pend);
		    } else mod_stat |= replace_ext(pstart, pend);
		}
	    }
	}

	return mod_stat;
}

sfsistat
mlfi_eom (SMFICTX *ctx) {
	struct mlfiPriv *priv = MLFIPRIV;
	char *bodyp;
	int mod_stat = 0;
	int add_headers = 0;
	int fd, body_size;

	/* check for private data presented */
        if (!priv) return SMFIS_TEMPFAIL;

	/* add headers only for incoming mail */
	if (!(priv->status & STAT_RELAY)) add_headers = 1;

	/* add checker headers */
	if (add_headers) {
	    if ((priv->status & STAT_WLIST)) {
		smfi_addheader(ctx, FLAG_HEADER, "NO");
		smfi_addheader(ctx, REPORT_HEADER, priv->dnsbl_report);
	    } else if ((priv->status & STAT_BLOCK)) {
		smfi_addheader(ctx, FLAG_HEADER, "YES");
		smfi_addheader(ctx, REPORT_HEADER, priv->dnsbl_report);
	    } else if ((priv->status & STAT_CHECK)) {
		smfi_addheader(ctx, FLAG_HEADER, "WARN");
		smfi_addheader(ctx, REPORT_HEADER, priv->check_report);
	    } else {
		smfi_addheader(ctx, FLAG_HEADER, "NO");
		smfi_addheader(ctx, REPORT_HEADER, priv->check_report);
	    }
	    smfi_addheader(ctx, CHECKER_HEADER, MILTER " " VERSION);
	}

	/* return if no message body is available */
	if (*priv->temp_file == '\0')
	    return mlfi_cleanup(ctx, SMFIS_CONTINUE, FALSE);

	if ((fd = open(priv->temp_file, O_RDWR|O_CREAT|O_SYNC)) < 0)
	    return mlfi_cleanup(ctx, SMFIS_TEMPFAIL, FALSE);

	/* get the temporary file size */
	body_size = lseek(fd, 0, SEEK_END);
	lseek(fd, 0, SEEK_SET);

	if (body_size > 0 && (bodyp = (char *) mmap(NULL, body_size + 1,
	    PROT_READ|PROT_WRITE, MAP_PRIVATE, fd, 0)) != MAP_FAILED) {
	    bodyp[body_size] = '\0';

	    /* parse message body */
	    if (extreplace || cleanhtml)
		mod_stat |= content_filter(bodyp, priv->boundary);
	    if (mod_stat) smfi_replacebody(ctx, bodyp, strlen(bodyp));

	    munmap(bodyp, body_size + 1);
	}

	close(fd);

	return mlfi_cleanup(ctx, SMFIS_CONTINUE, FALSE);
}

sfsistat
mlfi_cleanup (SMFICTX *ctx, sfsistat rc, bool ok) {
	struct mlfiPriv *priv = MLFIPRIV;

	/* check for private data presented */
        if (!priv) return rc;

	/* unlink temporary file */
	if (*priv->temp_file)
	    unlink(priv->temp_file);

	/* cleanup per-message data */
	*priv->from_host = '\0';
	*priv->from_addr = '\0';
	*priv->temp_file = '\0';
	*priv->boundary  = '\0';

	if (ok) {
	    /* release private memory */
	    free(priv);
	    smfi_setpriv(ctx, NULL);
	}

	return rc;
}

sfsistat
mlfi_abort (SMFICTX *ctx) {
	return mlfi_cleanup(ctx, SMFIS_CONTINUE, FALSE);
}

sfsistat
mlfi_close (SMFICTX *ctx) {
	return mlfi_cleanup(ctx, SMFIS_CONTINUE, TRUE);
}

void
parse_config (void) {
	FILE *fh = fopen(CONFIG_FILE, "r");
	char buffer[MAXLINE], value[MAXLINE];

	if (!fh) return;

	while (fgets(buffer, MAXLINE, fh))
	if (*buffer == '#') {
	    continue;
	} else if (strncasecmp("user", buffer, 4) == 0) {
	    sscanf(buffer, "%*s %s", value);
	    if (!smfiuser) smfiuser = strdup(value);
	} else if (strncasecmp("domain", buffer, 6) == 0) {
	    sscanf(buffer, "%*s %s", value);
	    domain = strdup(value);
	} else if (strncasecmp("extreplace", buffer, 10) == 0) {
	    sscanf(buffer, "%*s %s", value);
	    extreplace = strcasecmp(value, "yes")? 0 : 1;
	} else if (strncasecmp("cleanhtml", buffer, 9) == 0) {
	    sscanf(buffer, "%*s %s", value);
	    cleanhtml = strcasecmp(value, "yes")? 0 : 1;
	} else if (strncasecmp("lifetime", buffer, 8) == 0) {
	    sscanf(buffer, "%*s %d", &lifetime);
	} else if (strncasecmp("cachetime", buffer, 9) == 0) {
	    sscanf(buffer, "%*s %d", &cachetime);
	} else if (strncasecmp("maxdelay", buffer, 8) == 0) {
	    sscanf(buffer, "%*s %d", &maxdelay);
	} else if (strncasecmp("maxcount", buffer, 8) == 0) {
	    sscanf(buffer, "%*s %d", &maxcount);
	} else if (strncasecmp("maxrecs", buffer, 7) == 0) {
	    sscanf(buffer, "%*s %d", &maxrecs);
	}

	fclose(fh);
}

void
usage () {
	printf("Usage:\t" PROGNAME "\t[-fhv] [-p socket] [-t time] [-u username]\n\n");
}

void
version () {
	printf(PROJECT_NAME " Version " VERSION "\n" COPYRIGHT "\n\n");
}

void
help () {
	version();
	usage();
	printf("\t-f\t\t\tRun milter in the foreground.\n");
	printf("\t-p socket\t\tPath to create socket.\n");
	printf("\t-t time\t\t\tSocket connection timeout in seconds.\n");
	printf("\t-u username\t\tSpecifies the user the milter should\n");
	printf("\t\t\t\trun as after it initializes.\n");
	printf("\t-v\t\t\tShow program version.\n");
	printf("\t-h\t\t\tShow this help.\n\n");
	printf("Program recognises the following config file options:\n\n");
	printf("\t[user username]\t\tSpecifies the user the milter should\n");
	printf("\t\t\t\trun as after it initializes.\n");
	printf("\t[domain name]\t\tSMTP domain name.\n");
	printf("\t[extreplace (yes|no)]\tEnable or disable replacing\n");
	printf("\t\t\t\textensions of attached files.\n");
	printf("\t[cleanhtml (yes|no)]\tEnable or disable cleanup\n");
	printf("\t\t\t\tfeature for html messages.\n");
	printf("\t[lifetime time]\t\tDatabase records lifetime.\n");
	printf("\t[cachetime time]\tCache records lifetime.\n");
	printf("\t[maxdelay time]\t\tConnect counter lifetime.\n");
	printf("\t[maxcount count]\tConnect counter limit.\n");
	printf("\t[maxrecs count]\t\tDatabase records limit.\n\n");
	printf("Program default settings:\n\n");
	printf("\tdaemon\t\t\t[%s]\n", "yes");
	printf("\ttimeout\t\t\t[%d]\n", SOCKET_TIMEOUT);
	printf("\tuser\t\t\t[%s]\n", DEFAULT_USER);
	printf("\tdomain\t\t\t[%s]\n", DEFAULT_DOMAIN);
	printf("\textreplace\t\t[%s]\n", (extreplace) ? "yes" : "no");
	printf("\tcleanhtml\t\t[%s]\n", (cleanhtml) ? "yes" : "no");
	printf("\tlifetime\t\t[%d]\n", ACCESS_LIFETIME);
	printf("\tcachetime\t\t[%d]\n", CACHE_LIFETIME);
	printf("\tmaxdelay\t\t[%d]\n", MQUEUE_LIFETIME);
	printf("\tmaxcount\t\t[%d]\n\n", MQUEUE_LIMIT);
	printf("Default location of files:\n\n");
	printf("\tSocket file:\t\t%s\n", SOCKET_FILE);
	printf("\tConfig file:\t\t%s\n", CONFIG_FILE);
	printf("\tPolicy file:\t\t%s\n\n", POLICY_FILE);
	printf("%s\n\n", DEVELOPERS);
}

int
create_pid (char *pidfile, pid_t pid) {
	char buffer[16];
	pid_t spid;
	int fd;

	if (access(pidfile, R_OK) == 0) {
	    if ((fd = open(pidfile, O_RDONLY)) < 0)
		return -1;

	    if (read(fd, buffer, sizeof(buffer)) < 0) {
		close(fd);
		return -1;
	    }

	    close(fd);
	    sscanf(buffer, "%d", &spid);

	    if (spid == pid) return 0;
	    if ((kill(spid, 0) < 0) && errno == ESRCH)
		unlink(pidfile);
	    else return 1;
	}

	if (!pid) return 0;

	if ((fd = open(pidfile, O_RDWR|O_TRUNC|O_CREAT, 0644)) < 0)
	    return -1;

	/* put my pid in it */
	sprintf(buffer, "%d", pid);
	write(fd, buffer, strlen(buffer));
	close(fd);

	return 0;
}

struct smfiDesc smfilter = {
	MILTER,		/* filter name */
	SMFI_VERSION,	/* version code -- do not change */
	SMFIF_ADDHDRS|SMFIF_CHGBODY,	/* flags */
	mlfi_connect,	/* connection info filter */
	mlfi_helo,	/* SMTP HELO command filter */
	mlfi_envfrom,	/* envelope sender filter */
	mlfi_envrcpt,	/* envelope recipient filter */
	mlfi_header,	/* header filter */
	mlfi_eoh,	/* end of header */
	mlfi_body,	/* body block filter */
	mlfi_eom,	/* end of message */
	mlfi_abort,	/* message aborted */
	mlfi_close	/* connection cleanup */
};

void 
start_phoenix (void) {
	int i;
	pid_t pid;

start:
	/* remove old socket if found */
	unlink(smfisock);

	/* setup signals */
	signal_setup();

	/* specify the socket to use */
	if (smfi_setconn(smfisock) == MI_FAILURE) return;

	/* set socket timeout */
	if (smfi_settimeout(timeout) == MI_FAILURE) return;

	/* register the filter */
	if (smfi_register(smfilter) == MI_FAILURE) return;

	switch ((pid = fork())) {
	case -1:
	    syslog(LOG_ERR, "Could not fork new process: %s",
		strerror(errno));
	    return;
	case 0:
	    init_db (maxrecs);

	    /* open syslog */
	    openlog(PROGNAME, 0, LOG_DAEMON);

	    /* ignore control signals */
	    signal(SIGUSR1, SIG_IGN);
	    signal(SIGUSR2, SIG_IGN);

	    /* set file creation mask */
	    umask(S_IXUSR|S_IRWXG|S_IRWXO);

	    /* hand control to libmilter */
	    if (smfi_main() != MI_SUCCESS) {
		syslog(LOG_ERR, "Shutdown abnormally");
		exit(EX_UNAVAILABLE);
	    }

	    closelog();

	    close_db ();

	    exit(EX_OK);
	}

	while (!terminate && !nochild) sleep(1);

	if (terminate) {
	    syslog(LOG_INFO, "Stopping..");
	    kill(0, SIGTERM);
	    waitpid(0, NULL, 0);
	    return;
	}

	/* terminate processes */
	for (i = 0; i < 4; i++) {
	    if (kill(-pid, SIGTERM) < 0) {
		waitpid(-pid, NULL, 0);
		if (kill(pid, SIGTERM) < 0) {
		    waitpid(pid, NULL, 0);
		    sleep(1);
		    break;
		}
		sleep(1);
	    }
	}

	/* rip threads */
	kill(-pid, SIGKILL);
	waitpid(-pid, NULL, 0);

	/* rip child */
	kill(pid, SIGKILL);
	waitpid(pid, NULL, 0);

	goto start;
}

int
main (int argc, char **argv) {
	int c;
	extern int optind;
	struct passwd *pw;
	pid_t pid;

	/* process command line options */
	while ((c = getopt(argc, argv, "fhvp:t:u:")) != -1) {
	    switch (c) {
		case 'f':
		    runmode = 1;
		    break;
		case 'h':
		    help();
		    exit(EX_OK);
		case 'v':
		    version();
		    exit(EX_OK);
		case 'p':
		    smfisock = optarg;
		    break;
		case 't':
		    timeout = atoi(optarg);
		    break;
		case 'u':
		    smfiuser = optarg;
		    break;
		case ':':
		    fprintf(stderr, "%s: Option '%c' requires an argument.\n",
			PROGNAME, optopt);
		    exit(EX_USAGE);
		default:
		    usage();
		    exit(EX_USAGE);
	    }
	}

	/* get configuration file options */
	parse_config();

	if (!domain)	domain = strdup(DEFAULT_DOMAIN);
	if (!smfiuser)	smfiuser = strdup(DEFAULT_USER);
	if (!smfisock)	smfisock = strdup(SOCKET_FILE);
	if (!timeout)	timeout = SOCKET_TIMEOUT;

	if (timeout < 0) {
	    fprintf(stderr, "%s: Negative timeout value '%d'\n", PROGNAME, timeout);
            exit(EX_USAGE);
	}

	if ((pw = getpwnam(smfiuser)) == NULL) {
	    fprintf(stderr, "%s: User '%s' not found\n", PROGNAME, smfiuser);
	    exit(EX_USAGE);
	}

	/* avoid running as root user and/or group */
	if (getuid() == 0 && pw->pw_uid != 0 && pw->pw_gid != 0) {
	    (void) setgid(pw->pw_gid);
	    (void) setuid(pw->pw_uid);
	}

	/* check pid file */
	if (create_pid(PID_FILE, 0) == 1) {
	    fprintf(stderr, "%s: Filter is already running..\n", PROGNAME);
	    exit(EX_OK);
	}

	if (runmode == 0) {

	    /* ignore signals */
	    signal(SIGTTOU, SIG_IGN);
	    signal(SIGTTIN, SIG_IGN);
	    signal(SIGTSTP, SIG_IGN);
	    signal(SIGPIPE, SIG_IGN);

	    /* run in background */
	    if ((pid = daemon(0, 0)) < 0) {
		fprintf(stderr, "%s: Could not run filter in background, %s",
		    PROGNAME, strerror(errno));
		exit(EX_OSERR);
	    }

	    if (pid != 0) exit(EX_OK);
	}

	/* open syslog */
	openlog(PROGNAME, 0, LOG_USER | LOG_DAEMON);

	/* get new pid */
	pid = getpid();

	/* create pid file */
	if (create_pid(PID_FILE, pid)) {
	    syslog(LOG_INFO, "%s: Can't create pid file %s",
		PROGNAME, PID_FILE);
	    exit(EX_UNAVAILABLE);
	}


	syslog(LOG_INFO, "Running in %s as user '%s'",
	    (runmode) ? "foreground" : "background", pw->pw_name);

	/* start the filter */
	start_phoenix();

	if (domain)	free(domain);
	if (smfiuser)	free(smfiuser);
	if (smfisock)	free(smfisock);

	closelog();
	exit(EX_OK);
}

/* eof */
