/*
 * smtp.c
 *
 * Description:  SMC SMTP tools
 * Developed by: Alexander Djourik <sasha@iszf.irk.ru>
 *		 Anton Gorbunov <anton@iszf.irk.ru>
 *		 Pavel Zhilin <pzh@iszf.irk.ru>
 *
 * Copyright (c) 2003,2004,2005 Alexander Djourik. All rights reserved.
 *
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <arpa/inet.h>
#include <sysexits.h>
#include <resolv.h>
#include <ctype.h>
#include <syslog.h>
#include <errno.h>
#include "smc-milter.h"
#include "smtp.h"
#include "relay.h"

#define SMTP_PORT	25
#define MAXPACKET	8192
#define MAXPREF		66000
#ifndef MAXLINE
    #define MAXLINE	4096
#endif
#ifndef MAXMX
    #define MAXMX	8
#endif

#define SMTP_TIMEOUT	60
#define SEND_TIMEOUT	60
#define RECV_TIMEOUT	60

#define SMTP_CMD_OK(x)		(200 <= (x) && (x) < 300)
#define SMTP_CMD_PERM(x)	(500 <= (x) && (x) < 600)
#define SMTP_DATA_OK(x)		((x) == 354)

#if !defined MSG_NOSIGNAL
#define MSG_NOSIGNAL 0
#endif

typedef union {
    HEADER hdr;
    u_char buf[MAXPACKET];
} querybuf;

int smtp_connect (int sock, struct sockaddr *address, int addrlen);
int get_smtp_response (char *buffer);
void close_socket (int sock);
int smtp_send (int sock, char *buffer);
int smtp_recv (int sock, char *buffer, int size);
int ehlo_chat (int sock, char *cmd, int *dsn);
int smtp_chat (int sock, char *cmd);
void smtp_quit (int sock, int rset);

int
get_mxlist (char *domain_name, char mxlist[MAXMX][MAXLINE]) {
	querybuf answer;
	char mx[MAXDNAME];
	register u_char *cp, *eom;
	int count, type, dlen, pref, ret;
	int bestpref = MAXPREF;
	int mxcount = 0;
	struct __res_state res_local;
	
	/* check for required data presented */
	if (!domain_name) return -1;

	if (resolver_init(&res_local) < 0)
	    return -1;

	do { ret = res_nquery(&res_local, domain_name, C_IN, T_MX,
	    (u_char *) &answer, sizeof(answer));
	} while (ret < 0 && errno == EINTR);
	if (ret < 0) {
	    switch (res_local.res_h_errno) {
	    case HOST_NOT_FOUND:
		ret =  0;
		break;
	    case NO_DATA:
    		/* try null mx list */
		strcpy(mxlist[mxcount++], domain_name);
		ret = mxcount;
		break;
	    default:
		ret = -1;
	    }
	    resolver_close(&res_local);
	    return ret;
	}

	eom = answer.buf + ret;
	cp = answer.buf + sizeof(HEADER);
	cp += dn_skipname(cp, eom) + QFIXEDSZ;
	count = ntohs(answer.hdr.ancount) +
		ntohs(answer.hdr.nscount);

	while (--count >= 0 && cp < eom) {
	    cp += dn_skipname(cp, eom);

	    GETSHORT(type, cp);
	    cp += (INT16SZ + INT32SZ);
	    GETSHORT(dlen, cp);

	    if (type == T_MX && mxcount < MAXMX) {
		GETSHORT(pref, cp);
		if ((ret = dn_expand(answer.buf, eom,
		    cp, mx, MAXDNAME-1)) < 0) {
		    resolver_close(&res_local);
		    return -1;
		}

		if (pref < bestpref) {
		    strcpy(mxlist[mxcount++], mxlist[0]);
		    strcpy(mxlist[0], mx);
		    bestpref = pref;
		} else strcpy(mxlist[mxcount++], mx);
		cp += ret;		
	    } else cp += dlen;
	}

	resolver_close(&res_local);

	return mxcount;
}

int
smtp_connect (int sock, struct sockaddr *address, int addrlen) {
	int noblock;
	int ret;

	if (sock < 0) return -1;

	noblock = 1;
	if (ioctl(sock, FIONBIO, &noblock) < 0) return -1;

	if (connect(sock, address, addrlen) < 0) {
	    fd_set rfds, wfds;
	    struct timeval tv;
	    int optval;
	    int optlen = sizeof(int);
	    
	    if (errno != EAGAIN &&
		errno != EINPROGRESS) return -1;

	    FD_ZERO(&rfds);
	    FD_SET(sock, &rfds);
	    wfds = rfds;

	    tv.tv_sec = SMTP_TIMEOUT;
	    tv.tv_usec = 0;

	    do {
		ret = select(sock + 1, &rfds, &wfds, NULL, &tv);
	    } while (ret < 0 && errno == EINTR);
	    if (ret <= 0) return -1;

	    if (getsockopt(sock, SOL_SOCKET, SO_ERROR,
		&optval, &optlen) < 0) return -1;
	}

	return 0;
}

int
get_smtp_response (char *buffer) {
	int ret = 0;

	/* check for required data presented */
	if (strlen(buffer) < 4) return -1;
	sscanf(buffer, "%3d", &ret);

	return ret;
}

void
close_socket (int sock) {
	if (sock != -1) {
	    shutdown(sock, SHUT_RDWR);
	    close(sock);
	}
}

int
smtp_send (int sock, char *buffer) {
	struct timeval tv;
	fd_set wfds;
	int ret;
	int counter = ERR_COUNT;

	if (sock < 0) return -1;

	tv.tv_sec = SEND_TIMEOUT;
	tv.tv_usec = 0;

	FD_ZERO(&wfds);
	FD_SET(sock, &wfds);

	do {
	    ret = select(sock + 1, NULL, &wfds, NULL, &tv);
	} while (ret < 0 && errno == EINTR);
	if (ret <= 0 || !FD_ISSET(sock, &wfds)) return -1;

	do {
	    ret = send(sock, buffer, strlen(buffer), MSG_NOSIGNAL);
	    if (ret < 0) usleep(10000);
	} while (ret < 0 && errno == EAGAIN && --counter);
	
	if (ret <= 0) return -1;

	do {
	    ret = select(sock + 1, NULL, &wfds, NULL, &tv);
	} while (ret < 0 && errno == EINTR);

	if (ret <= 0) return -1;

	return 0;
}

int
smtp_recv (int sock, char *buffer, int size) {
	fd_set rfds;
	struct timeval tv;
	char *e;
	int ret;
	int counter = ERR_COUNT;

	if (sock < 0) return -1;

	tv.tv_sec = RECV_TIMEOUT;
	tv.tv_usec = 0;

	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);
	buffer[0] = '\0';

	do {
	    ret = select(sock + 1, &rfds, NULL, NULL, &tv);
	} while (ret < 0 && errno == EINTR);
	if (ret <= 0 || !FD_ISSET(sock, &rfds)) return -1;

	do {
	    ret = recv(sock, buffer, size - 1, MSG_NOSIGNAL);
	    if (ret < 0) usleep(10000);
	} while (ret < 0 && errno == EAGAIN && --counter);

	if (ret <= 0) return -1;
	else buffer[ret] = '\0';

	if ((e = strrchr(buffer, '\r')) != NULL) *e = '\0';
	if ((e = strrchr(buffer, '\n')) != NULL) *e = '\0';

	return 0;
}

int
smtp_chat (int sock, char *cmd) {
	int ret;
	char buffer[MAXLINE];

	if (sock < 0) return -1;

	if (cmd && smtp_send(sock, cmd) < 0) return -1;
	if (smtp_recv(sock, buffer, MAXLINE-1) < 0) return -1;
	ret = get_smtp_response(buffer);

	return ret;
}

int
ehlo_chat (int sock, char *domain, int *dsn) {
	int ret;
	char buffer[MAXLINE];
	char *cp, *e;

	if (sock < 0) return -1;

	dsn[0] = '\0';
	sprintf(buffer, "EHLO %s\r\n", domain);
	if (smtp_send(sock, buffer) < 0) return -1;
	if (smtp_recv(sock, buffer, MAXLINE-1) < 0) return -1;
	if ((ret = get_smtp_response(buffer)) < 0) return -1;

	cp = e = buffer + 4;
	if (SMTP_CMD_OK(ret)) do {
	    if ((e = strchr(cp, '\n')) != NULL) *e = '\0';
	    if (strlen(cp) > 7 && strncmp(cp + 4, "DSN", 3) == 0) {
		*dsn = get_smtp_response(cp);
		break;
	    }
	    cp = e + 1;
	} while (e);

	return ret;
}

void
smtp_quit (int sock, int rset) {
	char cmd[MAXLINE];

	switch (rset) {
	case 1:
	    strcpy(cmd, "RSET\r\n");
	    if (smtp_chat(sock, cmd) < 0) return;
	case 0:
	    strcpy(cmd, "QUIT\r\n");
	    smtp_chat(sock, cmd);
	}
}

int
mailer (char *domain, char *user, char *rcpt, char *mxhost, char *msg, int msg_size) {
	struct sockaddr_in address;
	struct hostent host_buf, *host = NULL;
	char cmd[MAXLINE];
	int ret, dsn;
	int sock = -1;
	int optval = 1;
	char buffer[8192];

#if defined __FreeBSD__
	host = gethostbyname_r(mxhost, &host_buf, buffer, sizeof(buffer), &ret);
	if (ret)
#else
	if ((gethostbyname_r(mxhost, &host_buf, buffer, sizeof(buffer),
	    &host, &ret)))
#endif /* __FreeBSD__ */
	    return -1;

	memset(&address, 0, sizeof(struct sockaddr));
	address.sin_addr.s_addr = *(uint32_t *)host->h_addr_list[0];
	address.sin_family = AF_INET;
	address.sin_port = htons(SMTP_PORT);

	sock = socket(PF_INET, SOCK_STREAM, 0);
	if (sock < 0)	{
	    syslog(LOG_ERR, "Unable to open socket: %s",
		strerror(errno));
	    exit(EX_IOERR);
	}
	setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE|SO_REUSEADDR,
	    &optval, sizeof(int));

	if (smtp_connect(sock, (struct sockaddr *)&address,
	    sizeof(struct sockaddr_in)) < 0) goto quit_fail;

	if ((ret = smtp_chat(sock, NULL)) < 0) goto quit_fail;
	if (!SMTP_CMD_OK(ret)) goto quit_temp;

	if ((ret = ehlo_chat(sock, domain, &dsn)) < 0) goto quit_fail;
	if (!SMTP_CMD_OK(ret)) {
	    sprintf(cmd, "HELO %s\r\n", domain);
	    if ((ret = smtp_chat(sock, cmd)) < 0) goto quit_fail;
	    if (SMTP_CMD_PERM(ret)) goto quit_perm;
	    if (!SMTP_CMD_OK(ret)) goto quit_temp;
	}

	sprintf(cmd, "MAIL From: <%s> SIZE=%d\r\n", user, msg_size);
	if ((ret = smtp_chat(sock, cmd)) < 0) goto quit_fail;
	if (SMTP_CMD_PERM(ret)) goto quit_perm;
	if (!SMTP_CMD_OK(ret)) goto quit_temp;

	if (SMTP_CMD_OK(dsn))
	     sprintf(cmd, "RCPT To: <%s> NOTIFY=NEVER\r\n", rcpt);
	else sprintf(cmd, "RCPT To: <%s>\r\n", rcpt);
	if ((ret = smtp_chat(sock, cmd)) < 0) goto quit_fail;
	if (SMTP_CMD_PERM(ret)) goto quit_perm;
	if (!SMTP_CMD_OK(ret)) goto quit_ok;	// 4XX OK

	if (msg) {
	    sprintf(cmd, "DATA\r\n");
	    if ((ret = smtp_chat(sock, cmd)) < 0) goto quit_fail;
	    if (SMTP_CMD_PERM(ret)) goto quit_perm;
	    if (!SMTP_DATA_OK(ret)) goto quit_temp;

	    if ((ret = smtp_chat(sock, msg)) < 0) goto quit_fail;
	    if (!SMTP_CMD_OK(ret)) goto quit_temp;

	    smtp_quit(sock, 0);
	} else

quit_ok:
	smtp_quit(sock, 1);
	close_socket(sock);
	return QUIT_OK;

quit_temp:
	smtp_quit(sock, 1);
	close_socket(sock);
	return QUIT_FAIL;

quit_fail:
	close_socket(sock);
	return QUIT_FAIL;

quit_perm:
	smtp_quit(sock, 1);
	close_socket(sock);
	return QUIT_PERM;
}

/* eof */
