/*
 * utils.c
 *
 * Description:  SMC Files utilities
 * Developed by: Alexander Djourik <sasha@iszf.irk.ru>
 *		 Pavel Zhilin <pzh@iszf.irk.ru>
 *
 * Copyright (c) 2003,2004,2005 Alexander Djourik. All rights reserved.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <syslog.h>
#include <pthread.h>
#include <sysexits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/file.h>
#include <limits.h>
#include <openssl/md5.h>
#include "smc-milter.h"
#include "utils.h"

#if !defined O_SYNC && defined O_FSYNC
	#define O_SYNC O_FSYNC
#endif

#ifndef MAXLINE
	#define MAXLINE	4096
#endif

#define HASH_SIZE 256

typedef struct _hash_leaf {
	char key[MD5_STRING_LENGTH+1];
	rec  value;
	struct _hash_leaf *next;
} hash_leaf;

typedef struct {
	hash_leaf *nodes[HASH_SIZE];
	int  count;
	pthread_mutex_t lock;
} db_rec;

static db_rec db_list[5];

void
md5sign (char *s1, char *s2, char *sign) {
	unsigned char md[MD5_DIGEST_LENGTH];
	MD5_CTX ctx;
	int i;

	MD5_Init(&ctx);
	if (s1) MD5_Update(&ctx, s1, strlen(s1));
	if (s2) MD5_Update(&ctx, s2, strlen(s2));
	MD5_Final(md, &ctx);

	for(i = 0; i < MD5_DIGEST_LENGTH; i++) {
	    sign[i*2] = HEX_DIGEST [*(md + i) >> 4];
	    sign[i*2+1] = HEX_DIGEST [*(md + i) % 16];
	}

	sign[MD5_STRING_LENGTH] = '\0';
}

int 
get_hash (char *key)
{
    	int hash = 0;

	if (key[0] >= 'A' && key[0] <= 'F')
	    hash += (key[0] - 'A') << 4;
	else
	    hash += (key[0] - '0') << 4;

	if (key[1] >= 'A' && key[1] <= 'F')
	    hash += key[1] - 'A';
	else
	    hash += key[1] - '0';

	return hash;
}

hash_leaf *
find_leaf (hash_leaf *leaf, char *key)
{
	for (; leaf; leaf = leaf->next) {
		if (memcmp (key, leaf->key, MD5_STRING_LENGTH) == 0)
			break;
	}
	return leaf;
}

hash_leaf *
delete_one_oldest(db_rec *dbr)
{
	hash_leaf *leaf;
	hash_leaf **node, **onode;
	int	  t;

	t = time(NULL) + 1;

	// find oldest record
	onode = dbr->nodes;
	for(node = dbr->nodes + HASH_SIZE - 1; node >= dbr->nodes; node--)
	{ 
	    if ((*node)->value.time1 < t) {
		t = (*node)->value.time1;
		onode = node;
	    }
	}

	// mark it as removed
	if ((leaf = *onode)) {
	    *onode = leaf->next;
	    dbr->count++;
	}
	return leaf;
}

int
insert_item(db_rec *dbr, char *key, rec *value)
{
	hash_leaf *leaf, *new_leaf;
	int hash;
	
	new_leaf = NULL;

	if (dbr->count <= 0)
	    new_leaf = delete_one_oldest(dbr); // reusing old leaf


	if (!new_leaf)
	    if (!(new_leaf = (hash_leaf *)malloc(sizeof (hash_leaf))))
		return -1;	

	memcpy (new_leaf->key, key, MD5_STRING_LENGTH+1);
	memcpy (&new_leaf->value, value, sizeof (*value));
	new_leaf->next = NULL;

	hash = get_hash(key);

	if (dbr->nodes[hash]) {
	    for (leaf = dbr->nodes[hash]; leaf->next; leaf = leaf->next);

	    // add new leaf
	    leaf->next = new_leaf; 
	} else dbr->nodes[hash] = new_leaf;

	dbr->count--;

	return 0;
}

rec *
find_item (db_rec *dbr, char *key)
{
	hash_leaf *leaf;
	int hash;
	
	hash = get_hash(key);
	if ((leaf = find_leaf (dbr->nodes[hash], key)))
		return &leaf->value;
	return NULL;
}

int
delete_item(db_rec *dbr, char *key)
{
	hash_leaf *leaf, *prev;
	int hash;
	
	hash = get_hash(key);
	for (leaf = dbr->nodes[hash], prev = NULL; leaf; leaf = leaf->next) {
	    if (memcmp (key, leaf->key, MD5_STRING_LENGTH) == 0) {
		if (prev)
			prev->next = leaf->next;
		else
			dbr->nodes[hash] = NULL;
		free (leaf);
		dbr->count++;
		return 0;
	    }
	    prev = leaf;
	}	
	return 1;
}

void
delete_all(db_rec *dbr)
{
	hash_leaf *leaf, *next_leaf, **node;
	
	for (node = dbr->nodes + HASH_SIZE - 1; node >= dbr->nodes; node--) {
	    for (leaf = *node, next_leaf = NULL; leaf;) {
		next_leaf = leaf->next;
		free (leaf);
		leaf = next_leaf;
	    }
	}
}

void
hashlist_purge(db_rec *dbr, int timeout)
{	
	hash_leaf *leaf, **node;
	time_t t;
	
	t = time(NULL) - timeout;

	for (node = dbr->nodes + HASH_SIZE - 1; node >= dbr->nodes; node--) {
	    if ((leaf = *node)) {
		if (0 < leaf->value.time1 && leaf->value.time1 < t) {
		    *node = leaf->next;
		    free (leaf);
		    dbr->count++;
		}
	    }
	}
}

void 
hashlist_print (db_rec *dbr)
{
	hash_leaf *leaf, **node;
	int leafs = 0, max_leafs = 0;
	
	for (node = dbr->nodes + HASH_SIZE - 1; node >= dbr->nodes; node--) {
		printf ("Node: %d %p\n\t", node - dbr->nodes, *node);
		for (leaf = *node, leafs = 0; leaf; leaf = leaf->next, leafs++)
			printf ("%s ", leaf->key);
		printf ("\n");
		if (leafs > max_leafs) max_leafs = leafs;
	}

	printf ("\nNodes %d, max leafs %d\n", HASH_SIZE, max_leafs);
}

void
init_db (int max_records)
{
    	int i;
	memset (db_list, 0, sizeof (db_list));
	
	for (i = 0; i < sizeof (db_list) / sizeof(db_rec); i++) {
	    pthread_mutex_init (&db_list[i].lock, NULL);
	    db_list[i].count = max_records;
	}
}

void
close_db () 
{
    	int i;
	for (i = 0; i < sizeof (db_list) / sizeof(db_rec); i++) {
	    pthread_mutex_destroy (&db_list[i].lock);
	    delete_all (&db_list[i]);
	}
}

int
add_record (int db_id, char *rkey, rec *rdata, int lifetime) 
{
	int ret = -1;

	if (pthread_mutex_lock(&db_list[db_id].lock))
	    return -1;

	hashlist_purge(db_list + db_id, lifetime);
	ret =  insert_item (db_list + db_id, rkey, rdata);

	if (pthread_mutex_unlock(&db_list[db_id].lock))
	    return -1;

	return ret;
}

int
del_record (int db_id, char *rkey) 
{
	int ret = -1;

	if (pthread_mutex_lock(&db_list[db_id].lock))
	    return -1;
	
	ret = delete_item (db_list + db_id, rkey);

	if (pthread_mutex_unlock(&db_list[db_id].lock))
	    return -1;

	return ret;
}

int
get_record (int db_id, char *rkey, rec *rdata) 
{
    rec *data; 
    int ret = 1;

    if (pthread_mutex_lock(&db_list[db_id].lock))
	return -1;
    
    if ((data = find_item (db_list + db_id, rkey))) {
	memcpy (rdata, data, sizeof (rec));
	ret = 0;
    }

    if (pthread_mutex_unlock(&db_list[db_id].lock))
	return -1;

    return ret;
}

/* eof */
