# $Id: colormanager.tcl,v 1.2 2001/08/19 12:25:42 issever Exp $

# UNDER CONSTRUCTION
#
# Colormanager
# ----------------------------------------------------------------------
# Keeps several colors in a widget and offers to configurate them. 
# Also remembers a list of commands to execute when a color is changed.
#
# ----------------------------------------------------------------------
#  AUTHOR: Selim Issever                    EMAIL: selim.issever@desy.de
#  Version: 1.0
#
# ----------------------------------------------------------------------
#            Copyright (c) 1999 by the author
# ======================================================================
# Permission to use, copy, modify, distribute and license this
# software and its documentation for any purpose, and without fee or
# written agreement with the author, is hereby granted, provided that
# the above copyright notice appears in all copies and that both the
# copyright notice and warranty disclaimer below appear in supporting
# documentation, and that the names of the author not be used in
# advertising or publicity pertaining to the software without
# specific, written prior permission.
# 
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
# INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, AND
# NON- INFRINGEMENT. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS,
# AND THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE
# MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
# CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
# OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
# ======================================================================

#
# Usual options.
#
itk::usual Colormanager {
    keep -background 
    keep -foreground
}

class iwidgets::Colormanager {
    inherit iwidgets::Dialogshell

    itk_option define -file                  file                  File        ""

    public {
	constructor {args} {} {
	    configure -title "Color Manager"

	    iwidgets::Dialogshell::add close -text "Close" -command [code $this _close]
	    iwidgets::Dialogshell::add reset -text "Reset" -command [code $this _reset]
	    iwidgets::Dialogshell::add save  -text "Save"  -command [code $this _save]
	    iwidgets::Dialogshell::add load  -text "Load"  -command [code $this _load]
	    iwidgets::Dialogshell::default close

	    set _colorwids [list]
	    set _dialogWId [childsite].c1
	    eval itk_initialize $args
	}
	method save {}
	method load {}
	method add {aLabel aColorId aDefColor {aColumn 0}} {
	    itk_component add color$aColorId {
		iwidgets::colorex [childsite].color$aColorId $aLabel \
		    -command [code $this adjustColor $aColorId]
	    } { keep -background ; keep -foreground}
	    grid $itk_component(color$aColorId) -column [_getColumn $aColumn] -row [_getRow $aColumn]
	    $itk_component(color$aColorId) configure -color $aDefColor
	    lappend _colorwids $itk_component(color$aColorId)
	    set _colorConfigArray($aColorId) $aDefColor
	    set _commandListArray($aColorId) [list]
	    eval iwidgets::Labeledwidget::alignlabels $_colorwids
	}
	method addCommand {aColorId aCommand} {
	    lappend _commandListArray($aColorId)  $aCommand
	}
	method color {aColorId} {
	    return [$itk_component(color$aColorId) cget -color]
	}
	method adjustColor {aColorId} {
	    set color [$itk_component(color$aColorId) cget -color]
	    set selcolor [tk_chooseColor \
			      -parent [winfo toplevel $itk_interior] \
			      -initialcolor $color \
			      -title "Choose Color" 
			 ]
	    if {[string compare "" $selcolor] != 0} {
		$itk_component(color$aColorId) configure -color $selcolor
		foreach com $_commandListArray($aColorId) {
		    regsub -all {%c} $com $selcolor helper
		    eval $helper
		}
	    }
	}
    }
    private {
	method  _getColorId {aWId} {
	    regexp -- {^.*\.color(.*)$} $aWId dummy retval
	    return $retval
	}
	method  _getColor {aWId} {
	    return [$aWId cget -color]
	}
	variable _commandListArray
	variable _colorConfigArray
	variable _dialogWId
	variable _colorwids
	variable _cols
	method   _getColumn {aColNum} { 
	    return [expr $aColNum*2]
	}
	method   _getRow {aColNum} { 
	    if {[info exists _cols($aColNum)]} {
		incr _cols($aColNum)
	    } else {
		itk_component add sep$aColNum {
		    iwidgets::line [childsite].sep$aColNum -xmargin 5
		} { 
		    keep -background
		}
		set cc  [expr 2*$aColNum - 1]
		if {$cc>0} {
		    grid $itk_component(sep$aColNum) -column $cc -row 0
		}
		set  _cols($aColNum) 0
	    }
	    return  $_cols($aColNum)
	}
	method   _close {}
	method   _reset {}
	method   _save {}
	method   _load {}

    }
}

# ------------------------------------------------------------------
#                            COLOREX
#
# Internally needed by Colormanager
# ------------------------------------------------------------------
itk::usual Colorex {
    keep -background -cursor -foreground
}
class iwidgets::Colorex {
    inherit iwidgets::Labeledwidget

    itk_option define -buttonsize buttonSize Size  30
    itk_option define -color      color      Color black
    public {
	constructor {aLabel args} {}
    }

}

#
# Provide a lowercased access method for the Colorex class.
# 
proc ::iwidgets::colorex {pathName args} {
    uplevel ::iwidgets::Colorex $pathName $args
}
#
# Provide a lowercased access method for the Colorex class.
# 
proc ::iwidgets::colormanager {pathName args} {
    uplevel ::iwidgets::Colormanager $pathName $args
}

#
# Use option database to override default resources of base classes.
#
option add *Colorex.labelPos w widgetDefault

# ------------------------------------------------------------------
#                        CONSTRUCTOR
# ------------------------------------------------------------------
body iwidgets::Colorex::constructor {aLabel args} {
    configure -labeltext    $aLabel
    configure -labeljustify right

    itk_component add f {
	frame [childsite].bf -width 30 -height 30 
    } {
	keep -background
    }
    pack $itk_component(f) -side right -fill none -expand 0 
    pack propagate $itk_component(f) 0
    itk_component add b { button $itk_component(f).bb } { 
	keep   -command
	ignore -background 	
    }
    pack $itk_component(b) -side right -fill both -expand 1
    eval itk_initialize $args
}

# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# OPTION: -buttonsize
#
# size of the button
# ------------------------------------------------------------------
configbody iwidgets::Colorex::buttonsize {
    $itk_component(f) configure -width  $itk_option(-buttonsize)
    $itk_component(f) configure -height $itk_option(-buttonsize)
}
configbody iwidgets::Colorex::color {
    $itk_component(b) configure -bg  $itk_option(-color)
}


# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# OPTION: -file
#
# Specifies the resourcefile
# ------------------------------------------------------------------
configbody iwidgets::Colormanager::file {
    if {[string compare "" $itk_option(-file)]} {
	show save
	show load
    } else {
	hide save
	hide load
    }
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _close
#
# Assosiated with the close button.
# ------------------------------------------------------------------
body iwidgets::Colormanager::_close {} {
    deactivate
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _reset
#
# Assosiated with the reset button.
# ------------------------------------------------------------------
body iwidgets::Colormanager::_reset {} {
    foreach c $_colorwids {
	set cid [_getColorId $c]
	$itk_component(color$cid) configure -color $_colorConfigArray($cid)
	foreach com $_commandListArray($cid) {
	    regsub -all {%c} $com $_colorConfigArray($cid) helper
	    eval $helper
	}
    }
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _save
#
# Assosiated with the save button. Will ask for confirmation
# ------------------------------------------------------------------
body iwidgets::Colormanager::_save {} {
    iwidgets::messagedialog $_dialogWId \
	-title "Confirm" \
	-text "Really save?" \
	-bitmap questhead \
	-modality application \
	-master $itk_interior \
	-foreground $itk_option(-foreground)
    $_dialogWId hide Help    
    if {[$_dialogWId activate]} {
	destroy $_dialogWId
	save
    } else {
	destroy $_dialogWId
    }
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _load
#
# Assosiated with the load button. Will ask for confirmation
# ------------------------------------------------------------------
body iwidgets::Colormanager::_load {} {
    iwidgets::messagedialog $_dialogWId \
	-title "Confirm" \
	-text "Really load?" \
	-bitmap questhead \
	-modality application \
	-master $itk_interior
    $_dialogWId hide Help    
    if {[$_dialogWId activate]} {
	destroy $_dialogWId
	load
    } else {
	destroy $_dialogWId
    }
}
# ------------------------------------------------------------------
# PUBLIC METHOD: save
#
# Saves all colors in the file $itk_option(-file)
# ------------------------------------------------------------------
body iwidgets::Colormanager::save {} {
    if {![string compare "" $itk_option(-file)]} {
	error "Please provide a filename for Colormanager.\nOption -file is an empty string."
    }
    set fid [open $itk_option(-file) w]
    foreach c $_colorwids {
	set cid [_getColorId $c]
	set col [_getColor   $c]
	puts $fid [list $cid $col]
    }
    close $fid
}

# ------------------------------------------------------------------
# PUBLIC METHOD: load
#
# Loads color configurations from the file $itk_option(-file)
# ------------------------------------------------------------------
body iwidgets::Colormanager::load {} {
    if {![string compare "" $itk_option(-file)]} {
	error "Please provide a filename for Colormanager.\nOption -file is an empty string."
    }
    set fid [open $itk_option(-file) r]
    while {![eof $fid]} {
	gets $fid line
	if {[llength $line]!=2} { continue }
	set cid [lindex $line 0]
	set col [lindex $line 1]
	$itk_component(color$cid) configure -color $col
	foreach com $_commandListArray($cid) {
	    regsub -all {%c} $com $col helper
	    eval $helper
	}
    }
    close $fid
}



# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: colormanager.tcl,v $
# Revision 1.2  2001/08/19 12:25:42  issever
# Added the cvs keywords Id at start of the file
# and Log at the end of the file
#
#
