# $Id: findreplacedialog.tcl,v 1.2 2001/08/19 12:25:42 issever Exp $

#
# Findreplacedialog
# ----------------------------------------------------------------------
#
# I've just extended the functionality for find-replace.
# Most of the work have been done by Mark L. Ulferts. The creation
# of such a widgets is based uppon his ideas.
# 
# (I found extending his code easier than implementing the extension 
# via inheritance etc.)
#
# 1) If Replace button is pressed, all found tagged textpieces are 
#    replaced by whatever is in itk_component(replace) and another
#    find is called.
#
# 2) There is an additional option -replace. On default its off.
#    If its on the additional widgets for replacement appear.
#
# 3) Copyright and all that stays as it was before
#
# Author: Selim Issever 
# Email:  selim.issever@desy.de
# Date:   14. Nov. 1999
# ----------------------------------------------------------------------
#
# Finddialog
# ----------------------------------------------------------------------
# This class implements a dialog for searching text.  It prompts the
# user for a search string and the method of searching which includes
# case sensitive, regular expressions, backwards, and all.
#
# ----------------------------------------------------------------------
#  AUTHOR: Mark L. Ulferts              EMAIL: mulferts@austin.dsccc.com
#
#  @(#) RCS: $Id: findreplacedialog.tcl,v 1.2 2001/08/19 12:25:42 issever Exp $
# ----------------------------------------------------------------------
#            Copyright (c) 1996 DSC Technologies Corporation
# ======================================================================
# Permission to use, copy, modify, distribute and license this software 
# and its documentation for any purpose, and without fee or written 
# agreement with DSC, is hereby granted, provided that the above copyright 
# notice appears in all copies and that both the copyright notice and 
# warranty disclaimer below appear in supporting documentation, and that 
# the names of DSC Technologies Corporation or DSC Communications 
# Corporation not be used in advertising or publicity pertaining to the 
# software without specific, written prior permission.
# 
# DSC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING 
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, AND NON-
# INFRINGEMENT. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, AND THE
# AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE MAINTENANCE, 
# SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS. IN NO EVENT SHALL 
# DSC BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR 
# ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, 
# WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION,
# ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
# SOFTWARE.
# ======================================================================

#
# Usual options.
#
itk::usual Findreplacedialog {
    keep -background -cursor -foreground -selectcolor
}

# ------------------------------------------------------------------
#                          IPRFINDDIALOG
# ------------------------------------------------------------------
class ::iwidgets::Findreplacedialog {
    inherit iwidgets::Dialogshell

    constructor {args} {}

    itk_option define -selectcolor selectColor Background {} 
    itk_option define -clearcommand clearCommand Command {}
    itk_option define -matchcommand matchCommand Command {}
    itk_option define -patternbackground patternBackground Background \#707070
    itk_option define -patternforeground patternForeground Foreground White
    itk_option define -searchbackground searchBackground Background \#c4c4c4
    itk_option define -searchforeground searchForeground Foreground Black
    itk_option define -textwidget textWidget TextWidget {}

    itk_option define -replace replace Replace {} false


    public {
	method clear {}
	method find {}
	method replace {}
    }

    protected {
	method _get {setting}
	method _textExists {}
	method _foundIndices {}
	method _replaceOff {}

	common _optionValues       ;# Current settings of check buttons.
	common _searchPoint        ;# Starting location for searches
	common _matchLen           ;# Matching pattern string length
	common _foundIndicesList   ;# Indices list of found text
    }
}

#
# Provide a lowercased access method for the ::finddialog class.
# 
proc ::iwidgets::findreplacedialog {pathName args} {
    uplevel ::iwidgets::Findreplacedialog $pathName $args
}

#
# Use option database to override default resources of base classes.
#
option add *Findreplacedialog.title "Find" widgetDefault

# ------------------------------------------------------------------
#                            CONSTRUCTOR
# ------------------------------------------------------------------
body ::iwidgets::Findreplacedialog::constructor {args} {
    #
    # Add the find pattern entryfield.
    #
    itk_component add pattern {
	iwidgets::Entryfield $itk_interior.pattern -labeltext "Find:"
    } 
    bind [$itk_component(pattern) component entry] \
	<Return> "[code $this invoke]; break"

    #
    # Add the replace pattern entryfield.
    #
    itk_component add replace {
	iwidgets::Entryfield $itk_interior.replace -labeltext "Replace with:"
    } 
    
    #
    # Add the find all checkbutton.
    #
    itk_component add all {
	checkbutton $itk_interior.all \
	    -variable [scope _optionValues($this-all)] \
	    -text "All"
    }

    #
    # Add the case consideration checkbutton.
    #
    itk_component add case {
	checkbutton $itk_interior.case \
	    -variable [scope _optionValues($this-case)] \
	    -text "Consider Case"
    }

    #
    # Add the regular expression checkbutton.
    #
    itk_component add regexp {
	checkbutton $itk_interior.regexp \
	    -variable [scope _optionValues($this-regexp)] \
	    -text "Use Regular Expression"
    }

    #
    # Add the find backwards checkbutton.
    #
    itk_component add backwards {
	checkbutton $itk_interior.backwards \
	    -variable [scope _optionValues($this-backwards)] \
	    -text "Find Backwards"
    }

    #
    # Add the find, clear, close and replace buttons, making find be the default.
    #
    add Find    -text Find    -command [code $this find]
    add Clear   -text Clear   -command [code $this clear]
    add Close   -text Close   -command [code $this deactivate 0]
    add Replace -text Replace -command [code $this replace]

    default Find

    #
    # Use the grid to layout the components.
    # Row 1 is allocated for the replace with entry
    #
    grid $itk_component(pattern) -row 0 -column 0 \
	-padx 10 -pady 10 -columnspan 4 -sticky ew

    grid $itk_component(all)       -row 2 -column 0
    grid $itk_component(case)      -row 2 -column 1
    grid $itk_component(regexp)    -row 2 -column 2
    grid $itk_component(backwards) -row 2 -column 3

    grid columnconfigure $itk_interior 0 -weight 1
    grid columnconfigure $itk_interior 1 -weight 1
    grid columnconfigure $itk_interior 2 -weight 1
    grid columnconfigure $itk_interior 3 -weight 1

    #
    # Initialize all the configuration options.
    #
    eval itk_initialize $args
}

# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# OPTION: -clearcommand
#
# Specifies a command to be invoked following a clear operation. 
# The command is meant to be a means of notification that the
# clear has taken place and allow other actions to take place such
# as disabling a find again menu.
# ------------------------------------------------------------------
configbody iwidgets::Findreplacedialog::clearcommand {}

# ------------------------------------------------------------------
# OPTION: -matchcommand
#
# Specifies a command to be invoked following a find operation. 
# The command is called with a match point as an argument.  Should
# a match not be found the match point is {}.
# ------------------------------------------------------------------
configbody iwidgets::Findreplacedialog::matchcommand {}

# ------------------------------------------------------------------
# OPTION: -patternbackground
#
# Specifies the background color of the text matching the search
# pattern.  It may have any of the forms accepted by Tk_GetColor.
# ------------------------------------------------------------------
configbody iwidgets::Findreplacedialog::patternbackground {}

# ------------------------------------------------------------------
# OPTION: -patternforeground
#
# Specifies the foreground color of the pattern matching a search
# operation.  It may have any of the forms accepted by Tk_GetColor.
# ------------------------------------------------------------------
configbody iwidgets::Findreplacedialog::patternforeground {}

# ------------------------------------------------------------------
# OPTION: -searchforeground
#
# Specifies the foreground color of the line containing the matching
# pattern from a search operation.  It may have any of the forms 
# accepted by Tk_GetColor.
# ------------------------------------------------------------------
configbody iwidgets::Findreplacedialog::searchforeground {}

# ------------------------------------------------------------------
# OPTION: -searchbackground
#
# Specifies the background color of the line containing the matching
# pattern from a search operation.  It may have any of the forms 
# accepted by Tk_GetColor.
# ------------------------------------------------------------------
configbody iwidgets::Findreplacedialog::searchbackground {}

# ------------------------------------------------------------------
# OPTION: -textwidget
#
# Specifies the scrolledtext or text widget to be searched.
# ------------------------------------------------------------------
configbody iwidgets::Findreplacedialog::textwidget {
    if {$itk_option(-textwidget) != {}} {
	set _searchPoint($itk_option(-textwidget)) 1.0
    }
}

# ------------------------------------------------------------------
# OPTION: -replace
#
# Specifies if replace opetion is available. It may have any form.
# The option doesnt care about the case. It is considered as on for:
# on, 1, true, yes. Otherwise it is turned off.
# ------------------------------------------------------------------
configbody iwidgets::Findreplacedialog::replace {
    if {[_replaceOff]} {
	grid forget $itk_component(replace)
	hide Replace
	iwidgets::Labeledwidget::alignlabels $itk_component(pattern)
    } else {
	grid $itk_component(replace) -row 1 -column 0 \
	    -padx 10 -pady 10 -columnspan 4 -sticky ew
	show Replace
	iwidgets::Labeledwidget::alignlabels $itk_component(pattern) \
	    $itk_component(replace) 
    }
}

# ------------------------------------------------------------------
#                            METHODS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# PUBLIC METHOD: clear 
#
# Clear the pattern entryfield and the indicators.  
# Dont forget to reset the _foundIndicesList.
# ------------------------------------------------------------------
body ::iwidgets::Findreplacedialog::clear {} {
    $itk_component(pattern) clear
    $itk_component(replace) clear
    
    if {[_textExists]} {
	set _searchPoint($itk_option(-textwidget)) 1.0

	$itk_option(-textwidget) tag remove search-line 1.0 end
	$itk_option(-textwidget) tag remove search-pattern 1.0 end

	set _foundIndicesList($itk_option(-textwidget)) [list]
    }

    if {$itk_option(-clearcommand) != {}} {
	$itk_option(-clearcommand)
    }
}


# ------------------------------------------------------------------
# PROTECTED METHOD: _replaceOff
#
# Returns 1, if no replace widgets should be displayed. Otherwise 0.
# ------------------------------------------------------------------
body ::iwidgets::Findreplacedialog::_replaceOff {} {
    if {![info exists itk_option(-replace)]} {
	return 1
    }
    switch -exact -- [string tolower [string trim $itk_option(-replace)]] {
	on - true - 1 - yes {
	    return 0
	}
	default {
	    return 1
	}
    }

}

# ------------------------------------------------------------------
# PROTECTED METHOD: _foundIndices
#
# Returns a list of index pairs, which define all ranges of found 
# text.
# ------------------------------------------------------------------
body ::iwidgets::Findreplacedialog::_foundIndices {} {    
    if {! [_textExists]} {
	return [list]
    }
    return $_foundIndicesList($itk_option(-textwidget))
}
# ------------------------------------------------------------------
# PUBLIC METHOD: replace
#
# Replaces all found text with the content of the replace entry
# ------------------------------------------------------------------
body ::iwidgets::Findreplacedialog::replace {} {
    if {! [_textExists]} {
	return
    }

    foreach indexpair [_foundIndices] {
	set i1 [lindex $indexpair 0]
	set i2 [lindex $indexpair 1]
	set what [$itk_option(-textwidget) get $i1 $i2]
	
	$itk_option(-textwidget) delete $i1 $i2
	$itk_option(-textwidget) insert $i1 [_get replace]
    }

    find
}
# ------------------------------------------------------------------
# PUBLIC METHOD: find
#
# Search for a specific text string in the text widget given by
# the -textwidget option.  Should this option not be set to an
# existing widget, then a quick exit is made. 
# ------------------------------------------------------------------
body ::iwidgets::Findreplacedialog::find {} {
    if {! [_textExists]} {
	return
    }

    #
    # Clear any existing indicators in the text widget.
    #
    $itk_option(-textwidget) tag remove search-line 1.0 end
    $itk_option(-textwidget) tag remove search-pattern 1.0 end

    #
    # also clear the foundIndicesList
    #
    set _foundIndicesList($itk_option(-textwidget)) [list]
    
    #
    # Make sure the search pattern isn't just blank.  If so, skip this.
    #
    set pattern [_get pattern]

    if {[string trim $pattern] == ""} {
	return
    }

    #
    # After clearing out any old highlight indicators from a previous
    # search, we'll be building our search command piece-meal based on 
    # the current settings of the checkbuttons in the find dialog.  The
    # first we'll add is a variable to catch the count of the length
    # of the string matching the pattern.
    #
    set precmd "$itk_option(-textwidget) search \
	    -count [list [scope _matchLen($this)]]"

    if {! [_get case]} {
	append precmd " -nocase"
    }

    if {[_get regexp]} {
	append precmd " -regexp"
    } else {
	append precmd " -exact"
    }

    #
    # If we are going to find all matches, then the start point for
    # the search will be the beginning of the text; otherwise, we'll
    # use the last known starting point +/- a character depending on
    # the direction.
    #
    if {[_get all]} {
	set _searchPoint($itk_option(-textwidget)) 1.0
    } else {
	if {[_get backwards]} {
	    append precmd " -backwards"
	} else {
	    append precmd " -forwards"
	}
    }

    #
    # Get the pattern to be matched and add it to the search command.
    # Since it may contain embedded spaces, we'll wrap it in a list.
    #
    append precmd " [list $pattern]"    

    #
    # If the search is for all matches, then we'll be performing the 
    # search until no more matches are found; otherwise, we'll break
    # out of the loop after one search.
    #
    while {1} {
	if {[_get all]} {
	    set postcmd " $_searchPoint($itk_option(-textwidget)) end"

	} else {
	    set postcmd " $_searchPoint($itk_option(-textwidget))"
	}

	#
	# Create the final search command out of the pre and post parts
	# and evaluate it which returns the location of the matching string.
	#
	set cmd {}
	append cmd $precmd $postcmd

	if {[catch {eval $cmd} matchPoint] != 0} {
	    set _searchPoint($itk_option(-textwidget)) 1.0
	    return {}
	}

	#
	# If a match exists, then we'll make this spot be the new starting
	# position.  Then we'll tag the line and the pattern in the line.
	# The foreground and background settings will lite these positions
	# in the text widget up.
	#
	if {$matchPoint != {}} {
	    set _searchPoint($itk_option(-textwidget)) $matchPoint 
	    
	    $itk_option(-textwidget) tag add search-line \
		"$_searchPoint($itk_option(-textwidget)) linestart" \
		"$_searchPoint($itk_option(-textwidget))" 
	    $itk_option(-textwidget) tag add search-line \
		"$_searchPoint($itk_option(-textwidget)) + \
               $_matchLen($this) chars" \
		"$_searchPoint($itk_option(-textwidget)) lineend"
	    $itk_option(-textwidget) tag add search-pattern \
		$_searchPoint($itk_option(-textwidget)) \
		"$_searchPoint($itk_option(-textwidget)) + \
                 $_matchLen($this) chars"

	    #
	    # build the _foundIndicesList
	    #
	    set i1 [$itk_option(-textwidget) index \
			$_searchPoint($itk_option(-textwidget))]
	    set i2 [$itk_option(-textwidget) index \
			"$_searchPoint($itk_option(-textwidget)) + $_matchLen($this) chars"]
	    lappend _foundIndicesList($itk_option(-textwidget)) [list $i1 $i2]
	}

	#
	# Set the search point for the next time through to be one
	# character more or less from the current search point based
	# on the direction.
	#
	if {[_get all] || ! [_get backwards]} {
	    set _searchPoint($itk_option(-textwidget)) \
		[$itk_option(-textwidget) index \
		     "$_searchPoint($itk_option(-textwidget)) + 1c"]
	} else {
	    set _searchPoint($itk_option(-textwidget)) \
		[$itk_option(-textwidget) index \
		     "$_searchPoint($itk_option(-textwidget)) - 1c"]
	}

	#
	# If this isn't a find all operation or we didn't get a match, exit.
	#
	if {(! [_get all]) || ($matchPoint == {})} {
	    break
	}
    }

    #
    # Configure the colors for the search-line and search-pattern.
    #
    $itk_option(-textwidget) tag configure search-line \
	-foreground $itk_option(-searchforeground)
    $itk_option(-textwidget) tag configure search-line \
	-background $itk_option(-searchbackground)
    $itk_option(-textwidget) tag configure search-pattern \
	-background $itk_option(-patternbackground)
    $itk_option(-textwidget) tag configure search-pattern \
	-foreground $itk_option(-patternforeground)

    #
    # Adjust the view to be the last matched position.
    #
    if {$matchPoint != {}} {
	$itk_option(-textwidget) see $matchPoint
    }

    #
    # There may be multiple matches of the pattern on a single line,
    # so we'll set the tag priorities such that the pattern tag is higher.
    #
    $itk_option(-textwidget) tag raise search-pattern search-line

    #
    # If a match command is defined, then call it with the match point.
    #
    if {$itk_option(-matchcommand) != {}} {
	$itk_option(-matchcommand) $matchPoint
    }

    #
    # Return the match point to the caller so they know if we found 
    # anything and if so where
    #
    return $matchPoint
}

# ------------------------------------------------------------------
# PROTECTED METHOD: _get setting
#
# Get the current value for the pattern, replace, case, regexp, 
# or backwards.
# ------------------------------------------------------------------
body ::iwidgets::Findreplacedialog::_get {setting} {
    switch $setting {
	pattern {
	    return [$itk_component(pattern) get]
	}
	replace {
	    return [$itk_component(replace) get]
	}
	case {
	    return $_optionValues($this-case)
	}
	regexp {
	    return $_optionValues($this-regexp)
	}
	backwards {
	    return $_optionValues($this-backwards)
	}
	all {
	    return $_optionValues($this-all)
	}
	default {
	    error "bad get setting: \"$setting\", should be pattern,\
		    replace, case, regexp, backwards, or all"
	}
    }
}

# ------------------------------------------------------------------
# PROTECTED METHOD: _textExists
#
# Check the validity of the text widget option.  Does it exist and
# is it of the class Text or Scrolledtext.
# ------------------------------------------------------------------
body ::iwidgets::Findreplacedialog::_textExists {} {
    if {$itk_option(-textwidget) == {}} {
	return 0
    }

    if {! [winfo exists $itk_option(-textwidget)]} {
	error "bad findreplacedialog text widget value: \"$itk_option(-textwidget)\",\
               the widget doesn't exist"
    }

    if {([winfo class $itk_option(-textwidget)] != "Text") &&
	([itcl::find objects -isa iwidgets::Scrolledtext *::$itk_option(-textwidget)] == "")} {
	error "bad findreplacedialog text widget value: \"$itk_option(-textwidget)\",\
               must be of the class Text or based on Scrolledtext"
    }

    return 1
}



# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: findreplacedialog.tcl,v $
# Revision 1.2  2001/08/19 12:25:42  issever
# Added the cvs keywords Id at start of the file
# and Log at the end of the file
#
#
