# $Id: fontconfig.tcl,v 1.2 2001/08/19 12:25:42 issever Exp $

#
# Fontconfig
# ----------------------------------------------------------------------
# Takes the name of a Tk font and pops up a UIF, in which the font can
# be configured.
#
# ----------------------------------------------------------------------
#  AUTHOR: Selim Issever                    EMAIL: selim.issever@desy.de
#  Version: 1.0
#
# ----------------------------------------------------------------------
#            Copyright (c) 1999 by the author
# ======================================================================
# Permission to use, copy, modify, distribute and license this
# software and its documentation for any purpose, and without fee or
# written agreement with the author, is hereby granted, provided that
# the above copyright notice appears in all copies and that both the
# copyright notice and warranty disclaimer below appear in supporting
# documentation, and that the names of the author not be used in
# advertising or publicity pertaining to the software without
# specific, written prior permission.
# 
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
# INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, AND
# NON- INFRINGEMENT. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS,
# AND THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE
# MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
# CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
# OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
# ======================================================================

#
# Usual options.
#
itk::usual Fontconfig {
    keep -background 
    keep -foreground
}

# ------------------------------------------------------------------
#                            FONTCONFIG
# ------------------------------------------------------------------
class iwidgets::Fontconfig {
    inherit iwidgets::Dialogshell

    itk_option define -titlefont titleFont TitleFont {helvetica 14 bold}
    public {
	constructor {args} {}
	destructor {}
	method activate {aFont aLabel} ;# call this method to configure font aFont
    }
    private {
	variable _font       ;# this one is used and changed in the widget
	variable _actfont    ;# this is the font, that should be changed - will take _font's settings after ok
	method   _reset {}
	method   _accept {}
	method   _cancel {}
	method   _adjustfont {}
	method   _setuif {aFont} 
	method   _increm {} 
	method   _decrem {} 
    }
}

#
# Provide a lowercased access method for the Fontconfig class.
# 
proc ::iwidgets::fontconfig {pathName args} {
    uplevel ::iwidgets::Fontconfig $pathName $args
}

#
# Use option database to override default resources of base classes.
#
option add *Fontconfig.title "Font Configuration" widgetDefault

# ------------------------------------------------------------------
#                        CONSTRUCTOR
# ------------------------------------------------------------------
body iwidgets::Fontconfig::constructor {args} {
    global iwidgets::_fontconfig
    iwidgets::Dialogshell::add accept  -text "Accept" -command [code $this _accept]
    iwidgets::Dialogshell::add reset   -text "Reset"  -command [code $this _reset]
    iwidgets::Dialogshell::add cancel  -text "Cancel" -command [code $this _cancel]
    iwidgets::Dialogshell::default accept

    set _font    [font create]
    set _actfont ""

    set wid [childsite]

    itk_component   add label {
	label $wid.titlabel -anchor center
    } { keep -background ; keep -foreground}
    itk_component   add lsep {
	iwidgets::line $wid.lsep -ymargin 5 -thickness 3
    } { keep -background  }
    pack $itk_component(label) $itk_component(lsep) \
	-side top -fill both -expand false \
	-ipadx 0 -ipady 0 -padx 0 -pady 0

    # --- family
    itk_component   add family {
	iwidgets::combobox $wid.family \
	    -labeltext "Family: " -labelpos w -width 20 \
	    -popupcursor hand1 -listheight 200  \
	    -editable 0 -textvariable iwidgets::_fontconfig($this,family) \
	    -selectioncommand [code $this _adjustfont]
    } {keep -background ; keep -foreground}
    foreach ffff [lsort [font families]] {
	$itk_component(family) insert list end $ffff
    }
    # --- weight: normal/bold
    itk_component add weight {
	iwidgets::optionmenu $wid.weight \
	    -labeltext "Weight: " \
	    -command [code $this _adjustfont]
    } { keep -background ; keep -foreground }
    $itk_component(weight) insert end normal bold

    # --- slant: roman/italic
    itk_component add slant {
	iwidgets::optionmenu $wid.slant \
	    -labeltext "Slant: " \
	    -command [code $this _adjustfont]
    } { keep -background ; keep -foreground }
    $itk_component(slant) insert end roman italic

    # --- underline: 0/1
    # --- overstrike: 0/1
    itk_component add lines {
	iwidgets::optionmenu $wid.lines \
	    -labeltext "Lines: " \
	    -command [code $this _adjustfont]
    } { keep -background ; keep -foreground }
    $itk_component(lines) insert end none underline overstrike both

    # --- size
    itk_component   add size { 
	iwidgets::spinint $wid.size \
	    -labeltext "Size: " -range {0 256} -width 3 \
	    -arroworient horizontal -textvariable iwidgets::_fontconfig($this,size)\
	    -increment [code $this _increm] -decrement [code $this _decrem] \
	    -command [code $this _adjustfont]
    } {keep -background ; keep -foreground}

    pack $itk_component(family) $itk_component(weight) $itk_component(slant) \
	$itk_component(lines) $itk_component(size) \
	-fill x -expand 0 -ipadx 0 -ipady 0 -padx 0 -pady 0 -side top

    # ====
    iwidgets::Labeledwidget::alignlabels \
	$itk_component(family) $itk_component(weight) $itk_component(slant) \
	$itk_component(lines) $itk_component(size) 

    # ====
    itk_component add extext {
	iwidgets::scrolledtext $wid.extext -height 2i \
	    -textfont $_font
    } { keep -foreground }
    $itk_component(extext) insert end "!@\#$%^&*()_+\n"
    $itk_component(extext) insert end "1234567890-=\n"
    $itk_component(extext) insert end "qwertyuiop[]\\\n"
    $itk_component(extext) insert end "QWERTYUIOP{}|\n"
    $itk_component(extext) insert end "sdfghjkl;'\n"
    $itk_component(extext) insert end "ASDFGHJKL:\"\n"
    $itk_component(extext) insert end "zxcvbnm,./\n"
    $itk_component(extext) insert end "ZXCVBNM<>?"
    pack $itk_component(extext) \
	-fill both -expand 1 -ipadx 0 -ipady 0 -padx 0 -pady 0 -side top
    
    eval itk_initialize $args
}

# ------------------------------------------------------------------
#                        DESTRUCTOR
# ------------------------------------------------------------------
body iwidgets::Fontconfig::destructor {} {
    unset iwidgets::_fontconfig($this,family)
    unset iwidgets::_fontconfig($this,size)
}

# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# OPTION: -titlefont
#
# Specifies the titlefont
# ------------------------------------------------------------------
configbody iwidgets::Fontconfig::titlefont {
    $itk_component(label) configure -font $itk_option(-titlefont)
}

# ------------------------------------------------------------------
#                            METHODS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# PUBLIC METHOD: activate aFont
#
# Popup the fontconfigurer for adjusting font aFont
# ------------------------------------------------------------------
body iwidgets::Fontconfig::activate {aFont aLabel} { 
    $itk_component(label) configure -text $aLabel
    # create the font, if it doesnt exists already
    catch "font create $aFont"
    # remember name of the font to be configured
    set _actfont $aFont
    # set the uif
    _setuif $aFont
    # launch uif
    iwidgets::Shell::activate
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _reset
#
# Associated with the reset button. The UIF will be reset to display
# the original font when activate was called
# ------------------------------------------------------------------
body iwidgets::Fontconfig::_reset {}  { 
    _setuif $_actfont
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _accept
#
# Associated with the accept button. The configuration is accepted
# and the font activate was called with will be configued according
# whats displayed in the UIF
# ------------------------------------------------------------------
body iwidgets::Fontconfig::_accept {} { 
    global iwidgets::_fontconfig
    set fam $iwidgets::_fontconfig($this,family)
    set wei [$itk_component(weight) get]
    set sla [$itk_component(slant) get]
    set lin [$itk_component(lines) get]
    set siz $iwidgets::_fontconfig($this,size)
    switch -exact -- $lin {
	none        { set uli 0; set ovs 0 }
	overstrike  { set uli 0; set ovs 1 }
	underline   { set uli 1; set ovs 0 }
	both        { set uli 1; set ovs 1 }
    }	    
    font configure $_actfont -family     $fam
    font configure $_actfont -weight     $wei
    font configure $_actfont -slant      $sla
    font configure $_actfont -underline  $uli
    font configure $_actfont -overstrike $ovs
    font configure $_actfont -size       $siz
    deactivate 
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _setuif aFont
#
# Adjust the UIF so it displays the properties of font aFont.
# ------------------------------------------------------------------
body iwidgets::Fontconfig::_setuif {aFont} {
    global iwidgets::_fontconfig
    # get its properties	
    set fam [font configure $aFont -family]
    set wei [font configure $aFont -weight]
    set sla [font configure $aFont -slant]
    set uli [font configure $aFont -underline]
    set ovs [font configure $aFont -overstrike]
    set siz [font configure $aFont -size]
    # pass the properties to the font on which the user can work	
    font configure $_font -family     $fam
    font configure $_font -weight     $wei
    font configure $_font -slant      $sla
    font configure $_font -underline  $uli
    font configure $_font -overstrike $ovs
    font configure $_font -size       $siz
    # set the rest of the uif
    set iwidgets::_fontconfig($this,family) $fam
    set iwidgets::_fontconfig($this,size)   $siz
    $itk_component(weight) select   $wei
    $itk_component(slant)  select   $sla
    switch -exact -- "$uli$ovs" {
	"00" {$itk_component(lines) select none}
	"01" {$itk_component(lines) select overstrike}
	"10" {$itk_component(lines) select underline}
	"11" {$itk_component(lines) select both}
    }	

}

# ------------------------------------------------------------------
# PRIVATE METHOD: _cancel
#
# Associated with the cancel button. Close the UIF. Changes dont 
# take effect.
# ------------------------------------------------------------------
body iwidgets::Fontconfig::_cancel {} { deactivate }

# ------------------------------------------------------------------
# PRIVATE METHOD: _adjustfont
#
# Each time the user changes properties in the UIF this method will
# be called, so the whole UIF can be updated.
# ------------------------------------------------------------------
body iwidgets::Fontconfig::_adjustfont {} {
    global iwidgets::_fontconfig
    set fam $iwidgets::_fontconfig($this,family)
    set wei [$itk_component(weight) get]
    set sla [$itk_component(slant) get]
    set lin [$itk_component(lines) get]
    set siz $iwidgets::_fontconfig($this,size)
    switch -exact -- $lin {
	none        { set uli 0; set ovs 0 }
	overstrike  { set uli 0; set ovs 1 }
	underline   { set uli 1; set ovs 0 }
	both        { set uli 1; set ovs 1 }
    }
    font configure $_font -family     $fam
    font configure $_font -weight     $wei
    font configure $_font -slant      $sla
    font configure $_font -underline  $uli
    font configure $_font -overstrike $ovs
    font configure $_font -size       $siz
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _increm
#
# Increments the font size in the UIF.
# ------------------------------------------------------------------
body iwidgets::Fontconfig::_increm {} {
    global iwidgets::_fontconfig
    incr iwidgets::_fontconfig($this,size)
    _adjustfont
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _decrem
#
# Decrements the font size in the UIF.
# ------------------------------------------------------------------
body iwidgets::Fontconfig::_decrem {} {
    global iwidgets::_fontconfig
    incr iwidgets::_fontconfig($this,size) -1
    _adjustfont
}



# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: fontconfig.tcl,v $
# Revision 1.2  2001/08/19 12:25:42  issever
# Added the cvs keywords Id at start of the file
# and Log at the end of the file
#
#
