# $Id: fontmanager.tcl,v 1.2 2001/08/19 12:25:42 issever Exp $

#
# Fontmanager
# ----------------------------------------------------------------------
# Keeps several fonts in a widget and offers to configurate them.
#
# ----------------------------------------------------------------------
#  AUTHOR: Selim Issever                    EMAIL: selim.issever@desy.de
#  Version: 1.0
#
# ----------------------------------------------------------------------
#            Copyright (c) 1999 by the author
# ======================================================================
# Permission to use, copy, modify, distribute and license this
# software and its documentation for any purpose, and without fee or
# written agreement with the author, is hereby granted, provided that
# the above copyright notice appears in all copies and that both the
# copyright notice and warranty disclaimer below appear in supporting
# documentation, and that the names of the author not be used in
# advertising or publicity pertaining to the software without
# specific, written prior permission.
# 
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
# INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, AND
# NON- INFRINGEMENT. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS,
# AND THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE
# MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
# CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
# OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
# ======================================================================

#
# Usual options.
#
itk::usual Fontmanager {
    keep -background 
    keep -foreground
}

# ------------------------------------------------------------------
#                            FONTEX
#
# Internally needed by Fontmanager
# ------------------------------------------------------------------
itk::usual Fontex {
    keep -background -cursor -foreground
}
class iwidgets::Fontex {
    inherit iwidgets::Entryfield

    public {
	constructor {aLabel aFont aFontConfigurer args} {}
    }

    private {
	variable _font
	variable _label
	variable _fontconfigurer
	method   _configurefont {} { $_fontconfigurer activate $_font $_label}
    }
}

#
# Provide a lowercased access method for the Fontex class.
# 
proc ::iwidgets::fontex {pathName args} {
    uplevel ::iwidgets::Fontex $pathName $args
}

#
# Use option database to override default resources of base classes.
#
option add *Fontex.labelPos w widgetDefault

# ------------------------------------------------------------------
#                        CONSTRUCTOR
# ------------------------------------------------------------------
body iwidgets::Fontex::constructor {aLabel aFont aFontConfigurer args} {
    set _fontconfigurer $aFontConfigurer
    set _font           $aFont
    set _label          $aLabel
    catch "font create $aFont"
    configure -labeltext $aLabel
    configure -textfont  $aFont
    insert    end "ABCabc123!@\#" 
    itk_component add b {
	button [childsite].bb -bitmap questhead -command [code $this _configurefont]
    } {keep -background ; keep -foreground}
    pack $itk_component(b) -side right -fill both -expand 0    
    eval itk_initialize $args
}

# ------------------------------------------------------------------
#                            FONTMANAGER
# ------------------------------------------------------------------
class iwidgets::Fontmanager {
    inherit iwidgets::Dialogshell

    itk_option define -file                  file                  File        ""
    itk_option define -dialogseparatorrelief dialogseparatorRelief Relief      sunken
    itk_option define -dialogseparatorwidth  dialogseparatorWidth  BorderWidth 2

    public {
	constructor {args} {}
	method add {aFontName aDefConf aLabel}
	method load {}
	method save {}
	method deactivate {}
    }

    private {
	variable _fontConfigArray
	variable _fontWidgets
	variable _sepWidgets
	variable _dialogWId
	method   _adjust {} 
	method   _close {}
	method   _reset {}
	method   _save {}
	method   _load {}
	method   _getFontname {aWId}
    }
}

#
# Provide a lowercased access method for the Fontmanager class.
# 
proc ::iwidgets::fontmanager {pathName args} {
    uplevel ::iwidgets::Fontmanager $pathName $args
}

#
# Use option database to override default resources of base classes.
#
option add *Fontmanager.title              "Font Manager" widgetDefault
option add *Fontmanager.dialogBorderWidth  3              widgetDefault
option add *Fontmanager.dialogRelief       sunken         widgetDefault

# ------------------------------------------------------------------
#                        CONSTRUCTOR
#
# Fontex children all have path: [childsite].a<xyz>
# Font separators all have path: [childsite].s<xyz>
# ------------------------------------------------------------------
body iwidgets::Fontmanager::constructor {args} {
    iwidgets::Dialogshell::add close -text "Close" -command [code $this _close]
    iwidgets::Dialogshell::add reset -text "Reset" -command [code $this _reset]
    iwidgets::Dialogshell::add save  -text "Save"  -command [code $this _save]
    iwidgets::Dialogshell::add load  -text "Load"  -command [code $this _load]
    iwidgets::Dialogshell::default close

    set _dialogWId [childsite].c1
    itk_component add fontconfig {
	iwidgets::fontconfig [childsite].c2 \
	    -master $itk_interior \
	    -tops $itk_interior
    } {usual}

    set _fontWidgets [list]
    set _sepWidgets [list]
    eval itk_initialize $args
}

# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# OPTION: -file
#
# Specifies the resourcefile
# ------------------------------------------------------------------
configbody iwidgets::Fontmanager::file {
    if {[string compare "" $itk_option(-file)]} {
	show save
	show load
    } else {
	hide save
	hide load
    }
}

# ------------------------------------------------------------------
# OPTION: -dialogborderwidth
#
# Specifies the width of the dialogchildsite border.
# ------------------------------------------------------------------
configbody iwidgets::Fontmanager::dialogseparatorrelief {
    foreach wid $_sepWidgets {
	$wid configure -relief $itk_option(-dialogseparatorrelief)
    }
}

# ------------------------------------------------------------------
# OPTION: -dialogseparatorwidth
#
# Specifies the thickness of the font separators.  It sets the width
# and height of the separator to the thickness value and the
# borderwidth
# ------------------------------------------------------------------
configbody iwidgets::Fontmanager::dialogseparatorwidth {
    foreach wid $_sepWidgets {
	$wid config -thickness $itk_option(-dialogseparatorwidth)
    }
}

# ------------------------------------------------------------------
#                            METHODS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# PUBLIC METHOD: add aFontName aDefConf aLabel 
#
# Adds font with name aFontName to the manager (to be managed by it).
# aLabel will be displayed in the UIF to describe the font. 
# aDefConf is the default configuration of the font - will be used
# by a reset.
# ------------------------------------------------------------------
body iwidgets::Fontmanager::add {aFontName aDefConf aLabel} {
    catch {eval "font create $aFontName $aDefConf"}

    if {[llength $_fontWidgets]>0} {
	set uns s$aFontName
	itk_component add $uns {
	    iwidgets::line [childsite].$uns \
		-thickness $itk_option(-dialogseparatorwidth) \
		-relief $itk_option(-dialogseparatorrelief) 
	} { keep -background }
	pack $itk_component($uns) -side top -fill x -padx 0 -pady 0 -ipadx 0 -ipady 0
	lappend _sepWidgets $itk_component($uns)
    }

    set un  a$aFontName
    itk_component add $un {
	iwidgets::fontex [childsite].$un $aLabel $aFontName $itk_component(fontconfig) 
    } { usual }
    pack $itk_component($un) -side top -fill x

    lappend _fontWidgets $itk_component($un)
    set     _fontConfigArray($aFontName) $aDefConf

    _adjust
}

# ------------------------------------------------------------------
# PUBLIC METHOD: save
#
# Saves all fonts in the file $itk_option(-file)
# ------------------------------------------------------------------
body iwidgets::Fontmanager::save {} {
    if {![string compare "" $itk_option(-file)]} {
	error "Please provide a filename for Fontmanager.\nOption -file is an empty string."
    }
    set fid [open $itk_option(-file) w]
    foreach f $_fontWidgets {
	set fn [_getFontname $f]
	puts $fid "font configure $fn [font configure $fn]"
    }
    close $fid
}

# ------------------------------------------------------------------
# PUBLIC METHOD: load
#
# Loads font configurations from the file $itk_option(-file)
# ------------------------------------------------------------------
body iwidgets::Fontmanager::load {} {
    if {![string compare "" $itk_option(-file)]} {
	error "Please provide a filename for Fontmanager.\nOption -file is an empty string."
    }
    set fid [open $itk_option(-file) r]
    while {![eof $fid]} {
	eval [gets $fid]
    }
    close $fid
}

# ------------------------------------------------------------------
# PUBLIC METHOD: deactivate
#
# ------------------------------------------------------------------
body iwidgets::Fontmanager::deactivate {} {
    $itk_component(fontconfig) deactivate
    iwidgets::Shell::deactivate
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _adjust 
#
# Adjust the labels
# ------------------------------------------------------------------
body iwidgets::Fontmanager::_adjust {} {
    eval iwidgets::Labeledwidget::alignlabels $_fontWidgets
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _getFontname
#
# Returns the fontname by analysing the widget id.
# ------------------------------------------------------------------
body iwidgets::Fontmanager::_getFontname {aWId} {
    regexp -- {^.*\.a(.*)$} $aWId dummy retval
    return $retval
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _close
#
# Assosiated with the close button.
# ------------------------------------------------------------------
body iwidgets::Fontmanager::_close {} {
    deactivate
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _reset
#
# Assosiated with the reset button.
# ------------------------------------------------------------------
body iwidgets::Fontmanager::_reset {} {
    foreach fwid $_fontWidgets {
	set fn [_getFontname $fwid]
	font delete $fn
	eval font create $fn $_fontConfigArray($fn)
    }
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _save
#
# Assosiated with the save button. Will ask for confirmation
# ------------------------------------------------------------------
body iwidgets::Fontmanager::_save {} {
    iwidgets::messagedialog $_dialogWId \
	-title "Confirm" \
	-text "Really save?" \
	-bitmap questhead \
	-modality application \
	-master $itk_interior
    $_dialogWId hide Help    
    if {[$_dialogWId activate]} {
	destroy $_dialogWId
	save
    } else {
	destroy $_dialogWId
    }
}

# ------------------------------------------------------------------
# PRIVATE METHOD: _load
#
# Assosiated with the load button. Will ask for confirmation
# ------------------------------------------------------------------
body iwidgets::Fontmanager::_load {} {
    iwidgets::messagedialog $_dialogWId \
	-title "Confirm" \
	-text "Really load?" \
	-bitmap questhead \
	-modality application \
	-master $itk_interior
    $_dialogWId hide Help    
    if {[$_dialogWId activate]} {
	destroy $_dialogWId
	load
    } else {
	destroy $_dialogWId
    }
}



# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: fontmanager.tcl,v $
# Revision 1.2  2001/08/19 12:25:42  issever
# Added the cvs keywords Id at start of the file
# and Log at the end of the file
#
#
