# $Id: line.tcl,v 1.2 2001/08/19 12:25:42 issever Exp $

#
# Line
# ----------------------------------------------------------------------
# A line between two widgets (in princ. only a frame, which doesnt 
# expand).
# Geometry managers: 
#   - Works best with the pack manager.
#   - For the grid manager it searches for the column or grid to use 
#     exclusivly. 
#
# ----------------------------------------------------------------------
#  AUTHOR: Selim Issever                    EMAIL: selim.issever@desy.de
#  Version: 1.0
#
# ----------------------------------------------------------------------
#            Copyright (c) 1999 by the author
# ======================================================================
# Permission to use, copy, modify, distribute and license this
# software and its documentation for any purpose, and without fee or
# written agreement with the author, is hereby granted, provided that
# the above copyright notice appears in all copies and that both the
# copyright notice and warranty disclaimer below appear in supporting
# documentation, and that the names of the author not be used in
# advertising or publicity pertaining to the software without
# specific, written prior permission.
# 
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
# INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, AND
# NON- INFRINGEMENT. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS,
# AND THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE
# MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
# CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
# OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
# ======================================================================

#
# Usual options.
#
itk::usual Line {
    keep -background 
}

# ------------------------------------------------------------------
#                            LINE
# ------------------------------------------------------------------
class iwidgets::Line {
    inherit itk::Widget

    itk_option define  -thickness  thickness  Thickness  3
    itk_option define  -relief     relief     Relief     sunken
    itk_option define  -xmargin    xMargin    Margin     0
    itk_option define  -ymargin    yMargin    Margin     0
    itk_option define  -linecolor  lineColor  Color      ""
    itk_option define  -orient     orient     Orient     "auto" 
    itk_option define  -anchor     anchor     Anchor     "" 

    constructor {args} {}

    private {
	variable _orow
	variable _ocol
	method _repack {} { 
	    set gman [winfo manager $itk_interior] 
	    switch -exact -- $gman {
		place {
		}
		grid {
		    set i      [grid info $itk_interior] 
		    set parent [lindex $i 1]
		    if {![info exists _orow]} {
			set _ocol      [lindex $i 3]
			set _orow      [lindex $i 5]			
		    }
		    set col    1
		    set row    1
		    set slaves [grid slaves $parent]
		    foreach s $slaves {
			if {![string compare $itk_interior $s]} {
			    continue
			}
			set i     [grid info $s]
			if {$_ocol==[lindex $i 3]} { set col 0 }
			if {$_orow==[lindex $i 5]} { set row 0 }
		    }
		    switch -glob -- $itk_option(-orient) {
			"h*" {
			    if {$row} {
				grid configure $itk_interior -sticky we$itk_option(-anchor) \
				    -column 0 -columnspan 1000 -row $_orow -rowspan 1
			    } else {
				grid configure $itk_interior -sticky we$itk_option(-anchor) \
				    -column $_ocol -columnspan 1 -row $_orow -rowspan 1
			    }
			}
			"v*" {
			    if {$col} {
				grid configure $itk_interior -sticky ns$itk_option(-anchor) \
				    -row 0 -rowspan 1000 -column $_ocol -columnspan 1
			    } else {
				grid configure $itk_interior -sticky ns$itk_option(-anchor) \
				    -row $_orow -rowspan 1 -column $_ocol -columnspan 1
			    }
			}
			"a*" {
			    if {$col} {
				grid configure $itk_interior -sticky ns$itk_option(-anchor) \
				    -row 0 -rowspan 1000 -column $_ocol -columnspan 1
			    } elseif {$row} {
				grid configure $itk_interior -sticky we$itk_option(-anchor) \
				    -column 0 -columnspan 1000 -row $_orow -rowspan 1
			    } else {
				error "Line cant be placed in column/row: $_ocol/$_orow"
			    }
			}
			default {
			    error "Line option -orient must be: auto, horizontal or vertical"
			}
		    }
		}
		pack {
		    pack configure $itk_interior -expand 0 
		}
		default {
		    error "Lines can't be used with $gman geometry manager."
		}
	    }
	}
    }

}

#
# Provide a lowercased access method for the Colorex class.
# 
proc ::iwidgets::line {pathName args} {
    uplevel ::iwidgets::Line $pathName $args
}

#
# Use option database to override default resources of base classes.
#

# ------------------------------------------------------------------
#                        CONSTRUCTOR
# ------------------------------------------------------------------
body iwidgets::Line::constructor {args} {
    itk_component add line {
	frame $itk_interior.f  
    } { ignore -background }
    pack $itk_component(line) -fill both -expand true
    bind $itk_component(line) <Map> [code $this _repack]
    eval itk_initialize $args
}

# ------------------------------------------------------------------
#                             OPTIONS
# ------------------------------------------------------------------

# ------------------------------------------------------------------
# OPTION: -thickness
#
# Specifies the thickness of the line.  It sets the width and
# height of the line to the thickness value and the borderwidth
# to half the thickness.
# ------------------------------------------------------------------
configbody iwidgets::Line::thickness {
    $itk_component(line) config -height $itk_option(-thickness)
    $itk_component(line) config -width $itk_option(-thickness)
    $itk_component(line) config \
	    -borderwidth [expr $itk_option(-thickness) / 2]
}

# ------------------------------------------------------------------
# OPTION: -relief
#
# Specifies the relief of the line. 
# ------------------------------------------------------------------
configbody iwidgets::Line::relief {
    $itk_component(line) config -relief $itk_option(-relief)
}

# ------------------------------------------------------------------
# OPTION: -orient
#
# Specifies the orientation of the line. 
# ------------------------------------------------------------------
configbody iwidgets::Line::orient {
    if {[winfo ismapped  $itk_interior]} {
	_repack
    }
}

# ------------------------------------------------------------------
# OPTION: -anchor
#
# Specifies the orientation of the line. 
# ------------------------------------------------------------------
configbody iwidgets::Line::anchor {
    if {[winfo ismapped  $itk_interior]} {
	_repack
    }
}

# ------------------------------------------------------------------
# OPTION: -margin
#
# Specifies the margin of the line. 
# ------------------------------------------------------------------
configbody iwidgets::Line::xmargin {
    pack configure $itk_component(line) -padx $itk_option(-xmargin)
}
configbody iwidgets::Line::ymargin {
    pack configure $itk_component(line) -pady $itk_option(-ymargin)
}

# ------------------------------------------------------------------
# OPTION: -background
#
# Specifies the color of the line. 
# ------------------------------------------------------------------
configbody iwidgets::Line::linecolor {
    if {![string compare "" $itk_option(-linecolor)]} {
	set color [$itk_interior cget -background]
	$itk_component(line) config -background $color
    } else {
	$itk_component(line) config -background $itk_option(-linecolor)
    }
}





# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: line.tcl,v $
# Revision 1.2  2001/08/19 12:25:42  issever
# Added the cvs keywords Id at start of the file
# and Log at the end of the file
#
#
