# $Id: TSesManager.tcl,v 1.2 2001/12/08 18:53:35 issever Exp $

class TSesManager {
    private common _sesList ;# is an array of objects

    private variable _fileManager    ;# the file system manager
    private variable _mudManager
    private variable _actualSession

    # ------------------------------------------------------
    constructor {aFileSystemManager aMudManager args} {} {}
    # This is the constructor. It will initialize all
    # class relevant data and load all session.
    # ------------------------------------------------------
    method exists {aName} { return [info exists _sesList($aName)] } 
    # This method checks, if a session named 'aName'
    # exists. Returns 1, if so, 0 otherwise.
    # ------------------------------------------------------
    method add {aName aPlayer aPassword aMudName} 
    # This method creates a session with the given
    # information and adds it to the session manager. If a
    # session with the name 'aName' exists, it will be
    # replaced by the new session. All session relevant
    # data will be wiped out first and recreated
    # afterwards.
    # ------------------------------------------------------
    method rem {aName}
    # With this method a session is permanently removed
    # from the session manager, which means that it will
    # be removed from the smm system. All session relevant
    # data is deleted as well
    # ------------------------------------------------------
    method sessionStarted {} { if {[string compare "" $_actualSession]} { return 1 } else { return 0 } }
    # Returns 1, if a session is already loaded and 0 otherwise.
    # ------------------------------------------------------
    method getSesNames {} { return [lsort [array names _sesList]] }
    # Returns a list with the names of all sessions, which
    # are managed by the session manager
    # ------------------------------------------------------
    method getMudNames {} { return [$_mudManager getMudNames] }
    # This method returns a list of names of all muds,
    # which are known to the mud-manager, which this
    # session manager is connected to.  The list should
    # contain just the muds, that are used by session of
    # this session manager. Additionally the list should
    # be non-redundant. The current implementation ensures
    # that, but is not error prone, if the mud or session
    # manager is reimplemented. E.g. If you have more than
    # one session manager and each one does not know all
    # muds of the system.
    # ------------------------------------------------------
    method getSes {aName} { return $_sesList($aName) } 
    # This method returns the pointer to the actual
    # running session.
    # ------------------------------------------------------
    method getSesId {aName} { return [$_sesList($aName) id]} 
    # This method returns the numerical ID of the actual
    # running session.
    # ------------------------------------------------------
    method newActSes {aName} { set _actualSession [getSes $aName] }
    # This method will signal the session manager, that a
    # new actual session is going to be used. This method
    # does not load the session yet.
    # ------------------------------------------------------
    method connectActSes {aMainWin aCon} 
    # This is the method, which will load the actual
    # session and connect it to the mud, if this was
    # requested
    # ------------------------------------------------------
    method sesDisconnected {} 
    # Via this method the main uif signals the session
    # manager that the running session was successfully
    # unloaded
    # ------------------------------------------------------
    method stopSession {} 
    # This triggers to stop and unload the session. first
    # the session itself is told to stop. Then the session
    # will signal the main uif that the session is
    # stopped, which again will tell the session manager -
    # via the method sesDisconnected - that the stop was
    # successfully penetrated to all relevant systems. The
    # session manager now logs, that no session is active.
    # ------------------------------------------------------
    method getActSes {} { return $_actualSession }
    # Returns the pointer of the actual running Session
    # ------------------------------------------------------
    method getActMudName {} 
    # Returns the name of the actual running mud.
    # ------------------------------------------------------
    method getActMud {} 
    # Returns mud pointer of the actual running mud.
    # ------------------------------------------------------
    private method _getSesIds {} 
    # Returns the list of all numerical session-ID's.
    # ------------------------------------------------------
    private method _getNewId {} 
    # Returns a new unique numerical ID.
    # ------------------------------------------------------

}

# ==========================================================
# === class code
# ==========================================================

# ----------------------------------------------------------
# --- constructor
# ----------------------------------------------------------
body TSesManager::constructor {aFileSystemManager aMudManager args} {
    set _fileManager $aFileSystemManager
    set _mudManager $aMudManager
    set _actualSession ""

    # laod the session informations from files
    foreach sf [$_fileManager getAllSesDataFiles] {
	set ses [TSession ::\#auto $_fileManager $sf]
	set _sesList([$ses name]) $ses
    }
}

# ----------------------------------------------------------
# --- other methods
# ----------------------------------------------------------
body TSesManager::add {aName aPlayer aPassword aMudName} {
    if {[exists $aName]} {
	::delete object $_sesList($aName)
	unset _sesList($aName)	
    } 
    set id [_getNewId]
    set _sesList($aName) [TSession ::\#auto $_fileManager $aName $aPlayer $aPassword $aMudName $id]
}
body TSesManager::rem {aName} {
    if {[exists $aName]} {
	$_sesList($aName) wipeout
	::delete object $_sesList($aName)
	unset _sesList($aName)	
    }
}
body TSesManager::connectActSes {aMainWin aCon} { 
    $_actualSession connect $_mudManager $aMainWin $aCon
}
body TSesManager::sesDisconnected {} {
    set _actualSession ""
}
body TSesManager::stopSession {} {
    if {[sessionStarted]} {
	$_actualSession disconnect
    }
}
body TSesManager::_getSesIds {} { 
    set idlist [list]
    foreach name [getSesNames] {
	lappend idlist [[getSes $name] id]
    }
    return $idlist
}
body TSesManager::getActMudName {} { 
    if {[sessionStarted]} { 
	return [$_actualSession mudName]
    } else {
	return ""
    }
}
body TSesManager::_getNewId {} {
    set id 0
    set idlist [_getSesIds]
    while {[lsearch -exact $idlist $id]!= -1} {
	incr id
    }
    return $id
}
body TSesManager::getActMud {} {
    return [$_mudManager getMud [getActMudName]]
}


# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: TSesManager.tcl,v $
# Revision 1.2  2001/12/08 18:53:35  issever
# Added and edited files for the namecreator.
#
# Revision 1.1  2001/12/08 13:38:54  issever
# reorganized session manager code.
# It is now in a separate file and has more comments.
# The functionality hasnt been extended.
#
#
