# $Id: TSession.tcl,v 1.17 2002/05/02 19:47:53 issever Exp $

# ##########################################################
# ### TSession
# ##########################################################
#
# A Session is on the one hand side a storage for
# information like: the relevant mud, player name, passwd,
# etc..
#
# On the other hand side it is responsible for creating the
# connection and mentaining various other things, which are
# needed while mudding: parsing of mud text, parsing of user
# input, logging, etc,....
#
# Rem: One could think of seperating both tasks: session
# (data) + connection.  Just in case it ever gets
# important,..

class TSession {
    inherit \
	TESCParser \
	TSes::TLogging

    # constructor and such,..
    private {
	variable CFilM
	variable CName    
	variable CPlayer
	variable CPassword
	variable CMudName
	variable CId 
	# CName: must be unique
	# CId:   must be unique too: this is needed to request files 
	#         and directories for the mud data to be stored
	#        => may never change again! 
    }
    public {
	constructor {args} { TESCParser::constructor } {}

	# .. file system
	method createFileSystem {} 
	method readFile {aFile} 
	method removeFileSystem {} 
	method wipeout {} 

	# .. general accessors
	method show     {} { puts "$CId -- $CName || $CPlayer $CMudName" }
	method name     {} { return $CName }
	method player   {} { return $CPlayer }
	method password {} { return $CPassword }
	method mudName  {} { return $CMudName }
	method id       {} { return $CId }

	method isLoaded {} { return $_sesLoaded }
    }




    private {
	variable _sesLoaded
	variable CSocket

	variable CSocketCompressionAvailable  ;# compression available?
	variable CSocketCompressed            ;# compression enabled?
	variable CSocketRead                  ;# socket used in readMud
	variable CSocketPipeWrite             ;# the pipe handles the
	variable CSocketPipeRead              ;#  MCCP uncompression
	variable CSocketHistory               ;# socket data

	variable CMainWindow
	variable CMudder
	variable CMapManager

	variable _output
	variable CInput
	variable CTicker

	common CNL
	common CCR
	common CRS
	common CNNL
	
	variable CTabM  ;# user tablist manager
	variable _actP     ;# array index: pattern -> value: level
	variable _actList  ;# level ordered list of patterns
	
	variable CCurrentNotParsedLine    ;# -> need for telnet and esc sequences; only the not parsed/printed part
	variable CCurrentAsciiLine        ;# -> need for actions/highlights/etc
	variable CParseMode               ;# t: telnet, l: line, e: esc
	variable CTelnetProtocol        


	variable CButtonArea
    }


    private method getCFile  {}  { return [$CFilM getMainSesConfFile $CId] }

    method getTabs {} { 
	set retval [$CTabM  getTabs]
	foreach tab [$_output getTabs] {
	    lappend retval $tab
	}
	return $retval
    } 
    

    method readMudFromFile {aFileName}
    method uncompressMud {}              ;# uncompress thread
    method readMud {} 
    method evalSingleLine {aPart args} 
    method evalSingleLineMode_t {aPart} 
    method evalSingleLineMode_l {aPart} 
    method writeMudText {aPart} 
    method writeMessage {aMsg} { $_output fromSMM $aMsg }


    #
    # printing
    #
    method printf {aStr} { if {$_showoutput} {$_output fromSMM $aStr} }
    method perror {aStr} 

    #
    # user interp: smm commands
    #
    private method _smmActionReadjust  {}
    public {
	method smmTab        {subspec args}
	method smmTick       {subspec args}
	method smmLog        {subspec args}
	method smmUserPrint  {string args}
	method smmKeyCom     {sequence command}
	method smmBell       {subspec args}
	method smmFile       {subspec args}
	method smmKillAll    {}

	method smmActionGet        {alias}
	method smmActionSet        {alias level}
	method smmActionUnset      {alias}
	method smmActionExists     {alias}
	method smmActionList       {}
	method smmActionNames      {pattern}
	
	method smmMapZoom    {subspec args}
	method smmMapExists  {subspec args}
	method smmMapSet     {subspec args}
	method smmMapGet     {subspec args}
	method smmMapCreate  {mapname maplevel}
	method smmMapSwitch  {mapname maplevel}
	method smmMapMove    {subspec args}

	method smmVSet       {varname value}
	method smmVUnset     {varname}
	method smmVGet       {varname}
	method smmVList      {}
	method smmVArray     {option arrayname args}

	method smmGUIAddRow             {rowname}
	method smmGUIRemRow             {rowname}
	method smmGUIAddButton          {rowname buttonname text command}
	method smmGUIRemButton          {rowname buttonname}
	method smmGUIAddCheckbutton     {rowname buttonname text command variablename}
	method smmGUIRemCheckbutton     {rowname buttonname}
	method smmGUIAddLabel           {rowname labelname text width}
	method smmGUIRemLabel           {rowname labelname}
	method smmGUISetLabelText       {rowname labelname text}
	method smmGUIAddVarEntry        {rowname varentryname variablename width } 
	method smmGUIRemVarEntry        {rowname varentryname} 
	method smmGUIAddVarLabel        {rowname varlabelname variablename width } 
	method smmGUIRemVarLabel        {rowname varlabelname} 
	method smmGUIAddVarChooser      {rowname varchoosername variablename width } 
	method smmGUIRemVarChooser      {rowname varchoosername} 
	method smmGUIAddVarChooserItem  {rowname varchoosername item} 
	method smmGUIRemVarChooserItem  {rowname varchoosername index}
	
	method smmOption {opt act}

	method smmGetSelection {{source {}}}

	method smmScroll {elem act}
    }
    private {
	variable _parse          1
	variable _showoutput     1
	variable _showerror      1
    }

    #
    # user interp general stuff
    #
    public {
	method connect {aMudM aMainWin aCon} 
	method cutConnection {}
	method disconnect {} 
	method killall {} 

	method toMud {arg} 
	method toSocket {arg} 
	method toSMM {arg} 

    }
    private {
	variable _showInput
	method   _createInterp {} 
	method   _deleteInterp {} 
    }

}

##################################################################################

#
# create/delete user interp
#
body TSession::_createInterp {} {

    # --- create a namespace, which will hold the gui variables
    namespace eval ::smm::ses {}

    # --- create userinterp
    interp create -- smmuser

    # --- set general interp behaviour
    smmuser alias tomud  [code $this toMud]
    smmuser eval {
	proc unknown {args} { 
	    if {[llength $args]>1} {
		::tomud $args 
	    } else {
		::tomud [lindex $args 0]
	    }
	}
    }
    smmuser eval {
	namespace eval ::ses {
	    ::foreach com [::lsort [::info commands]] {
		::eval "::proc $com {args} { ::eval ::unknown $com \$args }"
	    }
	}
    }

    # --- smm commands
    smmuser alias ::smm::getselection [code $this smmGetSelection]

    smmuser alias ::smm::option       [code $this smmOption]
    smmuser alias ::smm::scroll       [code $this smmScroll]
    smmuser alias ::smm::tick         [code $this smmTick]
    smmuser alias ::smm::tab          [code $this smmTab]
    smmuser alias ::smm::log          [code $this smmLog]
    smmuser alias ::smm::showme       [code $this smmUserPrint]
    smmuser alias ::smm::keycom       [code $this smmKeyCom]
    smmuser alias ::smm::bell         [code $this smmBell]
    smmuser alias ::smm::file         [code $this smmFile]
    smmuser alias ::smm::killall      [code $this smmKillAll]

    smmuser alias ::smm::action::list       [code $this smmActionList]
    smmuser alias ::smm::action::get        [code $this smmActionGet]
    smmuser alias ::smm::action::set        [code $this smmActionSet]
    smmuser alias ::smm::action::unset      [code $this smmActionUnset]
    smmuser alias ::smm::action::exists     [code $this smmActionExists]
    smmuser alias ::smm::action::names      [code $this smmActionNames]

    smmuser alias ::smm::map::zoom    [code $this smmMapZoom]
    smmuser alias ::smm::map::exists  [code $this smmMapExists]
    smmuser alias ::smm::map::set     [code $this smmMapSet]
    smmuser alias ::smm::map::get     [code $this smmMapGet]
    smmuser alias ::smm::map::create  [code $this smmMapCreate]
    smmuser alias ::smm::map::switch  [code $this smmMapSwitch]
    smmuser alias ::smm::map::move    [code $this smmMapMove]

    smmuser alias ::smm::gui::set         [code $this smmVSet]
    smmuser alias ::smm::gui::unset       [code $this smmVUnset]
    smmuser alias ::smm::gui::get         [code $this smmVGet]
    smmuser alias ::smm::gui::list        [code $this smmVList]
    smmuser alias ::smm::gui::array       [code $this smmVArray]

    smmuser alias ::smm::gui::addrow        [code $this smmGUIAddRow]
    smmuser alias ::smm::gui::remrow        [code $this smmGUIRemRow]
    smmuser alias ::smm::gui::addbutton     [code $this smmGUIAddButton]
    smmuser alias ::smm::gui::rembutton     [code $this smmGUIRemButton]
    smmuser alias ::smm::gui::addcheckbutton     [code $this smmGUIAddCheckbutton]
    smmuser alias ::smm::gui::remcheckbutton     [code $this smmGUIRemCheckbutton]
    smmuser alias ::smm::gui::addlabel      [code $this smmGUIAddLabel]
    smmuser alias ::smm::gui::remlabel      [code $this smmGUIRemLabel]
    smmuser alias ::smm::gui::setlabeltext  [code $this smmGUISetLabelText]
    smmuser alias ::smm::gui::addvarentry   [code $this smmGUIAddVarEntry]
    smmuser alias ::smm::gui::remvarentry   [code $this smmGUIRemVarEntry]
    smmuser alias ::smm::gui::addvarlabel   [code $this smmGUIAddVarLabel]
    smmuser alias ::smm::gui::remvarlabel   [code $this smmGUIRemVarLabel]
    smmuser alias ::smm::gui::addvarchooser      [code $this smmGUIAddVarChooser]
    smmuser alias ::smm::gui::remvarchooser      [code $this smmGUIRemVarChooser]
    smmuser alias ::smm::gui::addvarchooseritem  [code $this smmGUIAddVarChooserItem]
    smmuser alias ::smm::gui::remvarchooseritem  [code $this smmGUIRemVarChooserItem]

    smmuser alias ::smm::text::delete        [code $_output deleteLastLine]
    smmuser alias ::smm::text::deleterange   [code $_output deleteRangeOnLastLine]
    smmuser alias ::smm::text::subst         [code $_output replaceLastLine]
    smmuser alias ::smm::text::substrange    [code $_output replaceOnLastLine]
    smmuser alias ::smm::text::tag           [code $_output tagLastLine]
    smmuser alias ::smm::text::tagrange      [code $_output tagOnLastLine]
    smmuser alias ::smm::text::substtag      [code $_output tagNreplaceLastLine]
    smmuser alias ::smm::text::substtagrange [code $_output tagNreplaceOnLastLine]

}
body TSession::_deleteInterp {} {
    interp delete smmuser
    namespace delete ::smm::ses
}
body TSession::perror {aStr} { 
    if {!$_showerror} { return }
    regsub -- {tSession[0-7]* } $aStr {}                 aStr 
    regsub -- "$_output "       $aStr {}                 aStr 

    regsub -- {smmOption}       $aStr {smm::option}      aStr 
    regsub -- {smmTab}          $aStr {smm::tab}         aStr 
    regsub -- {smmTick}         $aStr {smm::tick}        aStr 
    regsub -- {smmLog}          $aStr {smm::log}         aStr 
    regsub -- {smmUserPrint}    $aStr {smm::showme}      aStr 
    regsub -- {smmKeyCom}       $aStr {smm::keycom}      aStr 
    regsub -- {smmBell}         $aStr {smm::bell}        aStr 
    regsub -- {smmFile}         $aStr {smm::file}        aStr 
    regsub -- {smmKillAll}      $aStr {smm::killall}     aStr 

    regsub -- {smmActionList}         $aStr {smm::action::list}        aStr 
    regsub -- {smmActionGet}          $aStr {smm::action::get}        aStr 
    regsub -- {smmActionSet}          $aStr {smm::action::set}        aStr 
    regsub -- {smmActionUnset}        $aStr {smm::action::unset}      aStr 
    regsub -- {smmActionExists}       $aStr {smm::action::exists}     aStr 
    regsub -- {smmActionNames}        $aStr {smm::action::names}      aStr 

    regsub -- {smmMapZoom}      $aStr {smm::map::zoom}   aStr 
    regsub -- {smmMapExists}    $aStr {smm::map::exists} aStr 
    regsub -- {smmMapSet}       $aStr {smm::map::set}    aStr 
    regsub -- {smmMapGet}       $aStr {smm::map::get}    aStr 
    regsub -- {smmMapCreate}    $aStr {smm::map::create} aStr 
    regsub -- {smmMapSwitch}    $aStr {smm::map::switch} aStr 
    regsub -- {smmMapMove}      $aStr {smm::map::move}   aStr 

    regsub -- {smmVSet}         $aStr {smm::gui::set}        aStr 
    regsub -- {smmVUnset}       $aStr {smm::gui::unset}      aStr 
    regsub -- {smmVGet}         $aStr {smm::gui::get}        aStr 
    regsub -- {smmVList}        $aStr {smm::gui::list}       aStr 
    regsub -- {smmVArray}       $aStr {smm::gui::array}      aStr 
    regsub -- {smmGUIAddVarChooserItem}    $aStr {smm::gui::addvarchooseritem}    aStr 
    regsub -- {smmGUIRemVarChooserItem}    $aStr {smm::gui::remvarchooseritem}    aStr 
    regsub -- {smmGUIAddVarChooser}        $aStr {smm::gui::addvarchooser}        aStr 
    regsub -- {smmGUIRemVarChooser}        $aStr {smm::gui::remvarchooser}        aStr
    regsub -- {smmGUIAddVarLabel}          $aStr {smm::gui::addvarlabel}          aStr 
    regsub -- {smmGUIRemVarLabel}          $aStr {smm::gui::remvarlabel}          aStr 
    regsub -- {smmGUIAddVarEntry}          $aStr {smm::gui::addvarentry}          aStr 
    regsub -- {smmGUIRemVarEntry}          $aStr {smm::gui::remvarentry}          aStr 
    regsub -- {smmGUIAddLabel}             $aStr {smm::gui::addlabel}             aStr 
    regsub -- {smmGUIRemLabel}             $aStr {smm::gui::remlabel}             aStr 
    regsub -- {smmGUISetLabelText}         $aStr {smm::gui::setlabeltext}         aStr 
    regsub -- {smmGUIAddButton}            $aStr {smm::gui::addbutton}            aStr 
    regsub -- {smmGUIRemButton}            $aStr {smm::gui::rembutton}            aStr 
    regsub -- {smmGUIAddCheckbutton}            $aStr {smm::gui::addcheckbutton}            aStr 
    regsub -- {smmGUIRemCheckbutton}            $aStr {smm::gui::remcheckbutton}            aStr 
    regsub -- {smmGUIAddRow}               $aStr {smm::gui::addrow}               aStr 
    regsub -- {smmGUIRemRow}               $aStr {smm::gui::remrow}               aStr 

    regsub -- {tagNreplaceOnLastLine}      $aStr {::smm::text::substtagrange}     aStr 
    regsub -- {tagNreplaceLastLine}        $aStr {::smm::text::substtag}          aStr 
    regsub -- {deleteRangeOnLastLine}      $aStr {::smm::text::deleterange}       aStr 
    regsub -- {deleteLastLine}             $aStr {::smm::text::delete}            aStr 
    regsub -- {replaceOnLastLine}          $aStr {::smm::text::substrange}        aStr 
    regsub -- {replaceLastLine}            $aStr {::smm::text::subst}             aStr 
    regsub -- {tagOnLastLine}              $aStr {::smm::text::tagrange}          aStr 
    regsub -- {tagLastLine}                $aStr {::smm::text::tag}               aStr 

    $_output fromSMM "ERROR: $aStr"
}

#
# Action
#

body TSession::smmActionSet {alias level} {
    if {[catch {set iLevel [expr double($level)]}]} {
	error "wrong level \"$level\""
    }
    set _actP($alias)  $iLevel
    _smmActionReadjust
}
body TSession::_smmActionReadjust {} {
    foreach a [array names _actP] {
	lappend hl($_actP($a)) $a
    }
    set _actList {}
    foreach lev [lsort -real -decreasing [array names hl]] {
	eval lappend _actList $hl($lev)
    }
    puts $_actList
}
body TSession::smmActionUnset {alias} {
    unset _actP($alias)
    _smmActionReadjust
}
body TSession::smmActionNames {pattern} {
    return [array names _actP $pattern]
}
body TSession::smmActionList {} {
    return $_actList
}
body TSession::smmActionGet {alias} {
    if {[info exists _actP($alias)]} {
	return $_actP($alias)
    } else {
	error "no such action: \"$alias\"."
    }
}
body TSession::smmActionExists {alias} {
    return [info exists _actP($alias)]
}


#
# GetSelection
#
body TSession::smmGetSelection {{sourcename {}}} {
  if { $sourcename == "" } {
    set sourcename _output
  }
  set allowed [list _output]
  if { [lsearch -exact $allowed $sourcename] == -1 } {
    error "bad window \"$sourcename\" to \"smm::getselection\": must be one of: $allowed."
    return {}
  }
  set source [set $sourcename]
  set ListSelection ""
  set TextSelection [$source getTextSelection]
  regsub -all -- "\n" "$TextSelection" "\} \{" ListSelection
  set ListSelection [concat "{" $ListSelection "}"]
  regsub " \{ \}$" "$ListSelection" "" ListSelection
  return $ListSelection
}


#
# GUI
#
body TSession::smmGUIAddRow {rowname} {
    $CButtonArea addButtonRow $rowname
}
body TSession::smmGUIRemRow {rowname} {
    $CButtonArea remButtonRow $rowname
}
body TSession::smmGUIAddButton {rowname buttonname text command} {
    $CButtonArea addButton $rowname $buttonname $text [code $this toSMM $command]
}
body TSession::smmGUIRemButton {rowname buttonname} {
    $CButtonArea remButton $rowname $buttonname
}
body TSession::smmGUIAddCheckbutton {rowname buttonname text command variablename} {
    $CButtonArea addCheckbutton $rowname $buttonname $text [code $this toSMM $command] $variablename
}
body TSession::smmGUIRemCheckbutton {rowname buttonname} {
    $CButtonArea remCheckbutton $rowname $buttonname
}
body TSession::smmGUIAddLabel {rowname labelname text width} {
    $CButtonArea addLabel $rowname $labelname $text $width
}
body TSession::smmGUIRemLabel {rowname labelname} {
    $CButtonArea remLabel $rowname $labelname
}
body TSession::smmGUISetLabelText {rowname labelname text} {
    $CButtonArea setLabelText $rowname $labelname $text 
}
body TSession::smmGUIAddVarEntry        {rowname varentryname variablename width } {
    $CButtonArea addVarEntry $rowname $varentryname $variablename $width
}
body TSession::smmGUIRemVarEntry        {rowname varentryname} {
    $CButtonArea remVarEntry $rowname $varentryname
} 
body TSession::smmGUIAddVarLabel        {rowname varlabelname variablename width } {
    $CButtonArea addVarLabel $rowname $varlabelname $variablename $width
} 
body TSession::smmGUIRemVarLabel        {rowname varlabelname} {
    $CButtonArea remVarLabel $rowname $varlabelname
} 
body TSession::smmGUIAddVarChooser      {rowname varchoosername variablename width } {
    $CButtonArea addVarChooser $rowname $varchoosername $variablename $width
} 
body TSession::smmGUIRemVarChooser      {rowname varchoosername} {
    $CButtonArea remVarChooser $rowname $varchoosername
} 
body TSession::smmGUIAddVarChooserItem  {rowname varchoosername item} {
    $CButtonArea addVarChooserItem $rowname $varchoosername $item
} 
body TSession::smmGUIRemVarChooserItem  {rowname varchoosername index} {
    $CButtonArea remVarChooserItem $rowname $varchoosername $index
}

#
# GUI-Var-Access
#
body TSession::smmVSet {varname value} {
    namespace eval ::smm::ses "set $varname [list $value]"
}
body TSession::smmVUnset {varname} {
    namespace eval ::smm::ses "unset $varname"
}
body TSession::smmVGet {varname} {
    namespace eval ::smm::ses "return \[set $varname\]"
}
body TSession::smmVList {} {
    set retlist [list]
    set vl [namespace eval ::smm::ses { info vars }]
    foreach varname $vl {
	set fullname [namespace eval ::smm::ses "namespace which -variable [list $varname]"]
	if {[string match {::smm::ses::*} $fullname]} {
	    lappend retlist $varname
	}
    }
    return $retlist
}
body TSession::smmVArray {option arrayname args} {
    namespace eval ::smm::ses "array $option $arrayname $args"
}

#
# Map-Get
#
body TSession::smmMapGet {subspec args} {
    switch -exact -- $subspec {
	"zoomname"    { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::get zoomname\"." }
	    return [[$CMapManager getZoomPos] getMapName]
	}
	"zoomlevel"    { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::get zoomlevel\"." }
	    return [[$CMapManager getZoomPos] getMapLevel]
	}
	"zoomx"    { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::get zoomx\"." }
	    return [[$CMapManager getZoomPos] getX]
	}
	"zoomy"    { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::get zoomy\"." }
	    return [[$CMapManager getZoomPos] getY]
	}
	"playername"    { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::get playername\"." }
	    return [[$CMapManager getPlayerPos] getMapName]
	}
	"playerlevel"    { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::get playerlevel\"." }
	    return [[$CMapManager getPlayerPos] getMapLevel]
	}
	"playerx"    { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::get playerx\"." }
	    return [[$CMapManager getPlayerPos] getX]
	}
	"playery"    { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::get playery\"." }
	    return [[$CMapManager getPlayerPos] getY]
	}
	"exitwithcommand" {
	    if {[llength $args]!=5} { error "wrong \# args: should be \"smm::map::get exitwithcommand command map-name map-level x y\"." }
	    return [$CMapManager getExitWithCommand [lindex $args  1] \
			[lindex $args  2] [lindex $args  3] [lindex $args  4] [lindex $args  0]]
	}
	"exitslist" {
	    if {[llength $args]!=4} { error "wrong \# args: should be \"smm::map::get exitslist map-name map-level x y\"." }
	    return [$CMapManager getExitsList [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3]]
	}
	"exitdest" {
	    if {[llength $args]!=5} { error "wrong \# args: should be \"smm::map::get exitdest map-name map-level x y pos\"." }
	    return [$CMapManager getExitDest [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"exitcmd" {
	    if {[llength $args]!=5} { error "wrong \# args: should be \"smm::map::get exitcmd map-name map-level x y pos\"." }
	    return [$CMapManager getExitCommand [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"exitdoor" {
	    if {[llength $args]!=5} { error "wrong \# args: should be \"smm::map::get exitdoor map-name map-level x y pos\"." }
	    return [$CMapManager getExitDoor [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"exitdooropen" {
	    if {[llength $args]!=5} { error "wrong \# args: should be \"smm::map::get exitdooropen map-name map-level x y pos\"." }
	    return [$CMapManager getExitDoorOpen [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"exitopen" {
	    # deprecated
	    if {[llength $args]!=5} { error "wrong \# args, and exitopen is deprecated: should be \"smm::map::get exitdooropen map-name map-level x y pos\"." }
	    return [$CMapManager getExitDoorOpen [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"exitdoorclose" {
	    if {[llength $args]!=5} { error "wrong \# args: should be \"smm::map::get exitdoorclose map-name map-level x y pos\"." }
	    return [$CMapManager getExitDoorClose [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"exitclose" {
	    # deprecated
	    if {[llength $args]!=5} { error "wrong \# args, and exitclose is deprecated: should be \"smm::map::get exitdoorclose map-name map-level x y pos\"." }
	    return [$CMapManager getExitDoorClose [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"exitcolor" {
	    if {[llength $args]!=5} { error "wrong \# args: should be \"smm::map::get exitcolor map-name map-level x y pos\"." }
	    return [$CMapManager getExitColor [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"exitthickness" {
	    if {[llength $args]!=5} { error "wrong \# args: should be \"smm::map::get exitthickness map-name map-level x y pos\"." }
	    return [$CMapManager getExitThickness [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"exitdrawto" {
	    if {[llength $args]!=5} { error "wrong \# args: should be \"smm::map::get exitdrawto map-name map-level x y pos\"." }
	    return [$CMapManager getDrawTo [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3] \
			[lindex $args  4]]
	}
	"text" {
	    if {[llength $args]!=4} { error "wrong \# args: should be \"smm::map::get text map-name map-level x y\"." }
	    return [$CMapManager getText [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3]]
	}
	"bg" {
	    if {[llength $args]!=4} { error "wrong \# args: should be \"smm::map::get bg map-name map-level x y\"." }
	    return [$CMapManager getBg [lindex $args  0] \
			[lindex $args  1] [lindex $args  2] [lindex $args  3]]
	}
	"maps" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::get maps\"." }
	    return [$CMapManager getMapNames]
	}
	"rooms" {
	    if {[llength $args]!=2} { error "wrong \# args: should be \"smm::map::get rooms map-name map-level\"." }
	    return [$CMapManager getRooms [lindex $args  0] [lindex $args  1]]
	}
	"xys" {
	    if {[llength $args]!=2} { error "wrong \# args: should be \"smm::map::get xys map-name map-level\"." }
	    return [$CMapManager getXYs [lindex $args  0] [lindex $args  1]]
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::map::get\": must be zoomname, zoomlevel, zoomx, zoomy, playername, playerlevel, playerx, playery, exitwithcommand, exitslist, exitdest, exitcmd, exitdoor, exitopen, exitclose, exitcolor, exitthickness, exitdrawto, text, maps, rooms or xys." 
	}
    }
}

#
# Map-Set
#
body TSession::smmMapSet {subspec args} {
    switch -exact -- $subspec {
	"zoompos"    { 
	    if {[llength $args]!=4} { error "wrong \# args: should be \"smm::map::set zoompos map-name map-level x y\"." }
	    set mapnamearg  [lindex $args 0]
	    set maplevelarg [lindex $args 1]
	    set posxarg     [lindex $args 2]
	    set posyarg     [lindex $args 3]
	    if {[$CMapManager existsMap $mapnamearg $maplevelarg]} {
		$CMapManager setZoomPos $mapnamearg $maplevelarg $posxarg $posyarg
	    } else {
		error "map '$mapnamearg' (level $maplevelarg) does not exist!"
	    }
	}
	"playerpos" {
	    if {[llength $args]!=4} { error "wrong \# args: should be \"smm::map::set playerpos map-name map-level x y\"." }
	    set mapnamearg  [lindex $args 0]
	    set maplevelarg [lindex $args 1]
	    set posxarg     [lindex $args 2]
	    set posyarg     [lindex $args 3]
	    if {[$CMapManager existsMap $mapnamearg $maplevelarg]} {
		$CMapManager setPlayerPos $mapnamearg $maplevelarg $posxarg $posyarg
	    } else {
		error "map '$mapnamearg' (level $maplevelarg) does not exist!"
	    }
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::map::set\": must be zoompos or playerpos." 
	}
    }
}

#
# Map-Move
#
body TSession::smmMapMove {subspec args} {
    switch -exact -- $subspec {
	"useexit"                {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::map::move $subspec exit-pos\"." }
	    $CMapManager movePlayer useexit [lindex $args  0] 
	}
	"useexitwithcommand"     {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::map::move $subspec command\"." }
	    $CMapManager movePlayer useexitwithcommand [lindex $args  0] 
	}
	"followexit"             {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::map::move $subspec exit-pos\"." }
	    $CMapManager movePlayer followexit [lindex $args  0] 
	}
	"followexitwithcommand"  {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::map::move $subspec command\"." }
	    $CMapManager movePlayer followexitwithcommand [lindex $args  0] 
	}
	"powerwalk" {
	    if {[llength $args]!=4} { error "wrong \# args: should be \"smm::map::move $subspec map-name map-level x y\"." }
	    $CMapManager CLPowerwalk [lindex $args  0] [lindex $args 1] [lindex $args  2] [lindex $args  3]
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::map::move\": must be useexit, useexitwithcommand, followexit, followexitwithcommand or powerwalk." 
	}
    }
}

#
# Map-Exists
#
body TSession::smmMapExists {subspec args} {
    switch -exact -- $subspec {
	"room"    { 
	    if {[llength $args]!=4} { error "wrong \# args: should be \"smm::map::exists room map-name map-level x y\"." }
	    return [$CMapManager existsRoom [lindex $args  0] [lindex $args 1] [lindex $args  2] [lindex $args  3]]
	}
	"map"    { 
	    if {[llength $args]!=2} { error "wrong \# args: should be \"smm::map::exists map map-name map-level\"." }
	    return [$CMapManager existsMap [lindex $args  0] [lindex $args 1]]
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::map::exists\": must be map or room." 
	}
    }
}

#
# Map-Create
#
body TSession::smmMapCreate {mapname maplevel} {
    $CMapManager createMap $mapname $maplevel
}

#
# killall
#
body TSession::smmKillAll {} {
    killall
}

#
# Map-Switch
#
body TSession::smmMapSwitch {mapname maplevel} {
    $CMapManager changeMap $mapname $maplevel
}

#
# Map-Zoom
#
body TSession::smmMapZoom {subspec args} {
    switch -exact -- $subspec {
	"accept"    { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::zoom accept\"." }
	    $CMapManager acceptRoom 
	}
	"reset"     { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::zoom reset\"." }
	    $CMapManager resetRoom 
	}
	"clear" { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::zoom clear\"." }
	    $CMapManager zifRemoveRoom 
	}
	"setbg"     { 
	    if {[llength $args]!=3} { error "wrong \# args: should be \"smm::map::zoom setbg bg-type bg-category bg-name\"." }
	    $CMapManager zifSetBg [lindex $args 0] [lindex $args 1] [lindex $args 2]
	}
	"setstuff"  { 
	    if {[llength $args]!=4} { error "wrong \# args: should be \"smm::map::zoom setstuff stuff-pos stuff-type stuff-category stuff-name\"." }
	    $CMapManager zifSetStuff [lindex $args 0] [lindex $args 1] [lindex $args 2] [lindex $args 3]
	}
	"remstuff"  { 
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::map::zoom remstuff stuff-pos\"." }
	    $CMapManager zifRemStuff [lindex $args 0]
	}
	"remtext"   { 
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::map::zoom remtext\"." }
	    $CMapManager zifRemText 
	}
	"settext"   { 
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::map::zoom settext text\"." }
	    $CMapManager zifSetText [lindex $args 0]
	}
	"remexit"   { 
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::map::zoom remexit exit-position\"." }
	    $CMapManager zifRemExit [lindex $args 0]
	}
	"setexit"   { 
	    if {[llength $args]!=13} { error "wrong \# args: should be \"smm::map::zoom setexit position command mapname maplevel connect-x connect-y draw-x draw-y door open-door-command close-door-command line-thickness line-color\"." }
	    $CMapManager zifSetExit \
		[lindex $args  0] [lindex $args 1] [lindex $args  2] [lindex $args  3] \
		[lindex $args  4] [lindex $args 5] [lindex $args  6] [lindex $args  7] \
		[lindex $args  8] [lindex $args 9] [lindex $args 10] [lindex $args 11] \
		[lindex $args 12] 
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::map::zoom\": must be accept, reset, clear, setbg, setstuff, remstuff, remtext, settext, remexit or setexit." 
	}
    }
}

#
# File
#
body TSession::smmFile {subspec args} {
    switch -exact -- $subspec {
	"readasmud" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::file readasmud file\"." }
	    readMudFromFile [lindex $args 0]	    
	}
	"eval" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::file eval file\"." }
	    toSMM "::source [lindex $args 0]"
	}
	"usereval" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::file usereval file\"." }
	    toSMM "::source [$CFilM getScriptFile [lindex $args 0]]"
	}
	"syseval" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::file syseval file\"." }
	    toSMM "::source [$CFilM getSysScriptFile [lindex $args 0]]"
	}
	"send" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::file send file\"." }
	    if {[string compare "" $CSocket]==0} { error "no connection established." }
	    set filename [lindex $args 0]
	    set fileId [open $filename r]
	    if {[catch {puts $CSocket [read $fileId]}]} {
		TInformer $itk_interior.mess \
		    -vanishafter 3 -information "Connection to the mud failed!"
	    }
	    close $fileId
	    printf "File sent: $filename"
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::file\": must be readasmud, eval, usereval, syseval or send." 
	}
    }
}

#
# Bell 
#
body TSession::smmBell {subspec args} {
    switch -exact -- $subspec {
	"1" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm:bell 1\"." }
	    exec xset b on
	}
	"0" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm:bell 0\"." }
	    exec xset b off
	}
	"ring" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm:bell ring\"." }
	    bell
	}
	"set" {
	    if {[llength $args]!=3} { error "wrong \# args: should be \"smm:bell set volume herz milliseconds\"." }
	    exec xset b [lindex $args 0] [lindex $args 1] [lindex $args 2] 
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::bell\": must be 1, 0, ring or set." 
	}
    }
}

#
# KeyCom
#
body TSession::smmKeyCom {sequence command} {
    $CInput bindevent $sequence [code $this toSMM $command]
}

#
# showme
#
body TSession::smmUserPrint {string {tags ""}} {
    $_output userPrint $string $tags
}

#
# Logging
#
body TSession::smmLog {subspec args} {
    switch -exact -- $subspec {
	"file" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::log file filename\"." }
	    logfile [lindex $args 0]
	}
	"flush" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::log flush\"." }
	    logFlush
	}
	"inject" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm::log inject\"." }
	    logInject [lindex $args 0]
	}
	"time" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::log time\"." }
	    logTimeOn
	}
	"notime" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::log notime\"." }
	    logTimeOff
	}
	"tomud" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::log tomud\"." }
	    logToMudOn
	}
	"notomud" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::log notomud\"." }
	    logToMudOff
	}
	"1" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::log 1\"." }
	    if {![logfileExists]}   { error "no logfile defined." }
	    logOn
	}
	"0" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::log 0\"." }
	    logOff
	}
	"!" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm::log !\"." }
	    if {![logfileExists]}   { error "no logfile defined." }
	    logToggle
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::log\": must be file, 1, 0, !, flush, inject, time, notime, tomud, notomud." 
	}
    }
    if {[isLogging]} {
	printf "SMM: logging turned on!"
    } else {
	printf "SMM: logging turned off!"
    }
}

#
# Option
#
body TSession::smmOption {option action} {
    switch -exact -- $option {
	"parse" {
	    switch -- $action {
		0 { set _parse 0 }
		1 { set _parse 1 }
		! { set _parse [expr !$_parse] }
		? {  }
		default { error "bad action \"$action\" to \"smm::option parse\": must be 0, 1, ! or ?."}
	    }
	    return $_parse
	}
	"output" {
	    switch -- $action {
		0 { set _showoutput 0 }
		1 { set _showoutput 1 }
		! { set _showoutput [expr !$_showoutput] }
		? {  }
		default { error "bad action \"$action\" to \"smm::option output\": must be 0, 1, ! or ?."}
	    }
	    return $_showoutput
	}
	"error" {
	    switch -- $action {
		0 { set _showerror 0 }
		1 { set _showerror 1 }
		! { set _showerror [expr !$_showerror] }
		? {  }
		default { error "bad action \"$action\" to \"smm::option error\": must be 0, 1, ! or ?."}
	    }
	    return $_showerror
	}
	"devmode" {
	    switch -- $action {
		0 { $CMapManager setDevMode 0 }
		1 { $CMapManager setDevMode 1 }
		! { $CMapManager setDevMode [expr ![$CMapManager devMode]] }
		? {  }
		default { error "bad action \"$action\" to \"smm::option devmode\": must be 0, 1, ! or ?."}
	    }
	    return [$CMapManager devMode]
	}
	default { 
	    error "bad option \"$option\" to \"smm::option\": must be parse, devmode, output or error." 
	    return {}
	}
    }
}


#
# Scroll
#
body TSession::smmScroll {elem act} {
    set ElemList ""
    switch -exact -- $elem {
	"lock" -
	"evaluation" -
	"input" -
	"readasmud" -
	"showme" -
	"line" -
	"packet" {
	    set ElemList [list $elem]
	}
	"socket" {
	    set ElemList [list line packet]
	}
	"all" {
	    set ElemList [list input readasmud showme line packet evaluation]
	}
	default {
	    error "bad element \"$elem\" to \"smm::scroll\": must be lock, evaluation, input, readasmud, showme, line, packet, all or socket."
	}
    }
    set el ""
    switch -- $act {
	! {
	    foreach el $ElemList {
		$_output setScrollLevel $el [expr ![$_output getScrollLevel $el]]
		# debug printf "smmScroll acting ! on $el.n"
	    }
	}
	? { set el [lindex $ElemList end] }
	default {
	    foreach el $ElemList {
		# debug printf "smmScroll acting $act on $el.n"
		if { [expr ([$_output setScrollLevel $el $act] == -1)] } {
		    error "bad action \"$act\" to \"smm::scroll $elem\": must be a non-negative number, ! or ?."
		    return [$_output getScrollLevel lock]
		}
	    }
	}
    }
    if { $el != "" } { return [$_output getScrollLevel $el] }
}



#
# Tab
#
body TSession::smmTab {subspec args} {
    switch -exact -- $subspec {
	"add" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm:tab add string\"." }
	    $CTabM add [lindex $args 0]
	}
	"rem" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm:tab rem string\"." }
	    $CTabM rem [lindex $args 0]
	} 
	"list" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm:tab list\"." }
	    printf "Tabs: [lsort [$CTabM getAllNames]]"
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::tab\": must be add, rem or list." 
	}
    }
}
#
# Tick
#
body TSession::smmTick {subspec args} {
    switch -exact -- $subspec {
	"get" {
	    if {[llength $args]!=0} { error "wrong # args: should be \"smm:tick get\"." }
	    return [$CTicker getTick]
	}
	"set" {
	    if {[llength $args]!=1} { error "wrong # args: should be \"smm:tick set count\"." }
	    $CTicker setTick [lindex $args 0]
	}
	"on" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm:tick on\"." }
	    $CTicker start
	}
	"off" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm:tick off\"." }
	    $CTicker stop
	} 
	"reset" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm:tick reset\"." }
	    $CTicker reset
	}
	"clear" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm:tick clear\"." }
	    $CTicker clear
	}
	"size" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm:tick size size\"." }
	    $CTicker setSize [lindex $args 0]
	}
	"add" {
	    if {[llength $args]!=2} { 
		error "wrong \# args: should be \"smm:tick add second command\"." }
	    $CTicker addTickCommand [lindex $args 0] [lindex $args 1]
	}
	"remove" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm:tick remove second\"." }
	    $CTicker remTickCommand [lindex $args 0]
	}
	"show" {
	    if {[llength $args]!=1} { error "wrong \# args: should be \"smm:tick show second\"." }
	    set sec [lindex $args 0]
	    if {[$CTicker existsCommand $sec]} { 
		set com [$CTicker getCommand $sec]
	    } else {
		set com "No tick command for second \"$sec\"!"
	    }
	    printf "Tick [format {%4d} $sec]: $com"
	}
	"list" {
	    if {[llength $args]!=0} { error "wrong \# args: should be \"smm:tick list\"." }
	    set secl [$CTicker getAllTimes] 
	    if {[llength $secl] == 0} {
		printf "No tick commands defined!"
	    } else {
		foreach sec $secl {
		    printf "Tick [format {%4d} $sec]: [$CTicker getCommand $sec]"
		}
	    }
	}
	default { 
	    error "bad subspec \"$subspec\" to \"smm::tick\": must be on, off, reset, get, set, clear, size, add, remove, show or list." 
	}
    }
}


#
# --- user interp
#
body TSession::toMud {arg} {
    if {$_showInput} { $_output fromInput $arg }
    if {[string compare "" $CSocket]!=0} {
	if {[catch {puts $CSocket $arg}]} {
	    TInformer $itk_interior.mess \
		-vanishafter 3 -information "Connection to the mud failed!"
	}
    	if {[isLogTiming]} {
	    if {[isLogToMud]} {
		logInject "\xff\xfa\x1b"
		logInject "t,[clock clicks -milliseconds]\x1d$arg\xff\xf0"
	    } {
		logInject "\xff\xfa\x1b"
		logInject "t,[clock clicks -milliseconds]\xff\xf0"
	    }
    	} {
	    if {[isLogToMud]} {
		logInject "\xff\xfa\x1d$arg\xff\xf0"
	    }
	}
    }
}
body TSession::toSocket {arg} {
    if {[string compare "" $CSocket]!=0} {
	if {[catch {puts -nonewline $CSocket $arg}]} {
	    TInformer $itk_interior.mess \
		-vanishafter 3 -information "Connection to the mud failed!"
	}
    }
}
body TSession::toSMM {arg} {
    if {$_parse} {
	if { ![string compare "" [string trim $arg]] } {	
	    toMud $arg    
	} else {
	    if { [catch { smmuser eval "::namespace eval ::ses { $arg }" } errtext] } {
		perror $errtext
	    } else {
		return $errtext
	    }
	}
    } else {
	if {[string compare $arg "::smm::option parse 1"]} {
	    toMud $arg
	} else {
	    if { [catch { smmuser eval "::namespace eval ::ses { $arg }" } errtext] } {
		perror $errtext
	    }
	}
    }
}








# --- connect -----------------------------------------------------
body TSession::connect {aMudM aMainWin aCon} {
    set mud [$aMudM getMud $CMudName]
    if {!$_sesLoaded} {
	set CMainWindow $aMainWin
	set CMudder     [$aMainWin getMudderUif]
	set CMapManager [$aMainWin getMapManager]
	$CMapManager assignSession $this
	set _output     [$CMudder getOutput]      
	set CInput      [$CMudder getEntryX]      
	set CButtonArea [$CMudder getButtonArea]

	set CTicker     [$aMainWin getTicker]
	$CTicker bindToSes $this

	set _showInput 1
	set _parse     1

	$CMudder bindToSes $this
    }
    if {$aCon} {
	catch {::http::geturl "http://www.issever.de/smm/counter.php" -timeout 5000}

	cutConnection
	set mudport [$mud port]
	if {[string length $mudport]==0} {
	    set mudport 23
	}
	set CSocket [socket [$mud host] $mudport]

	# socket not compressed when just opened.
	set CSocketCompressed 0
	set CSocketCompressionAvailable 0
	set CSocketHistory ""
	if {![catch {package present Tclx}] && ![catch {package present Trf}]} {
	    set CSocketCompressionAvailable 1
	    pipe CSocketPipeRead CSocketPipeWrite
	    fconfigure $CSocketPipeRead -blocking 0 -translation {lf lf}
	    if {[catch {zip -attach $CSocketPipeWrite -mode decompress}]} {
		# error will occur if zlib.dll isn't available.
		set CSocketCompressionAvailable 0
	    }
	    fconfigure $CSocketPipeWrite -translation binary -buffering none -blocking 0
	    fileevent $CSocketPipeRead readable [code $this readMud]
	}

	fconfigure $CSocket -blocking 0 -buffering none -translation {lf lf}
	fileevent  $CSocket readable [code $this readMud]
	set CSocketRead $CSocket
	$_output fromSMMImportant "Connection opened!"
	if {[isLogTiming]} {
	    logInject "\xff\xfa\x1b"
	    logInject "o[clock seconds],[clock clicks -milliseconds]\xff\xf0"
	}
    } else {
	set CSocket ""
    }
    if {!$_sesLoaded} {
	_createInterp	
	smmFile eval [getCFile]
    }
    set _sesLoaded 1
}

body TSession::cutConnection {} {
    if {[string compare "" $CSocket]!=0} {
	close $CSocket
	set CSocket ""
	catch {close $CSocketWrite}  ;# might not have been opened
	catch {close $CSocketRead}   ;# ditto
	$_output fromSMMImportant "Connection closed!"
	if {[isLogTiming]} {
	    logInject "\xff\xfa\x1b"
	    logInject "c[clock seconds],[clock clicks -milliseconds]\xff\xf0"
	}
	logFlush
    }
}
body TSession::disconnect {} {
    cutConnection

    $CMainWindow sesDisconnected
    $CButtonArea clear
    $CInput unbindFromSes
    $CTicker unbindFromSes

    _deleteInterp
    set _sesLoaded 0
}
body TSession::killall {} {
    $CTabM clear
    set _actList {}
    foreach i [array names _actP] { unset _actP($i) }
}

# -------------------------------------------------------------------
body TSession::readMudFromFile {aFileName} {
    if {![file exists $aFileName]} {
	bell
	return
    }
    set fid [open $aFileName "r"]
    set CCurrentNotParsedLine ""
#     set c 0
    while {![eof $fid]} {
	gets $fid line
	evalSingleLine "\n$line" 1

	update idletasks

# 	incr c
# 	if {$c>9} {
# 	    $_output see end
# 	    update idletasks
# 	    set c 0
# 	}
    }

    close $fid
    $_output seeEnd readasmud
    # scrolllock impl: $_output see end
    update idletasks
}
#  uncompress thread (generates readable event activating readMud)
body TSession::uncompressMud {} {
    set x [read $CSocket]
    if {[string compare "" $x]==0} {
	cutConnection
	return
    }
    puts -nonewline $CSocketPipeWrite $x
    flush $CSocketPipeWrite    
}
body TSession::readMud {} {
    global errorInfo errorCode
    set x [read $CSocketRead]
    if ([string compare "" $x]==0) {
    	cutConnection
	#disconnect
   	return
    }

    # we need the last 5Kb of data or so (about 2 full tcpip packages)
    # as a history to search for the MCCP enable sequence.
    set CSocketHistory [string range $CSocketHistory$x end-5000 end]

    regsub -all -- $CCR $x {} x
    regsub -all -- $CNL $x $CNNL x

    set p $CCurrentNotParsedLine$x
    set CCurrentNotParsedLine ""
#    set c 0
    foreach line [split $p $CRS ] {
	set CompStat $CSocketCompressed
	if {[catch {evalSingleLine "$line" 1} errMsg]} {
	    set savedInfo $errorInfo
	    set savedCode $errorCode
	    if {[string compare $savedCode MCCP]} {
		error $errMsg $savedInfo $savedCode
	    }
	}
	# scroll lock impl $_output see end
	$_output seeEnd line
	update idletasks
	set CompStat [expr $CSocketCompressed - $CompStat]
	if {$CompStat==1} {
	    # compression was just enabled, now find rest of package
	    # write it to $CSocketPipeWrite. Now, assuming that the
	    # first occurence of the MCCP enable sequence in
	    # $CSocketHistory really is the true MCCP enable sequence
	    # (very likely), we can simply search $CSocketHistory for
	    # that sequence and when found (it must be there), split
	    # $CSocketHistory and take the last part and write it to
	    # $CSocketPipeWrite. IAC SB COMPRESS WILL SE
	    set i [string first \xff\xfa\x55\xfb\xf0 $CSocketHistory]
	    if {$i>=0} {
		set rest [string range $CSocketHistory [expr $i+5] end]
		if {[string length $rest]>0} {
		    puts -nonewline $CSocketPipeWrite $rest
		}
	    }
	    break  ;# dont process rest of $p (is taken care of)
	}
# 	incr c
# 	if {$c>19} {
# 	    $_output see end
# 	    update idletasks
# 	    set c 0
# 	}
    }

    $_output seeEnd packet
    #scroll lock impl $_output see end
    if {[isLogTiming]} {
	logInject "\xff\xfa\x1b"
	logInject "f,[clock clicks -milliseconds]\xff\xf0"
    }
    update idletasks
}

#
# eval single lines
#
body TSession::evalSingleLineMode_t {aPart} {
    # parsed seq's:
    #  \xff\xfb\x01 -> echo input
    #  \xff\xfc\x01 -> dont echo input	    
    switch -exact -- $CTelnetProtocol {
	null {
	    if {[string length $aPart]>0} {
		set nextChar [string index $aPart 0]
		switch -exact -- $nextChar {
		    \xfa {
			#nodebug puts Subnegotiation
			set CTelnetProtocol sb
		    }
		    \xfb {
			#nodebug puts will!
			set CTelnetProtocol will 
		    }
		    \xfc {
			#nodebug puts wont!;
			set CTelnetProtocol wont
		    }
		    \xfd {
			#nodebug puts do!
			set CTelnetProtocol do
		    }
		    \xfe {
			#nodebug puts dont!
			set CTelnetProtocol dont
		    }
		    default {
			puts "ERROR: telnet parse, unknown protocol, '[charToInt $nextChar]'"
			set CParseMode l
		    }
		}
		evalSingleLine [string range $aPart 1 end]
	    }
	}
	will {
	    if {[string length $aPart]>0} {
		set nextChar [string index $aPart 0]
		# the response to "will" can be "do" or "dont"
		# do=253=\xfd, dont=254=\xfe
		switch -exact -- $nextChar {
		    \x01 { # will echo
			#nodebug puts "rcvd: will echo"
			$CInput hideInput
			set _showInput 0
			toSocket \xff\xfd\x01
			#nodebug puts "sent: do echo"
		    }
		    \x03 { # will suppress go ahead
			#nodebug puts "rcvd: will suppress go ahead"
			# do suppress go ahead
			toSocket \xff\xfd\x03
			#nodebug puts "sent: do suppress go ahead"
		    }
		    \x05 { # will status
			#nodebug puts "rcvd: will status"
			# dont status
			toSocket \xff\xfc\x05
			#nodebug puts "sent: dont status"
		    }
		    \x55 {
			#nodebug puts "rcvd: will compress"
			if {$CSocketCompressionAvailable} {
			    # do compress
			    toSocket \xff\xfd\x55
			    #nodebug puts "sent: do compress"
			    # the server may begin compression at any time
			    # by sending a IAC SB COMPRESS WILL SE
			    # sequence, immediately followed by the start
			    # of the compressed stream.
			    # We will be watching for the enabling sequence.
			} else {
			    # dont compress
			    toSocket \xff\xfe\x55
			    #nodebug puts "sent: dont compress"
			}
		    }
		    \xf0 {
			#nodebug puts "rcvd: will se"
			# assumption: IAC SB COMPRESS preceeded WILL SE
			# now we need to enable decompression
			set CSocketCompressed 1
			fconfigure $CSocket -translation binary
			# redirect socket read events
			fileevent $CSocket readable [code $this uncompressMud]
			# make readMud read from pipe instead of socket
			set CSocketRead $CSocketPipeRead
			#nodebug puts "decompression enabled"
			$_output fromSMMImportant "Compression enabled!"
			# we need to terminate parsing of the current
			# tcpip package as the rest is compressed
			# data.
			set CTelnetProtocol null
			set CParseMode l
			error {MCCP enabled} {This error should not be seen} MCCP
		    }
		    default {
			puts "ERROR: telnet parse, unknown will command, '[charToInt $nextChar]'"
		    }
		}
		set CTelnetProtocol null
		set CParseMode l
		evalSingleLine [string range $aPart 1 end]
	    }
	}
	wont {
	    if {[string length $aPart]>0} {
		set nextChar [string index $aPart 0]
		# the response to "wont" can only be "dont"
		# dont=254=\xfe
		switch -exact -- $nextChar {
		    \x01 { 
			#nodebug puts "rcvd: wont echo"
			$CInput showInput
			set _showInput 1
			toSocket \xff\xfe\x01
			#nodebug puts "sent: dont echo"
		    }
		    default {
			puts "ERROR: telnet parse, unknown wont command, '[charToInt $nextChar]'"
		    }
		}
		set CTelnetProtocol null
		set CParseMode l
		evalSingleLine [string range $aPart 1 end]
	    }
	}
	do {
	    if {[string length $aPart]>0} {
		set nextChar [string index $aPart 0]
		# the repsonse to "do" can only be "will" or "wont".
		# will=251=\xfb, wont=252=\xfc
		switch -exact -- $nextChar {
		    \x01 { # do echo
			#nodebug puts "rcvd: do echo"
			# wont echo (with suppress go ahead enabled this
			# enables kludge line mode). Besides, echo is
			# not implemented.
			toSocket \xff\xfc\x01
			#nodebug puts "sent: wont echo"
		    }
		    \x18 { # do terminal type
			#nodebug puts "rcvd: do terminal type"
			# wont terminal type
			toSocket \xff\xfc\x18
			#nodebug puts "sent: wont terminal type"
		    }
		    \x1f { # do terminal size
			#nodebug puts "rcvd: do terminal size"
			# wont terminal size
			toSocket \xff\xfc\x1f
			#nodebug puts "sent: wont terminal size"
		    }
		    \x20 { # do terminal speed
			#nodebug puts "rcvd: do terminal speed"
			# wont terminal speed
			toSocket \xff\xfc\x20
			#nodebug puts "sent: wont terminal speed"
		    }
		    \x21 { # do remote flow control
			#nodebug puts "rcvd: do remote flow control"
			# wont remote flow control
			toSocket \xff\xfc\x21
			#nodebug puts "sent: wont remote flow control"
		    }
		    \x23 { # do "unknown"
			#nodebug puts "rcvd: do unknown 35"
			# wont "unknown"
			toSocket \xff\xfc\x23
			#nodebug puts "sent: wont unknown 35"
		    }
		    \x24 { # do environment variables
			#nodebug puts "rcvd: do environment variables"
			# wont env. vars
			toSocket \xff\xfc\x24
			#nodebug puts "sent: wont environment variables"
		    }
		    \x27 { # do "unknown"
			#nodebug puts "rcvd: do unknown 39"
			# wont "unknown"
			toSocket \xff\xfc\x27
			#nodebug puts "sent: wont unknown 39"
		    }
		    default {
			puts "ERROR: telnet parse, unknown do command, '[charToInt $nextChar]'"
		    }
		}
#		toSocket \xff\xfc$nextChar
		set CTelnetProtocol null
		set CParseMode l
		evalSingleLine [string range $aPart 1 end]
	    }
	}
	dont {
	    if {[string length $aPart]>0} {
		set nextChar [string index $aPart 0]
		# the repsonse to "dont" can only "wont".
		# will=251=\xfb, wont=252=\xfc
		switch -exact -- $nextChar {
		    \x01 { # dont echo
			#nodebug puts "rcvd: dont echo"
			# wont echo
			toSocket \xff\xfc\x01
			#nodebug puts "sent: wont echo"
		    }
		    \x18 { # dont terminal type
			#nodebug puts "rcvd: dont terminal type"
			# wont trminal type
			toSocket \xff\xfc\x18
			#nodebug puts "sent: wont terminal type"
		    }
		    \x1f { # dont windows size
			#nodebug puts "rcvd: dont window size"
			# wont window size
			toSocket \xff\xfc\x1f
			#nodebug puts "sent: wont window size"
		    }
		    \x23 { # dont "unknown 35"
			#nodebug puts "rcvd: dont \"unknown 35\""
			# wont "unknown"
			toSocket \xff\xfc\x23
			#nodebug puts "sent: wont \"unknown 35\""
		    }
		    \x24 { # dont environment variables
			#nodebug puts "rcvd: dont environment variables"
			# wont environment variables
			toSocket \xff\xfc\x24
			#nodebug puts "sent: wont environment variables"
		    }
		    \x27 { # dont "unknown 39"
			#nodebug puts "rcvd: dont \"unknown 39\""
			# wont "unknown 39"
			toSocket \xff\xfc\x27
			#nodebug puts "sent: wont \"unknown 39\""
		    }
		    default {
			puts "ERROR: telnet parse, unknown dont command, '[charToInt $nextChar]'"
		    }
		}
#		toSocket \xff\xfc$nextChar
		set CTelnetProtocol null
		set CParseMode l
		evalSingleLine [string range $aPart 1 end]
	    }
	}
	sb {
	    if {[string length $aPart]>0} {
		set nextChar [string index $aPart 0]
		# the repsonse to "sb" can be many things
		switch -exact -- $nextChar {
		    \x55 {
			#nodebug puts "rcvd: sb compress"
			# now we expect: will se (251 240=\xfb \xf0)
			set CTelnetProtocol null
			set CParseMode t
		    }
		    default {
			puts "ERROR: telnet parse, unknown sb command, '[charToInt $nextChar]'"
			set CTelnetProtocol null
			set CParseMode l
		    }
		}
		evalSingleLine [string range $aPart 1 end]
	    }		
	}

    }
}
body TSession::evalSingleLineMode_l {aPart} {
    if {[regexp -- "(\[^\x1b\xff\]*)(\[\x1b\xff\])(.*)" $aPart dummy b c d]} {
	#b: contains all before the command seq (esc or telnet)
	#c: contains the first command seq char -> may be \x1b or \xff
	#d: contains the rest of the string

	#. first write away the first part
	writeMudText $b			
	if {[string compare "\x1b" $c] == 0} {
	    # esc seq
	    
	    #. check if the complete sequence is in
	    if {[escAnalyse $d]} {
		#nodebug puts "<<$CESCFinal>>$CESCParameters<<"
		switch -exact -- $CESCFinal {
		    m { $_output setColors $CESCParameters }
		    s { $_output rememberCursorPosition }
		    u { $_output returnToCursorPosition }
		    f - H {
			$_output moveCursorTo $CESCParameters
		    }
		    h { 
			# cursor on??
		    }
		    l { 
			# cursor off??
		    }
		    r { $_output setScrollRegion $CESCParameters	}
		    J { 
			switch -- $CESCParameters {
			    ""      { $_output clearToStart }
			    "1"     { $_output clearToEnd }
			    default { $_output clear }
			}
		    }
		    K { 
			switch -- $CESCParameters {
			    "1"     { $_output clearToStartOfLine }
			    default { $_output clearToEndOfLine }
			}
		    }
		    P { $_output delChars $CESCParameters	}
		    L { $_output addLinesBeforeCurPos $CESCParameters	}
		    M { $_output remLinesAfterCurPos $CESCParameters	}
		    A { $_output moveup $CESCParameters	}
		    B { $_output movedown $CESCParameters	}
		    C { $_output moveright $CESCParameters	}
		    D { $_output moveleft $CESCParameters }
		    7 - 8 {
			# ??
		    }
		    default {
			msgError "ESC Error:\nFinal: '$CESCFinal'\nParameter: '$CESCParameters'\nIntermediate: '$CESCIntermediates'\nFollowed by: \[$CESCParsedText\]"
		    }
		}
		evalSingleLine "$CESCParsedText"
	    } else {			    
		set CCurrentNotParsedLine "$c$d"
		if {[string length $CCurrentNotParsedLine]>100} {
		    set CCurrentNotParsedLine [string range $CCurrentNotParsedLine 1 end]
		    msgError "ERROR: seq not complete or not known: ESC+'$d'\n--------------------------------\n'\[$c$d'\]"
		}
	    }


	} else {
	    # telnet seq
	    set CParseMode t
	    evalSingleLineMode_t $d
	}
    } else {
	writeMudText $aPart
    }
}
body TSession::evalSingleLine {aPart args} {
    # first check if a esc for telnet sequence is
    # in the line: \xff (telnet:iac) \x1b (esc) 

    #nodebug puts "|$aPart|"

    evalSingleLineMode_$CParseMode $aPart

    if {[llength $args]>0} {
	foreach act $_actList {
	    set line [$_output getLastLine]
	    catch {if {[toSMM [list $act $line]]} { break }}
	}
    }
}
body TSession::writeMudText {aPart} {
    # -----------------------------------------------
    # screen output
    $_output fromMud $aPart
    # file logging
    if {[isLogging]} {
	logInject $aPart
    }
}

# ----------------------------------------------------------------------
body TSession::createFileSystem {} {
    file mkdir    [$CFilM getSesDir    $CId]

    # session information
    set    fid [open [$CFilM getSesDataFile $CId] "w+"]
    puts  $fid 1 ;# file version
    puts  $fid [id]
    puts  $fid [name]
    puts  $fid [player]
    puts  $fid [password]
    puts  $fid [mudName]
    close $fid

    # main session configuration file
    set    fid [open [$CFilM getMainSesConfFile $CId] "CREAT RDWR"]
    close $fid
}
body TSession::readFile {aFile} {
    set    fid [open $aFile "r"]
    gets  $fid version
    gets  $fid CId
    gets  $fid CName
    gets  $fid CPlayer
    gets  $fid CPassword
    gets  $fid CMudName
    close $fid
}
body TSession::removeFileSystem {} { 
    file delete -force -- [$CFilM getSesDir    $CId]
}
body TSession::wipeout {} {
    removeFileSystem
}

# ----------------------------------------------------------------------------
body TSession::constructor {args} {
    set CButtonArea ""
    set CSocket ""
    set _sesLoaded 0
    set CCurrentNotParsedLine ""
    set CCurrentAsciiLine     ""
    set CSocketHistory ""
    set CParseMode            l
    set CTelnetProtocol       null

    set CTicker               ""
    
    set CNL   "\n"
    set CCR   "\r"
    set CRS   [format "%c" 30]
    set CNNL  "$CRS$CNL"

    # init managers
    set CTabM      [TTabManager  ::\#auto]
    set _actList {}
    switch [llength $args] {
	2 {
	    set CFilM [lindex $args 0]
	    readFile  [lindex $args 1]
	}
	6 {
	    set CFilM     [lindex $args 0]
	    set CName     [lindex $args 1]
	    set CPlayer   [lindex $args 2]
	    set CPassword [lindex $args 3]
	    set CMudName  [lindex $args 4]
	    set CId       [lindex $args 5]
	    createFileSystem
	}
	default {
	    error "Wrong number of arguments to constructor!"
	}
    }
}









# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: TSession.tcl,v $
# Revision 1.17  2002/05/02 19:47:53  issever
# debugged logging and reorganized its code.
#
# Revision 1.16  2002/04/28 13:43:31  issever
# implemented Hamlets ticker overhaul.
# Important: setting tickersize has changed.
# set size x, sets the maxcount to x-1. before it
# was set to x and the loop took x+1 seconds.
#
# Revision 1.15  2002/04/27 20:58:54  issever
# again changed some gui colors and
# implmented Hamlets getselection
#
# Revision 1.14  2002/04/25 21:34:35  moth
# make exitdooropen and exitdoorclose actually work
#
# Revision 1.13  2002/04/24 20:42:22  moth
# use clock clicks -milliseconds for high resolution time logging, instead of
# the platform dependent variant
#
# Revision 1.12  2002/04/23 21:31:07  moth
# provided exitdooropen and exitdoorclose to be used in place of exitopen and exitclose
#
# Revision 1.11  2002/04/23 04:53:42  moth
# Got rid of bogus default on ::smm::map::get
#
# Revision 1.10  2002/04/23 00:36:53  moth
# Added some missing backslashes in usage messages for smm::scroll
#
# Revision 1.9  2002/04/22 19:35:49  issever
# implemented Hamlets scroll lock
#
# Revision 1.8  2002/04/06 17:35:35  moth
# added a bunch of map query submethods to ::smm::map::get
#
# Revision 1.7  2002/03/27 00:12:03  moth
# logging enhancements:  flush log when connection closes, add options to log
# commands sent to mud, and/or when things happen.
#
# Revision 1.6  2002/03/04 20:01:47  issever
# bugfix: map access from the commandline
#
# Revision 1.5  2001/12/23 02:04:57  issever
# debug: exitwithcommand
#
# Revision 1.4  2001/12/23 01:17:24  issever
# Support for muds without a portnumber
#
# Revision 1.3  2001/12/08 22:59:08  issever
# Session counter added.
#
# Revision 1.2  2001/12/08 13:38:54  issever
# reorganized session manager code.
# It is now in a separate file and has more comments.
# The functionality hasnt been extended.
#
# Revision 1.1  2001/12/07 23:58:05  issever
# The code structure has changed while implementing the mud and session manager
#
# Revision 1.6  2001/11/25 20:30:31  issever
# Removed the -foreground bug.
# This was a quite widespread bug and I hope I caught
# every appereance. If I've missed one, please notify
# us again.
#
# Revision 1.5  2001/09/24 14:43:27  moth
# added catch around execution of user code.
#
# Revision 1.4  2001/08/26 09:19:54  issever
# Removed a bug with ::smm::gui::set.
# Values were not properly backslashed/packaged in the
# procedure call-flow, so that {[1]} was interpreted as
# the command 1.
#
# Revision 1.3  2001/08/19 12:25:42  issever
# Added the cvs keywords Id at start of the file
# and Log at the end of the file
#
#
