# $Id: TMMClipboard.tcl,v 1.10 2002/05/02 19:47:57 issever Exp $

# ==========================================================
# TMapManager::TClipboard
#     This class is a helper class for the TMapManager, thus
#     it was placed into the TMapManager namespace.
#
# --- This is the mapclipboard for cut/copy/pasting
#     maps.
#
# --- First the user selects several rooms. But selected
#     rooms are not yet in the clipboard. The clipboard 
#     is filled after the user cuts or copies the
#     selection to the clipboard.
#
# --- The clipboard is filled, if the class variable _clip
#     exists. So this variable has to be unset, if the
#     clipboard is cleared.
#
# --- BTW: $this points also to the mapmanager,.. ehh
#     but for any reason not in the constructor(, which
#     we dont have).
# ==========================================================

namespace eval TMapManager {
    
    class TClipboard {
	inherit TMapManager::TSelManager 

	protected {
	    # ----------------------------------------------
	    variable _clipx 
	    variable _clipy
	    # Cut or copy of map parts is alway done
	    # relative to the position of the zoom at the
	    # time of cut or copy. When pasted later, these
	    # parts are copied to the map, which has now the
	    # zoom and relative to the new zoom position.
	    # These variables remember the zoomposition at
	    # the time of cut or copy.
	    # ----------------------------------------------
	    variable _clip
	    # The clipboard. This is an array. Each 
	    # array-variable points to a room object. The
	    # indices correspond those of the selection
	    # list: it is the xy-position as 
	    # TCoordinate::GETXY returns it.
	    # ----------------------------------------------
	    variable _cliprex
	    # This is an array. It has the same index as 
	    # _clip. Foreach room in the clipboard it is
	    # recorded which exits need to be copied
	    # relatively. It is a list of exit positions
	    # for each room.
	    # ----------------------------------------------
	    variable _clipaex
	    # This is an array. It has the same index as 
	    # _clip. Foreach room in the clipboard it is
	    # recorded which exits need to be copied
	    # absolutely. Each array element is a list.
	    # It is a list of (exit postion, map coord of
	    # destination room, connection point at
	    # destination room).
	    # ----------------------------------------------
	    variable _clipoex
	    # This is an array. It has the same index as 
	    # _clip. Foreach room in the clipboard it is
	    # recorded which exits are pointing to
	    # another map. Each array element is a list.
	    # It is a list of (exit postion, map coord of
	    # destination room, connection point at
	    # destination room, destination map name,
	    # destination map level).
	    # ----------------------------------------------
	    variable _clipmap
	    # This holds the pointer from which the 
	    # clipboard was filled
	    # ----------------------------------------------
	    variable _clipDrawMap
	    # Map on which the clip-squares are drawn
	    # ----------------------------------------------
	    method _clipCleanUp {}
	    # Clears the clipboard. All rooms are removed
	    # and _clip and _cliprex and _clipaex are unset.
	    # ----------------------------------------------
	}
	# --------------------------------------------------
	method drawClip {aMapNew aX aY}
	# --------------------------------------------------
	method clipExists {}
	# Tests, if the clipboard exists (is filled)
	# or not.
	# --------------------------------------------------
	method selClear {}
	# This method clears the selections and the
	# clipboard.
	# --------------------------------------------------
	method selCopy {aClear}
	# This copies the selected rooms into the clipboard.
	# If aClear is true, the selection (not the
	# clipboard) is cleared. This is the way selCopy is
	# to be used, if the user wants to copy parts of
	# the map. If he wants to cut, then the selection
	# shoul not be cleared, as we still need the 
	# information to delete the selected parts. Copy is
	# always done relative of the zoom position.
	# --------------------------------------------------
	method selCut {} 
	# This copies the selected parts of the map and then
	# deletes these parts.
	# --------------------------------------------------
	method selPaste {}
	# This method pastes what ever is in the clipboard
	# to the map. Again relative to the zooms position.
	# --------------------------------------------------
	variable _dndDidntStarted       1
	variable _dndDidntReallyStarted 1
	variable _dndNeedClip           1
	variable _dndMap
	variable _dndStartX
	variable _dndStartY
	variable _dndX
	variable _dndY
	variable _dndEndX
	variable _dndEndY
	variable _dndDragPlayer
	method dragNDropStart {aMap aX aY} {
	    set _dndDidntStarted       0
	    set _dndDidntReallyStarted 1
	    set _dndNeedClip           1
	    set _dndMap                $aMap
	    set _dndStartX             $aX
	    set _dndStartY             $aY
	    set _dndX                  $aX
	    set _dndY                  $aY
	    set _dndEndX               $aX
	    set _dndEndY               $aY
	    if {[$this isPlayerposEqualTo $aMap $aX $aY]} {
		set _dndDragPlayer 1
	    } else {
		set _dndDragPlayer 0
	    }
	}
	method dragNDropMid {aX aY} {
	    if {$_dndDidntStarted} { return }
	    if {$aX==$_dndX && $aY==$_dndY} { return }
	    set  _dndDidntReallyStarted 0
	    set _dndX                   $aX
	    set _dndY                   $aY

	    if {$_dndNeedClip} {
		_clipCleanUp
		if {![_selExists]} {
		    mapSelect $_dndMap s $_dndStartX $_dndStartY
		}
		if {[$_selMap !=  $_dndMap]} {
		    mapSelect $_dndMap s $_dndStartX $_dndStartY
		}
		set _dndNeedClip 0
		selCut
	    }
	    drawClip $_dndMap $_dndX $_dndY
	}
	method dragNDropEnd {aX aY} {
	    if {$_dndDidntReallyStarted} { return }
	    $this newZoomPos $_dndMap $aX $aY
	    set _dndEndX $aX
	    set _dndEndY $aY
	    dragNDropAbort
	}
	method dragNDropAbort {} {
	    if {$_dndDidntReallyStarted} { return }
	    set _dndDidntStarted 1
	    set _dndDidntReallyStarted 1
	    selPaste
	    $this newZoomPos $_dndMap $_dndEndX $_dndEndY
	    if {$_dndDragPlayer} {
		$this newPlayerPos $_dndMap $_dndEndX $_dndEndY		
	    }
	    _clipCleanUp
	}
    }




    # ======================================================
    # === class code
    # ======================================================

    # ------------------------------------------------------
    # --- private helpers
    # ------------------------------------------------------
    body TClipboard::clipExists {} {
	return [info exist _clip]
    }
    body TClipboard::_clipCleanUp {} {
	if {![clipExists]} { return }
	catch {$_clipDrawMap undrawAllClip}
	foreach xy [array names _clip] {
	    ::delete object $_clip($xy)
	}
	catch {unset _clip}
	catch {unset _cliprex}
	catch {unset _clipaex}
	catch {unset _clipoex}
	catch {unset _clipmap}
    }

    # ------------------------------------------------------
    # --- misc.
    # ------------------------------------------------------
    body TClipboard::selClear {} {
	TSelManager::_selCleanUp
	TClipboard::_clipCleanUp	
    }
    body TClipboard::selCopy {aClear} {
	# Copy is only possible, if a selection exists
	if {![_selExists]} { return }
	# Now first clear the clipboard.
	_clipCleanUp
	
	# remember the map from which the clipboard was filled
	set _clipmap $_selMap

	# Now remember where the zoom position was.
	set _clipx [[$this getZoomPos] getX]
	set _clipy [[$this getZoomPos] getY]

	set _clipDrawMap $_selMap
	# Now copy all the rooms.
	foreach xy [array names _selRooms] {
	    if {[$_selMap exists $xy]} {
		set _clip($xy) [TRoom ::\#auto [$_selMap getRoom $xy]]

		set x [$_clip($xy) getX]
		set y [$_clip($xy) getY]
		$_selMap drawClip $x $y

		# we have to remember, which exits are dragged
		# relatively and which keep their destination points.
		# 1. All exits are dragged relatively,
		# 2. except exits with these attributes (all)
		#    a) destination is on the same map.
		#    a) destination is outside of the selection
		#    - destination does point back with an exit
		set _clipaex($xy) [list]
		set _clipoex($xy) [list]
		set _cliprex($xy) [list]
		if {$aClear} {
		    foreach exp [$_clip($xy) TExitList::getPoss] {
			lappend _cliprex($xy) $exp
		    }
		} else {
		    foreach exp [$_clip($xy) TExitList::getPoss] {
			set ex        [$_clip($xy) TExitList::getExit $exp]
			set conxy     [[$ex getConnectTo] getXY]
			if {[$ex isntPointingToSameMap]} {
			    # exit points to another map
			    set mhelper [$ex getMap]
			    set lhelper [$ex getMapLevel]
			    if {[$this existsMap $mhelper $lhelper]} {
				# the other map exists
				set otherMap [$this getMap $mhelper $lhelper]
				if {[$otherMap exists $conxy]} {
				    # a room exists at the pointed position
				    set destroom  [$otherMap getRoom $conxy]
				    set pointback [$destroom getExitPosTo2 $_clip($xy)]
				    lappend _clipoex($xy) [list $exp [$destroom getXY] $pointback $mhelper $lhelper]
				} else {
				    # no room exists at the pointed position
				    lappend _cliprex($xy) $exp
				}
			    } else {
				# the other map does not exist
				lappend _cliprex($xy) $exp
			    }
			} else {
			    catch {$_selMap getRoom $conxy} destroom
			    catch {$destroom getExitPosTo $_clip($xy)} pointback
			    # exit points to the same map
			    if {[info exists _selMap($conxy)]} { 
				# the coordinates the exit points to is in the selection
				lappend _cliprex($xy) $exp
			    } else {
				# the coordinates the exit points to isnt in the selection
				if {[$_selMap exists $conxy]} {
				    # at those coordinates a room exists
				    if {[string match cc $pointback]} {
					# this room points back to the room, which is going to be moved
					lappend _cliprex($xy) $exp
				    } else {
					# this room does not point back to the room, which is going to be moved
					lappend _clipaex($xy) [list $exp [$destroom getXY] $pointback]
				    }
				} else {
				    # at those coordinates no room exists
				    lappend _cliprex($xy) $exp
				}
			    }
			}
		    }
		}
	    }
	}
	
	# If this was a pure copy, then we can now clear
	# the selection (not the clipboard!). If this is not
	# a put copy, but a cut, then we still need the
	# selection list, so we can delete the rooms after
	# we have them in the clipboard.
	if {$aClear} { _selCleanUp }
    }

    body TClipboard::selCut {} {
	selCopy 0
	selDelete
	# selDelete will clear the selection list.
    }
    body TClipboard::drawClip {aMapNew aX aY} {
	$_clipDrawMap undrawAllClip

	set dx  [expr {$aX - $_clipx}]
	set dy  [expr {$aY - $_clipy}]
	set _clipDrawMap $aMapNew

	foreach xy [array names _clip] {
	    set x [$_clip($xy) getX]
	    set y [$_clip($xy) getY]
	    $aMapNew drawClip [expr {$x+$dx}] [expr {$y+$dy}]
	}

    }
    body TClipboard::selPaste {} {
	set dx  [expr [[$this getZoomPos] getX] - $_clipx]
	set dy  [expr [[$this getZoomPos] getY] - $_clipy]
	set map [[$this getZoomPos] getMap]

	foreach xy [array names _clip] {
	    # get room
	    set h [TRoom ::\#auto $_clip($xy)]
	    # move the room to the right position
	    $h moveby $dx $dy
	    # adjust exits
	    foreach exp $_cliprex($xy) {
		[$h TExitList::getExit $exp] moveby $dx $dy
	    }
	    foreach x $_clipaex($xy) {
		set exp       [lindex $x 0]
		set destroom  [lindex $x 1]
		set pointback [lindex $x 2]
		if {[$map == $_clipmap]} {
		    if {![$map exists $xy]} {
			if {[$map exists $destroom]} {
			    set r [$map getRoom $destroom]
			    if {[$r TExitList::exists $pointback]} {
				set e [$r TExitList::getExit $pointback]
				set d [$e getDrawTo]
				if {[string compare [$d getXY] $xy]==0} {
				    $e moveby $dx $dy
				    $map reDraw $r
				    continue
				}
			    }
			}
		    }		    
		} 
		[$h TExitList::getExit $exp] moveby $dx $dy
	    }
	    foreach x $_clipoex($xy) {
		set exp       [lindex $x 0]
		set destroom  [lindex $x 1]
		set pointback [lindex $x 2]
		set mhelper   [lindex $x 3]
		set lhelper   [lindex $x 4]
		if {[$this existsMap $mhelper $lhelper]} {
		    set otherMap [$this getMap $mhelper $lhelper]
		    if {[$otherMap exists $destroom]} {
			set r [$otherMap getRoom $destroom]
			if {[$r TExitList::exists $pointback]} {
			    set e [$r TExitList::getExit $pointback]
			    set c [$e getConnectTo]
			    $c setXYc       $xy
			    $c moveby       $dx $dy
			    $e setMap      [$map name]
			    $e setMapLevel [$map level]
			}
		    }		    
		}
		[$h TExitList::getExit $exp] moveDrawBy $dx $dy
	    }

	    # add to map
	    $map put $h
	}
    }


    # --- namespace end    
}


# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: TMMClipboard.tcl,v $
# Revision 1.10  2002/05/02 19:47:57  issever
# debugged logging and reorganized its code.
#
# Revision 1.9  2002/04/21 18:04:33  issever
# removed puts debugger
#
# Revision 1.8  2002/04/21 18:03:16  issever
# made ccp und dnd much smarter about exits leading to
# and coming from other maps.
#
# Revision 1.7  2002/04/17 20:51:04  issever
# removed another dnd bug.
#
# Revision 1.6  2002/04/17 20:48:29  issever
# d&d debug:
# 1) another _clipx bug
# 2) the zif shows the room at zoompos after d&d
#
# Revision 1.5  2002/04/16 21:05:01  issever
# removed the _clipx bug by adding the return statement into the method TMMClipboard::dragNDropAbort
#
#         method dragNDropAbort {} {
#             if {$_dndStoped} { return }
#             set _dndStoped 1
#             selPaste
#             _clipCleanUp
#         }
#
# Revision 1.4  2002/04/06 22:49:38  issever
# drag&drop bound to cut/copy/paste (in maps).
# This allows to d&d more then a single room at once.
#
# Revision 1.3  2002/04/06 01:31:16  issever
# debugged cut/copy/paste in maps. In fact it was quite a redo
# of the code, but now it works fine.
# Area to be pasted will be indicated.
#
# Revision 1.2  2002/04/05 20:35:27  issever
# splited the file TExit.tcl
# removed a small bug
#
# Revision 1.1  2002/02/11 20:24:32  issever
# cut/copy/paste for maps, drag&drop, code cleanup of mapmanager
#
