# $Id: TMMSelManager.tcl,v 1.1 2002/02/11 20:24:32 issever Exp $

# ==========================================================
# TMapManager::TSelManager
#     This class is a helper class for the TMapManager, thus
#     it was placed into the TMapManager namespace.
#
# --- The TSelManager manages the selection of rooms
#     in a map, that are going to be cut/copy/pasted.
# 
# --- One can only make selections in one map. As soon
#     as you start to select on a map the selection on 
#     the previos map (if it existed at all) is cleared.
# 
# --- A selection exists, if the class variable _selMap
#     exists. So this variable has to be unset, if the
#     selection is cleared.
#
# --- BTW: $this points also to the mapmanager,.. ehh
#     but for any reason not in the constructor(, which
#     we dont have).
# ==========================================================

namespace eval TMapManager {

    class TSelManager {
	protected {
	    # ----------------------------------------------
	    variable _selMap
	    # _selMap is the pointer to the map, in which 
	    # the user has selected rooms for ccp
	    # ----------------------------------------------
	    variable _selRooms
	    # This is an array of the selected rooms. Only 
	    # the indices are important. If a variable 
	    # _selRoom(x,y) exists the room at (x,y) is in
	    # the selection. 
	    # ----------------------------------------------
	    variable _selsx
	    variable _selex
	    variable _selsy
	    variable _seley
	    # If regions are selected the loop over the 
	    # rooms is going from _selsx,_selsy to 
	    # _selex,_seley
	    # ----------------------------------------------
	    method   _selExists {}
	    # Returns, if a selection exists or not.
	    # ----------------------------------------------
	    method   _setMapSelect {aMap}
	    # _setMapSelect takes a pointer to a map as 
	    # argument and checks, if this map and the map,
	    # which already has the selection are the same.
	    # If not the selection is cleared and the 
	    # pointer _selMap is set to aMap
	    #
	    # You have to use this procedure, before adding 
	    # rooms of a specific map to the clipboard.
	    # ----------------------------------------------
	    method   _selAdjustLoop {aX aY} 
	    # Needed to set counters, if rectangular regions
	    # are selected. The region between the zoom-pos
	    # and (aX,aY) will be walked thrue.
	    # ----------------------------------------------
	    method _selCleanUp {}
	    # Clears the selection. All selected rooms are
	    # removed from the selection list and _selMap is
	    # unset. This does not touch the clipboard. The 
	    # markers on the map are removed too.
	    # ----------------------------------------------
	}

	# --------------------------------------------------
	method setMapManager {aMapM}
	# This class must know the mapmanager it belongs to.
	# Usually this argument should have been passed, 
	# with the constructor, but it didnt work together
	# with being inherited by the mapmanager.
	# --------------------------------------------------
	method mapSelect {aMap aAct aX aY} 
	# mapSelect is called upon a mouseevent. The event
	# is bound to maps and the map itself passes this
	# event to its mapmanager.
	# --------------------------------------------------
	method selDelete {}
	# This deletes every room, which has been selected.
	# --------------------------------------------------
	method selFillZif {} 
	# This fills every room, which has been selected,
	# with what is actually in the zif. Exit 
	# destinations are relative.
	# --------------------------------------------------

    }


    # ======================================================
    # === class code
    # ======================================================

    # ------------------------------------------------------
    # --- private helpers
    # ------------------------------------------------------
    body TSelManager::_selExists {} {
	return [info exist _selMap]
    }
    body TSelManager::_setMapSelect {aMap} {
	if {[_selExists] && [$_selMap != $aMap]} {
	    _selCleanUp
	}
	set _selMap $aMap
    }
    body TSelManager::_selAdjustLoop {aX aY} {
	set zoomX [[$this getZoomPos] getX]
	set zoomY [[$this getZoomPos] getY]
	if {$zoomX<$aX} {
	    set _selsx $zoomX
	    set _selex $aX
	} else {
	    set _selsx $aX
	    set _selex $zoomX
	}
	if {$zoomY<$aY} {
	    set _selsy $zoomY
	    set _seley $aY
	} else {
	    set _selsy $aY
	    set _seley $zoomY
	}
	incr _selex
	incr _seley
    }
    body TSelManager::_selCleanUp {} {
	# signal the map to remove the selection graphics
	catch {$_selMap unselAll}
	# clear selectionlist
	catch {unset    _selMap}
	catch {unset    _selRooms}
    }

    # ------------------------------------------------------
    # --- misc
    # ------------------------------------------------------    
    body TSelManager::selDelete {} {
	if {![_selExists]} { return }
	foreach xy [array names _selRooms] {
	    # no need to check, if the room really exists,
	    # as this is done by the map itself.
	    # Furthermore I know, that I have listed the
	    # coordinates of the rooms in the selection list
	    # as they are used by the TMap::rem method
	    $_selMap remRoomAt $xy
	}
	_selCleanUp
    }
    body TSelManager::selFillZif {} {
	if {![_selExists]} { return }
	if {![[$this getZif] exists]} {
	    return
	}
	set zroom [TRoom ::\#auto [$this getZif]]
	set zx    [$zroom getX]
	set zy    [$zroom getY]
	set posl  [$zroom TExitList::getPoss] 
	foreach exp $posl {
	    set ex  [$zroom TExitList::getExit $exp]
	    set exc [$ex getConnectTo]
	    set exd [$ex getDrawTo]
	    set off($exp,c,x) [expr [$exc getX]-$zx]
	    set off($exp,c,y) [expr [$exc getY]-$zy]
	    set off($exp,d,x) [expr [$exd getX]-$zx]
	    set off($exp,d,y) [expr [$exd getY]-$zy]
	}
	foreach xy [array names _selRooms] {
	    regexp -- {(.*),(.*)} $xy dummy x y
	    set room [TRoom ::\#auto $zroom]
	    $room setXY $x $y
	    foreach exp $posl {
		set ex  [$room TExitList::getExit $exp]
		set exc [$ex getConnectTo]
		set exd [$ex getDrawTo]
		$exc setX [expr $off($exp,c,x)+$x]
		$exc setY [expr $off($exp,c,y)+$y]
		$exd setX [expr $off($exp,d,x)+$x]
		$exd setY [expr $off($exp,d,y)+$y]
	    }
	    # Here we put the room into the map. So we
	    # are not allowed to delete it, as not the
	    # map has taken responsibility over this
	    # room. The map will add the room, whose
	    # pointer it got to the rest of the rooms.
	    # The map will not create an additional
	    # new room object.
	    $_selMap put $room
	}
	_selCleanUp
    }
    body TSelManager::mapSelect {aMap aAct aX aY} {
	_setMapSelect $aMap
	switch $aAct {
	    s {
		set xy [TCoordinate::GETXY $aX $aY]
		if {![info exists _selRooms($xy)]} {
		    set _selRooms($xy) {}
		    $_selMap drawSelection $aX $aY
		}
	    }
	    u {
		set xy [TCoordinate::GETXY $aX $aY]
		if {[info exists _selRooms($xy)]} {
		    unset _selRooms($xy) 
		    $_selMap undrawSelection $aX $aY
		}
	    }
	    t {
		set xy [TCoordinate::GETXY $aX $aY]
		if {[info exists _selRooms($xy)]} {
		    unset _selRooms($xy) 
		    $_selMap undrawSelection $aX $aY
		} else {
		    set _selRooms($xy) {}
		    $_selMap drawSelection $aX $aY
		}
	    }
	    sr {
		_selAdjustLoop         $aX $aY
		for {set x $_selsx} {$x<$_selex} {incr x} {
		    for {set y $_selsy} {$y<$_seley} {incr y} {
			set xy [TCoordinate::GETXY $x $y]

			if {![info exists _selRooms($xy)]} {
			    set _selRooms($xy) {}
			    $_selMap drawSelection $x $y
			}
		    }
		}	    			     
	    }
	    ur {
		_selAdjustLoop         $aX $aY
		for {set x $_selsx} {$x<$_selex} {incr x} {
		    for {set y $_selsy} {$y<$_seley} {incr y} {
			set xy [TCoordinate::GETXY $x $y]

			if {[info exists _selRooms($xy)]} {
			    unset _selRooms($xy) 
			    $_selMap undrawSelection $x $y
			}
		    }
		}	    			     
	    }
	    tr {
		_selAdjustLoop         $aX $aY
		for {set x $_selsx} {$x<$_selex} {incr x} {
		    for {set y $_selsy} {$y<$_seley} {incr y} {
			if {$x==$aX && $y==$aY} { continue }
			set xy [TCoordinate::GETXY $x $y]

			if {[info exists _selRooms($xy)]} {
			    unset _selRooms($xy) 
			    $_selMap undrawSelection $x $y
			} else {
			    set _selRooms($xy) {}
			    $_selMap drawSelection $x $y
			}
		    }
		}		
	    }
	}
    }


    # --- namespace end    
}

# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: TMMSelManager.tcl,v $
# Revision 1.1  2002/02/11 20:24:32  issever
# cut/copy/paste for maps, drag&drop, code cleanup of mapmanager
#
