# $Id: TMap.tcl,v 1.10 2002/04/28 14:30:56 issever Exp $

# ==========================================================
# TMap
#
# --- This is the data structure for mentaining maps. TMap 
#     will also create and manage the canvas, in which the
#     map is drawn.
#
# --- Maps are bound to muds, but managed by a mapmanager.
#     Each map can only be bound to one mud and will never
#     loose this binding. As long as the session of the
#     corresponding mud is not active maps dont need a 
#     mapmanager. As soon the session gets active each map
#     must know its mapmanager, as it has to signal back
#     changes etc. The mapmanager is the interface to the
#     user _and_ for many smm-map calls.
#
# --- The index-pair (name,level) for maps is unique in each
#     mud and thus unique with in the mapmanager. This is
#     the way how the user can definetly identify a map.
#     However technically maps are identified by an unique
#     id.
#
# --- As loading maps can take quite some time and not all
#     map of a map might be needed for the session, maps
#     are not loaded at the start of the session. They are
#     loaded, then they are accessed the first time.
#
# ==========================================================




# ==========================================================
# === TMap
# ==========================================================

class TMap {

    # ======================================================
    # === static class members
    # ======================================================

    # ------------------------------------------------------
    # --- calculate map coordinates into pixels and vice 
    #     versa.
    # ------------------------------------------------------
    private common _grid  32
    # _grid defines the spacing of the grid in pixels. Maps
    # are drawn on a grid. The grid spacing is in x and y
    # the same.
    # ------------------------------------------------------
    # These procedures take a coordinate (in pixel or 
    # map-grid) and a direction (1 for x, -1 for y) as
    # arguments.
    proc m2p {x d} 
    # This procedure takes mapcoordinates and returns the 
    # center pixel of the corresponding map grid.
    proc m2pmin {x d} 
    # As above, but returns the lower edge pixel.
    proc m2pmax {x d} 
    # As above, but returns the upper edge pixel.
    proc p2m {x d} 
    # This procedure takes a pixel coordinate and calculates
    # it into map coordinates. The returned arguments is
    # a floating point number!
    proc p2mi {x d} 
    # The procedures takes a pixel coordinate and returns
    # the corresponding map coordinate.

    # ------------------------------------------------------
    # --- namel
    # ------------------------------------------------------
    proc getNamel   {aName aLevel}
    # getNamel takes a mud-name and -level and returns the
    # unique "namel". The namel can be used to extract the
    # mud-name and -level.
    # ------------------------------------------------------
    proc namelname  {aNamel}
    # This procedure takes a namel and returns the name,
    # which has been used to create the namel.
    # ------------------------------------------------------
    proc namellevel {aNamel}
    # This procedure takes a namel and returns the level,
    # which has been used to create the namel.
    # ------------------------------------------------------

    # ------------------------------------------------------
    # --- misc
    # ------------------------------------------------------
    proc returnMapNameLevelFromFile {aFile}
    proc getMapNLIFromFile {aFile}





    # ======================================================
    # --- con- and destructor
    #     arguments 1: mapname, 2: maplevel
    #     rem: i am not sure about the arguments anymore
    # ------------------------------------------------------
    constructor {args} {}
    destructor {}
    # ------------------------------------------------------
    private variable CName      
    private variable CLevel     
    private variable CId        
    # Name, level and the unique id of the map
    # ------------------------------------------------------
    private variable CMapM      
    # the Map manager, that needs to be notified of changes
    private variable CParentWid 
    # widget ID of the parent
    private variable CCanvasWid 
    # widget ID of the scrolled canvas
    private variable CCBWid     
    # widget ID of the canvas
    # ------------------------------------------------------

    private variable CMud

    private variable CRoomList  ;# this is an array of objects
    private variable CChanged

    variable CZoomX
    variable CZoomY
    variable CPlayerX
    variable CPlayerY
    variable CHasZoom
    variable CHasPlayer





    # ======================================================
    # --- map size
    # --- the default mapsize is 101x101 from -50 to +50
    #     in x and y
    # ------------------------------------------------------
    private variable CXmin "-50"
    private variable CXmax "50"
    private variable CYmin "-50"
    private variable CYmax "50"
    # ------------------------------------------------------
    # getters
    method xmin {} { return $CXmin }
    method xmax {} { return $CXmax }
    method ymin {} { return $CYmin }
    method ymax {} { return $CYmax }
    method size {} { return [list $CXmin $CXmax $CYmin $CYmax]}
    # ------------------------------------------------------
    # setters
    # Resizing involves more than just setting x and y. E.g.
    # the map must be redrawn-
    method resize {a b c d}


    # ======================================================
    # --- building maps
    # ------------------------------------------------------
    method add {aRoom}     
    # Adds a room to the Map. A copy of the room aRoom is
    # added to the map.
    # ------------------------------------------------------
    method put {aRoom}     
    # Adds a room to the Map. The room aRoom itself is
    # added to the map. So do not delete the room object
    # after adding it to the map, as the map is now 
    # responsible for its management.
    # ------------------------------------------------------
    method = {source}      
    # Copies the contents of another map into this one.
    # Each room from the other map is copied and added to
    # the map.
    # ------------------------------------------------------
    method remRoom   {aRoom} 
    method remRoomAt {aXY} 
    # Removes the room from the map
    # ------------------------------------------------------
    method rem {aXY}       
    method remAt {aX aY}       
    # Removes the room with the coresponding coordinates


    method reDraw {aRoom} 




    private variable CUnLoaded 0
    private variable CBGIM


    method powerWalkTo {a b} 

    method getCanvas {} { return $CCanvasWid }

    method clickCon {aCursor} 
    method clickConClick {a b} 
    method clickDraw {aCursor} 
    method clickDrawClick {a b} 
    method defaultBindings {} 

    method touch {}       ;# sign the map as 'changed'
    method untouch {}     ;# sign the map as 'hasn't changed'
    method hasChanged {}  ;# Q: if the map has changed


    method exists {aXY}   ;# Q: if the room exists

    method getXYs {}      ;# R: return the coordinates of all rooms
    method getRoom {aXY}  ;# RO: return the room with the corresponding coordinate

    method == {source}    ;# test for equality
    method != {source}    ;# test for not equality

    method show {}
    method hide {}
    method createCanvas {}
    private method removeCanvas {}

    method remMapM {aMapM} 


    method namel {}       ;# returns a string that combines level and name
    method name {} { return $CName }
    method level {} { return $CLevel }
    method id {}    { return $CId }

    private variable _xyCursorExists 0
    method createXYCursor {} {
	$CCanvasWid delete xycursor	
        $CCanvasWid create text 0 0 \
	    -tag xycursor
	set _xyCursorExists 1
    }
    method removeXYCursor {} {
	$CCanvasWid delete xycursor
	$CMapM dragNDropAbort
	set _xyCursorExists 0
    }
    method updateXYCursor {a b} {
	set x    [$CCBWid canvasx $a]
	set y    [$CCBWid canvasy $b]
	set xval [p2mi $x  1]
	set yval [p2mi $y -1]
    
	if {$xval<$CXmin || $xval>$CXmax || $yval<$CYmin || $yval>$CYmax} {
	    removeXYCursor
	    return
	}
	if {[expr !$_xyCursorExists]} { createXYCursor }

	$CCanvasWid coords        xycursor [expr {$x+16}] [expr {$y-8}]
	$CCanvasWid itemconfigure xycursor -text "$xval,$yval"
    }


    method dragNDropMid {a b}
    method dragNDropEnd {a b}

    method xminpix {} { return [m2pmin $CXmin  1]}
    method xmaxpix {} { return [m2pmax $CXmax  1]}
    method yminpix {} { return [m2pmax $CYmax -1]}
    method ymaxpix {} { return [m2pmin $CYmin -1]}

    method canvasWidth  {} { return [expr {[xmaxpix] - [xminpix]}] }
    method canvasHeight {} { return [expr {[ymaxpix] - [yminpix]}] }

    method newZoomPos {a b} 
    method newPlayerPos {a b} 

    method mapSelect {act a b} 
    method unselAll {} 


    method drawGrid {}
    method drawZoomPos {aX aY}
    method drawPlayerPos {aX aY}
    method drawSelection {aX aY} 
    method undrawSelection {aX aY} 
    method drawClip {aX aY} 
    method undrawClip {aX aY} 
    method undrawAllClip {} 
    method remZoomPos {}
    method remPlayerPos {}

    method adjustViewPoint {aX aY}


    method getConPos {from to} {
	if {[exists [$to getXY]]} {
	    return [[getRoom [$to getXY]] getExitPosTo $from]
	} else {
	    return cc
	}
    }

    method export {aFileName} 
    method save {} 
    method saveMain {fid} 
    method load {} 
    method loadRooms {fid version} 
    method import {aFileName} 
    method loadIfNeeded {}
    method unload {}
}






















# ==========================================================
# === class code
# ==========================================================

# ----------------------------------------------------------
# --- constructor
# ----------------------------------------------------------
body TMap::constructor {args} {
    switch [llength $args] {
	5 {
	    # this is called for new created maps?
	    set CZoomX     ""
	    set CZoomY     ""
	    set CPlayerX   ""
	    set CPlayerY   ""
	    set CHasZoom   0
	    set CHasPlayer 0

	    set CName      [lindex $args 0]
	    set CLevel     [lindex $args 1]
	    set CId        [lindex $args 2]
	    set CMud       [lindex $args 3]
	    set CUnLoaded  [lindex $args 4]
	    
	    set CMapM [list]
	    untouch
	}
	7 {
	    set sw [lindex $args 0]
	    switch -- $sw {
		"import" {
		    set CZoomX     ""
		    set CZoomY     ""
		    set CPlayerX   ""
		    set CPlayerY   ""
		    set CHasZoom   0
		    set CHasPlayer 0

		    set fn         [lindex $args 1]
		    set CName      [lindex $args 2]
		    set CLevel     [lindex $args 3]
		    set CId        [lindex $args 4]
		    set CMud       [lindex $args 5]	    
		    set CMapM      [lindex $args 6]

		    set CUnLoaded  0
		    import $fn
		    touch
		}
		default {
		    error "Wrong numer of arguments to constructor! BP2"
		}
	    }
	}
	default {
	    error "Wrong numer of arguments to constructor!"
	}
	
    }	
}

body TMap::destructor {} {
    foreach pos [TMap::getXYs] {
	::delete object [TMap::getRoom $pos]
    }	
    removeCanvas
}






# ----------------------------------------------------------
# --- other
# ----------------------------------------------------------

body TMap::m2p    {x d} { return [expr {($d*$x*$_grid)+$_grid/2}] }
body TMap::m2pmin {x d} { return [expr {($d*($x-0.5)*$_grid)+$_grid/2}] }
body TMap::m2pmax {x d} { return [expr {($d*($x+0.5)*$_grid)+$_grid/2}] }
body TMap::p2m    {x d} { return [expr {$d*($x-($_grid/2))/$_grid}] }
body TMap::p2mi   {x d} { return [expr {round($d*($x-($_grid/2))/$_grid)}] }


body TMap::powerWalkTo {a b} {
    if {[$CMapM roomHasChanged]} {
	if {[$CMapM devMode]} {
	    $CMapM acceptRoom
	} else {
	    $CMapM informRoomHasChanged
	    return
	}
    }
    
    set x [p2mi [$CCBWid canvasx $a]  1]
    set y [p2mi [$CCBWid canvasy $b] -1]
    $CMapM powerWalkTo $this $x $y
}

body TMap::export {aFileName} {
    set       fid [open $aFileName "w+"]
    saveMain $fid
    close    $fid
}
body TMap::save {} {
    set       fid [open [$CMud getMapFile $CId] "w+"]
    saveMain $fid
    close    $fid
    untouch
}
body TMap::saveMain {fid} {
    # do saving here
    puts $fid "2" ; # version
    puts $fid $CId
    puts $fid $CName
    puts $fid $CLevel
    puts $fid $CXmin
    puts $fid $CXmax
    puts $fid $CYmin
    puts $fid $CYmax

    foreach rpos [getXYs] {
	[getRoom $rpos] save $fid
    }
}


body TMap::getMapNLIFromFile {aFile} {
    set fid [open $aFile "r"]
    gets $fid version
    gets $fid id
    gets $fid name
    gets $fid level
    close $fid    
    return [list $name $level $id]
}
body TMap::loadIfNeeded {} {
    if {$CUnLoaded} {
	load
	set CUnLoaded 0
    }
}
body TMap::returnMapNameLevelFromFile {aFile} {
    ## map version changes affect here too!!!!!
    set fid [open $aFile "r"]
    # 
    gets $fid version
    gets $fid id
    gets $fid name
    gets $fid level

    close $fid

    return [list $name $level]
}
body TMap::load {} {
    set fid [open [$CMud getMapFile $CId] "r"]
    # do loading here
    gets $fid version
    gets $fid CId
    gets $fid CName
    gets $fid CLevel
    gets $fid CXmin
    gets $fid CXmax
    gets $fid CYmin
    gets $fid CYmax

    loadRooms $fid $version
    
    close $fid
}
body TMap::import {aFileName} {
    set fid [open $aFileName "r"]
    # do loading here
    gets $fid version
    gets $fid fileId
    gets $fid fileName
    gets $fid fileLevel
    gets $fid CXmin
    gets $fid CXmax
    gets $fid CYmin
    gets $fid CYmax

    loadRooms $fid $version
    
    close $fid
}
body TMap::loadRooms {fid version} {
    switch -- $version {
	1 {
	    createCanvas	    
	    while {[gets $fid line]>0} {
		set room [TRoom ::\#auto]
		$room load_V1ConvertToV2 $line [[$CMud getMapM] getBGIM]
		put $room
	    } 
	    touch
	}
	2 {
	    createCanvas	    
	    while {[gets $fid line]>0} {
		set room [TRoom ::\#auto]
		$room load $line [[$CMud getMapM] getBGIM]
		put $room
	    } 
	}
	default {
	    error "unknown version for map: [namel]"
	}
    }
}



body TMap::namelname {aNamel} {
    if {[regexp -- {(.+); Level: ([\+|\-])(\ *)([0-9]+)$} "$aNamel" all name sign sp val]} {
	return $name
    } else {
	return $aNamel
    }
}
body TMap::namellevel {aNamel} {
    if {[regexp -- {(.+); Level: ([\+|\-])(\ *)([0-9]+)$} "$aNamel" all name sign sp val]} {
	if {[string match "+" $sign]} {
	    return [expr { 1*$val}]
	} else {
	    return [expr {-1*$val}]
	}
    } else {
	return 0
    }
}
body TMap::namel {} {
    if {$CLevel==0} {
	return $CName
    } else {
	if {$CLevel>0} {
	    set level [format "%3d" $CLevel]
	    return "$CName; Level: +$level"
	} else {
	    set level [format "%3d" [expr {-1*$CLevel}]]
	    return "$CName; Level: -$level"
	}
    }
}
body TMap::getNamel {aName aLevel} {
    if {$aLevel==0} {
	return $aName
    } else {
	if {$aLevel>0} {
	    set level [format "%3d" $aLevel]
	    return "$aName; Level: +$level"
	} else {
	    set level [format "%3d" [expr {-1*$aLevel}]]
	    return "$aName; Level: -$level"
	}
    }
}
body TMap::drawZoomPos {aX aY} {
    remZoomPos
    
    $CCanvasWid create rectangle \
	[expr {[m2pmin $aX 1]-0}] [expr {[m2pmin $aY -1]+0}] \
	[expr {[m2pmax $aX 1]+0}] [expr {[m2pmax $aY -1]-0}] \
	-outline red -width 3 -tag zoompos
    $CCanvasWid lower zoompos grid
    set CHasZoom 1
    set CZoomX   $aX
    set CZoomY   $aY
}

body TMap::drawSelection {aX aY} {
    $CCanvasWid create rectangle \
	[expr {[m2pmin $aX 1]+5}] [expr {[m2pmin $aY -1]-5}] \
	[expr {[m2pmax $aX 1]-5}] [expr {[m2pmax $aY -1]+5}] \
	-outline yellow -width 2 -tag "sel sel$aX,$aY"
    #$CCanvasWid lower "sel$aX,$aY" grid
}
body TMap::undrawSelection {aX aY} {
    $CCanvasWid delete "sel$aX,$aY"
}
body TMap::drawClip {aX aY} {
    $CCanvasWid create rectangle \
	[expr {[m2pmin $aX 1]+7}] [expr {[m2pmin $aY -1]-7}] \
	[expr {[m2pmax $aX 1]-7}] [expr {[m2pmax $aY -1]+7}] \
	-outline orange -width 2 -tag "clip clip$aX,$aY"
    #$CCanvasWid lower "sel$aX,$aY" grid
}
body TMap::undrawClip {aX aY} {
    $CCanvasWid delete "clip$aX,$aY"
}
body TMap::undrawAllClip {} {
    $CCanvasWid delete "clip"
}
body TMap::unselAll {} {
    $CCanvasWid delete sel
}

body TMap::drawPlayerPos {aX aY} {
    remPlayerPos
    
    $CCanvasWid create rectangle \
	[expr {[m2pmin $aX 1]+1}] [expr {[m2pmin $aY -1]-1}] \
	[expr {[m2pmax $aX 1]-1}] [expr {[m2pmax $aY -1]+1}] \
	-outline blue -width 3 -tag playerpos
    catch "$CCanvasWid lower playerpos zoompos"

    adjustViewPoint $aX $aY

    set CHasPlayer 1
    set CPlayerX   $aX
    set CPlayerY   $aY
}
body TMap::adjustViewPoint {aX aY} {
    set xp [winfo width  $CCBWid]
    set yp [winfo height $CCBWid]
    set Xp [expr {double($CXmax - $CXmin)*$_grid}]
    set Yp [expr {double($CYmax - $CYmin)*$_grid}]
    set xa [expr {double($aX - $CXmin)*$_grid}]
    set ya [expr {double($CYmax - $aY)*$_grid}]
    set xf [expr {($xa - $xp/2)/$Xp}]
    set yf [expr {($ya - $yp/2)/$Yp}]

    $CCBWid xview moveto $xf
    $CCBWid yview moveto $yf
}
body TMap::remZoomPos {} {
    $CCanvasWid delete zoompos
    set CHasZoom 0    
}
body TMap::remPlayerPos {} {
    ::catch "$CCanvasWid delete playerpos"
    set CHasPlayer 0
}
body TMap::drawGrid {} {
    
    $CCanvasWid delete grid
    
    #draw horizontal lines in x direction
    set start [expr {$CYmin}]
    set ende  [expr {$CYmax+2}]
    for {set i $start} {$i<$ende} {incr i 1} {
        $CCanvasWid create line [xminpix] [m2pmin $i -1] [xmaxpix] [m2pmin $i -1] \
	    -fill grey70 -tag grid
    }
    #draw vertical lines in y direction
    set start [expr {$CXmin}]
    set ende  [expr {$CXmax +2}]
    for {set i $start} {$i < $ende} {incr i 1} {
        $CCanvasWid create line [m2pmin $i 1] [yminpix] [m2pmin $i 1] [ymaxpix] \
	    -fill grey70 -tag grid
    }
    $CCanvasWid lower grid
}

body TMap::mapSelect {act a b} {
    set x [p2mi [$CCBWid canvasx $a]  1]
    set y [p2mi [$CCBWid canvasy $b] -1]
    $CMapM mapSelect $this $act $x $y
}
body TMap::dragNDropMid {a b} {
    set x [p2mi [$CCBWid canvasx $a]  1]
    set y [p2mi [$CCBWid canvasy $b] -1]
    $CMapM dragNDropMid $x $y
}
body TMap::dragNDropEnd {a b} {
    set x [p2mi [$CCBWid canvasx $a]  1]
    set y [p2mi [$CCBWid canvasy $b] -1]
    $CMapM dragNDropEnd $x $y
}
body TMap::newZoomPos {a b} {
    if {[$CMapM roomHasChanged]} {
	if {[$CMapM devMode]} {
	    $CMapM acceptRoom
	} else {
	    $CMapM informRoomHasChanged
	    return
	}
    }

    set x [p2mi [$CCBWid canvasx $a]  1]
    set y [p2mi [$CCBWid canvasy $b] -1]
    $CMapM newZoomPos $this $x $y
    drawZoomPos $x $y
    $CMapM dragNDropStart $this $x $y
}
body TMap::newPlayerPos {a b} {
    if {[$CMapM roomHasChanged]} {
	if {[$CMapM devMode]} {
	    $CMapM acceptRoom
	} else {
	    $CMapM informRoomHasChanged
	    return
	}
    }

    set x [p2mi [$CCBWid canvasx $a]  1]
    set y [p2mi [$CCBWid canvasy $b] -1]
    $CMapM newPlayerPos $this $x $y
    drawPlayerPos $x $y

}
body TMap::remMapM {aMapM} {
    set CMapM ""
    TMap::removeCanvas
    TMap::unload
}
body TMap::resize {a b c d} {
    set CXmin $a
    set CXmax $b
    set CYmin $c
    set CYmax $d

    TMap::save
    TMap::removeCanvas
    TMap::unload

    TMap::loadIfNeeded
    $CMapM newActMap $this

    if {$CHasZoom} {
	drawZoomPos $CZoomX $CZoomY
    }
    if {$CHasPlayer} {
	drawPlayerPos $CPlayerX $CPlayerY
    }

}
body TMap::unload {} {
    if {!$CUnLoaded} {
	foreach xy [TMap::getXYs] {
	    TMap::rem $xy
	}
	set CUnLoaded 1
	untouch
    }
}
body TMap::removeCanvas {} {
    if {[info exists CCanvasWid]} {
	destroy $CCanvasWid
	unset CCanvasWid
    }
}
body TMap::createCanvas {} {
    if {[info exists CCanvasWid]} {
	return
    }
    set CMapM      [$CMud getMapM]
    set CBGIM      [$CMapM getBGIM]
    set CParentWid [$CMapM getMapFrame]
    set CCanvasWid "$CParentWid.mapc$CId"

    iwidgets::scrolledcanvas $CCanvasWid \
	-relief ridge -sbwidth 10 \
	-background $TColor::BGdef -textbackground $TColor::BGdef -selectbackground $TColor::BGdef
    set CCBWid [$CCanvasWid childsite]
    
    TMap::drawGrid
    
    # cursor
    bind    $CCBWid <Enter>  [code $this createXYCursor]
    bind    $CCBWid <Leave>  [code $this removeXYCursor]
    bind    $CCBWid <Motion> [code $this updateXYCursor %x %y]
    bind    $CCBWid <<DragNDroping>> [code $this dragNDropMid %x %y]
    bind    $CCBWid <<DragNDropEnd>> [code $this dragNDropEnd %x %y]
    # selections 
    bind    $CCBWid <<MapSelect>>               "[code $this mapSelect s  %x %y]"
    bind    $CCBWid <<MapSelectRegion>>         "[code $this mapSelect sr %x %y]"
    bind    $CCBWid <<MapUnSelect>>             "[code $this mapSelect u  %x %y]"
    bind    $CCBWid <<MapUnSelectRegion>>       "[code $this mapSelect ur %x %y]"
    bind    $CCBWid <<MapToggleSelect>>         "[code $this mapSelect t  %x %y]"
    bind    $CCBWid <<MapToggleSelectRegion>>   "[code $this mapSelect tr %x %y]"
    # primary actions
    bind    $CCBWid <<MapPrimary>>     "[code $this newZoomPos %x %y]"
    bind    $CCBWid <<MapPlacePlayer>> "[code $this newPlayerPos %x %y]"
    bind    $CCBWid <<MapPowerwalk>>   "[code $this powerWalkTo %x %y]"
}
body TMap::clickCon {aCursor} {
    $CCBWid config -cursor $aCursor
    bind    $CCBWid <<MapPrimary>> "[code $this clickConClick %x %y]"
}
body TMap::clickConClick {a b} {
    set x [p2mi [$CCBWid canvasx $a]  1]
    set y [p2mi [$CCBWid canvasy $b] -1]
    $CMapM clickConClick $x $y
    defaultBindings
}
body TMap::clickDraw {aCursor} {
    $CCBWid config -cursor $aCursor
    bind    $CCBWid <<MapPrimary>> "[code $this clickDrawClick %x %y]"
}
body TMap::clickDrawClick {a b} {
    set x [p2mi [$CCBWid canvasx $a]  1]
    set y [p2mi [$CCBWid canvasy $b] -1]
    $CMapM clickDrawClick $x $y
    defaultBindings
}
body TMap::defaultBindings {} {
    $CCBWid config -cursor {}
    bind    $CCBWid <<MapPrimary>> "[code $this newZoomPos %x %y]"
}
body TMap::touch {} {
    set CChanged 1
    foreach mm $CMapM {
	$CMapM mapChanged $this
    }
}
body TMap::untouch {} {
    set CChanged 0
    foreach mm $CMapM {
	$CMapM mapUnchanged $this
    }
}
body TMap::hasChanged {} {return $CChanged}
body TMap::reDraw {aRoom} {
    $aRoom drawRem $CCanvasWid $this
    $aRoom draw $CCanvasWid $this    
}
body TMap::add {aRoom} {
    # will get their touch with calling 'rem'
    set xy [$aRoom getXY]
    TMap::rem $xy
    set CRoomList($xy) [TRoom ::\#auto $aRoom]
    $aRoom draw $CCanvasWid $this
}
body TMap::put {aRoom} {
    # will get their touch with calling 'rem'
    set xy [$aRoom getXY]
    TMap::rem $xy
    set CRoomList($xy) $aRoom
    $aRoom draw $CCanvasWid $this
}
body TMap::remRoomAt {aXY} {
    # will get their touch with calling 'rem'
    if {[TMap::exists $aXY]} {	
	remRoom $CRoomList($aXY)
    }
}
body TMap::remRoom {aRoom} {
    # will get their touch with calling 'rem'

    set xy [$aRoom getXY]
    set x  [$aRoom getX]
    set y  [$aRoom getY]
    TMap::rem $xy
    set CRoomList($xy) [TRoom ::\#auto]
    $CRoomList($xy) setXY $x $y
    
    $CRoomList($xy) drawRem $CCanvasWid $this
    TMap::rem $xy
}
body TMap::remAt {aX aY} {
    rem [TCoordinate::GETXY $aX $aY]
}
body TMap::rem {aXY} {
    if {[TMap::exists $aXY]} {
	::delete object $CRoomList($aXY)
	unset CRoomList($aXY)
    }
    if {!$CUnLoaded} {
	# only mark as touched, if map is not first newly loaded!
	touch
    }
}
body TMap::exists {aXY} {
    return [info exists CRoomList($aXY)]
}
body TMap::getXYs {} {
    return [array names CRoomList]
}
body TMap::getRoom {aXY} {
    return $CRoomList($aXY)
}
body TMap::== {source} {
    if {[string match $CName [$source name]] && [string match $CLevel [$source level]]} {
	return 1
    } else {
	return 0
    }
}
body TMap::!= {source} {
    if {[string match $CName [$source name]] && [string match $CLevel [$source level]]} {
	return 0
    } else {
	return 1
    }
}
body TMap::= {source} {
    foreach xy [$source getXYs] {
	TMap::add [$source getRoom $xy]
    }
}

body TMap::show {} {
    createCanvas
    pack $CCanvasWid -fill both -expand true
}
body TMap::hide {} {
    pack forget $CCanvasWid
}


#################################################################################
class TNewMapUif {
    inherit iwidgets::Texteddialog

    itk_option define -command command Command "return 1" {}
    itk_option define -name    name    Name    "main" {}
    itk_option define -level   level   Level   "0" {}
    # the command given to the command option _must_ return a boolean!
    #  -> 0: uif stays
    #  -> 1: = success, uif goes

    constructor {args} {
	configure -foreground black
	addFGCom [code $this configure -foreground %c] 

	global TV
	set TV($this,level) 0
	set TV($this,name) ""

	configure -title "Create New Map"
	configure -modality application
	configure -text "Please, enter a name and a level for the map!"
	configure -aspect 333
	buttonconfigure OK     -text "Create" -command "[code $this oki]"
	buttonconfigure Cancel -text "Cancel" -command "[code $this canc]"

	#addFGCom [code $this buttonconfigure OK       -foreground %c] 
	#addFGCom [code $this buttonconfigure Cancel   -foreground %c] 


	#########################################################################
	itk_component add   name {
	    iwidgets::entryfield [childsite].name -labeltext "Name: " \
		-width 20 -labelmargin 0 -textvariable TV($this,name)
	} {
	    keep -background
	    keep -foreground
	}
	itk_component add   level {
	    iwidgets::entryfield [childsite].level -labeltext "Level: " -validate integer \
		-width 4 -labelmargin 0 -textvariable TV($this,level)
	} {
	    keep -background
	    keep -foreground
	}
	pack $itk_component(name) $itk_component(level) -side top -fill both -expand 1
	iwidgets::Labeledwidget::alignlabels $itk_component(name) $itk_component(level) 

	eval itk_initialize $args
	#launchUif
    } 
    private method canc {}        { 
	global TV
	unset TV($this,name)
	unset TV($this,level)
	deactivate
	#::delete object $this
    }
    private method oki {}        { 
	global TV

	set name [string trim $TV($this,name)]
	if {![string compare "" $name]} {
	    TInformer $itk_interior.mess \
		-information "You have to give a 'name' for the map!"	    
	    return
	} 
	scan $TV($this,level) "%d" level
	if {![info exists level]} {
	    TInformer $itk_interior.mess \
		-information "You have to give a 'level' for the map!"	    
	    return
	} 
	
	regsub -all {%n} $itk_option(-command) $name helper
	regsub -all {%l} $helper $level theCom
	if {[eval $theCom]} {
	    canc
	}
    }
    method launchUif {}   { clearUif ; activate }
    private method clearUif  {}   { 
	global TV
	set TV($this,name)  $itk_option(-name)
	set TV($this,level) $itk_option(-level) 
    }

}



#################################################################################
class TResizeMapUif {
    inherit iwidgets::Texteddialog

    itk_option define -command command Command "return 1" {}
    itk_option define -size    size    Size    "0 0 0 0"  {}
    # the command given to the command option _must_ return a boolean!
    #  -> 0: uif stays
    #  -> 1: = success, uif goes

    constructor { args } {} {
	global _TResizeMapUif
	configure -title "Resize Map"
	configure -modality application
	configure -text "Please, enter the new dimensions of the map.\nResizing will also save the map automatically!"
	configure -aspect 333
	buttonconfigure OK     -text "Resize" -command "[code $this oki]"
	buttonconfigure Cancel -text "Cancel" -command "[code $this canc]"

	#########################################################################
	foreach aaa {xmin xmax ymin ymax} {
	    itk_component   add $aaa { 
		iwidgets::spinint  [childsite].$aaa \
		    -labeltext "[string range $aaa 0 0] [string range $aaa 1 end]: " \
		    -step 5 -width 5 -validate integer \
		    -increment [code $this increm $aaa] -decrement [code $this decrem $aaa] \
		    -arroworient horizontal -textvariable _TResizeMapUif($this,$aaa)
	    } {keep -background}
	}

	pack $itk_component(xmin) $itk_component(xmax) $itk_component(ymin) $itk_component(ymax) \
	    -side top -fill both -expand 1
	iwidgets::Labeledwidget::alignlabels \
	    $itk_component(xmin) $itk_component(xmax) $itk_component(ymin) $itk_component(ymax) 


	eval itk_initialize $args

	set _TResizeMapUif($this,xmin) [lindex $itk_option(-size) 0]
	set _TResizeMapUif($this,xmax) [lindex $itk_option(-size) 1]
	set _TResizeMapUif($this,ymin) [lindex $itk_option(-size) 2]
	set _TResizeMapUif($this,ymax) [lindex $itk_option(-size) 3]

	activate 
    } 
    method increm {aaa} {
	global _TResizeMapUif
	if {![string compare "" $_TResizeMapUif($this,$aaa)]} {
	    set _TResizeMapUif($this,$aaa) 0
	}
	incr _TResizeMapUif($this,$aaa) 5
    }
    method decrem {aaa} {
	global _TResizeMapUif
	if {![string compare "" $_TResizeMapUif($this,$aaa)]} {
	    set _TResizeMapUif($this,$aaa) 0
	}
	incr _TResizeMapUif($this,$aaa) -5
    }
    private method canc {}        { 
	global _TResizeMapUif
	unset _TResizeMapUif($this,xmin)
	unset _TResizeMapUif($this,xmax)
	unset _TResizeMapUif($this,ymin)
	unset _TResizeMapUif($this,ymax)
	::delete object $this
    }
    private method oki {}        { 
	global _TResizeMapUif
	set theCom $itk_option(-command)
	foreach aaa {xmin xmax ymin ymax} {
	    if {![string compare "" $_TResizeMapUif($this,$aaa)]} {
		set _TResizeMapUif($this,$aaa) 0
	    }
	}
	
	regsub -all {%xmin} $theCom $_TResizeMapUif($this,xmin) theCom
	regsub -all {%xmax} $theCom $_TResizeMapUif($this,xmax) theCom
	regsub -all {%ymin} $theCom $_TResizeMapUif($this,ymin) theCom
	regsub -all {%ymax} $theCom $_TResizeMapUif($this,ymax) theCom
	if {[eval $theCom]} {
	    canc
	}
    }


}












# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: TMap.tcl,v $
# Revision 1.10  2002/04/28 14:30:56  issever
# minor enchancements of the xy-cursor
#
# Revision 1.9  2002/04/06 22:49:38  issever
# drag&drop bound to cut/copy/paste (in maps).
# This allows to d&d more then a single room at once.
#
# Revision 1.8  2002/04/06 01:31:16  issever
# debugged cut/copy/paste in maps. In fact it was quite a redo
# of the code, but now it works fine.
# Area to be pasted will be indicated.
#
# Revision 1.7  2002/04/05 20:35:27  issever
# splited the file TExit.tcl
# removed a small bug
#
# Revision 1.6  2002/04/04 21:07:03  issever
# Optimized the expr's, which are used in the mapping code.
# Splited the File TEXit.tcl.
#
# Revision 1.5  2002/02/11 20:24:32  issever
# cut/copy/paste for maps, drag&drop, code cleanup of mapmanager
#
# Revision 1.4  2001/11/25 20:30:33  issever
# Removed the -foreground bug.
# This was a quite widespread bug and I hope I caught
# every appereance. If I've missed one, please notify
# us again.
#
# Revision 1.3  2001/09/24 15:02:46  moth
# got rid of black in "hand2 black" as it crashes smm under windows
#
# Revision 1.2  2001/08/19 12:25:42  issever
# Added the cvs keywords Id at start of the file
# and Log at the end of the file
#
#
