# $Id: TMapManager.tcl,v 1.9 2002/04/28 11:13:35 issever Exp $

# ==========================================================
# TMapManager
#     includes the Room- or Zoom-Interface (ZIF)
#
# --- The TMapManager manages all the maps of an active 
#     session. As maps are bound to muds, the TMapManager 
#     manages the maps of the mud of the active session.
#
# --- Besides that the TMapManager provides the UIF, with 
#     the user can interact with maps. The User always
#     interacts indirectly with maps over the TMapManager.
#
# --- The important userinterface ZIF is part of the 
#     TMapManager. 
#     (It is the graphical part: inherit itk::Widget)
# ==========================================================

class TMapManager {
    inherit itk::Widget \
	TMapManager::TDevMode \
	TMapManager::TClipboard \
	TMapManager::TCommandLineIF 

    private {
	# --------------------------------------------------
	variable _bgIM
	# This is the pointer to the imange manager
	# --------------------------------------------------
	variable _mud
	# Type: TMud-pointer.
	# This is the pointer to the mud, whose maps the
	# mapmanager manages. We need the corresponding 
	# mud, because we need to know where to save all
	# the map e.g..
	#
	# If no session is active and thus the mapmanager
	# is not connected to a mud this variable is unset.
	# This can be queried via the method 
	# 'mappingStarted'
	# --------------------------------------------------
	variable _playerPos
	# Type: TPosition-pointer
	# The position SMM thinks the player is in the mud.
	# --------------------------------------------------
	variable _zoomPos
	# Type: TPosition-pointer
	# The position of the zoom.
	# --------------------------------------------------
	variable _ses
	# The session the mapmanager helps to manage its
	# maps.
	#
	# If no session is active and thus the mapmanager
	# is not connected to a session this variable is
	# unset.
	# This can be queried via the method 
	# 'mappingStarted'
	# --------------------------------------------------

	# ==================================================
	# methods
	# ==================================================

	# --------------------------------------------------
	method existsRoomAtPlayer {} 
	# Checks, if a room at the player position exists.
	# --------------------------------------------------
	method getRoomAtPlayer {}
	# Returns the room at the player position.
	# --------------------------------------------------
    }


    # ------------------------------------------------------
    # --- con-and destructor
    # ------------------------------------------------------
    constructor {aBGIM args} {} {}
    destructor  {}
    # ------------------------------------------------------

    # ------------------------------------------------------
    # --- powerwalking
    # ------------------------------------------------------
    method powerWalkTo {aMap aX aY}
    # powerWalkTo contains the powerwalk algorithm 
    # and will be invoked from the map, which calls 
    # the powerwalk.
    # ------------------------------------------------------
    method movePlayer {aHow aWhere}
    # Moves the player on the map by using the ZIF from
    # the commandline. Thismethod should be in 
    # TCommandLineIF.
    # ------------------------------------------------------

    # ------------------------------------------------------
    # --- user map queries
    # ------------------------------------------------------
    method getExitsList {aMap aLevel aX aY} {
	[getMap $aMap $aLevel] loadIfNeeded
	return [[getRoom $aMap $aLevel $aX $aY] TExitList::getPoss]}
    method getExitDest {aMap aLevel aX aY aPos} {
	[getMap $aMap $aLevel] loadIfNeeded
	set exit [[getRoom $aMap $aLevel $aX $aY] TExitList::getExit $aPos]
	set map [$exit getMap]
	if {"" == $map} {set map $aMap}
	set lev [$exit getMapLevel]
	if {"" == $lev} {set lev $aLevel}
	set to [$exit getConnectTo]
	return [list $map $lev [$to getX] [$to getY]]
    }
    method getExitCommand {aMap aLevel aX aY aPos} {
	[getMap $aMap $aLevel] loadIfNeeded
	return [[[getRoom $aMap $aLevel $aX $aY] TExitList::getExit $aPos] getCommand]
    }
    method getExitDoor {aMap aLevel aX aY aPos} {
	[getMap $aMap $aLevel] loadIfNeeded
	return [[[getRoom $aMap $aLevel $aX $aY] TExitList::getExit $aPos] getDoor]
    }
    method getExitDoorOpen {aMap aLevel aX aY aPos} {
	[getMap $aMap $aLevel] loadIfNeeded
	set exit [[getRoom $aMap $aLevel $aX $aY] TExitList::getExit $aPos]
	if {[$exit getDoor]} {
	    return [$exit getDoorOpen]
	}
    }
    method getExitDoorClose {aMap aLevel aX aY aPos} {
	[getMap $aMap $aLevel] loadIfNeeded
	set exit [[getRoom $aMap $aLevel $aX $aY] TExitList::getExit $aPos]
	if {[$exit getDoor]} {
	    return [$exit getDoorClose]
	}
    }
    method getExitColor {aMap aLevel aX aY aPos} {
	if {[regexp {^[ud]} $aPos]} {
	    return $aPos
	} {
	    [getMap $aMap $aLevel] loadIfNeeded
	    return [[[getRoom $aMap $aLevel $aX $aY] TExitList::getExit $aPos] getConColor]
	}
    }
    method getExitThickness {aMap aLevel aX aY aPos} {
	if {[regexp {^[ud]} $aPos]} {
	    return 7
	} {
	    [getMap $aMap $aLevel] loadIfNeeded
	    return [[[getRoom $aMap $aLevel $aX $aY] TExitList::getExit $aPos] getConThickness]
	}
    }
    method getExitDrawTo {aMap aLevel aX aY aPos} {
	if {[regexp {^[ud]} $aPos]} {
	    return [list $x $y]
	} {
	    [getMap $aMap $aLevel] loadIfNeeded
	    set draw [[[getRoom $aMap $aLevel $aX $aY] TExitList::getExit $aPos] getDrawTo]
	    return [list [$draw getX] [$draw getY]]
	}
    }
    method getText {aMap aLevel aX aY} {
	[getMap $aMap $aLevel] loadIfNeeded
	return [[[getRoom $aMap $aLevel $aX $aY] text] text]
    }
    method getBg {aMap aLevel aX aY} {
	[getMap $aMap $aLevel] loadIfNeeded
	return [[[getRoom $aMap $aLevel $aX $aY] bg] streamout]
    }
    method getMapNames {} {return [$_mud getMapNames]}
    method getRooms {aMap aLevel} {
	set map [getMap $aMap $aLevel]
	$map loadIfNeeded
	set xys [$map getXYs]
	set rooms {}
	foreach xy $xys {
	    regexp {^(.*),(.*)$} $xy ignore x y
	    lappend rooms [list $aMap $aLevel $x $y]
	}
	return $rooms
    }
    method getXYs {aMap aLevel} {
	set map [getMap $aMap $aLevel]
	$map loadIfNeeded
	set xys [$map getXYs]
	set rooms {}
	foreach xy $xys {
	    regexp {^(.*),(.*)$} $xy ignore x y
	    lappend rooms [list $x $y]
	}
	return $rooms
    }
    # ------------------------------------------------------



    # ------------------------------------------------------
    # --- GUI stuff
    # ------------------------------------------------------
    # widgetID in which the maps should be drawn
    method getMapFrame {}
    # ------------------------------------------------------
    method getLayout {}
    # returns a string, that describes the current layout
    # of the mapmanager.
    # ------------------------------------------------------
    method setLayout {aLayout}
    # gets the layout string from getLayout and adjusts the 
    # mapmanager to this layout.
    # ------------------------------------------------------


    # ------------------------------------------------------
    # --- map handling
    # ------------------------------------------------------
    private {
	# the actual shown map: object pointer
	variable _actMap    
    }
    # ------------------------------------------------------
    method newActMap {aMap}
    # ------------------------------------------------------
    method getActMap {}         
    # RO: returns the actual map
    # ------------------------------------------------------
    method selActMap {}         
    # get selection of actmap and make it the new act map
    # ------------------------------------------------------
    method hasMap {}            
    # 0/1, if a map exist
    # ------------------------------------------------------
    method showMap {} 
    # ------------------------------------------------------
    method hideMap {}
    # ------------------------------------------------------
    method getMap {aName aLevel}
    # Returns the pointer to the map with the given name
    # and level
    # ------------------------------------------------------
    method existsMap {aName aLevel}
    # Checks, if the map with the given name and level 
    # exists.
    # ------------------------------------------------------



    # ------------------------------------------------------
    # --- helpers
    # ------------------------------------------------------
    method informRoomHasChanged {}
    # This pops up an dialog, which informs that the rooms
    # has changed and needs to be reset or saved before
    # continuing.
    # ------------------------------------------------------








    variable CExitSelector
    variable CClickConMap
    variable CClickMode
    method clickCon {aExitSelector}
    method clickConClick {x y}
    method clickDraw {aExitSelector}
    method clickDrawClick {x y}



    method useExit {aPos} 

    method getZif {} { return $itk_component(zoomUif) }

    method getBGIM {} { return $_bgIM }
    variable CChangedMapNames
    method startMapping {aMud} ;# "connects" to a Mud -> sets _mud variable
    method startMappingBase {}
    method stopMapping {}
    method mappingStarted {} 
    method reallyStopMapping {aId aWid} 
    method saveAllMaps {} 

    method cleanup {}


    method getActCoord {} { return $_zoomPos}        ;# RO: actual coordinate and map -> position
    method getZoomPos {} { return $_zoomPos }        ;	# used by TEntryX
    method getPlayerPos {} { return $_playerPos }    ;	# used by TEntryX


    method importMap {aFileName} 
    method reallyImportMap { aFileName aName aLevel }
    method exportMap {aFileName} 
    method saveMap {} 
    method remMap {} 
    method removeMap {} 

    method existsRoom {aName aLevel aX aY} {
	# used by TEntryX
	if {[existsMap $aName $aLevel]} {
	    set map [$_mud getMap $aName $aLevel]
	    $map loadIfNeeded
	    set xy  [TCoordinate::GETXY $aX $aY]
	    if {[$map exists $xy]} {
		return 1
	    }
	}
	return 0
    }
    method assignSession {aSes} { 
	set _ses $aSes 
	TCommandLineIF::_setSes $aSes
    }


    method getRoom {aName aLevel aX aY} {
	set map [$_mud getMap $aName $aLevel]
	set xy  [TCoordinate::GETXY $aX $aY]
	return [$map getRoom $xy]
    }

    method newZoomPos {aMap aX aY} 
    method newPlayerPos {aMap aX aY} 



    method zoomIsOnActMap {} {
	if {![$_zoomPos exists]} { return 0 }
	if {![hasMap]} { return 0 }
	return [$_actMap == [$_zoomPos getMap]]
    }
    method playerIsOnActMap {} {
	if {![$_playerPos exists]} { return 0}
	if {![hasMap]} { return 0}
	return [$_actMap == [$_playerPos getMap]]
    }
    method isPlayerposEqualTo {aMap aX aY} {
	if {![$_playerPos exists]} { return 0}
	if {![hasMap]} { return 0}
	return [$_playerPos cmp $aMap $aX $aY]
    }


    method newMapReq {} 
    method resizeMapReq {} 
    method reallyResizeMap {a b c d} 
    method addRoom {aRoom aMap} 
    method remRoom {aRoom aMap} 
    method reallyCreateNewMapAndWalk {aName aLevel aPos}
    method reallyCreateNewMap {aName aLevel} 



    method mapChanged {aMap} {
	set namel [$aMap namel]
	set ls [$itk_component(mapname) size]
	for {set i 0} { $i<$ls } {incr i} {
	    set nnnn [$itk_component(mapname) get $i]
	    set patt "\[*.\] $namel"
	    if {[string match $patt $nnnn]} {
		break
	    }
	}
	sortMapSelector
    }
    method mapUnchanged {aMap} {
	set namel [$aMap namel]
	set ls [$itk_component(mapname) size]
	for {set i 0} { $i<$ls } {incr i} {
	    set nnnn [$itk_component(mapname) get $i]
	    set patt "\[*.\] $namel"
	    if {[string match $patt $nnnn]} {
		break
	    }
	}
	sortMapSelector
    }
    method updateMapsList {} {
	# this is called from outside
	set _actMap ""
	startMappingBase
    }
    method sortMapSelector {} {
	$itk_component(mapname) delete list 0 end
	foreach a [$_mud getAllMapNamesWithIndicator] {
	    $itk_component(mapname) insert list end $a
	}
    }

    method roomHasChanged {} {
	return [$itk_component(zoomUif) hasChanged]
    }

}











# ==========================================================
# === class code
# ==========================================================

# ----------------------------------------------------------
# --- constructor
# ----------------------------------------------------------

body TMapManager::constructor {aBGIM args} {
    set CExitSelector ""
    set CClickConMap  ""
    set CClickMode    "normal"

    set _actMap      ""
    set _zoomPos     [TPosition ::\#auto]
    set _playerPos   [TPosition ::\#auto]
    set _bgIM        $aBGIM


    # newmap gui
    itk_component add  newmap {
    TNewMapUif $itk_interior.newmap \
	-master [winfo toplevel $itk_interior] \
	-tops [winfo toplevel $itk_interior] 
    } {
	keep -background
	keep -foreground
    }    


    # mainframe
    itk_component add   pw {
	iwidgets::panedwindow $itk_interior.mainf -background $TColor::BGdef \
	    -orient horizontal
    } {keep -background}
    pack $itk_component(pw) -fill both -expand true -ipadx 0 -ipady 0 -padx 0 -pady 0 -side left

    $itk_component(pw) add  zif    -margin 0
    $itk_component(pw) add  maps   -margin 0


    # zoom IF/RoomUif
    itk_component add zoomUif {
	TRoomUif [$itk_component(pw) childsite zif].zoom $this $_bgIM -background $TColor::BGdef 
    } {
	keep -background
	keep -foreground
    }
    pack  $itk_component(zoomUif) -side top -expand 1 -fill both -ipadx 0 -ipady 0 -padx 0 -pady 0


    itk_component add secframe {
	frame [$itk_component(pw) childsite maps].secframe -background $TColor::BGdef -bd 0 -relief ridge
    } {
	keep -background
    }
    pack  $itk_component(secframe) -side top -expand 1 -fill both -ipadx 0 -ipady 0 -padx 0 -pady 0


    # mapname
    itk_component add mapnamef {
	frame $itk_component(secframe).mapname -background $TColor::BGdef -bd 2 -relief ridge
    } {keep -background}
    pack  $itk_component(mapnamef) -side top -expand 0 -fill x -ipadx 0 -ipady 0 -padx 0 -pady 0
    itk_component add   mapname {
	iwidgets::combobox $itk_component(mapnamef).cb \
	    -popupcursor hand1 -listheight 100  \
	    -textbackground $TColor::BGlight -selectbackground $TColor::BGdark \
	    -labeltext "Map: " -textfont def -labelfont defBold \
	    -selectioncommand [code $this selActMap] -editable 0 \
	    -textvariable TV($this,mapname) 
    } {
	keep -background
	keep -foreground
    }
    pack  $itk_component(mapname) -side top -expand 0 -fill x -ipadx 0 -ipady 0 -padx 0 -pady 0
 
    # mapframe
    itk_component add mapframe {
	frame $itk_component(secframe).map -background $TColor::BGdef -bd 2 -relief ridge
    } {
	keep -background
    }
    pack  $itk_component(mapframe) -side top -expand 1 -fill both -ipadx 0 -ipady 0 -padx 0 -pady 0

    #----- no showuppers
    #- Room Changed confirmer
    eval itk_initialize $args

    $itk_component(pw) fraction 14 86

    TCommandLineIF::_init $itk_component(zoomUif) $_bgIM

}
body TMapManager::destructor {} {
}

# ----------------------------------------------------------
# --- private helpers
# ----------------------------------------------------------
body TMapManager::existsRoomAtPlayer {} {
    return [existsRoom \
		[$_playerPos getMapName]  \
		[$_playerPos getMapLevel] \
		[$_playerPos getX]        \
		[$_playerPos getY]         ]
}
body TMapManager::getRoomAtPlayer {} {
    return [getRoom \
		[$_playerPos getMapName]  \
		[$_playerPos getMapLevel] \
		[$_playerPos getX]        \
		[$_playerPos getY]         ]
}


# ----------------------------------------------------------
# --- powerwalking
# ----------------------------------------------------------
body TMapManager::powerWalkTo {aMap aX aY} {
    if {![existsRoomAtPlayer]} {
	bell
	return
    }
    if {![existsRoom [$aMap name] [$aMap level] $aX $aY]} {
	bell
	after 300
	bell
	after 300
	bell
	return
    }

    update idletasks

    # init
    pqueue yellowpq
    set  start            [getRoomAtPlayer]
    set  dist($start)     0
    set  rmap($start)     [$_playerPos getMap]
    set  father($start)   ""
    set  fatherEx($start) ""  
    set  yellow($start)   [yellowpq insert $start $dist($start)]

    set found 0
    # start search
    while {[yellowpq isntEmpty]} {
	# get the nearest not visited room
	set nearestYellow         [yellowpq deleteMin]
	set green($nearestYellow) ""
	
	# test, if to far
	if {$dist($nearestYellow)>1000} {
	    # not found!
	    set found 0
	    break
	}
	# test, if found
	if { [$aMap == $rmap($nearestYellow)]} {
	    if { [$nearestYellow TCoordinate::getX] == $aX } {
		if { [$nearestYellow TCoordinate::getY] == $aY } {
		    #found!
		    set found 1
		    break
		}
	    }
	}

	# markvisited

	# follow its connections
	foreach expos [$nearestYellow TExitList::getPoss] {		
	    #nodebug puts "  search: $expos"
	    set ex [$nearestYellow TExitList::getExit $expos]
	    set mn [$ex getMap]
	    set ml [$ex getMapLevel]
	    set cc [$ex getConnectTo]
	    set mx [$cc getX]
	    set my [$cc getY]
	    
	    if {![string compare "" $ml]} {
		set ml 0
	    }
	    if {![string compare "" $mn]} {
		set mn [$rmap($nearestYellow) name]
		set ml [$rmap($nearestYellow) level]
	    }

	    #nodebug puts "  $expos: '$mn' -- '$ml'"
	    
	    # this is special to smm graphs -> connection may lead to void
	    if {![existsRoom $mn $ml $mx $my]} { continue }
	    set room        [getRoom $mn $ml $mx $my]
	    #nodebug puts "      room yes"
	    
	    # skip if its green
	    if {[info exists green($room)]} { continue }

	    #nodebug puts "     found yellow: $mn $ml $mx $my"
	    if {[info exists yellow($room)]} {
		# readjust its position, if needed (when dist now is smaller)
		set nd [expr $dist($nearestYellow) + 1]
		if { $nd < $dist($room)} {
		    # its needed
		    set dist($room)     $nd
		    set father($room)   $nearestYellow
		    set fatherEx($room) $expos
		    yellowpq rekey $yellow($room) $dist($room)
		    
		    set s $yellow($room)
		    while {[yellowpq exchangeNeeded $s]} {
			set f      [yellowpq exchange $s]
			set yellow([yellowpq getValue $s])   $s
		    }
		    set yellow($room) $f
		    
		}
	    } else {
		# put it into the pqueue, if its a new yellow
		# and also remeber its pos in the pqueue
		set  dist($room)     [expr $dist($nearestYellow) + 1]
		set  father($room)   $nearestYellow
		set  fatherEx($room) $expos
		set  yellow($room)   [yellowpq insert $room $dist($room)]
		set  rmap($room)     [getMap $mn $ml]
	    }
	}

    }


    if {$found} {
	update idletasks
	set exl [list]
	set ar  $nearestYellow
	while {[string compare "" $fatherEx($ar)]} {
	    set exl [linsert $exl 0 $fatherEx($ar)]
	    set ar $father($ar)
	}
	foreach expos $exl {
	    useExit $expos
	    update idletasks
	}

    }

    ::delete object yellowpq
}

body TMapManager::movePlayer {aHow aWhere} {
    if {[roomHasChanged]} {
	if {[devMode]} {
	    acceptRoom
	} else {
	    bell
	    $_ses writeMessage "Current room in zoom interface hasnt been saved!"
	    return
	}
    }

    set map [$_playerPos getMap]
    set x   [$_playerPos getX]
    set y   [$_playerPos getY]
    set xy  [TCoordinate::GETXY $x $y]

    #firstof set the zoompos to the playerpos
    if {[$_zoomPos != $_playerPos]} {
	newZoomPos $map $x $y
    } 
    if {![$map exists $xy]} {
	bell
	$_ses writeMessage "There is no map!"
	return
    }
    set room [$map getRoom $xy]	

    switch -- $aHow {
	useexitwithcommand {
	    set aPos [$room getPosOfExitWithCommand $aWhere]
	    set sendCommands 1
	}
	followexitwithcommand {
	    set aPos [$room getPosOfExitWithCommand $aWhere]
	    set sendCommands 0
	}
	useexit {
	    set aPos $aWhere
	    set sendCommands 1
	}	    
	followexit {
	    set aPos $aWhere
	    set sendCommands 0
	}	    
	default {
	    bell
	    $_ses writeMessage "No such command!"
	    return
	}
    }

    if {![$room TExitList::exists $aPos]} {
	bell
	$_ses writeMessage "No such exit!"
	return
    }
    
    # get the map to where to go
    set ex [$room getExit $aPos]
    set ml [$ex getMapLevel]
    set mn [$ex getMap]
    if {![string compare "" $ml]} {
	set ml 0
    }
    if {![string compare "" $mn]} {
	set mn [$map name]
	set ml [$map level]
    }

    # walk on the map
    if {[$_mud existsMap $mn $ml]} {
	# send the command to the mud
	if {$sendCommands} {
	    if {[$ex hasDoor]} {
		$_ses toSMM [$ex getDoorOpen]
		$_ses toSMM [$ex getCommand]
		$_ses toSMM [$ex getDoorClose]
	    } else {
		$_ses toSMM [$ex getCommand]
	    }
	}
	
	set nmap [$_mud getMap $mn $ml]
	set nx   [[$ex getConnectTo] getX]
	set ny   [[$ex getConnectTo] getY]
	newZoomPos   $nmap $nx $ny
	newPlayerPos $nmap $nx $ny
    } else {
	$_ses writeMessage "The map the exit leads to does not exist! You have to create it before!"
    }
}

    
# ----------------------------------------------------------
# --- map handling
# ----------------------------------------------------------
body TMapManager::getMap {aName aLevel} { 
    return [$_mud getMap $aName $aLevel] 
}
body TMapManager::existsMap {aName aLevel} { 
    return [$_mud existsMap $aName $aLevel] 
}


# ----------------------------------------------------------
# --- helpers
# ----------------------------------------------------------
body TMapManager::informRoomHasChanged {} { 
    TInformer $itk_interior.rcc \
	-master [winfo toplevel $itk_interior] \
	-information "Room changed! Either 'accept' or 'reset' first."
}



# ----------------------------------------------------------
# --- GUI stuff
# ----------------------------------------------------------
body TMapManager::getMapFrame {} { 
    return $itk_component(mapframe) 
}
body TMapManager::getLayout {} {
    set retval [list]
    lappend retval [$itk_component(pw) getFractions]
    lappend retval [$itk_component(zoomUif) getLayout]
    return $retval
}
body TMapManager::setLayout {aLayout} {
    set frac [lindex $aLayout 0]
    eval $itk_component(pw) fraction $frac
    $itk_component(zoomUif) setLayout [lindex $aLayout 1]
}


# ----------------------------------------------------------
# --- other
# ----------------------------------------------------------
body TMapManager::newZoomPos {aMap aX aY} {
    if {[roomHasChanged]} {
	if {[devMode]} {
	    acceptRoom
	} else {
	    informRoomHasChanged
	    return
	}
    }

    if {[clipExists]} {
	drawClip $aMap $aX $aY
    }

    $_zoomPos setXY $aX $aY
    if { [$aMap != [$_zoomPos getMap]] } {
	[$_zoomPos getMap] remZoomPos
	$_zoomPos setMap $aMap
	newActMap $aMap
    }

    if {[$aMap exists [$_zoomPos getXY]]} {
	$itk_component(zoomUif) setUif   [$aMap getRoom [$_zoomPos getXY]] [getActMap]
    } else {
	$itk_component(zoomUif) resetUif $aMap $aX $aY
    }
    $aMap drawZoomPos $aX $aY
}
body TMapManager::newPlayerPos {aMap aX aY} {
    if {[roomHasChanged]} {
	if {[devMode]} {
	    acceptRoom
	} else {
	    informRoomHasChanged
	    return
	}
    }

    $_playerPos setXY $aX $aY
    if { [$aMap != [$_playerPos getMap]] } {
	[$_playerPos getMap] remPlayerPos
	$_playerPos setMap $aMap
    }
    $aMap drawPlayerPos $aX $aY
}
body TMapManager::newMapReq {} {
    if {![mappingStarted]} {
	TInformer $itk_interior.mess \
	    -information \
	    "No Session has been started!
           \nYou need to have a active session in order to use mapping."
	return
    }
    $itk_component(newmap) configure -command "[code $this reallyCreateNewMap \{%n\} \{%l\}]"
    $itk_component(newmap) launchUif
#     TNewMapUif $itk_interior.newmap \
# 	-command "[code $this reallyCreateNewMap \{%n\} \{%l\}]" \
# 	-master [winfo toplevel $itk_interior] \
# 	-tops [winfo toplevel $itk_interior] 
}
body TMapManager::resizeMapReq {} {
    if {![hasMap]} {
	TInformer $itk_interior.mess \
	    -information \
	    "Somehow there is no map,..\ndoesn't it sound logic to have a map before trying to resize it?!?"
	return
    }
    TResizeMapUif $itk_interior.resizemap -size [$_actMap size] \
	-command "[code $this reallyResizeMap %xmin %xmax %ymin %ymax]" \
	-master [winfo toplevel $itk_interior] \
	-tops [winfo toplevel $itk_interior] 
}
body TMapManager::reallyResizeMap {a b c d} {
    $_actMap resize $a $b $c $d
    return 1
}
body TMapManager::addRoom {aRoom aMap} {
    if {![mappingStarted]} {
	TInformer $itk_interior.mess \
	    -information \
	    "Funny,.. and where is your map?!\nStart a session first and create/activate a map."
	return

    }
    if {[string compare "" $aMap]==0} { 
	$_actMap add $aRoom
    } else {
	$aMap add $aRoom
    }
}
body TMapManager::remRoom {aRoom aMap} {
    if {![mappingStarted]} {
	TInformer $itk_interior.mess \
	    -information \
	    "Funny,.. and where is your map?!\nStart a session first and create/activate a map."
	return

    }
    if {[string compare "" $aMap]==0} { 
	$_actMap remRoom $aRoom 
    } else {
	$aMap remRoom $aRoom 
    }
}
##############################3
body TMapManager::reallyCreateNewMapAndWalk {aName aLevel aPos} {
    reallyCreateNewMap $aName $aLevel
    useExit $aPos
}
body TMapManager::reallyCreateNewMap {aName aLevel} {
    regexp "^{(.*)}$" $aName  h name
    regexp "^{(.*)}$" $aLevel h level

    if {![$_mud addMap $name $level]} {
	TInformer $itk_interior.mess \
	    -information "The map '$name' (level: $level)\n already exists!"
	return 0
    }

    set map [$_mud getMap $name $level]    
    $map touch

    sortMapSelector

    newActMap $map
    
    if {[expr ![$_zoomPos exists]]} {
	$_zoomPos setMap      $map
	$_zoomPos setXY       0 0
	$map      drawZoomPos 0 0
    }
    if {[expr ![$_playerPos exists]]} {
	$_playerPos setMap $map
	$_playerPos setXY 0 0
	update
	$map        drawPlayerPos 0 0
    }
    return 1
}
body TMapManager::useExit {aPos} {    
    if {[roomHasChanged]} {
	if {[devMode]} {
	    acceptRoom
	} else {
	    informRoomHasChanged
	    return
	}
    }

    set map [$_playerPos getMap]
    set x   [$_playerPos getX]
    set y   [$_playerPos getY]
    set xy  [TCoordinate::GETXY $x $y]

    #firstof set the zoompos to the playerpos
    if {[$_zoomPos != $_playerPos]} {
	newZoomPos $map $x $y
    } 
    if {![$map exists $xy]} {
	bell
	return
    }
    set room [$map getRoom $xy]	
    if {![$room TExitList::exists $aPos]} {
	bell
	return
    }
    
    # get the map to where to go
    set ex [$room getExit $aPos]
    set ml [$ex getMapLevel]
    set mn [$ex getMap]
    if {![string compare "" $ml]} {
	set ml 0
    }
    if {![string compare "" $mn]} {
	set mn [$map name]
	set ml [$map level]
    }

    # walk on the map
    if {[$_mud existsMap $mn $ml]} {
	# send the command to the mud
	if {[$ex hasDoor]} {
	    $_ses toSMM [$ex getDoorOpen]
	    $_ses toSMM [$ex getCommand]
	    $_ses toSMM [$ex getDoorClose]
	} else {
	    $_ses toSMM [$ex getCommand]
	}
	
	set nmap [$_mud getMap $mn $ml]
	set nx   [[$ex getConnectTo] getX]
	set ny   [[$ex getConnectTo] getY]
	newZoomPos   $nmap $nx $ny
	newPlayerPos $nmap $nx $ny
    } else {
	TConfirmer $itk_interior.useexit \
	    -command "[code $this reallyCreateNewMapAndWalk \{$mn\} \{$ml\}] $aPos" \
	    -question "Do you want to create the following map:\nname: $mn\nlevel: $ml?" 
    }
}

body TMapManager::importMap {aFileName} {
    set nl [TMap::returnMapNameLevelFromFile $aFileName]
    
    $itk_component(newmap) configure -command "[code $this reallyImportMap $aFileName \{%n\} \{%l\}]"
    $itk_component(newmap) configure -name [lindex $nl 0] 
    $itk_component(newmap) configure -level [lindex $nl 1]
    $itk_component(newmap) launchUif

#     TNewMapUif $itk_interior.newmap \
# 	-name [lindex $nl 0] -level [lindex $nl 1] \
# 	-command "[code $this reallyImportMap $aFileName \{%n\} \{%l\}]" \
# 	-master [winfo toplevel $itk_interior] \
# 	-tops [winfo toplevel $itk_interior] 
}

body TMapManager::reallyImportMap { aFileName aName aLevel } {
    regexp "^{(.*)}$" $aName  h name
    regexp "^{(.*)}$" $aLevel h level

    if {![$_mud importMap $aFileName $name $level]} {
	TInformer $itk_interior.mess \
	    -information "The map '$name' (level: $level)\n already exists!"
	return 0
    }

    set map [$_mud getMap $name $level]    
    $map touch

    sortMapSelector

    newActMap $map
    
    if {[expr ![$_zoomPos exists]]} {
	$_zoomPos setMap      $map
	$_zoomPos setXY       0 0
	$map      drawZoomPos 0 0
    }
    if {[expr ![$_playerPos exists]]} {
	$_playerPos setMap $map
	$_playerPos setXY 0 0
	update
	$map        drawPlayerPos 0 0
    }
    return 1
}
body TMapManager::exportMap {aFileName} {
    if {[string match "" $_actMap]} {
	TInformer $itk_interior.mess \
	    -information "No map available!"
	return
    }

    $_actMap export $aFileName
}
body TMapManager::saveMap {} {
    if {[string match "" $_actMap]} {
	TInformer $itk_interior.mess \
	    -information "No map available!"
	return
    }

    $_actMap save  
}
body TMapManager::removeMap {} {
    if {[zoomIsOnActMap]} { $_zoomPos setNull } 
    if {[playerIsOnActMap]} { $_playerPos setNull } 
    $_mud deleteMap $_actMap
}
body TMapManager::remMap {} {
    if {[string match "" $_actMap]} {
	TInformer $itk_interior.mess \
	    -information "No map available!"
	return
    }
    TConfirmer $itk_interior.remmap -command "[code $this removeMap]" \
	-question "Really remove actual map?"
}
body TMapManager::saveAllMaps {} {
    set CChangedMapNames [$_mud getChangedMapNames]
    foreach ml $CChangedMapNames {
	[$_mud getMap [lindex $ml 0] [lindex $ml 1]] save
    }    
}
body TMapManager::reallyStopMapping {aId aWid} {
    switch $aId {
	s {
	    foreach ml $CChangedMapNames {
		[$_mud getMap [lindex $ml 0] [lindex $ml 1]] save
	    }
	    $_mud remMapM $this
	    cleanup
	    $aWid deactivate
	    destroy $aWid
	}
	d {
	    $_mud remMapM $this
	    cleanup
	    $aWid deactivate
	    destroy $aWid
	}
	a {
	    $aWid deactivate
	    destroy $aWid

 	    set wid $itk_interior.savediscdialog
 	    iwidgets::messagedialog $wid \
 		-modality application \
  		-master [winfo toplevel $itk_interior] \
  		-tops [winfo toplevel $itk_interior] \
 		-bitmap questhead \
 		-title "Question: save map?" 

	    $wid buttonconfigure OK -text "Yes"
	    $wid buttonconfigure Cancel -text "No"

	    foreach mapi $CChangedMapNames {
		set name  [lindex $mapi 0]
		set level [lindex $mapi 1]

		$wid configure -text "Do you want to save the map\n'$name (level: $level)'?"
		$wid center 
		if {[$wid activate]} {
		    [$_mud getMap $name $level] save
		}
	    }
	    destroy $wid

	    $_mud remMapM $this
	    cleanup
	}
    }
}
body TMapManager::cleanup {} {
    global TV
    catch {unset _mud}
    set _actMap ""

    $itk_component(mapname) delete list  0 end
    set TV($this,mapname) ""
    $_zoomPos setNull
    $_playerPos setNull
    $itk_component(zoomUif) resetUif "" 0 0    
    selClear
}
body TMapManager::stopMapping {} {
    if {[mappingStarted]} {
	set CChangedMapNames [$_mud getChangedMapNames]
	set l [llength $CChangedMapNames]

	if {$l>0} {
 	    set wid $itk_interior.unsavedmapsdialog
 	    iwidgets::messagedialog $wid \
 		-modality application \
		-buttonboxpos e \
  		-master [winfo toplevel $itk_interior] \
  		-tops [winfo toplevel $itk_interior] \
 		-bitmap questhead \
 		-title "Question: unsaved maps" \
 		-text "There is/are $l unsaved map(s)!\nWhat do you want to do?"
 	    $wid hide Cancel
 	    $wid hide OK
 	    $wid add saveall -text "save all maps" -command "[code $this reallyStopMapping s $wid]"
 	    $wid add discall -text "discard all changes" -command "[code $this reallyStopMapping d $wid]"
 	    $wid add askeach -text "ask for each map separatly" -command "[code $this reallyStopMapping a $wid]"
 	    $wid default saveall

 	    $wid center 
 	    $wid activate
	    destroy $wid
	    return
	} 
	$_mud remMapM $this
    } 
    cleanup
}
body TMapManager::mappingStarted {} {
    return [info exists _mud]
}
body TMapManager::hasMap {} {
    return [expr ![string match "" $_actMap]]
}
body TMapManager::startMappingBase {} {
    set maps [$_mud getAllMaps]	
    
    if {[llength $maps] >0} { 
	sortMapSelector
	
	set map [lindex $maps 0]	
	newActMap $map	
	if {[expr ![$_zoomPos exists]]} {
	    $_zoomPos setMap      $map
	    $_zoomPos setXY       0 0
	    $map      drawZoomPos 0 0
	}
	if {[expr ![$_playerPos exists]]} {
	    $_playerPos setMap $map
	    $_playerPos setXY 0 0
	    update
	    $map        drawPlayerPos 0 0
	}
	newPlayerPos $map 0 0
	newZoomPos   $map 0 0
    } else {
	$itk_component(mapname) configure -editable 1
	$itk_component(mapname) clear
	$itk_component(mapname) clear entry
	set TV($this,mapname) ""
	$itk_component(mapname) configure -editable 0
    }
}
body TMapManager::startMapping {aMud} {
    if {![TMapManager::mappingStarted]} {
	set _mud    $aMud
	TCommandLineIF::_setMud $aMud

	$_mud addMapM $this

	startMappingBase
    } else {
	TInformer $itk_interior.mess \
	    -information "There is already a mud active!"
    }
}


body TMapManager::clickCon {aExitSelector} {
    set CClickMode con
    set CExitSelector $aExitSelector
    set CClickConMap  [getActMap]
    focus $itk_component(secframe)
    grab $itk_component(secframe)
    [getActMap] clickCon hand2	;# moth was "hand2 black"
}
body TMapManager::clickConClick {x y} {
    set CClickMode normal
    set am [getActMap]
    grab  release $itk_component(secframe)
    if {[$CClickConMap == $am]} {
	$CExitSelector clickConClick $x $y "" ""
    } else {
	newActMap $CClickConMap
	$CExitSelector clickConClick $x $y [$am name] [$am level]
    }
    set CExitSelector ""
}
body TMapManager::clickDraw {aExitSelector} {
    set CClickMode draw
    set CExitSelector $aExitSelector
    set CClickConMap  [getActMap]
    focus $itk_component(secframe)
    grab $itk_component(secframe)
    [getActMap] clickDraw hand2	;# moth was "hand2 black"
}
body TMapManager::clickDrawClick {x y} {
    set CClickMode normal
    set am [getActMap]
    grab  release $itk_component(secframe)
    if {[$CClickConMap == $am]} {
	$CExitSelector clickDrawClick $x $y 
    } else {
	newActMap $CClickConMap
	$CExitSelector clickDrawClick $x $y 
    }
    set CExitSelector ""
}
body TMapManager::selActMap {} {
    global TV
    
    set nnnn  [string range $TV($this,mapname) 2 end]
    if {![string compare "" $nnnn]} {
	return
    }

    set name  [TMap::namelname  $nnnn]
    set level [TMap::namellevel $nnnn]
    set map   [$_mud getMap $name $level]
    
    switch -- $CClickMode {
	con {
	    [getActMap] defaultBindings
	    newActMap $map
	    [getActMap] clickCon hand2	;# moth was "hand2 black"
	    [getActMap] adjustViewPoint [$_zoomPos getX] [$_zoomPos getY]
	}
	draw {
	    [getActMap] defaultBindings
	    newActMap $map
	    [getActMap] clickDraw hand2	;# moth was "hand2 black"
	    [getActMap] adjustViewPoint [$_zoomPos getX] [$_zoomPos getY]
	}
	default {
	    newActMap $map
	}
    }
}

body TMapManager::newActMap {aMap} {
    global TV

    if {![string match "" $_actMap]} {
	TMapManager::hideMap
    }
    
    set _actMap $aMap
    $_actMap loadIfNeeded

    set TV($this,mapname) [$aMap namel]

    TMapManager::showMap  
}





body TMapManager::getActMap {} {
    return $_actMap
}

body TMapManager::showMap {} {
    $_actMap show
}
body TMapManager::hideMap {} {
    $_actMap hide
}





# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: TMapManager.tcl,v $
# Revision 1.9  2002/04/28 11:13:35  issever
# put menu->stop session back in place
# debug: sesManagerUif buttons showed wrong state,
#   when the menu was used to stop the session
# added save all maps menu entry
#
# Revision 1.8  2002/04/21 18:03:16  issever
# made ccp und dnd much smarter about exits leading to
# and coming from other maps.
#
# Revision 1.7  2002/04/06 22:49:38  issever
# drag&drop bound to cut/copy/paste (in maps).
# This allows to d&d more then a single room at once.
#
# Revision 1.6  2002/04/06 17:33:38  moth
# Added a variety of routines to support user queries of map data
#
# Revision 1.5  2002/04/06 01:31:16  issever
# debugged cut/copy/paste in maps. In fact it was quite a redo
# of the code, but now it works fine.
# Area to be pasted will be indicated.
#
# Revision 1.4  2002/04/04 21:07:03  issever
# Optimized the expr's, which are used in the mapping code.
# Splited the File TEXit.tcl.
#
# Revision 1.3  2002/03/04 20:01:49  issever
# bugfix: map access from the commandline
#
# Revision 1.2  2002/02/17 11:19:21  issever
# resolving merge conflicts
#
# Revision 1.1  2002/02/11 20:24:32  issever
# cut/copy/paste for maps, drag&drop, code cleanup of mapmanager
#
