# $Id: pqueue.tcl,v 1.2 2001/08/19 12:25:42 issever Exp $

#
# pqueue
# ----------------------------------------------------------------------
# Priority Queues
# source: Datenstructuren und Algorithmen
#         Ralf Hartmut Gueting
#         B.G. Teubner Stuttgart
#
# PQ's are sets of elements which are priority ordered. Highest
# priority means generally the element with the smallest value. Good
# examples are queues in a hospital ambulanz. The one which needs most
# urgently help is helped first no matter when he arrived. For
# computer science purposes PQ's are used by OS'es: processes are
# priority ordered and get resources (e.g. CPU time) depending on
# their priority.
#
# This implementation uses heaps (implemented as arrays) and both
# insert and deletemin need O(log n) time, if n is the number of
# elements ion the pqueue. O(n) memory is needed.
#
# ----------------------------------------------------------------------
#  AUTHOR: Selim Issever                    EMAIL: selim.issever@desy.de
#  Version: 1.0
#
# ----------------------------------------------------------------------
#            Copyright (c) 1999 by the author
# ======================================================================
# Permission to use, copy, modify, distribute and license this
# software and its documentation for any purpose, and without fee or
# written agreement with the author, is hereby granted, provided that
# the above copyright notice appears in all copies and that both the
# copyright notice and warranty disclaimer below appear in supporting
# documentation, and that the names of the author not be used in
# advertising or publicity pertaining to the software without
# specific, written prior permission.
# 
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
# INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, AND
# NON- INFRINGEMENT. THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS,
# AND THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE
# MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
# CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
# OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
# ======================================================================

# ------------------------------------------------------------------
#                            PQUEUE
# ------------------------------------------------------------------
class pqueue {
    
    # corresponds to empty (the method name in 'Datenstructuren und Algorithmen')
    constructor {} {}
    method empty {}
    method isEmpty {}
    method isntEmpty {}
    method insert {aValue aKey}  ; # puts an element into the pqueue
    method deleteMin {}          ; # returns the value with the smallest
                                   #  key and removes this v/k pair

    # - pqueues are originally designed just to take v/k pairs and
    # - just return the value with the smallest key.
    # - For my application (algorithm of Dijkstra: single source 
    # - shortest path problem) I needed to change the keyvalues of
    # - of already inserted v/k elements, thus I needed to have
    # - extended access to the pqueue
    method getValue {aPos}
    method exchangeNeeded {s}
    method exchange {s} 
    method rekey {aPos aKey} 

    private {
	variable CTree
	variable CN	
	method getFather {aNum} 
	method hasASon {aNum} 
	method hasTwoSons {aNum} 
	method getSmallerSon {aNum} 
    }

}

# ------------------------------------------------------------------
#                        CONSTRUCTOR
# ------------------------------------------------------------------
body pqueue::constructor {} {
    empty
}
body pqueue::empty {} {
    set CN 0
}
body pqueue::isEmpty {} {
    return [expr $CN==0]
}
body pqueue::isntEmpty {} {
    return [expr $CN!=0]
}

body pqueue::getFather {aNum} { 
    # this returns 0, if no father is available
    return [expr int($aNum/2)] 
}

body pqueue::exchangeNeeded {s} {
    set f [getFather $s]
    return [expr $CTree($f,k)<$CTree($s,k)]
}
body pqueue::exchange {s} {
    set f [getFather $s]
    set v            $CTree($s,v)
    set k            $CTree($s,k)
    set CTree($s,v)  $CTree($f,v)
    set CTree($s,k)  $CTree($f,k)
    set CTree($f,v)  $v
    set CTree($f,k)  $k
    return $f
}
body pqueue::getValue {aPos} { return $CTree($aPos,v) }
body pqueue::insert {aValue aKey} {
    incr CN
    set s $CN
    set f [getFather $s]

    set CTree($s,v) $aValue
    set CTree($s,k) $aKey
    
    
    while { $f } {
	if {$CTree($f,k)>$CTree($s,k)} {
	    # remember
	    set v            $CTree($s,v)
	    set k            $CTree($s,k)
	    # exchange
	    set CTree($s,v)  $CTree($f,v)
	    set CTree($s,k)  $CTree($f,k)
	    set CTree($f,v)  $v
	    set CTree($f,k)  $k

	    set s $f
	    set f [getFather $s]
	} else {
	    break
	}
    }
    return $s
}
body pqueue::rekey {aPos aKey} {
    set CTree($aPos,k) $aKey
}
body pqueue::hasASon {aNum} {
    return [expr (2*$aNum)<=$CN]
}
body pqueue::hasTwoSons {aNum} {
    return [expr (2*$aNum)<$CN]
}
body pqueue::getSmallerSon {aNum} {
    set s1 [expr 2*$aNum]
    
    if {$s1<$CN} {
	# two sons
	set s2 [expr $s1+1]
	if {$CTree($s1,k)<$CTree($s2,k)} {
	    return $s1
	} else {
	    return $s2
	}		
    } else {
	# one son
	return $s1
    }
}
body pqueue::deleteMin {} {
    set retval $CTree(1,v)

    set v             $CTree(1,v)
    set k             $CTree(1,k)
    set CTree(1,v)    $CTree($CN,v)
    set CTree(1,k)    $CTree($CN,k)
    set CTree($CN,v)  $v
    set CTree($CN,k)  $k
    incr CN -1

    set f 1
    while {[hasASon $f]} {
	set ss [getSmallerSon $f]
	if {$CTree($f,k)>$CTree($ss,k)} {
	    set v             $CTree($f,v)
	    set k             $CTree($f,k)
	    set CTree($f,v)   $CTree($ss,v)
	    set CTree($f,k)   $CTree($ss,k)
	    set CTree($ss,v)  $v
	    set CTree($ss,k)  $k
	    
	    set f $ss
	} else {
	    break
	}
    }
    return $retval
}




# ##############################################################################
# ### LOG MESSAGES
# ### As suggested by the CVS-manual this region is put to the end of the file.
# ##############################################################################
#
# $Log: pqueue.tcl,v $
# Revision 1.2  2001/08/19 12:25:42  issever
# Added the cvs keywords Id at start of the file
# and Log at the end of the file
#
#
