/*
 * 	daemon.c
 *
 * 	Copyright (C) 2004-2005 Bartomiej Korupczynski <bartek@klolik.org>
 *
 * 	This program is free software; you can redistribute it and/or 
 * 	modify it under the terms of the GNU General Public License 
 * 	as published by the Free Software Foundation; either 
 * 	version 2 of the License, or (at your option) any later 
 * 	version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the Free Software
 * 	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE
#define _DAEMON_C_

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <pwd.h>
#include <grp.h>

#include "confvars.h"
#include "conffile.h"
#include "daemon.h"
#include "util.h"



int foreground = 0;


/*
 * 	pid file
*/

int create_pidfile()
{
	int fd;

	if (EMPTY_STRING(config.pidfile)) return 0;

	fd = open(config.pidfile, O_CREAT|O_WRONLY|O_EXCL|O_NOCTTY|O_TRUNC, PID_FILE_MODE);
	if (fd == -1) {
		log_action(LOG_CRIT, "Error creating pid file [%s]: %s", config.pidfile, strerror(errno));
		return -1;
	}
	fdprintf(fd, "%d\n", getpid());
	SAFE_CLOSE(fd);

	return 0;
}

int remove_pidfile()
{
	if (EMPTY_STRING(config.pidfile)) return 0;

	if (strcmp(config.pidfile, "") == 0) return 0;
	if (strcmp(config.pidfile, "/") == 0) return 0;

	if (unlink(config.pidfile) != 0) {
		log_action(LOG_CRIT, "Error removing .pid file [%s]: %s", config.pidfile, strerror(errno));
		return -1;
	}

	return 0;
}


/*
 *	daemonize process
*/

int daemonize()
{
	struct passwd *pw;
	struct group *grp;
	gid_t new_gid;
	int res;


	if (!foreground) {
		openlog("smtp-gated", LOG_PID, config.log_facility);

		if  ((res = fork()) == -1) {
			log_action(LOG_CRIT, "fork(): %s", strerror(errno));
			return -1;
		}

		if (res) exit(0);

		if ((res = setsid()) == -1) {
			log_action(LOG_CRIT, "setsid(): %s", strerror(errno));
			return -1;
		}
	}

	if (!EMPTY_STRING(config.chroot_path)) {
		if (chdir(config.chroot_path) != 0) {
			log_action(LOG_CRIT, "chdir(%s): %s", config.chroot_path, strerror(errno));
			return -1;
		}
		if (chroot(config.chroot_path) != 0) {
			log_action(LOG_CRIT, "chroot(%s): %s", config.chroot_path, strerror(errno));
			return -1;
		}

		chdir("/");
		log_action(LOG_DEBUG, "chroot(%s) successful, reopening syslog", config.chroot_path);
		closelog();
		openlog("smtp-gated", LOG_PID, LOG_DAEMON);
		log_action(LOG_DEBUG, "syslog reopened after chroot(%s), continuing", config.chroot_path);
	}

	if (!foreground) {
		if (!freopen("/dev/null", "r", stdin) || !freopen("/dev/null", "w", stdout) || !freopen("/dev/null", "w", stderr)) {
			log_action(LOG_CRIT, "freopen(stdxxx): %s", strerror(errno));
			return -1;
		}
	}

	if (chdir(config.spool_path) != 0) {
		log_action(LOG_CRIT, "chdir(%s): %s", config.spool_path, strerror(errno));
		return -1;
	}


	if (create_pidfile() != 0) return -1;

	if (config.priority) {
		if (setpriority(PRIO_PROCESS, 0, config.priority) < 0) {
			log_action(LOG_WARNING, "setpriority(%d) failed: %s", config.priority, strerror(errno));
		}
	}

	grp = NULL;

	if (strcmp(config.set_group, "") != 0) {
		grp = getgrnam(config.set_group);

		if (grp == NULL) {
			log_action(LOG_CRIT, "getgrnam('%s') failed. no such group?", config.set_group);
			goto err_rem_pid;
		}
		if (setgid(grp->gr_gid) != 0) {
			log_action(LOG_CRIT, "setgid() failed: %s", strerror(errno));
			goto err_rem_pid;
		}
		log_action(LOG_INFO, "Changed GID to %d (%s)", grp->gr_gid, config.set_group);
	}

	if (strcmp(config.set_user, "") != 0) {
		pw = getpwnam(config.set_user);

		if (pw == NULL) {
			log_action(LOG_CRIT, "getpwnam(%s) failed. no such user?", config.set_user);
			goto err_rem_pid;
		}
		new_gid = (grp != NULL) ? grp->gr_gid : pw->pw_gid;
		
		if (initgroups(pw->pw_name, new_gid) != 0) {
			log_action(LOG_CRIT, "initgroups() failed: %s", strerror(errno));
			goto err_rem_pid;
		}

		// change pid-file owner to new user id, we must be able to delete it
		if (chown(config.pidfile, pw->pw_uid, (gid_t) -1) != 0) {
			log_action(LOG_CRIT, "chown(%s) failed: %s", config.pidfile, strerror(errno));
			goto err_rem_pid;
		}

		if ((grp == NULL) && (setgid(new_gid) != 0)) {
			log_action(LOG_CRIT, "setgid() for default group failed: %s", strerror(errno));
			goto err_rem_pid;
		}

		if (setuid(pw->pw_uid) != 0) {
			log_action(LOG_CRIT, "setuid(%d) failed: %s", pw->pw_uid, strerror(errno));
			goto err_rem_pid;
		}

		log_action(LOG_INFO, "Changed UID to %d (%s)", pw->pw_uid, config.set_user);
	}

	return 0;

err_rem_pid:
	remove_pidfile();
	return -1;
}


