/*
 *	dump.c
 *
 *	Copyright (C) 2004-2005 Bartomiej Korupczynski <bartek@klolik.org>
 *
 *	This program is free software; you can redistribute it and/or 
 *	modify it under the terms of the GNU General Public License 
 *	as published by the Free Software Foundation; either 
 *	version 2 of the License, or (at your option) any later 
 *	version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*
 *	headers
*/

#define _GNU_SOURCE

#define _DUMP_C_

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>


#include "confvars.h"
#include "smtp-gated.h"
#include "scan.h"
#include "dump.h"
#include "util.h"

void dump_state()
{
	char tmp_dumpfile[FILENAME_MAX+1];
	struct conn_info *ci;
	int fd;

	time_t now;
	int i, td, d, h, m, s;
#ifdef USE_SHARED_MEM
	double speed;
#endif
#ifdef HAVE_GETRUSAGE
	struct rusage ru;
#endif

	snprintf(tmp_dumpfile, sizeof(tmp_dumpfile), "%s.tmp", config.dumpfile);
	TERMINATE_STRING(tmp_dumpfile);
	
	fd = open(tmp_dumpfile, O_WRONLY | O_CREAT | O_TRUNC, config.dumpfile_perm);
	if (fd == -1) {
		log_action(LOG_ERR, "can't open dump file [%s]: %s", config.dumpfile, strerror(errno));
		return;
	}

	now = time(NULL);
	time2dhms(now - stats->started, &d, &h, &m, &s);	// elapsed

#ifndef USE_SHARED_MEM
	stats->rejects_ident =
		stats->viruses =
		stats->spams =
		stats->requests_empty = -1;
#endif
	fdprintf(fd, "Dump time:    %s\n", time2str(now));
	fdprintf(fd, "Start time:   %s\n", time2str(stats->started));
	fdprintf(fd, "Restart time: %s\n", time2str(stats->restarted));
	fdprintf(fd, "Uptime:   %dd %dh %dm %ds\n", d, h, m, s);
#ifdef HAVE_GETRUSAGE
	if (getrusage(RUSAGE_SELF, &ru) == 0) {
		fdprintf(fd, "Resource: %ld/%ld/%ld/%ld (maxrss/ixrss/idrss/isrss)\n",
			ru.ru_maxrss, ru.ru_ixrss, ru.ru_idrss, ru.ru_isrss);
	} else {
		fdprintf(fd, "Resource: getrusage() error: %s\n", strerror(errno));
	}
#endif
	fdprintf(fd, "Found:    %d/%d (viruses/spam)\n", stats->viruses, stats->spams);
	fdprintf(fd, "Children: %d/%d (current/max)\n", children, stats->max_children);
	fdprintf(fd, "Requests: %d/%d/%d (total/direct/empty)\n",
		stats->requests, stats->requests_direct, stats->requests_empty);
	fdprintf(fd, "Rejects:  %d/%d/%d/%d (host/ident/lock/other)\n",
		stats->rejects_host, stats->rejects_ident, stats->rejects_lock, stats->rejects);

	fdprintf(fd, "\n");
	fdprintf(fd, "%4s %-5s %-8s %-5s %-15s %-15s %4s %8s %8s %7s %s\n", "slot", "pid", "state", "time", "source", "target", "trns", "cli_rx", "srv_rx", "kbps", "ident");
	for (i=0; i<max_connections_real; i++) {
		ci = &connections[i];
		if (!pids[i]) continue;

		td = now - ci->start_time;
#ifdef USE_SHARED_MEM
		if (td) {
			speed = ((float) ci->cli_rx)/(td*1024/8);
		} else {
			speed = 0;
		}
#endif
		
		fdprintf(fd, "%4d %-5" FORMAT_PID_T " ", i, pids[i]);
#ifdef USE_SHARED_MEM
		fdprintf(fd, "%-8s", conn_states[ci->state]);
#else
		fdprintf(fd, "%-8s", "-");
#endif
		fdprintf(fd, " %02d:%02d", td / 60, td % 60);
		fdprintf(fd, " %-15s ", inet_ntoa(UINT32_TO_SIN(ci->src)));
#ifdef USE_SHARED_MEM
		fdprintf(fd, "%-15s %4d %8d %8d %7.1f", inet_ntoa(UINT32_TO_SIN(ci->dst)), ci->transaction, ci->cli_rx, ci->srv_rx, speed);
		fdprintf(fd, " %s", (ci->ident_ok) ? ci->ident : "");
#else
		fdprintf(fd, "%-15s %4s %8s %8s %6s", "-", "-", "-", "-", "-");
#endif

		fdprintf(fd, "\n");
	}

	close(fd);

	// move fresh state file to its nominal name
	if (rename(tmp_dumpfile, config.dumpfile) != 0) {
		log_action(LOG_ERR, "can't rename temporary state file [%s] to [%s]: %s", tmp_dumpfile, config.dumpfile, strerror(errno));
		return;
	}
} /* dump_state() */


void dump_ver(int verbose)
{
	printf("SMTP Transparent AV proxy\n");
	CONF_SS2("version", VERSION);
	CONF_SS2("build date", __DATE__ " " __TIME__);
	
	if (!verbose) {
		printf("\nThis program is distributed under GNU GPL license, without\n");
	       	printf("any warranty. For more information see COPYING file.\n");
		return;
	}

	printf("Defines:\n");
#ifdef DEFAULT_CONFIG_FILE
	CONF_S2(DEFAULT_CONFIG_FILE);
#endif
	CONF_S2(SCANNER);
#ifdef USE_NAT
	printf("  %-30s: %s (%s)\n", "USE_NAT", "yes", USE_NAT);
#else
	CONF_SS2("USE_NAT", "no");
#endif
#ifdef USE_SHARED_MEM
	CONF_SS2("USE_SHARED_MEM", "yes");
#else
	CONF_SS2("USE_SHARED_MEM", "no");
#endif
/*
#ifdef USE_REGEX
	CONF_SS2("USE_REGEX", "yes");
#else
	CONF_SS2("USE_REGEX", "no");
#endif
#ifdef USE_PGSQL
	CONF_SS2("USE_PGSQL", "yes");
#else
	CONF_SS2("USE_PGSQL", "no");
#endif
*/
#ifdef FILTER_CHUNKING
	CONF_SS2("FILTER_CHUNKING", "yes");
#else
	CONF_SS2("FILTER_CHUNKING", "no");
#endif
#ifdef SILENT_ECONNRESET
	CONF_SS2("SILENT_ECONNRESET", "yes");
#else
	CONF_SS2("SILENT_ECONNRESET", "no");
#endif
	CONF_D2(CONN_REJ_CODE);
	CONF_D2(HELO_LENGTH);
	CONF_D2(IDENT_SIZE);
	CONF_D2(PIPELINE_SIZE_MIN);
#ifndef HAVE_SETENV
	CONF_SS2("HAVE_SETENV", "no");
#endif
#ifdef MEMLEAK_TESTING
	CONF_SS2("MEMLEAK_BUILDIN", "YES -- FOR TESTING ONLY!");
#endif
	printf("Extra:\n");
	CONF_D2(sizeof(struct proc_data));
	CONF_D2(sizeof(struct conn_info));
	CONF_D2(sizeof(struct pipeline_entry));
} /* dump_ver() */


