/*
 *	lookup.c
 *
 *	Copyright (C) 2004-2005 Bartomiej Korupczynski <bartek@klolik.org>
 *
 *	This program is free software; you can redistribute it and/or 
 *	modify it under the terms of the GNU General Public License 
 *	as published by the Free Software Foundation; either 
 *	version 2 of the License, or (at your option) any later 
 *	version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE
#define _LOOKUP_C_


#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#ifdef HAVE_ERR_H
#include <err.h>
#endif
#include <syslog.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>

#ifdef HAVE_LINUX_NETFILTER_IPV4_H
#include <limits.h>
#include <linux/netfilter_ipv4.h>
#endif

#include "confvars.h"
#include "conffile.h"
#include "smtp-gated.h"
#include "lookup.h"
#include "util.h"


/*
	identyfikacja adres serwera oryginalnego
	=0	ok
	-1	error, (->errno)
	>0	LOOKUP_x
*/

/*
 * 	used data members:
 * 	- client
 * 	- target, target_port, target_str
 * 	- target
*/

#ifdef USE_NAT
int nat_lookup(struct proc_data *data)
{
	socklen_t size;

	data->target.sin_family = AF_INET;
	size = sizeof(data->target);

#ifdef USE_NAT_NETFILTER
	if (getsockopt(data->client, SOL_IP, SO_ORIGINAL_DST, &data->target, &size) != 0)
		return -1;
#endif
#ifdef USE_NAT_GETSOCKNAME
	if (getsockname(data->client, &data->target, &size) != 0) 
		return -1;
#endif

	snprintf(data->target_str, sizeof(data->target_str), "%s", inet_ntoa(data->target.sin_addr));
	TERMINATE_STRING(data->target_str);
	data->target_port = ntohs(data->target.sin_port);

	return 0;
} /* netfilter_lookup() */
#endif



int remote_lookup(struct proc_data *data)
{
	char buf[1024];
	char ident_tmp[IDENT_SIZE+1];	// fixed 128!
	int res, client_port, pos;
	int rport, lport, local_port;
	int ip1, ip2, ip3, ip4;
	int sock;

	client_port = ntohs(data->origin.sin_port);
	local_port = config.port;
	SET_TIMEOUT(config.timeout_lookup);

	if ((sock = connect_host(data->origin, config.lookup_port, data->local_addr)) == -1) {
		goto fail;
	}

	if (fdprintf(sock, "%d , %d\r\n", client_port, local_port) == -1) {
		if (errno == EINTR) errno = ETIMEDOUT;
		goto fail;
	}

	for (pos=0;;) {
		if (timedout) {
			CLEAR_TIMEOUT();
			errno = ETIMEDOUT;
			goto fail;
		}
		if ((res = read(sock, buf+pos, sizeof(buf)-pos)) == -1) {
			goto fail;
		}
		if (res == 0) break;
		pos += res;
	}

	buf[(pos < sizeof(buf)) ? pos : sizeof(buf)-1] = '\0';

	SAFE_CLOSE(sock);
	sock = -1;
	CLEAR_TIMEOUT();

#if IDENT_SIZE != 64
#warning Please update numbers in two places:
#warning 1. in sscanf format below (last number)
#warning 2. in #if above
#warning Both of them must much IDENT_SIZE!
#error
#endif
	res = sscanf(buf, "%d , %d : ORIGIN : %d.%d.%d.%d : %d : %64s", 
		&rport, &lport, &ip1, &ip2, &ip3, &ip4, &data->target_port, ident_tmp);

	untaint(buf, sizeof(buf));

	if (res != 8) goto lookup_mismatch;
	TERMINATE_STRING(ident_tmp);

	if ((rport != client_port) || (lport != local_port) ||
		(data->target_port < 0) || (data->target_port > 65535) ||
		(ip1 < 0) || (ip1 > 255) || (ip2 < 0) || (ip2 > 255) ||
		(ip3 < 0) || (ip3 > 255) || (ip4 < 0) || (ip3 > 255)) 
		goto lookup_mismatch;

	res = snprintf(data->target_str, sizeof(data->target_str), "%d.%d.%d.%d", ip1, ip2, ip3, ip4);
	if (res == -1 || res > sizeof(data->target_str)) return LOOKUP_NOMEM;

	TERMINATE_STRING(data->target_str);

	data->target.sin_family = AF_INET;
	data->target.sin_port = htons(data->target_port);
	if (!inet_aton(data->target_str, &data->target.sin_addr))
		goto lookup_mismatch;

	strncpy(data->ident, ident_tmp, sizeof(data->ident));
	TERMINATE_STRING(data->ident);

	// LOOKUP_RFC, LOOKUP_EXTENDED
	return LOOKUP_OK;

lookup_mismatch:
	log_action(LOG_ERR, "Lookup mismatch from %s [%s]", data->origin_str, buf);
	return LOOKUP_MISMATCH;

fail:
	res = errno;
	CLEAR_TIMEOUT();
	if (sock != -1) SAFE_CLOSE(sock);

	if (res == EINTR || res == ETIMEDOUT) return LOOKUP_TIMEOUT;
	errno = res;
	return -1;
} /* ident_lookup() */




