/*
 *	scan.c
 *
 *	Copyright (C) 2004-2005 Bartomiej Korupczynski <bartek@klolik.org>
 *
 *	This program is free software; you can redistribute it and/or 
 *	modify it under the terms of the GNU General Public License 
 *	as published by the Free Software Foundation; either 
 *	version 2 of the License, or (at your option) any later 
 *	version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE

#include <stdio.h>
#include <unistd.h>
#include <sys/un.h>
#ifdef HAVE_ERR_H
#include <err.h>
#endif
#include <syslog.h>
#include <errno.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


#define _SCAN_C_
#include "confvars.h"
#include "conffile.h"
#include "smtp-gated.h"
#include "scan.h"
#include "util.h"


#ifdef SCANNER_MKSD
#include "libmksd.h"
#endif



/*
 *	all scanners must be prepared to receive EINTR (which means timeout)
 *	and check 'timedout' variable
*/

/*
 *	SpamAssassin
*/

spam_result spam_scanner(char *filename, double *score)
{
	int sock = -1;
	int file, res, len, pos;
	char buf[32768];
	char version[32];
	double temp_thr;
	struct stat st;
	long int size;


	// spool open
	if ((file = open(filename, O_RDONLY)) == -1) {
		log_action(LOG_CRIT, "spamd:open(%s): %s", filename, strerror(errno));
		goto fail;
	}

	if (fstat(file, &st) == -1) {
		log_action(LOG_CRIT, "spamd:stat(%s): %s", filename, strerror(errno));
		goto fail;
	}

	/* BSD compat */
	size = st.st_size;

	if ((sock = connect_path(config.spamd_path)) == -1) {
		log_action(LOG_CRIT, "spamd:connect_path(%s) error: %s", config.spamd_path, strerror(errno));
		goto fail;
	}

	// lookup.c:71: warning: passing arg 2 of `getsockname' from incompatible pointer type
	if (fdprintf(sock, "CHECK SPAMC/1.2\r\nContent-length: %ld\r\n\r\n", size) == -1) {
		log_action(LOG_CRIT, "spamd:fdprintf(sock) failed: %s", strerror(errno));
		goto fail;
	}

#ifdef USE_SHARED_MEM
	set_dump_state(CONN_SPAM1);
#endif

	for (;;) {
		if (timedout) {
			log_action(LOG_WARNING, "spamd:read(file):timeout");
			errno = ETIMEDOUT;
			goto fail;
		}

		if ((len = read(file, buf, sizeof(buf))) == -1) {
			// SIGALRM sets timedout flags
			if (errno == EINTR) continue;
			
			log_action(LOG_CRIT, "spamd:read(file) error: %s", strerror(errno));
			goto fail;
		}

		if (len == 0) break;

//		if ((res = send(sock, buf, len, 0)) != len) {
		for (pos=0;;) {
			if (timedout) {
				log_action(LOG_WARNING, "spamd:write(sock):timeout");
				errno = ETIMEDOUT;
				goto fail;
			}
			if ((res = write(sock, buf+pos, len)) == -1) {
				if (errno == EINTR) continue;

				log_action(LOG_CRIT, "spamd:write(sock) error: %s", strerror(errno));
				goto fail;
			}

			if (res == 0) break;

			len -= res;
			if (len == 0) break;
			pos += res;
		}
	}

	SAFE_CLOSE(file);
	file = -1;
	shutdown(sock, SHUT_WR);

#ifdef USE_SHARED_MEM
	set_dump_state(CONN_SPAM2);
#endif
	memset(buf, 0, sizeof(buf));
	for (pos=0;;) {
		if (timedout) {
			log_action(LOG_WARNING, "spamd:read(sock) timeout");
			errno = ETIMEDOUT;
			break;
		}
		if ((res = read(sock, buf+pos, sizeof(buf)-pos)) == -1) {
			if (errno == EINTR) continue;

			log_action(LOG_CRIT, "spamd:read(sock) error: %s", strerror(errno));
			goto fail;
		}

		if (res == 0) break;
		pos += res;
	}

	TERMINATE_STRING(buf);
	SAFE_CLOSE(sock);
	sock = -1;

//	log_action(LOG_DEBUG, "spamd: returned [%s]", buf);

	if (sscanf(buf, "SPAMD/%30s 0 EX_OK\r\nSpam: %*s ; %lf / %lf \r\n", version, score, &temp_thr) != 3) {
		log_action(LOG_CRIT, "spamd:sscanf() cannot parse output [%s]", buf);
		goto fail;
	}

//	alahutdoown
	//	m(0);
//	return (*score > spam_threshold) ? SPAM_YES : SPAM_NO;
	return SPAM_OK;

fail:
	res = (errno == EINTR || errno == ETIMEDOUT) ? SPAM_TIMEOUT : SPAM_FAILED;
//	alarm(0);
	if (file != -1) SAFE_CLOSE(file);
	if (sock != -1) SAFE_CLOSE(sock);

	return res;
}


/*
 *	Clam Antivirus daemon
*/

#ifdef SCANNER_CLAMD
av_result av_scanner(char *filename, char **result)
{
	static char buf[4800];
	int sock, pos, res;
	char *p;


	*result = NULL;

	if ((sock = connect_path(config.clamd_path)) == -1) {
		log_action(LOG_CRIT, "clamd:connect_path(%s) error: %s", config.clamd_path, strerror(errno));
		goto fail;
	}
#if 0
	if ((sock = socket(AF_UNIX, SOCK_STREAM, 0)) == -1) {
		log_action(LOG_CRIT, "clamd: unable to create socket (%s)", strerror(errno));
		goto fail;
	}

	sun.sun_family = AF_UNIX;
	strncpy(sun.sun_path, clamd_path, sizeof(sun.sun_path));
	TERMINATE_STRING(sun.sun_path);

	if (connect(sock, (struct sockaddr *) &sun, sizeof(sun)) != 0) {
		log_action(LOG_CRIT, "clamd:connect(%s) error: %s", clamd_path, strerror(errno));
		goto fail;
	}
#endif

	if (fdprintf(sock, "SCAN %s\n", filename) == -1) {
		if (errno != EINTR && errno != ETIMEDOUT)
			log_action(LOG_CRIT, "clamd:fdprintf(SCAN...): %s", strerror(errno));
		goto fail;
	}

#ifdef USE_SHARED_MEM
	set_dump_state(CONN_SCAN1);
#endif

	// konieczne!
//	memset(buf, '\0', sizeof(buf));
	for (pos=0;;) {
		if (timedout) {
			CLEAR_TIMEOUT();
//			log_action(LOG_ERR, "clamd:timeout");
			errno = ETIMEDOUT;
			goto fail;
		}

		if ((res = read(sock, buf+pos, sizeof(buf)-pos)) == -1) {
			if (errno == EINTR) continue;
			log_action(LOG_CRIT, "clamd:read(sock) error: %s", strerror(errno));
			goto fail;
		}

		if (res == 0) break;
		pos += res;
	}

//	alarm(0);
	SAFE_CLOSE(sock);
	sock = -1;

	if (pos >= sizeof(buf)) {
		log_action(LOG_CRIT, "clamd:buffer to small");
		return SCAN_FAILED;
	}
	buf[pos-1] = '\0';
	TERMINATE_STRING(buf);

/*
	"%s: %s FOUND\n", filename, virname
	"%s: %s ERROR\n", filename, cl_strerror(ret)
	"%s: OK\n"
	"%s: Empty file\n", filename
*/
	// usuwamy '\n' z konca
	if ((p = strrchr(buf, '\n'))) {
		*p = '\0';
	}

	// szukamy pierwszego dwukropka
	p = strchr(buf, ':');
	if (p == NULL) {
		log_action(LOG_CRIT, "clamd:parse malformed response (%s)", buf);
		return SCAN_FAILED;
	}

	p++;
	while (*p == ' ') p++;
	*result = p;

	// szukamy ostatniej spacji
	p = strrchr(buf, ' ');
	if (p == NULL) {
		log_action(LOG_CRIT, "clamd:parse malformed response (%s)", buf);
		return SCAN_FAILED;
	}
	*p = '\0';
	p++;

	if (strcmp(p, "ERROR") == 0) return SCAN_FAILED;
	if (strcmp(p, "FOUND") == 0) return SCAN_VIRUS;
	if (strcmp(p, "OK") == 0) {
		*result = NULL;
		return SCAN_OK;
	}

	log_action(LOG_WARNING, "clamd:returned (%s %s %s)", buf, *result, p);
	return SCAN_OK;

fail:
	res = (errno == EINTR || errno == ETIMEDOUT) ? SCAN_TIMEOUT : SCAN_FAILED;
//	alarm(0);
	if (sock != -1) SAFE_CLOSE(sock);

	return res;
}
#endif



/*
 * 	mks_vir daemon
*/

#ifdef SCANNER_MKSD
av_result av_scanner(char *filename, char **result)
{
	char opts[] = "S\0";
	static char buf[4200];
	char *cur;
	int code;


	*result = NULL;
	memset(buf, '\0', sizeof(buf));

//	alarm(timeout_scanner);

	if (mksd_connect() < 0) {
		log_action(LOG_CRIT, "MKSD: connect failed");
		return SCAN_FAILED;
	}


	memset(buf, 0, sizeof(buf));
	if (mksd_query(filename, opts, buf) < 0) {
		log_action(LOG_CRIT, "MKSD:query failed");
		return SCAN_FAILED;
	}

	mksd_disconnect();
//	alarm(0);

	if (memcmp(buf, "OK ", 3) == 0) return SCAN_OK;
	if (memcmp(buf, "CLN ", 4) == 0) return SCAN_OK;
	if (memcmp(buf, "DEL ", 4) == 0) return SCAN_OK;

	cur = NULL;
	code = 0;
	if (memcmp(buf, "ERR ", 4) == 0) {
		cur = buf + 4;
		code = SCAN_FAILED;
	}

	if (memcmp(buf, "VIR ", 4) == 0) {
		cur = buf + 4;
		code = SCAN_VIRUS;
	}

	if (cur != NULL) {
		*result = cur;
		for (;; cur++) {
			if (*cur == ' ') break;
			if (*cur == '\0') break;

			if (cur+1 >= buf+sizeof(buf)) break;
		}
		*cur = '\0';

		return code;
	}

	buf[16] = '\0';
	log_action(LOG_ERR, "MKSD:unknown result [%s]", buf);
	return SCAN_FAILED;

}
#endif


/*
 *	mks_vir stand-alone
*/

#ifdef SCANNER_MKS32
#error SIGCHLD removed from signal handling
av_result av_scanner(char *filename, char **name)
{
	struct timeval tv;
	int pid, res;
	char *arg[] = { scanner_path, "--exit", filename, NULL};
	char *env[] = { "PATH=/usr/local/bin:/usr/bin:/bin", NULL };

	name = NULL;

	pid = fork();
	if (pid < 0) {
		log_action(LOG_CRIT, "Fork failed: %s", strerror(errno));
		return SCAN_FAILED;
	}

	child_status = 0;

	if (!pid) { // dzieciak
		res = execve(scanner_path, arg, env);

		// nie powinno tu dotrzec
		log_action(LOG_CRIT, "execve failed: %s", strerror(errno));
		exit(255);
	} else {
		tv.tv_sec = timeout_scanner;
		tv.tv_usec = 0;
		res = select(0, NULL, NULL, NULL, &tv);
		if (res == 0) {
			// timeout

			kill(pid, SIGTERM);
			return SCAN_TIMEOUT;
		}

		// EINTR <= SIGCHLD
		if (errno != EINTR) {
			log_action(LOG_ERR, "select: %s", strerror(errno));
			return SCAN_FAILED;
		}
	}

	child_reaper();
	if (WIFEXITED(child_status) == 0) return SCAN_FAILED;
	
	res = WEXITSTATUS(child_status);
	if (res > 0x07) return SCAN_FAILED;
	if (res & 0x07) return SCAN_VIRUS;

	if (res == 0) return SCAN_OK;
	return SCAN_FAILED;
}
#endif


