/*
 * 	smtp-gated.c
 *
 *	SMTP Transparent Proxy
 *	Bartlomiej Korupczynski
 *	http://smtp-proxy.klolik.org
 *	(c) Warszawa 2004-2005
 *	GNU GPL License
 *
 *	Copyright (C) 2004-2005 Bartomiej Korupczynski <bartek@klolik.org>
 *
 *	This program is free software; you can redistribute it and/or 
 *	modify it under the terms of the GNU General Public License 
 *	as published by the Free Software Foundation; either 
 *	version 2 of the License, or (at your option) any later 
 *	version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*
 * 	headers
*/

#define _GNU_SOURCE

#define _SMTP_GATED_C_

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include <locale.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <assert.h>
//#include <err.h>

#ifdef USE_PGSQL
#include <libpq-fe.h>
#error USE_PGSQL not yet implemented
#endif

#ifdef USE_REGEX
#error USE_REGEX not yet implemented
#endif

#include "confvars.h"
#include "smtp-gated.h"
#include "lang.h"
#include "conffile.h"
#include "scan.h"
#include "dump.h"
#include "lockfile.h"
#include "spool.h"
#include "lookup.h"
#include "util.h"
#include "daemon.h"
#include "md5.h"
#include "compat.h"
#include "action.h"

//#define MEMLEAK_TESTING		4

/*
 * 	constants
*/

#define CONN_REJ_ER		(struct response_code) {CONN_REJ_CODE,3,0}

#define FORE_LOG_TRAFFIC	0x01
#define FORE_SIGNAL_PPID	0x02


//#define NO_FORK
static const int one = 1;

// max. 8 chars
char *conn_states[] = {
       	"start", "helo",
       	"ident", "connect", "pre",
       	"data", "bdat", "direct",
	"scan", "scan1", "scan2", "spam", "spam1", "spam2",
	"post", "rset", "quit"
};


/*
	global variables
*/

char *config_file = NULL;

int child_status = 0;
int children = 0;
int i_am_a_child = 0;	

volatile sig_atomic_t force_reconfig = 0;
volatile sig_atomic_t force_finish = 0;
volatile sig_atomic_t force_dump = 0;
volatile sig_atomic_t child_died = 0;
volatile sig_atomic_t timedout = 0;

#ifdef USE_SHARED_MEM
// child_slot i i_am_a_child sie poniekad dubluja
#define I_AM_A_CHILD()	(child_slot != -1)
int child_slot = -1;
int conn_shmid = -1;
int stat_shmid = -1;
#endif

pid_t *pids = NULL;
struct conn_info *connections = NULL;
struct stat_info *stats = NULL;

int max_connections_real;



/*
 * 	configuration variables
*/

char *msg_success = "Success / Sukces";


char **lookup_errors[] = {
	&msg_success,		// LOOKUP_OK
	&config.msg_lookup_timeout,	// LOOKUP_TIMEOUT
	&config.msg_lookup_mismatch,	// LOOKUP_MISMATCH
	&config.msg_lookup_nomem	// LOOKUP_NOMEM
};


/*
 * 	forward functions
*/

void cleanup();
void lockfile_action(struct proc_data *data, char *cause);
void dump_state();



/*
 * 	deklaracje funkcji wprzod
*/

void wait_for_quit(struct proc_data *data, char* format, ...);

/*
 * 	funkcje pomocnicze
*/

inline int min(int a, int b)
{
	return (a < b) ? a : b;
}


inline void upcase(char *str)
{
	if (str == NULL) return;

	for (; str[0] != '\0'; str++) {
		if (str[0] >= 'a' && str[0] <= 'z') str[0] -= ('a'-'A');
	}
}

inline int resp_code(char *s)
{
	if (s[0]<'0' || s[0]>'9' || s[1]<'0' || s[1]>'9' || s[2]<'0' || s[2]>'9') return -1;

	return 100*((int) (s[0]-'0')) + 10*((int) (s[1]-'0')) + ((int) (s[2]-'0'));
}



/*
 * 	Logowanie
*/

#ifdef USE_SHARED_MEM
void set_dump_state(conn_state s)
{
	connections[child_slot].state = s;
}
#endif


/*
	return:
	=0	success
	<0	errno
	1	data truncated
*/

// wyslij odpowiedz code " " [ class "." subject "." detail ]
// code = tradycyjny kod odpowiedzi
// esc2 = subject kodu rozszerzonego
// esc3 = detail kodu rozszerzonego
//
#define NR(generic)			(struct response_code) {generic, -1, -1}
#define ER(generic,subject,detail)	(struct response_code) {generic, subject, detail}

int response(struct proc_data *data, struct response_code resp, char *format, ...)
	__attribute__ ((format (printf, 3, 4)));


int response(struct proc_data *data, struct response_code resp, char *format, ...)
{
	int class, res, len;
	va_list ap;
	char buf[PRINTF_SIZE];
	char *cur;

	class = resp.generic / 100;

	if (data->enhancedstatuscodes && (class == 2 || class == 4 || class == 5) && (resp.subject != -1)) {
		// kod rozszerzony
		len = snprintf(buf, sizeof(buf), "%03d %d.%d.%d ", resp.generic, class, resp.subject, resp.detail);
	} else {
		// kod zwykly
		len = snprintf(buf, sizeof(buf), "%03d ", resp.generic);
	}

	if (len == -1 || len > sizeof(buf)) {	// glibc 2.0.6 and glibc 2.1
		errno = ENOMEM;
		return -1;
	}

	va_start(ap, format);
	res = vsnprintf(buf+len, sizeof(buf)-len, format, ap);
	va_end(ap);

	if (res == -1 || res > sizeof(buf)-len) {
		errno = ENOMEM;
		return -1;
	}

	len += res;
	cur = buf;
	while (len > 0) {
		if (timedout) {
			CLEAR_TIMEOUT();
			errno = ETIMEDOUT;
			return -1;
		}
		
	       	if ((res = write(data->client, cur, len)) == -1) {
//			if (errno == EINTR) continue;
			return -1;
		}

		if (res == 0) {
			// a nie ECONNRESET?
			errno = ETIMEDOUT;
			return -1;
		}

		len -= res;
		cur += res;
	}

	return res;
} /* response() */

/*
 * 	usuwanie i przywracanie CRLF
*/

inline void save_remove_crlf(char *buffer, char *pos, char **replace_pos, char *replace_char)
{
	if ((pos > buffer) && (*(pos-1) == '\r')) {
		*replace_pos = pos-1;
		*replace_char = **replace_pos;
		**replace_pos = '\0';
	} else {
		*replace_pos = pos;
		*replace_char = **replace_pos;
		**replace_pos = '\0';
	}
} /* rave_remove_crlf() */

inline void restore_crlf(/*@unused@*/ char *buffer, /*@unused@*/ char *pos, char **replace_pos, char *replace_char)
{
	**replace_pos = *replace_char;
} /* restore_crlf() */

/*
 * 	pipeline - kolejkowanie
*/

int queue_commandp(smtp_command command, struct proc_data *data, pipeline_parm_t parm1, pipeline_parm_t parm2)
{
	struct pipeline_entry *ce;
	
	assert(data->command_pos_client < config.pipeline_size);

//	log_action(LOG_DEBUG, "QUEUE:QUEUE %d at %d", command, data->command_pos_client);
	ce = &data->pipeline[data->command_pos_client];
	ce->cmd = command;
	ce->parm1 = parm1;
	ce->parm2 = parm2;

	data->command_pos_client++;
	data->command_pos_client %= config.pipeline_size;

	// queue full
	if (data->command_pos_client == data->command_pos_server) return -1;
	return 0;
} /* queue_commandp() */

inline int queue_command(smtp_command command, struct proc_data *data)
{
	return queue_commandp(command, data, 0, 0);
} /* queue_command() */

smtp_command poll_commandp(struct proc_data *data, pipeline_parm_t *parm1, pipeline_parm_t *parm2)
{
	smtp_command command;
	struct pipeline_entry *ce;

	assert(data->command_pos_server < config.pipeline_size);

	if (data->command_pos_client == data->command_pos_server) {
//		log_action(LOG_DEBUG, "QUEUE:POLL no commands queued");
		return COMMAND_NONE;
	}

	ce = &data->pipeline[data->command_pos_server];
	command = ce->cmd;
	if (parm1) *parm1 = ce->parm1;
	if (parm2) *parm2 = ce->parm2;

//	log_action(LOG_DEBUG, "QUEUE:POLL %d at %d", command, data->command_pos_server);
	return command;
} /* poll_commandp() */

inline void dequeue_command(struct proc_data *data)
{
	// move to next
	data->command_pos_server++;
	data->command_pos_server %= config.pipeline_size;
} /* dequeue_command() */


/*
 * 	signals
*/

void child_reaper()
{
	pid_t res;
	int i;

	for (;;) {
		res = wait3((int *) &child_status, WNOHANG, NULL);

		// brak kolejnych bachorow (teoretycznie)
		if (res == 0) break;

		// error
		if (res == -1) {
			if (errno == ECHILD) break;
			if (errno == EINTR) continue;
			
			log_action(LOG_ERR, "!ERR! wait3() returned: %s", strerror(errno));
			break;
		}

		if (WIFSIGNALED(child_status)) {
			log_action(LOG_ALERT, "!BUG! Child %" FORMAT_PID_T " exited on signal %d", res, WTERMSIG(child_status));
		}

		if (!i_am_a_child && connections) {
			if ((foreground & FORE_SIGNAL_PPID) != 0) {
				if (kill(getppid(), SIGUSR1) != 0) {
					log_action(LOG_ERR, "kill(%" FORMAT_PID_T ",SIGUSR1) [on SIGCHLD] error: %s", getppid(), strerror(errno));
				} else {
					log_action(LOG_ERR, "parent pid %" FORMAT_PID_T " signaled, child gone", getppid());
				}	
			}

			for (i=0; i<max_connections_real; i++) {
				if (pids[i] != res) continue;
				pids[i] = 0;
#ifdef USE_SHARED_MEM
				connections[i].ident_ok = 0;
#endif
			}
		}

		if (children > 0) children--;
	}
} /* child_reaper() */

void got_signal(int sig)
{
	if (sig == SIGCHLD) {
		child_died = 1;
		return;
	} else if (sig == SIGALRM) {
		timedout = 1;
		return;
	} else if (sig == SIGHUP) {
		force_reconfig = 1;
		return;
	} else if (sig == SIGUSR1) {
		force_dump = 1;
		return;
	} else if (sig == SIGUSR2) {
		return;
	} else if (sig == SIGTERM) {
		force_finish = 1;
	} else if (sig == SIGQUIT) {
		force_finish = 1;
	} else {
		log_action(LOG_INFO, "Received signal %s [%d], exiting.", strsignal(sig), sig);
		cleanup();
		exit(0);
	}
} /* got_signal() */

void cleanup()
{
	if (!i_am_a_child) {
		remove_pidfile();
#ifdef USE_SHARED_MEM
		shmfreeid(conn_shmid);
		shmfreeid(stat_shmid);
	} else {
		if (child_slot != -1) {
			SHARED_CONN_STATUS(ident_ok, 0);
		}
#endif
	}
} /* cleanup() */

int setup_signal()
{
	struct sigaction sa;

	sa.sa_handler = &got_signal;
	sigfillset(&sa.sa_mask);
	sa.sa_flags = 0;

	if (sigaction(SIGHUP, &sa, NULL) != 0) return -1;
	if (sigaction(SIGINT, &sa, NULL) != 0) return -1;
	if (sigaction(SIGQUIT, &sa, NULL) != 0) return -1;
	if (sigaction(SIGUSR1, &sa, NULL) != 0) return -1;
	if (sigaction(SIGUSR2, &sa, NULL) != 0) return -1;

	if (sigaction(SIGTERM, &sa, NULL) != 0) return -1;
	if (sigaction(SIGALRM, &sa, NULL) != 0) return -1;
	if (sigaction(SIGCHLD, &sa, NULL) != 0) return -1;

	sa.sa_handler = SIG_IGN;

	if (sigaction(SIGPIPE, &sa, NULL) !=0) return -1;

	return 0;
} /* setup_signal() */



/*
 * 	line reading
*/


// TODO flush_buffers()
// TODO: wysle drugi raz np. BDAT ...
// mysi pominac to co juz trafilo do func()
void flush_buffers(struct proc_data *data)
{
	ssize_t res;
	char *buf;

	buf = data->cli_buf;
	while (data->cli_size > 0) {
		res = write(data->server, buf, data->cli_size);
		if (res == -1) {
			if (errno == EINTR) continue;
			break;
		}

		data->cli_size -= res;
		buf += res;
	}

	buf = data->srv_buf;
	while (data->srv_size > 0) {
		res = write(data->client, buf, data->srv_size);
		if (res == -1) {
			if (errno == EINTR) continue;
			break;
		}

		data->srv_size -= res;
		buf += res;
	}
} /* flush_buffers() */



inline void helo(int client)
{
	fdprintf(client, "220 %s SMTP\r\n", config.proxy_name);
} /* helo() */

/*
 * oczekiwanie na QUIT
 * symuluje sesje SMTP, w oczekiwaniu na QUIT klienta
*/

line_status local_callback(char* buffer, char* pos, /*@unused@*/ int size, void *ptr)
{
	char *replace_pos, replace_char;
	struct proc_data *data = ptr;

	save_remove_crlf(buffer, pos, &replace_pos, &replace_char);

	if ((foreground & FORE_LOG_TRAFFIC) != 0) fprintf(stderr, "[%" FORMAT_PID_T "] <= %s\n", getpid(), buffer);

	// close after couple of meaningless commands
	data->command_count++;
	if (data->command_count > 10) {
//		fdprintf(data->client, "554 %s %s\r\n", config.proxy_name, config.msg_transaction_failed);
		response(data, ER(554,5,0), "%s %s\r\n", config.proxy_name, config.msg_transaction_failed);
		return LINE_CLOSED;
	}

	if (strcasecmp(buffer, "QUIT") == 0) {
//		fdprintf(data->client, "221 %s %s\r\n", config.proxy_name, config.msg_sign_off);
		response(data, ER(221,0,0), "%s %s\r\n", config.proxy_name, config.msg_sign_off);
		return LINE_CLOSED;
	} else if ((strcasecmp(buffer, "NOOP") == 0) || 
		   (strcasecmp(buffer, "RSET") == 0)) {
//		fdprintf(data->client, "250 OK\r\n");
		response(data, ER(250,0,0), "OK\r\n");
	} else if ((strncasecmp(buffer, "EHLO ", 5) == 0) ||
		   (strncasecmp(buffer, "HELO ", 5) == 0)) {
//		fdprintf(data->client, "250 %s %s\r\n", config.proxy_name, config.msg_hello);
		response(data, ER(250,0,0), "%s %s\r\n", config.proxy_name, config.msg_hello);
	} else if (((strncasecmp(buffer, "RCPT TO:", 5) == 0)) ||
		   ((strcasecmp(buffer, "DATA") == 0))) {
//		fdprintf(data->client, "503 %s %s\r\n", config.proxy_name, config.msg_proto_error);
		response(data, ER(503,5,1), "%s %s\r\n", config.proxy_name, config.msg_proto_error);
	} else if ((strncasecmp(buffer, "MAIL FROM:", 5) == 0)) {
//		fdprintf(data->client, "451 %s %s\r\n", config.proxy_name, data->message);
		response(data, ER(451,3,2), "%s %s\r\n", config.proxy_name, data->message);
	} else {
//		fdprintf(data->client, "502 %s %s\r\n", config.proxy_name, config.msg_unimpl_command);
		response(data, ER(502,5,2), "%s %s\r\n", config.proxy_name, config.msg_unimpl_command);
	}

	return LINE_OK;
} /* local_callback() */

// bez \r\n na koncu!

void wait_for_quit(struct proc_data* data, char* format, ...)
	__attribute__ ((format (printf, 2, 3)));
	
void wait_for_quit(struct proc_data* data, char* format, ...)
{
	va_list ap;
	char cli_buf[256];
	int cli_size, res;

	// inicjacja unii dla wait_for_quit
	data->command_count = 0;

	va_start(ap, format);
	if (vasprintf(&data->message, format, ap) == -1) {
		data->message = NULL;
	}
	va_end(ap);

	SHARED_CONN_STATUS(state, CONN_QUIT);
	log_action(LOG_WARNING, "SESSION TAKEOVER: src=%s, ident=%s, trns=%d, reason=%s",
		data->origin_str, data->ident, data->transaction, data->message);

	cli_size = 0;
	SET_TIMEOUT(config.timeout_session);

	for (;;) {
		assert(cli_size < sizeof(cli_buf));

		if (timedout) {
			CLEAR_TIMEOUT();
			res = LINE_EINTR;
			response(data, ER(421,5,0), "%s %s\r\n", config.proxy_name, config.msg_session_timeout);
			log_action(LOG_INFO, "CLOSE:TAKEN %s", config.msg_session_timeout);
			break;
		}

		res = fdgetline_cb(data->client, cli_buf, sizeof(cli_buf), &cli_size, &local_callback, data, NULL);
/*
		if (res == LINE_EINTR) {
			CLEAR_TIMEOUT();
			response(data, ER(421,5,0), "%s %s\r\n", config.proxy_name, config.msg_session_timeout);
			log_action(LOG_INFO, "CLOSE:TAKEN %s", config.msg_session_timeout);
			break;
		}
*/
		if (res == LINE_CLOSED) break;
	}

	SAFE_CLOSE(data->client);
	cleanup();
	
	if (res != LINE_EINTR) log_action(LOG_DEBUG, "CLOSE:TAKEN");
	exit(5);
} /* wait_for_quit() */


/*
 * polaczenie pass-thru
 * zestawia "tunel" pomiedzy serwerem i klientem, nie ingeruje w przesylane dane.
*/

void direct_proxy(struct proc_data *data)
{
	fd_set rfds;
	struct timeval tv;
	int max_fd, size;
	int res;
	char buffer[config.buffer_size];


	CLEAR_TIMEOUT();

	// deskryptor dla select-a
	max_fd = (data->client > data->server) ? data->client : data->server;

	SHARED_STATS_INC(requests_direct);
	SHARED_CONN_STATUS(state, CONN_DIRECT);

	for (;;) {
		if (force_finish) {
			res = LINE_CLOSED_ADMIN;
			break;
		}

		FD_ZERO(&rfds);
		FD_SET(data->client, &rfds);
		FD_SET(data->server, &rfds);
		tv.tv_sec = config.timeout_direct;
		tv.tv_usec = 0;

		if ((res = select(max_fd+1, &rfds, NULL, NULL, &tv)) == -1) {
			log_action(LOG_ERR, "select: %s", strerror(errno));
			continue;
		}

		// timeout
		if (res == 0) {
			res = LINE_CLOSED_TIMEOUT;
			break;
		}

		// dane od klienta
		if (FD_ISSET(data->client, &rfds)) {
			if ((size = read(data->client, buffer, sizeof(buffer))) == -1) {
				log_action(LOG_ERR, "direct:client:read error: %s", strerror(errno));
				res = LINE_ERROR;
				break;
			} else if (size == 0) {
				res = LINE_CLOSED;
				break;
			} else {
				data->cli_rx += size;
				SHARED_CONN_STATUS(cli_rx, data->cli_rx);
				write(data->server, buffer, size);
			}
		}

		// dane z serwera
		if (FD_ISSET(data->server, &rfds)) {
			if ((size = read(data->server, buffer, sizeof(buffer))) == -1) {
				log_action(LOG_ERR, "direct:server:read error: %s", strerror(errno));
				res = LINE_ERROR;
				break;
			} else if (size == 0) {
				res = LINE_CLOSED_SERVER;
				break;
			} else {
				data->srv_rx += size;
				SHARED_CONN_STATUS(srv_rx, data->srv_rx);
				write(data->client, buffer, size);
			}
		}

	}

	SAFE_CLOSE(data->client);
	SAFE_CLOSE(data->server);

	cleanup();
	
	log_action(LOG_NOTICE|LOG_ALWAYS, "CLOSE:DIRECT by=%s rcv=%d/%d, time=%" FORMAT_TIME_T ", src=%s, ident=%s",
		line_closed_cause(res), data->cli_rx, data->srv_rx, time(NULL)-data->time,
		data->origin_str, data->ident);
	exit(0);
} /* direct_proxy */


/*
 * 	logowanie adresow z naglowka
 *	zakladamy: line w postaci zakonczonej '\0'
*/

/*
 *	naglowek wiadomosci
*/

void prepare_nat_header(struct proc_data *data)
{
//	char *buf_from, *buf_ident, *buf_with, *buf_abuse;
	char buf_from[128], buf_ident[16+IDENT_SIZE], buf_with[128], buf_abuse[128];

	if (config.nat_header_type == 2) {
		data->header_size = asprintf(&data->header, "%s: %s\r\n", config.nat_header, inet_ntoa(data->origin.sin_addr));
		if (data->header_size == -1) {
			data->header = NULL;
			data->header_size = 0;
		}

		return;
	}
		
	snprintf(buf_from, sizeof(buf_from), "%s: from [%s]:%d ",
		config.nat_header, inet_ntoa(data->origin.sin_addr), data->origin_port);
	TERMINATE_STRING(buf_from);

	if (!EMPTY_STRING(data->ident)) {
		snprintf(buf_ident, sizeof(buf_ident), "[ident %s]\r\n", data->ident);
	} else {
		snprintf(buf_ident, sizeof(buf_ident), "[ident-empty]\r\n");
	}
	TERMINATE_STRING(buf_ident);

	snprintf(buf_with, sizeof(buf_with), "\tby %s with TPROXY id %" FORMAT_TIME_T ".%" FORMAT_PID_T "\r\n",
		config.proxy_name, data->time, getpid());
	TERMINATE_STRING(buf_with);

	if (!EMPTY_STRING(config.abuse)) {
		snprintf(buf_abuse, sizeof(buf_abuse), "\tabuse-to %s\r\n", config.abuse);
	} else {
		buf_abuse[0] = '\0';
	}
	TERMINATE_STRING(buf_abuse);

	data->header_size = asprintf(&data->header, "%s%s%s%s", buf_from, buf_ident, buf_with, buf_abuse);
	if (data->header_size == -1) {
		data->header = NULL;
		data->header_size = 0;
	}
} /* prepare_nat_header() */


/*
 * 	obsluga logowania mail from/rcpt to
*/


void destroy_mail_from(struct proc_data *data)
{
	if (data->mail_from) {
		free(data->mail_from);
		data->mail_from = NULL;
		data->mail_from_logged = 0;
	}
} /* destroy_mail_from() */

void destroy_rcpt_to(struct proc_data *data)
{
	int i;

	for (i=0; i<RCPTS_ONE_TIME; i++) {
		if (!data->rcpt_to[i]) continue;

		free(data->rcpt_to[i]);
		data->rcpt_to[i] = NULL;
	}
	data->rcpts = 0;
} /* destroy_rcpt_to() */

inline void destroy_addresses(struct proc_data *data)
{
	destroy_mail_from(data);
	destroy_rcpt_to(data);
}

void flush_addresses(struct proc_data *data)
{
	if ((!data->mail_from || data->mail_from_logged) && !data->rcpts) return;

	if (data->mail_from && !data->mail_from_logged) {
		if (!data->rcpts) {
			log_action(LOG_INFO|LOG_ALWAYS, "MAIL FROM <%s>", data->mail_from);
			data->mail_from_logged = 1;
			return;
		} else {
			log_action(LOG_INFO|LOG_ALWAYS, "MAIL FROM <%s> RCPT TO: %03d<%s>", 
				data->mail_from, data->rcpt_to_code[0], data->rcpt_to[0]);
			data->mail_from_logged = 1;
			
			free(data->rcpt_to[0]);
			data->rcpt_to[0] = data->rcpt_to[1];
			data->rcpt_to_code[0] = data->rcpt_to_code[1];
			data->rcpt_to[1] = NULL;
			
			data->rcpts--;
			return;
		}
	}

	// data->rcpts > 0
	if (data->rcpts == 1) {
		log_action(LOG_INFO|LOG_ALWAYS, "RCPT TO: %03d<%s>",
			data->rcpt_to_code[0], data->rcpt_to[0]);
	} else {
		log_action(LOG_INFO|LOG_ALWAYS, "RCPT TO: %03d<%s>, %03d<%s>",
			data->rcpt_to_code[0], data->rcpt_to[0],
			data->rcpt_to_code[1], data->rcpt_to[1]);
	}

//	destroy_addresses(data);
	destroy_rcpt_to(data);
} /* flush_addresses() */

// email is malloc-ed
void new_mail_from(struct proc_data *data, char *email, int code)
{
	char *tmp;
	int accepted = (code == 250);

	
	if ((config.log_mail_from & (accepted ? LOG_MAIL_ACCEPTED : LOG_MAIL_REJECTED)) == 0)
		return;

	if (config.log_mail_from & LOG_MAIL_BASE64) {
		tmp = strdup(md5_string_base64(email));
		free(email);
		email = tmp;
	} else {
		untaint(email, -1);
	}

	if (!accepted) {
		log_action(LOG_NOTICE|LOG_ALWAYS, "MAIL FROM <%s> rejected [%d]", email, code);
		free(email);
		return;
	}
	
	if (data->mail_from) free(data->mail_from);
	data->mail_from = email;
	data->mail_from_logged = 0;
} /* net_mail_from() */

// email is malloc-ed
void add_rcpt_to(struct proc_data *data, char *email, int code)
{
	char *tmp;

	if ((config.log_rcpt_to & (code == 250 ? LOG_MAIL_ACCEPTED : LOG_MAIL_REJECTED)) == 0)
		return;
	
	if (config.log_rcpt_to & LOG_MAIL_BASE64) {
		tmp = strdup(md5_string_base64(email));
		free(email);
		email = tmp;
	} else {
		untaint(email, -1);
	}

	data->rcpt_to[data->rcpts] = email;
	data->rcpt_to_code[data->rcpts] = code;
	data->rcpts++;

	if ((data->rcpts >= RCPTS_ONE_TIME) || (data->mail_from && !data->mail_from_logged))
		flush_addresses(data);
} /* add_rcpt_to() */


/*
 * 	funkcje obslugi sesji
*/

void transaction_reset(struct proc_data *data)
{
	data->data_used = 0;
	data->bdat_used = 0;
	data->bdat_togo = 0;
	data->bdat_last = 0;

	data->size = 0;
	data->extra_size = 0;
	data->rcpts = 0;
	data->mail_from_logged = 0;

	destroy_addresses(data);
} /* transaction_reset() */

line_status client_callback(char *buffer, char *pos, int size, void *ptr)
{
	struct proc_data *data = ptr;
	char *replace_pos, replace_char;
	char *msg, tmp[32];
	int p1, p2, res;
	int stolen = 0;
	line_status ret_code = LINE_OK;

	// nie zakonczona linia => nic sie nie da zrobic
	if (!pos) goto write_ok;

	save_remove_crlf(buffer, pos, &replace_pos, &replace_char);
//	if ((foreground & FORE_LOG_TRAFFIC) != 0) fprintf(stderr, "[%d] => %s\n", getpid(), buffer);

	if (data->data_going == GOING_NONE) {
		if ((foreground & FORE_LOG_TRAFFIC) != 0) fprintf(stderr, "[%" FORMAT_PID_T "] <= %s\n", getpid(), buffer);
		if (strcasecmp(buffer, "DATA") == 0 || strncasecmp(buffer, "DATA ", 5) == 0) {
			// zostanie odrzucony przez serwer
			// if (data->bdat_used) {}
			log_action(LOG_DEBUG, "DATA:REQUEST");
			if (spool_create(data) != 0) {
				res = errno;
				fdprintf(data->server, "QUIT\r\n");
				SAFE_CLOSE(data->server);

				response(data, ER(451,3,1), "%s %s: %s\r\n", config.proxy_name, config.msg_spool_open_fail, strerror(res));
				wait_for_quit(data, "%s %s: %s", config.proxy_name, config.msg_spool_open_fail, strerror(res));
				exit(5);
			}
			// possibly receives "go ahead" [354] response
			res = queue_command(COMMAND_DATA, data);
			// pseudo-command, receives transaction ack [250]
			if (!res) res = queue_command(COMMAND_DATA_ACK, data);
		} else if (strncasecmp(buffer, "XEXCH50 ", 8) == 0) {
			// filter-out
			buffer[3] = '*';
			res = queue_command(COMMAND_OTHER, data);
		} else if (strncasecmp(buffer, "BDAT ", 5) == 0) {
			strncpy(tmp, buffer, sizeof(tmp));
			TERMINATE_STRING(tmp);
			upcase(tmp);
			// unsigned int
			if (sscanf(tmp, "BDAT %u LAST", &p1) == 1) {
				p2 = 1;
			} else if (sscanf(tmp, "BDAT %d", &p1) == 1) {
				p2 = 0;
			} else {
				p1 = p2 = 0;
			}

			if (data->chunking) {
				log_action(LOG_DEBUG, "BDAT:REQUEST %d %s", p1, p2 ? "LAST" : "");
				// TODO: 
				// "BDAT p1 LAST" => "BDAT p1", "BDAT 0 LAST"/"QUIT"
				// stolen = 1;
				data->bdat_togo = p1;
				data->bdat_last = p2;

				if (p2) {
					fdprintf(data->server, "BDAT %u\r\n", p1);
					stolen = 1;
//					res = queue_commandp(COMMAND_BDAT_PRELAST, data, p1, p2);
//					res = queue_commandp(COMMAND_BDAT_LAST, data, p1, p2);
					res = queue_commandp(COMMAND_BDAT, data, p1, p2);
				} else {
					res = queue_commandp(COMMAND_BDAT, data, p1, p2);
				}
			} else {
				log_action(LOG_DEBUG, "BDAT without CHUNKING support src=%s, ident=%s",
					data->origin_str, data->ident);

				buffer[3] = '*';
//				response(data, ER(500,5,1), "[%s] Command used when not advertised\r\n", config.proxy_name);
//				stolen = 1;
				res = queue_command(COMMAND_OTHER, data);
			}
			ret_code = LINE_BINARY;
		} else if (strcasecmp(buffer, "STARTTLS") == 0 || strncasecmp(buffer, "STARTTLS ", 9) == 0) {
			log_action(LOG_DEBUG, "STARTTLS:REQUEST");
			res = queue_command(COMMAND_STARTTLS, data);
		} else if (strcasecmp(buffer, "RSET") == 0 || strncasecmp(buffer, "RSET ", 5) == 0) {
			res = queue_command(COMMAND_RSET, data);
		} else if (strcasecmp(buffer, "QUIT") == 0 || strncasecmp(buffer, "QUIT ", 5) == 0) {
			res = queue_command(COMMAND_QUIT, data);
		} else if (strncasecmp(buffer, "RCPT TO:", 8) == 0) {
			res = queue_commandp(COMMAND_RCPT, data, (int) alloc_msg_mail(buffer+8, config.email_length), 0);
		} else if (strncasecmp(buffer, "MAIL FROM:", 10) == 0) {
			res = queue_commandp(COMMAND_MAIL, data, (int) alloc_msg_mail(buffer+10, config.email_length), 0);
		} else if (strncasecmp(buffer, "HELO ", 5) == 0) {
			res = queue_commandp(COMMAND_HELO, data, (int) strndup(buffer+5, HELO_LENGTH), 0);
		} else if (strncasecmp(buffer, "EHLO ", 5) == 0) {
			res = queue_commandp(COMMAND_EHLO, data, (int) strndup(buffer+5, HELO_LENGTH), 0);
		} else {
			res = queue_command(COMMAND_OTHER, data);
		}

		if (res != 0) {
			fdprintf(data->server, "QUIT\r\n");
			SAFE_CLOSE(data->server);
			data->server = -1;

			spool_close(data);
			// no sense to leave spool, it probably contains proxy header only
//			if (!config.leave_on_error) spool_remove(data);
			spool_remove(data);

#if PIPELINE_SIZE_MIN < 10
#error Invalid PIPELINE_SIZE_MIN, set to less than 10
#else
#define PIPE(x)	(data->pipeline[data->command_pos_server+(x)].cmd)
			log_action(LOG_DEBUG, "pipeline trace: %d,%d,%d,%d,%d,%d,%d,%d,%d,%d",
				PIPE(0), PIPE(1), PIPE(2), PIPE(3), PIPE(4),
			       	PIPE(5), PIPE(6), PIPE(7), PIPE(8), PIPE(9));
#undef PIPE
#endif
			// TODO: a moze od razu rozlaczyc?
			response(data, ER(503,5,0), "%s %s\r\n", config.proxy_name, config.msg_pipeline_full);
			wait_for_quit(data, "%s", config.msg_pipeline_full);
			exit(5);
		}
	} else {
		if (strcmp(buffer, ".") == 0) {
			if ((foreground & FORE_LOG_TRAFFIC) != 0)
				fprintf(stderr, "[%" FORMAT_PID_T "] <= %s\n", getpid(), buffer);

			spool_close(data);
			msg = spool_scan(data);
			if (!EMPTY_STRING(msg)) {
				SAFE_CLOSE(data->server);
				data->server = -1;

				response(data, ER(550,7,1), "%s %s\r\n", config.proxy_name, msg);
				wait_for_quit(data, "%s", msg);
				exit(5);
			}

			SHARED_CONN_STATUS(state, CONN_POST);
		} else {
			if (data->data_going == GOING_HEADER) {
				// ... header 
				if (buffer[0] == '\0') {
					// empty line => message header complete
					data->data_going = GOING_BODY;
				}
			}

//			if ((foreground & FORE_LOG_TRAFFIC) != 0) fprintf(stderr, "<#  %s\n", buffer);
			restore_crlf(buffer, pos, &replace_pos, &replace_char);
			if (buffer[0] == '.') {
				// wycinamy pierwsza kropke [channel transparency]
				spool_write(data, buffer+1, size-1);
			} else {
				spool_write(data, buffer, size);
			}
		}
	}


	// komenda usunieta ze strumienia, nie wysylaj
	if (stolen) return ret_code;

	restore_crlf(buffer, pos, &replace_pos, &replace_char);

write_ok:
	for (;;) {
		if (timedout) {
			CLEAR_TIMEOUT();
			errno = ETIMEDOUT;
			return LINE_CLOSED;
		}
		if ((res = write(data->server, buffer, size)) == -1) {
			if (errno == EINTR) continue;
#ifdef SILENT_ECONNRESET
			if (errno == ECONNRESET) return LINE_CLOSED;
#endif
			log_action(LOG_ERR, "client:write(%d) error: %s", size, strerror(errno));
			return LINE_CLOSED;
		}

		if (res == 0) {
			log_action(LOG_ERR, "client:write(%d) returned 0 (connection lost)", size);
			return LINE_CLOSED;
		}
		
		break;
	}

	return ret_code;
} /* client_callback() */

line_status server_callback(char *buffer, char *pos, int size, void *ptr)
{
	char *replace_pos, replace_char;
	struct proc_data *data = ptr;
	smtp_command command;
	pipeline_parm_t parm1, parm2;
	int res;
	int code;		// numeric response code
	int cont;		// response continues on next line
	int direct = 0;		// enter direct proxy mode
	int ret_code = LINE_OK;	// function return code
	int ack;


	// dluga linia/krotka linia => nie ma co sprawdzac
	// jesli nieprawidlowa odpowiedz z serwera to tez nie ma co robic
	if (!pos || size<3 || (code = resp_code(buffer)) == -1) goto write_ok;

	// status bedzie kontynuowany?
	cont = ((size > 3) && (buffer[3] == '-'));

	// co nas czeka
	command = poll_commandp(data, &parm1, &parm2);
//	if (foreground) log_action(LOG_DEBUG, "POLL:COMMAND %d", command);

#ifdef FILTER_CHUNKING
	// paskudny hack, length("250 CHUNKING")=12[+crlf]
	if (command == COMMAND_EHLO && size>=12 && code == 250) {
		save_remove_crlf(buffer, pos, &replace_pos, &replace_char);

		if (strcasecmp(buffer+4, "CHUNKING") == 0) {
			buffer[7] = '*';
			log_action(LOG_DEBUG, "ESMTP CHUNKING filtered-out dst=%s:%d",
				data->target_str, data->target_port);
		} else if (strcasecmp(buffer+4, "BINARYMIME") == 0) {
			buffer[7] = '*';
			log_action(LOG_DEBUG, "ESMTP BINARYMIME filtered-out dst=%s:%d",
				data->target_str, data->target_port);
		} else if (strcasecmp(buffer+4, "XEXCH50") == 0) {
			buffer[7] = '*';
			log_action(LOG_DEBUG, "ESMTP XEXCH50 filtered-out dst=%s:%d",
				data->target_str, data->target_port);
		}

		restore_crlf(buffer, pos, &replace_pos, &replace_char);
	}
#else
#warning CHUNKING support possibly broken! use -DFILTER_CHUNKING to filter this extension
#endif

	// przygotowania
	save_remove_crlf(buffer, pos, &replace_pos, &replace_char);
	if ((foreground & FORE_LOG_TRAFFIC) != 0) fprintf(stderr, "[%" FORMAT_PID_T "] => %s\n", getpid(), buffer);

	// process both continued and ending responses
	switch (command) {
	case COMMAND_EHLO:	// SMTP service extensions
		if (code == 250) {
			if (strcasecmp(buffer+4, "ENHANCEDSTATUSCODES") == 0) {
				data->enhancedstatuscodes = 1;
				log_action(LOG_DEBUG, "ESMTP ENHANCEDSTATUSCODES supported dst=%s:%d",
					data->target_str, data->target_port);
			} else if (strcasecmp(buffer+4, "CHUNKING") == 0) {
				data->chunking = 1;
				log_action(LOG_DEBUG, "ESMTP CHUNKING supported dst=%s:%d",
					data->target_str, data->target_port);
			}
		}
	default:
		break;
	}

	// odpowiedz nie zakonczona nas w tym momencie nie interesuje
	if (cont) goto restore_write_ok;

	// odpowiedz nie bedzie kontynuowana
	if (command != COMMAND_NONE) dequeue_command(data);
	
	// pipeline: liczenie odpowiedzi,
	// STARTTLS,DATA jest juz synchronizowane
	switch (command) {
	case COMMAND_DATA:
		if (code == 354) {
			data->data_used = 1;
			flush_addresses(data);
			data->data_going = GOING_HEADER;
			data->transaction++;
			
			SHARED_CONN_STATUS(transaction, data->transaction);
			SHARED_CONN_STATUS(state, CONN_DATA);

			if (config.nat_header_type && data->header) fdprintf(data->server, "%s", data->header);
			log_action(LOG_DEBUG, "DATA:GOING");
		} else {
			destroy_addresses(data);	// 20050311

			spool_remove(data);
			data->data_going = GOING_NONE;

			// dequeue pseudo-command COMMAND_DATA_ACK
			// there won't be any ack cause it's already failed
			dequeue_command(data);
			log_action(LOG_DEBUG, "DATA:CANCELLED [%d]", code);
		}
		break;

	case COMMAND_DATA_ACK:
		transaction_reset(data);
		data->data_going = GOING_NONE;
//		SHARED_CONN_STATUS(state, CONN_ACK);
		log_action(LOG_DEBUG, "DATA:FINISHED [%d]", code);
		break;

	case COMMAND_BDAT:
		if (code == 250) {
			flush_addresses(data);
			log_action(LOG_DEBUG, "BDAT:CHUNK ok [%d] %s", parm1, parm2 ? "LAST" : "");
			if (!data->bdat_used) {
				if (data->data_used) {}
				data->bdat_used = 1;
				// tymczasowe rozwiazanie
				flush_buffers(data);
				log_action(LOG_DEBUG, "BDAT:DIRECT");
				direct = 1;
				break;
				// TODO: inject for BDAT!
				//if (config.nat_header_type && data->header) inject_nat_header(data);
			}

			// BDAT last
			if (parm2) {


			}
		} else {
			destroy_addresses(data);	// 20050311
			log_action(LOG_DEBUG, "BDAT:CHUNK rejected [%d]", code);
			data->bdat_togo = 0;
			data->bdat_last = 0;
//`			data->bdat_used = 0;	// TODO: napewno?
		}
		break;

	case COMMAND_MAIL:
		transaction_reset(data);
		ack = (code == 250);
		new_mail_from(data, (char *) parm1, code);
		break;

	case COMMAND_RCPT:
		ack = (code == 250);
		add_rcpt_to(data, (char *) parm1, code);
		if (ack) data->rcpts_total++;
		break;

	case COMMAND_STARTTLS:
		if (code == 220) {
			destroy_addresses(data);
			spool_remove(data);
			log_action(LOG_DEBUG, "DIRECT:GOING");
			// direct_proxy() after write()
			direct = 1;
		} else {
			log_action(LOG_DEBUG, "DIRECT:CANCELLED [%d]", code);
		}
		break;

	case COMMAND_RSET:
		transaction_reset(data);
		SHARED_CONN_STATUS(state, CONN_RSET);
		break;

	case COMMAND_HELO:
	case COMMAND_EHLO:
		transaction_reset(data);
		SHARED_CONN_STATUS(state, CONN_HELO);
		if (config.log_helo) {
			untaint((char *) parm1, 0);
			log_action(LOG_INFO|LOG_ALWAYS, "%s src=%s, ident=%s, helo=%s\n",
				(command == COMMAND_EHLO) ? "EHLO" : "HELO",
				data->origin_str, data->ident, (char *) parm1);
		}
		free((char *) parm1);
		break;
	default:
		break;
	}

restore_write_ok:
	restore_crlf(buffer, pos, &replace_pos, &replace_char);

write_ok:
	for (;;) {
		if (timedout) {
			CLEAR_TIMEOUT();
			errno = ETIMEDOUT;
			return LINE_CLOSED;
		}
		if ((res = write(data->client, buffer, size)) == -1) {
			if (errno == EINTR) continue;
#ifdef SILENT_ECONNRESET
			if (errno == ECONNRESET) return LINE_CLOSED;
#endif
			log_action(LOG_ERR, "server:write(%d) error: %s", size, strerror(errno));
			return LINE_CLOSED;
		}
		
		if (res == 0) {
			log_action(LOG_ERR, "server:write(%d) returned 0 (connection lost)", size);
			return LINE_CLOSED;
		}
		
		break;
	}

	if (direct) {
		direct_proxy(data);
		exit(0);
	}
	
	return ret_code;
} /* server_callback() */


void context_free(struct proc_data *data)
{
	FREE_NULL(data->cli_buf);
	FREE_NULL(data->srv_buf);
	FREE_NULL(data->pipeline);

	FREE_NULL(data->header);
	FREE_NULL(data->spool_name);
	FREE_NULL(data->lockfile);
	FREE_NULL(data->virus_name);
	destroy_addresses(data);
	FREE_NULL(data->message);
}

void context_init_1(struct proc_data *data, int client, struct sockaddr_in origin)
{
	memset(data, '\0', sizeof(struct proc_data));

	data->spool_fd = -1;
	data->origin = origin;
	data->client = client;
	data->server = -1;
	data->time = time(NULL);
	
	// adres zrodlowy (tekst)
	snprintf(data->origin_str, sizeof(data->origin_str), "%s", inet_ntoa(data->origin.sin_addr));
	TERMINATE_STRING(data->origin_str);
	data->origin_port = ntohs(data->origin.sin_port);

	// utworz nazwe lockfile, jesli nie trzeba do tego identa
	if (!EMPTY_STRING(config.fixed_server) || config.use_netfilter) {
		if (asprintf(&data->lockfile, "%s/%s", config.lock_path, data->origin_str) == -1) {
			data->lockfile = NULL;
		}
//		log_action(LOG_DEBUG, "LOCK:DEBUG using simple lock file [%s]", data->lockfile);
	}
} /* context_init_1() */

void context_init_2(struct proc_data *data)
{
	int res, len;

	if ((data->cli_buf = malloc(config.buffer_size)) == NULL) {
		goto no_mem;
	}
	if ((data->srv_buf = malloc(config.buffer_size)) == NULL) {
		goto no_mem;
	}
	if ((data->pipeline = malloc(config.pipeline_size * sizeof(struct pipeline_entry))) == NULL) {
		goto no_mem;
	}

	// nazwa pliku spool-a
	if (asprintf(&data->spool_name, "%s/%" FORMAT_TIME_T ".%" FORMAT_PID_T, config.spool_path, data->time, getpid()) == -1) {
		data->spool_name = NULL;
		if (!config.ignore_errors) {
			context_free(data);
			helo(data->client);
			wait_for_quit(data, "%s", config.msg_spool_problem);
			exit(5);
		}
	}

	// adres lokalny potrzebny do ident-a
	// data->local_addr_len;
	len = sizeof(data->local_addr);
	if (getsockname(data->client, (struct sockaddr *) &data->local_addr, &len) != 0) {
		res = errno;
		context_free(data);
		helo(data->client);
		wait_for_quit(data, "%s: %s", config.msg_lookup_failed, strerror(res));
		exit(5);
	}

	return;

no_mem:
	context_free(data);

	helo(data->client);
	wait_for_quit(data, "%s", config.msg_nomem);
	exit(5);
} /* context_init_2() */


int target_lookup(struct proc_data *data)
{
	int res;
#ifdef USE_SHARED_MEM
	int i;
#endif

	res = 0;

	// lookups
	// must fill: target, target_str, ident
	if (!EMPTY_STRING(config.fixed_server)) {
		strncpy(data->target_str, config.fixed_server, sizeof(data->target_str));
		TERMINATE_STRING(data->target_str);
		data->target_port = config.fixed_server_port;

		data->target.sin_family = AF_INET;
		data->target.sin_port = htons(data->target_port);
		if (!inet_aton(data->target_str, &data->target.sin_addr))
			res = LOOKUP_MISMATCH;
//		snprintf(data->ident, sizeof(data->ident), "fixed");
//		TERMINATE_STRING(data->ident);
		// normalny ident?
#ifdef USE_NAT
	} else if (config.use_netfilter) {
		res = nat_lookup(data);
#endif

//		snprintf(data->ident, sizeof(data->ident), "local");
//		TERMINATE_STRING(data->ident);

		// normalny ident?
	} else {
		res = remote_lookup(data); //, data->origin, config.port, data->local_addr /*, data->local_addr_len*/);

#ifdef USE_SHARED_MEM
		if (res == 0) {
			data->ident_count = 0;
			untaint(data->ident, sizeof(data->ident));
			// strncpy zeruje pozostalosc stringa,
			strncpy(connections[child_slot].ident, data->ident, sizeof(connections[child_slot].ident));
			TERMINATE_STRING(connections[child_slot].ident);
			SHARED_CONN_STATUS(ident_ok, 1);

			for(i=0; i<max_connections_real; i++) {
				// if (!connections[i].pid) continue;
				if (!connections[i].ident_ok) continue;
				if (connections[i].src != SIN_TO_UINT32(data->origin.sin_addr)) continue;
				if (strcmp(connections[i].ident, data->ident) != 0) continue;

				data->ident_count++;
			}

			if (data->ident_count > config.max_per_ident) {
				/* no need to be nice for flooder */
				response(data, CONN_REJ_ER, "%s %s\r\n", config.proxy_name, config.msg_max_per_ident);
				SAFE_CLOSE(data->client);
				SHARED_STATS_INC(rejects_ident);
				log_action(LOG_INFO, "Rejecting (ident %d) connection from %s:%d [%s]",
					data->ident_count, data->origin_str, data->origin_port, data->ident);
				/* lockfile_ident_present needed for data->lockfile setup */
				data->found = FOUND_MAX_IDENT;
				if (config.lock_duration && config.max_per_ident_lock && !lockfile_ident_present(data))
				       	lockfile_action(data, "MAX_IDENT");
				cleanup();
				exit(0);
			}
		}
#endif
	}

	if (res == -1) {
		res = errno;
		helo(data->client);
		wait_for_quit(data, "%s: %s", config.msg_lookup_failed, strerror(res));
		exit(5);
	} else if (res > 0) {
		helo(data->client);
		wait_for_quit(data, "%s: %s", config.msg_lookup_failed, *lookup_errors[res]);
		exit(5);
	}

	return 1;
} /* target_lookup() */


int target_connect(struct proc_data *data)
{
	int res;
	struct sockaddr_in src;


	SHARED_CONN_STATUS(dst, SIN_TO_UINT32(data->target.sin_addr));
	SHARED_CONN_STATUS(state, CONN_CONNECT);
	
	inet_aton(config.source_addr, &src.sin_addr);
	SET_TIMEOUT(config.timeout_connect);

	if ((data->server = connect_host(data->target, ntohs(data->target.sin_port), src)) == -1) {
		res = errno;
		helo(data->client);
		if (res == EINTR || res == ETIMEDOUT) {
			wait_for_quit(data, "%s [%s]", config.msg_connect_timeout, data->target_str);
			exit(5);
		} else {
			wait_for_quit(data, "%s [%s]: %s", config.msg_cannot_connect, data->target_str, strerror(res));
			exit(5);
		}
		exit(1);
	}

	CLEAR_TIMEOUT();
	return 0;
} /* target_connect() */



/*
 * 	CHUNKING
*/

line_status bdat_chunk(struct proc_data *data, int allow_read)
{
	int size;
	int res;
	
	// TODO: co jesli klient da od razu "BDAT 0 LAST"?
	
	// nie mozemy zrobic od razu read(), bo moglismy tu trafic z danymi
	// pozostalymi z fdgetline_cb() i kolejny read() moglby zablokowac
	size = min(config.buffer_size - data->cli_size, data->bdat_togo);
	if (allow_read && size) {
		res = read(data->client, data->cli_buf + data->cli_size, size);
		if (res == -1) {
#ifdef SILENT_ECONNRESET
			if (errno == ECONNRESET) return LINE_CLOSED;
#endif
			log_action(LOG_ERR, "bdat_chunk:read(%d): %s", size, strerror(errno));
			return LINE_CLOSED;
		}

		data->cli_size += res;
		data->cli_rx += res;
		// === if (res == 0)
		if (!data->cli_size) return LINE_CLOSED;	// a moze blad?
	}
	
	size = min(data->bdat_togo, data->cli_size);
	spool_write(data, data->cli_buf, size);

	for (;;) {
		if ((res = write(data->server, data->cli_buf, size)) == -1) {
			if (errno == EINTR) continue;
#ifdef SILENT_ECONNRESET
			if (errno == ECONNRESET) return LINE_CLOSED;
#endif
			log_action(LOG_ERR, "bdat_chunk:write(%d):error: %s", size, strerror(errno));
			return LINE_CLOSED;
		}
		
		if (res == 0) {
			log_action(LOG_ERR, "bdat_chunk:write(%d) returned 0 (connection lost)", size);
			return LINE_CLOSED;
		}
		
		break;
	}

	data->bdat_togo -= size;
	data->cli_size -= size;
	if (data->cli_size) memmove(data->cli_buf, data->cli_buf+size, data->cli_size);

	if (!data->bdat_togo && data->bdat_last) {
			spool_close(data);
			// TODO: scanning, BDAT 0 LAST/QUIT
	}

	return LINE_OK;
} /* bdat_chunk() */

/*
 * 	obsluga sesji
*/

void connection(struct proc_data *data)
{
	fd_set rfds;
	struct timeval tv;
	int res, max_fd;


	// pid, src filled by parent process
	SHARED_CONN_STATUS(state, CONN_IDENT);
	SHARED_CONN_STATUS(cli_rx, 0);
	SHARED_CONN_STATUS(srv_rx, 0);

	context_init_2(data);

	// dane serwera docelowego
	if (!target_lookup(data)) exit(1);

	if (lockfile_ident_present(data)) {
		SHARED_STATS_INC(rejects_lock);
		fdprintf(data->client, "%03d %s %s\n", CONN_REJ_CODE, config.proxy_name, config.msg_virus_locked);
		SAFE_CLOSE(data->client);
		exit(0);
	}
	
	// log
	log_action(LOG_NOTICE|LOG_ALWAYS, "NEW (%d/%d) src=%s:%d, ident=%s, dst=%s:%d, id=%" FORMAT_TIME_T ".%" FORMAT_PID_T, 
		children, data->ident_count, data->origin_str, data->origin_port,
		data->ident, data->target_str, data->target_port, data->time, getpid());

	// nie bedzie petli?
	// dla testow wylacz (albo: wlacz jesli !fixed)
	if (!foreground) {
		if (data->origin.sin_addr.s_addr == data->target.sin_addr.s_addr) {
			log_action(LOG_WARNING, "Avoiding loop, exiting.");
			SAFE_CLOSE(data->client);
			cleanup();
			exit(1);
		}
	}

	// sygnaly
	setup_signal();

	// podlaczenie do docelowego serwera SMTP
	if (target_connect(data) != 0) exit(1);

	// przygotowanie naglowka nat
	prepare_nat_header(data);

	// no real command, but we get initial 'response' from MTA
	queue_command(COMMAND_OTHER, data);

	// deskryptor dla select-a
	max_fd = (data->client > data->server) ? data->client : data->server;

	SHARED_CONN_STATUS(state, CONN_PRE);

	// petla obslugi sesji
	for (;;) {
		if (force_finish) {
			res = LINE_CLOSED_ADMIN;
			break;
		}

		FD_ZERO(&rfds);
		FD_SET(data->client, &rfds);
		FD_SET(data->server, &rfds);
		tv.tv_sec = config.timeout_idle;
		tv.tv_usec = 0;

		// oczekiwanie na dane lub timedout
		if ((res = select(max_fd+1, &rfds, NULL, NULL, &tv)) == -1) {
			if (errno == EINTR) continue;
			log_action(LOG_ERR, "select: %s", strerror(errno));
			continue;
		}

		// timedout
		if (res == 0) {
			res = LINE_CLOSED_TIMEOUT;
			break;
		}

		// dane od klienta
		if (FD_ISSET(data->client, &rfds)) {
			if (data->bdat_togo) {
				if ((res = bdat_chunk(data, 1)) == LINE_CLOSED) break;
			}

			// nie moze byc w "else", bo po bdat_chunk moga zostac jakies dane w buforze
			if (!data->bdat_togo) {
				assert(data->cli_size < config.buffer_size);
				res = fdgetline_cb(data->client, data->cli_buf, config.buffer_size, &data->cli_size,
					&client_callback, data, &data->cli_rx);
				if (res == LINE_CLOSED) break;

				// cos moglo zostac w budorze
				// if (res == LINE_BINARY?) ?
				if (data->cli_size && data->bdat_togo) {
					if ((res = bdat_chunk(data, 0)) == LINE_CLOSED) break;
				}
			}

			SHARED_CONN_STATUS(cli_rx, data->cli_rx);
		}

		// dane z serwera
		if (FD_ISSET(data->server, &rfds)) {
			assert(data->srv_size < config.buffer_size);
			res = fdgetline_cb(data->server, data->srv_buf, config.buffer_size, &data->srv_size,
				&server_callback, data, &data->srv_rx);
			
			SHARED_CONN_STATUS(srv_rx, data->srv_rx);
			
			if (res == LINE_CLOSED) {
				res = LINE_CLOSED_SERVER;
				break;
			}
		}
	}

	// porzadki

	SAFE_CLOSE(data->client);
	SAFE_CLOSE(data->server);

	if (!data->transaction) SHARED_STATS_INC(requests_empty);

	spool_remove(data);
	cleanup();

	log_action(LOG_NOTICE|LOG_ALWAYS, "CLOSE by=%s, rcv=%d/%d, trns=%d, rcpts=%d, time=%" FORMAT_TIME_T ", src=%s, ident=%s",
		line_closed_cause(res), data->cli_rx, data->srv_rx, data->transaction, data->rcpts_total,
		time(NULL)-data->time, data->origin_str, data->ident);
	exit(0);
} /* connection() */


int setup()
{
	// config.max_connections defined during startup only,  this setting is ignored during reload
	max_connections_real = config.max_connections;

	pids = malloc(sizeof(pid_t) * max_connections_real);
	if (pids == NULL) return -1;

#ifdef USE_SHARED_MEM
	connections = shmalloc(sizeof(struct conn_info) * max_connections_real, &conn_shmid);
	if (connections == NULL) return -1;

	stats = shmalloc(sizeof(struct stat_info), &stat_shmid);
	if (stats == NULL) return -1;
#else
	connections = malloc(sizeof(struct conn_info) * max_connections_real);
	if (connections == NULL) {
		log_action(LOG_CRIT, "connections malloc failed: %s", strerror(errno));
		return -1;
	}

	stats = malloc(sizeof(struct stat_info));
	if (stats == NULL) {
		log_action(LOG_CRIT, "stats malloc failed: %s", strerror(errno));
		return -1;
	}
#endif

	return 0;
} /* setup */


/*
 * 	post-reconfiguration, called:
 * 	1. after daemonize() on start
 * 	2. after reading configuration on HUP
*/

void post_config()
{
	spool_max_size = (config.scan_max_size > config.spam_max_size) ? config.scan_max_size : config.spam_max_size;
	if (config.pipeline_size < PIPELINE_SIZE_MIN) config.pipeline_size = PIPELINE_SIZE_MIN;

	if (!EMPTY_STRING(config.locale)) {
		setlocale(LC_MESSAGES, config.locale);
//		log_action(LOG_DEBUG, "Changed locale to %s ['Success'=>'%s']", config.locale, strerror(0));
	}
} /* post_config() */


int start()
{
	int res, i, host_count, slot;
	int client, proxy;
	struct timeval tv;
	struct sockaddr_in origin;
	socklen_t originlen;
	double cur_load;
	struct proc_data data;


	memset(connections, 0, sizeof(struct conn_info) * max_connections_real);
	memset(pids, 0, sizeof(pid_t) * max_connections_real);
	memset(stats, 0, sizeof(struct stat_info));
	memset(&data, 0, sizeof(struct proc_data));

	stats->started = time(NULL);
	proxy = -1;

restart:
	stats->restarted = time(NULL);

	if ((res = setup_signal()) < 0) {
		log_action(LOG_CRIT, "setup_signal: %s", strerror(errno));
		cleanup();
		exit(1);
	}

	if (proxy == -1) {
		proxy = setup_listen(config.bind_address, config.port, config.connect_queue);
		if (proxy == -1) {
			cleanup();
			exit(1);
		}
		
		log_action(LOG_INFO, "SMTP-Proxy %s listening on %s:%d [queue: %d]", VERSION, config.bind_address, config.port, config.connect_queue);
	}

	/* to improve test-suite performance */
	if ((foreground & FORE_SIGNAL_PPID) != 0) {
		if (kill(getppid(), SIGUSR1) != 0) {
			log_action(LOG_ERR, "kill(%" FORMAT_PID_T ",SIGUSR1) error: %s", getppid(), strerror(errno));
		} else {
			log_action(LOG_ERR, "parent pid %" FORMAT_PID_T " signaled, ready for test", getppid());
		}	
	}

	/* main server loop */
	for (;;) {
		if (force_finish) {
			force_finish = 0;
			break;
		}
		if (child_died) {
			// nie ma wyscigow, bo child_reaper obsluzy wszystkie zakonczone procesy
			child_died = 0;
			child_reaper();
		}
		if (force_reconfig) {
			force_reconfig = 0;
			log_action(LOG_INFO, "Reloading configuration...");
			if (read_config(config_options, config_file) != 0) {
				log_action(LOG_INFO, "Error parsing configuration file...");
				// inconsistent configuration 
				break;
			}

			post_config();
			if ((config.port >= 1024) || (getuid() == 0)) {
				SAFE_CLOSE(proxy);
				proxy = -1;
			} else {
				log_action(LOG_NOTICE, "NOTICE: listening socket won't be reopened...");
			}

			goto restart;
		}
		if (force_dump) {
			force_dump = 0;
			dump_state();
		}

		originlen = sizeof(origin);
		client = accept(proxy, (struct sockaddr*) &origin, &originlen);
		if (client == -1) {
			if (errno == EINTR) continue;
			log_action(LOG_CRIT, "accept error: %s", strerror(errno));
			break;
		}

		stats->requests++;

		/* memleak testing */
#ifdef MEMLEAK_TESTING
		#warning MEMLEAK TESTING
		malloc(MEMLEAK_TESTING);
#endif

		/* fast DoS checks */
		if (children >= max_connections_real) {
			log_action(LOG_INFO, "Rejecting (%d) connection from %s:%d", children, inet_ntoa(origin.sin_addr), ntohs(origin.sin_port));
			fdprintf(client, "%03d %s %s\n", CONN_REJ_CODE, config.proxy_name, config.msg_max_reached);
			SAFE_CLOSE(client);
			stats->rejects++;
			continue;
		}

		if (is_load_above(config.max_load, &cur_load) == 1) {
			log_action(LOG_INFO, "Rejecting connection from %s:%d, load=%f", inet_ntoa(origin.sin_addr), ntohs(origin.sin_port), cur_load);
			fdprintf(client, "%03d %s %s\n", CONN_REJ_CODE, config.proxy_name, config.msg_system_load);
			SAFE_CLOSE(client);
			stats->rejects++;
			continue;
		}

		slot = -1;
		host_count = 1;
		for (i=0; i<max_connections_real; i++) {
			if (pids[i] == 0) {
				slot = i;
			} else {
				if (connections[i].src == SIN_TO_UINT32(origin.sin_addr)) host_count++;
			}
		}

		// free previous context
		context_free(&data);
		context_init_1(&data, client, origin);

		if (host_count > config.max_per_host) {
			log_action(LOG_INFO, "Rejecting (host %d) connection from %s:%d", host_count, inet_ntoa(origin.sin_addr), ntohs(origin.sin_port));
			fdprintf(client, "%03d %s %s\n", CONN_REJ_CODE, config.proxy_name, config.msg_max_per_host);
			data.found = FOUND_MAX_HOST;
			if (config.lock_duration && config.max_per_host_lock && !lockfile_present(&data))
			       lockfile_action(&data, "MAX_HOST");
			SAFE_CLOSE(client);
			stats->rejects_host++;
			continue;
		}

		if (slot == -1) {
			log_action(LOG_ALERT, "!BUG! No free slot, but max_connections_real=%d, rejecting connection from %s:%d", max_connections_real, inet_ntoa(origin.sin_addr), ntohs(origin.sin_port));
			fdprintf(client, "%03d %s %s\n", CONN_REJ_CODE, config.proxy_name, config.msg_temp_unavail);
			SAFE_CLOSE(client);
			stats->rejects++;
			continue;
		}

		if (config.lock_duration && lockfile_present(&data)) {
			fdprintf(client, "%03d %s %s\n", CONN_REJ_CODE, config.proxy_name, config.msg_virus_locked);
			SAFE_CLOSE(client);
			stats->rejects_lock++;
			continue;
		}

//		connections[slot].pid = -1;
		connections[slot].src = SIN_TO_UINT32(origin.sin_addr);
		connections[slot].start_time = time(NULL);
#ifdef USE_SHARED_MEM
		connections[slot].ident_ok = 
			connections[slot].dst =
			connections[slot].cli_rx =
			connections[slot].srv_rx =
			connections[slot].transaction = 0;
//		connections[slot].ident[]
		connections[slot].state = CONN_START;
#endif

		/* fork & service request */
#ifndef NO_FORK
		res = fork();
#else
		res = 0;
#endif
		if (res < 0) {
			log_action(LOG_CRIT, "fork error: %s", strerror(errno));
			break;
		}
		if (res) {	// parent
			pids[slot] = res;
			SAFE_CLOSE(client);
			children++;
			if (children > stats->max_children) stats->max_children = children;
			continue;
		}

		SAFE_CLOSE(proxy);

		i_am_a_child = 1;
#ifdef USE_SHARED_MEM
		child_slot = slot;
#endif
		children++;
		connection(&data);
		cleanup();
		exit(0);
	}

	SAFE_CLOSE(proxy);

		
	if (children) {
		remove_pidfile();
		// hack => cleanup() nie bedzie probowal usunac ponownie
		if (config.pidfile) config.pidfile[0] = '\0';
		log_action(LOG_INFO, "Listening-socket closed, waiting for %d children(s) to finish", children);
	}

	while (children) {
		if (force_dump) {
			force_dump = 0;
			dump_state();
		}

		if (force_finish) break;

		if (child_died) {
			child_died = 0;
			child_reaper();
			if (!children) break;
		}
		
		tv.tv_sec = 3;
		tv.tv_usec = 0;
		if (select(0, NULL, NULL, NULL, &tv) == -1 && errno != EINTR) {
			log_action(LOG_ERR, "select: %s", strerror(errno));
			break;
		}
	}

	cleanup();
	log_action(LOG_INFO, "Children finished, exiting");
	return 0;
} /* start() */

int cat(char *fn)
{
	char buf[4096];
	int size, res, pos, fd;

	if ((fd = open(fn, O_RDONLY)) == -1) {
		fprintf(stderr, "open(%s) error: %s\n", fn, strerror(errno));
		return -1;
	}
	
	for (;;) {
		if ((size = read(fd, buf, sizeof(buf))) == -1) {
			if (errno == EINTR) continue;

			fprintf(stderr, "read() error: %s\n", strerror(errno));
			SAFE_CLOSE(fd);
			return -1;
		}

		if (size == 0) break;

		for (pos=0; size>0; size-=res, pos+=res) {
			if ((res = fwrite(buf+pos, 1, size, stdout)) == -1) {
				if (errno == EINTR) continue;

				fprintf(stderr, "cat:write() error: %s\n", strerror(errno));
				SAFE_CLOSE(fd);
				return -1;
			}
		}
	}

	SAFE_CLOSE(fd);
	return 0;
} /* cat() */


char* chrootize_path(char *path)
{
	static char buf[256];

	snprintf(buf, sizeof(buf), "%s%s%s", config.chroot_path, EMPTY_STRING(config.chroot_path) ? "" : "/./", path);
	TERMINATE_STRING(buf);
	
	return buf;
} /* chrootize_path () */

int signal_bg_process(int signum)
{
	char *real_pidfile;
	FILE *f;
	int bg_pid;

	real_pidfile = chrootize_path(config.pidfile);
	
	f = fopen(real_pidfile, "r");
	if (!f) {
		fprintf(stderr, "Cannot open .pid file [%s]: %s\n", real_pidfile, strerror(errno));
		return 1;
	}

	if (fscanf(f, "%d", &bg_pid) != 1) {
		fclose(f);
		fprintf(stderr, "Cannot parse .pid file [%s]\n", real_pidfile);
		return 1;
	}

	fclose(f);

//	fprintf(stderr, "Sending signal %d to process %d...\n", signal, bg_pid);
	if (kill(bg_pid, signum) != 0) {
		fprintf(stderr, "kill(%d, %d) error: %s\n", bg_pid, signum, strerror(errno));
		return 1;
	}

	return 0;
} /* signal_bg_process() */


void dump_help()
{
	printf("SMTP Transparent AV proxy %s\n", VERSION);
	printf("Usage: smtp-gated [-f] [ -h | -s | -S | -r | -t | -T | -v | -V ] config_file\n");
	printf("	-f	run foreground (debugging only)\n");
	printf("	-h	this command reference\n");
	printf("	-s	prepare & dump process status\n");
	printf("	-S	prepare & show process status filename\n");
	printf("	-r	reload configuration (simple SIGHUP)\n");
	printf("	-t	syntax check & dump configuration (except messages)\n");
	printf("	-T	syntax check & dump configuration (including messages)\n");
	printf("	-v	show version\n");
	printf("	-V	show version & compiled-in options\n");
	printf("Signals:\n");
	printf("	HUP	reload configuration\n");
	printf("	USR1	dump statistics\n\n");
} /* dump_help() */

// petla glowna

typedef enum {MODE_DEFAULT=0, MODE_MANY, MODE_H, MODE_V, MODE_VV, MODE_T, MODE_TT, MODE_R, MODE_S, MODE_SS} run_mode;


#define SET_MODE(x) if (mode == MODE_DEFAULT) mode = x; else mode = MODE_MANY;

int main(int argc, char* argv[])
{
	run_mode mode = MODE_DEFAULT;
	int i;

	confvars_init(&config);

	for (i=1; i<argc; i++) {
		if ((strcmp(argv[i], "-h") == 0) || (strcmp(argv[i], "--help") == 0)) {
			SET_MODE(MODE_H);
		} else if (strcmp(argv[i], "-v") == 0) {
			SET_MODE(MODE_V);
		} else if (strcmp(argv[i], "-V") == 0) {
			SET_MODE(MODE_VV);
		} else if (strcmp(argv[i], "-t") == 0) {
			SET_MODE(MODE_T);
		} else if (strcmp(argv[i], "-T") == 0) {
			SET_MODE(MODE_TT);
		} else if (strcmp(argv[i], "-r") == 0) {
			SET_MODE(MODE_R);
		} else if (strcmp(argv[i], "-s") == 0) {
			SET_MODE(MODE_S);
		} else if (strcmp(argv[i], "-S") == 0) {
			SET_MODE(MODE_SS);
		} else if (strncmp(argv[i], "-f", 2) == 0) {
			foreground += strlen(argv[i])-1;
		} else {
			if (*argv[i] == '-') {
				fprintf(stderr, "Fatal: Unknown option: %s\n", argv[i]);
				exit(1);
			} else {
				if (config_file) {
					fprintf(stderr, "Fatal: You can supply only one configuration file!\n");
					exit(1);
				}

				config_file = argv[i];
			}
		}
	}

	if (mode == MODE_MANY) {
		fprintf(stderr, "Fatal: You can specify only one mode\n");
		exit(3);
	}

#ifdef DEFAULT_CONFIG_FILE
	if (!config_file) config_file = DEFAULT_CONFIG_FILE;
#endif

	foreground++;
	if (config_file) {
		if (read_config(config_options, config_file) != 0) {
			if (config_file[0] == '-') fprintf(stderr, "Try %s -h for help\n", argv[0]);
			exit(2);
		}
/*
	} else {
		if ((res = config_set_defaults(config_options)) != 0) {
			fprintf(stderr, "Fatal: !BUG! invalid default (hardcoded) config [%d]\n", res);
			exit(10);
		}
*/
	}
	foreground--;

	if (foreground) fprintf(stderr, "Foreground level: 0x%02x\n", foreground);

	switch (mode) {
		case MODE_DEFAULT:
			break;
		case MODE_H:
			dump_help();
			exit(10);
		case MODE_V:
		case MODE_VV:
			dump_ver(mode == MODE_VV);
			exit(0);
		case MODE_T:
		case MODE_TT:
			dump_config(config_options, (mode == MODE_TT));
			exit(0);
		case MODE_S:
		case MODE_SS:
			if (signal_bg_process(SIGUSR1) != 0) exit(1);
			sleep(1);
			if (mode == MODE_SS) {
				printf("%s\n", chrootize_path(config.dumpfile));
			} else {
				i = cat(chrootize_path(config.dumpfile));
				if (i) exit(1);
			}
			exit(0);
		case MODE_R:
			if (signal_bg_process(SIGHUP) != 0) exit(1);
			exit(0);
		default:
			fprintf(stderr, "Fatal: !BUG! unknown exec mode [%d]\n", mode);
			exit(3);
	}
	
	if (!config_file) {
		fprintf(stderr, "Fatal: Config file not supplied.\n");
		exit(1);
	}

	if (daemonize() != 0) exit(1);
	post_config();

	if (setup() != 0) {
		cleanup();
		exit(1);
	}

	start();
	exit(0);
} /* main() */

// : vim: 

