/*
 * 	util.c
 *
 * 	Copyright (C) 2004-2005 Bartomiej Korupczynski <bartek@klolik.org>
 *
 * 	This program is free software; you can redistribute it and/or 
 * 	modify it under the terms of the GNU General Public License 
 * 	as published by the Free Software Foundation; either 
 * 	version 2 of the License, or (at your option) any later 
 * 	version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the Free Software
 * 	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE
#define _UTIL_C_


#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <syslog.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/un.h>

#ifdef HAVE_SYS_LOADAVG_H
#include <sys/loadavg.h>
#endif

#ifdef USE_SHARED_MEM
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

#include "confvars.h"
#include "daemon.h"
#include "util.h"
//#include "smtp-gated.h"



/*
 * 	constants
*/

static const int one = 1;


/*
 *	connect host
*/

/*
 * alarm(...)
 * connect_host()
 * alarm(0);
*/

int connect_host(struct sockaddr_in dst, int dst_port, struct sockaddr_in src)
{
	int res;
	int sock;


	sock = socket(PF_INET, SOCK_STREAM, getprotobyname("tcp")->p_proto);
	if (sock == -1) return -1;

	// konieczne?!
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one)) != 0)
		goto fail;

	src.sin_family = AF_INET;
	src.sin_port = 0;
	if (bind(sock, (struct sockaddr *) &src, sizeof(src)) != 0)
		goto fail;

	dst.sin_family = AF_INET;
	dst.sin_port = htons(dst_port);

	if (connect(sock, (struct sockaddr *) &dst, sizeof(dst)) != 0)
		goto fail;

	return sock;

fail:
	res = errno;
	if (sock != -1) SAFE_CLOSE(sock);
/*
	if (timedout) {
		errno = ETIMEDOUT;
	} else {
		errno = res;
	}
*/
	return -1;
} /* connect_host() */

int connect_path(char *path)
{
	int sock;
	int port, save;
	char *port_pos;
	struct sockaddr_un us;
	struct sockaddr_in sin;

	
	sock = -1;

	if (path[0] == '/') {
		if ((sock = socket(AF_UNIX, SOCK_STREAM, 0)) == -1) {
			goto fail;
		}
	
		us.sun_family = AF_UNIX;
		strncpy(us.sun_path, path, sizeof(us.sun_path));
		TERMINATE_STRING(us.sun_path);
		
		if (connect(sock, (struct sockaddr*) &us, sizeof(us)) != 0) {
			goto fail;
		}
	} else {
		port_pos = strchr(path, ':');
		if (port_pos == NULL) {
			errno = EINVAL;
			goto fail;
		}

		*port_pos = '\0';
		if (inet_aton(path, &sin.sin_addr) == 0) {
			*port_pos = ':';
			errno = EINVAL;
			goto fail;
		}

		*port_pos = ':';
		port = atoi(port_pos+1);

		if (port < 1 || port > 65535) {
			errno = ERANGE;
			goto fail;
		}
		
		sin.sin_family = AF_INET;
		sin.sin_port = htons(port);

		if ((sock = socket(PF_INET, SOCK_STREAM, getprotobyname("tcp")->p_proto)) == -1) {
			goto fail;
		}

		if (connect(sock, (struct sockaddr *) &sin, sizeof(sin)) != 0) {
			goto fail;
		}
	}

	// errno = ENXIO;
	return sock;

fail:
	save = errno;
	if (sock != -1) SAFE_CLOSE(sock);
	errno = save;
	return -1;
} /* connect_path() */


/*
 * 	setup listening socket
*/

int setup_listen(char *listen_ip, int listen_port, int backlog)
{
	int fd;
	struct sockaddr_in addr;

	if ((fd = socket(PF_INET, SOCK_STREAM, getprotobyname("tcp")->p_proto)) == -1) {
		log_action(LOG_CRIT, "socket: %s", strerror(errno));
		return -1;
	}

	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one)) != 0) {
		log_action(LOG_CRIT, "setsockopt(SO_REUSEADDR): %s", strerror(errno));
		SAFE_CLOSE(fd);
		return -1;
	}

	addr.sin_family = AF_INET;
	addr.sin_port = htons(listen_port);
	if (inet_aton(listen_ip, &addr.sin_addr) == 0) {
		log_action(LOG_CRIT, "inet_aton: %s", strerror(errno));
		SAFE_CLOSE(fd);
		return -1;
	}

	if (bind(fd, (struct sockaddr*) &addr, sizeof(addr)) != 0) {
		log_action(LOG_CRIT, "bind: %s", strerror(errno));
		SAFE_CLOSE(fd);
		return -1;
	}

	if (listen(fd, backlog) != 0) {
		log_action(LOG_CRIT, "listen: %s", strerror(errno));
		SAFE_CLOSE(fd);
		return -1;
	}

	return fd;
} /* setup_listen() */



/*
 * 	get line from fd
*/

// callback moze zwrocic: LINE_OK, LINE_CLOSED, LINE_BINARY
line_status fdgetline_cb(int fd, char* buf, size_t sizeof_buf, int *buf_size, line_callback callback, void *ptr, int *rx_bytes)
{
	int size;
	char *current, *pos, *buf_end;
	line_status res = LINE_OK;

	if ((size = read(fd, buf + (*buf_size), sizeof_buf - (*buf_size))) == -1) {
#ifdef SILENT_ECONNRESET
		if (errno == ECONNRESET) return LINE_CLOSED;
#endif
		if (errno == EINTR || errno == ETIMEDOUT) return LINE_EINTR;
		log_action(LOG_ERR, "fdgetline_cb:read error: %s", strerror(errno));
		return LINE_CLOSED;
	}
	if (size == 0) return LINE_CLOSED;

	if (rx_bytes) (*rx_bytes) += size;
	(*buf_size) += size;
	buf_end = buf + (*buf_size) - 1;

	for (current = buf;;) {
		for (pos = current; 1; pos++) {
			if (pos > buf_end) {
				pos = NULL;
				break;
			}
			if (*pos == '\n') break;
		}

		if (pos == NULL) {
			// dluga linia bez zakonczenia (ani CRLF ani \0)
			// stara sie najpierw skompletowac caly bufor
			// wlasciwie to drugi warunek zawiera pierwszy
			if (current == buf && *buf_size == sizeof_buf) {
				res = callback(current, NULL, *buf_size, ptr);
				*buf_size = 0;
			}
			break;
		} else {
			// pos na koncu linii: [pos] == '\n'
			size = pos - current + 1;
			res = callback(current, pos, size, ptr);
			*buf_size -= size;

			// jesli res != LINE_OK to cos sie dzieje
			// albo blad, albo LINE_BINARY
			if (res != LINE_OK || *buf_size == 0) break;
			current += size;
		}
	}

	// przesun dane na poczatek bufora
	if (*buf_size) memmove(buf, current, *buf_size);
	return res;
} /* fdgetline_cb() */



/*
 *	file descriptor printf()
*/

int fdprintf(int fd, char* format, ...)
{
	va_list ap;
	char buf[PRINTF_SIZE];
	int size, res, pos;

	va_start(ap, format);
	size = vsnprintf(buf, sizeof(buf), format, ap);
	va_end(ap);

	if (size == -1 || size > sizeof(buf)) {
		errno = ENOMEM;
		return -1;
	}

	for (pos = 0;;) {
/*
		if (timedout) {
			errno = ETIMEDOUT;
			return -1;
		}
*/
		if ((res = write(fd, buf+pos, size-pos)) == -1) {
			if (errno == EINTR) continue;
//			if (errno == ECONNRESET) return -1;
			return -1;
		}

		if (res == 0) break;

		pos += res;
		if (pos == size) break;
	}

	return res;
} /* fdprintf() */


/*
 * 	(sys)logging
*/

// does not break errno (!!), really necessary?
void log_action(int prio, char* format, ...)
{
	va_list ap;
	char buf[1024];
	int save;

	// log level check
	if (prio & LOG_ALWAYS) {
		prio &= ~(int) LOG_ALWAYS;
	} else {
		if (prio > config.log_level) return;
	}

	if (prio > LOG_DEBUG) prio = LOG_DEBUG;

	save = errno;
	if (foreground) {
		// does fprintf return -1 and set errno?
		fprintf(stderr, "[%d] ", getpid());
	
		va_start(ap, format);
		vfprintf(stderr, format, ap);
		va_end(ap);
	
		fprintf(stderr, "\n");
	} else {
		va_start(ap, format);
		vsnprintf(buf, sizeof(buf), format, ap);
		va_end(ap);
		TERMINATE_STRING(buf);

		syslog(prio, buf);
	}
	errno = save;
} /* log_action() */


/*
 * 	shared memory
*/

#ifdef USE_SHARED_MEM
void* shmalloc(size_t size, int *id)
{
	void *ptr;

	*id = shmget(IPC_PRIVATE, size, IPC_CREAT | 0600);
	if (*id == -1) {
		log_action(LOG_CRIT, "shmget(%d) failed: %s", size, strerror(errno));
		return NULL;
	}

	ptr = shmat(*id, NULL, 0);
	if (ptr == (void *) -1) {
		log_action(LOG_CRIT, "shmat() failed: %s", strerror(errno));
		shmctl(*id, IPC_RMID, NULL);
		return NULL;
	}

/*
	if (shmctl(*id, IPC_RMID, NULL) != 0) {
		log_action(LOG_CRIT, "shmctl(IPC_RMID) failed: %s", strerror(errno));
		return NULL;
	}
*/
	
	return ptr;
} /* shmalloc() */

int shmfreeid(int id)
{
	if (id == -1) return 0;

	if (shmctl(id, IPC_RMID, NULL) == 0) return 0;

	log_action(LOG_ERR, "shmctl(IPC_RMID) failed: %s", strerror(errno));
	return -1;
} /* shmfreeid() */
#endif


/*
 * 	removing unwanted characters from strings
*/

void untaint(/*@null@*/char *str, int len)
{
	if (str == NULL) return;

	if (len > 0) str[len-1] = '\0';

	for (; *str; str++) {
//		if (*str < 32 || *str > 127) *str = '#';
		if (*str < 32) *str = '#';
	}
} /* untaint() */

void untaint_for_filename(/*@null@*/char *str, int len)
{
	if (str == NULL) return;

	if (len > 0) str[len-1] = '\0';

	for (; *str; str++) {
		if (*str >= 'A' && *str <= 'Z') continue;
		if (*str >= 'a' && *str <= 'z') continue;
		if (*str >= '0' && *str <= '9') continue;
		switch (*str) {
			case '-':
			case '=':
			case '+':
			case ':':
				continue;

			default:
				*str = '_';
		}
	}
} /* untaint_for_filename() */


/*
 * 	other functions
*/

int is_load_above(double max, double *current)
{
	int res;
	double load[2];

	load[0] = load[1] = 0.0;
	if ((res = getloadavg(load, 2)) < 1) {
		log_action(LOG_ERR, "getloadavg() failed (%d)", res);
		if (current != NULL) *current = -1;
		return -1;
	}

	if (current != NULL) *current = load[0];
	if (max <= 0) return 0;
	
	if (load[0] > config.max_load) return 1;
	return 0;
} /* is_load_above() */

char* time2str(time_t t)
{
	static char buf[128];
	char c[] = "%c";	// jako zmienna, bo inaczej kompilator sie pluje

	// glibc warning
	strftime(buf, sizeof(buf), c, localtime(&t));
	TERMINATE_STRING(buf);

	return buf;
} /* time2str() */

void time2dhms(int t, int *d, int *h, int *m, int *s)
{
	*d = t / (3600*24);
	t %= (3600*24);
	*h = t / 3600;
	t %= 3600;
	*m = t / 60;
	t %= 60;

	*s = t;
} /* time2dhms */


char* line_closed_cause(line_status sta)
{
	switch (sta) {
		case LINE_OK:
			return "ok";
		case LINE_BINARY:
			return "binary";
		case LINE_EINTR:
			return "eintr";
		case LINE_CLOSED:
			return "client";
		case LINE_CLOSED_SERVER:
			return "server";
		case LINE_CLOSED_TIMEOUT:
			return "timeout";
		case LINE_CLOSED_ADMIN:
			return "admin";
		case LINE_ERROR:
			return "error";
	}

	return "unknown-bug";
} /* line_closed_cause */

// MAIL FROM: bartek@isp.pl
// MAIL FROM: <bartek@isp.pl>
// MAIL FROM:<bartek@isp.pl   > BODY=8BIT
char* alloc_msg_mail(char *line, int len)
{
	char buf[len+1];
	register unsigned char c;
	int i;

	if (!line) return NULL;

	// pomin spacje i tabulatory i '<' na poczatku
	while ((line[0] == ' ') || (line[0] == '\t') || (line[0] == '<')) line++;

	for (i=0; i<sizeof(buf); i++) {
		c = line[i];

		switch (c) {
		case '>':
		case ' ':
		case '\0':
		case '\t':
			buf[i] = '\0';
			goto quit;
		default: 
			/* do not untaint here, just copy */
			buf[i] = c;
		}
	}

quit:
	TERMINATE_STRING(buf);

	return strdup(buf);
} /* alloc_msg_mail() */


char* alloc_str_crlf(char *line, int len)
{
	char buf[len+1];
	register unsigned char c;
	int i;

	if (!line) return NULL;

	for (i=0; i<sizeof(buf); i++) {
		c = line[i];

		switch (c) {
		case '\r':
		case '\n':
			buf[i] = '#';
			break;
		case '\0':
			buf[i] = '\0';
			goto quit;
		default:
			/* do not untaint here, just copy */
			buf[i] = c;
		}
	}
	
quit:
	TERMINATE_STRING(buf);
	return strdup(buf);
} /* alloc_str_crlf */




