#!/usr/bin/perl -w
#@PERL@ -w

use strict;
use IO::Socket;
use IO::Select;
use IO::File;
use POSIX;
use Cwd;


# script path root
#my $path = '/home/bartek/prog/c/smtp-gated/tests';
#our $path = cwd();

# script connections (fds)
my %conns;
# some variables
my ($fd_sp);
my ($pid_clamd_emu, $pid_spamd_emu);
my $test_name;
# default client/server connection name

# script settings, config template, initial config template, 'smtp-gated -V' definitions
our (%set, %conf, %initconf, %defs);
our ($prev, @pipe);
our ($cli, $srv);
our $pid_sp;


my $reuse = 'Reuse';
#my $reuse = 'ReuseAddr';

#
# misc
#

sub mdebug($@)
{
	return unless $set{'debug'};
	my $format = shift;

	printf STDERR " # $format\n", @_;
}

sub t_sleep($)
{
	my ($time) = @_;
	select(undef, undef, undef, $time);
}

sub t_debug($@)
{
	my $format = shift;
	mdebug($format, @_);
}

sub t_echo($@)
{
	my $format = shift;
	printf $format, @_;
}

sub i_printable($)
{
	my $str = $_[0];

	$str =~ s%\r%\\r%go;
	$str =~ s%\n%\\n%go;

	return $str;
}

sub i_progress($)
{
	my ($val) = @_;
	printf "\rrunning %-32s ... %s", $test_name, $val;
}

sub t_progress($$)
{
	i_progress("\[$_[0]/$_[1]\]");
}

#
# process
#

sub t_spawn_proxy()
{
	my $arg;
	$arg = "$set{bin} $set{args} $set{conf} $set{redir}";
	mdebug('t_spawn [%s]', $arg);

	$fd_sp = new IO::File;

	eval {
		local $SIG{'USR1'} = sub {die "OK\n"};

		$pid_sp = open($fd_sp, "$arg|");
		die "t_spawn: $!" unless $pid_sp;

		sleep($set{'process_timeout'});
		die "t_spawn: timeout!\n";
	};

	die $@ if ($@ ne "OK\n");
	
	open(PID, $conf{'pidfile'}) || die "open($conf{pidfile}): $!\n";
	$pid_sp = <PID>;
	close(PID);

	chomp($pid_sp);

	mdebug('t_spawn:pid: %s', $pid_sp);
}

sub t_signal($)
{
	my $signal = $_[0];
	mdebug('t_signal %s, %s', $signal, $pid_sp);

	kill($signal, $pid_sp) || die "t_signal($signal, $pid_sp): $!\n";
}

sub t_wait()
{
	mdebug('t_wait');
	eval {
		local $SIG{'ALRM'} = sub { die "t_wait timeout!\n"; };
		alarm($set{'process_timeout'});
		wait();
		alarm(0);
	};
}

sub i_save_config()
{
	open(CONF, ">$set{conf}") || die "t_reload: save config($set{conf}): $!";
	foreach (keys %conf) {
		printf CONF "%-24s\t%s\n", $_, $conf{$_};
	}
	close(CONF);
}

sub t_reload()
{
	mdebug('t_reload');

	i_save_config();

	eval {
		local $SIG{'USR1'} = sub {die "OK\n"};

		t_signal(1); # SIGHUP

		sleep($set{'process_timeout'});
		die "t_spawn: timeout!\n";
	};

	die $@ if ($@ ne "OK\n");
}

sub t_init_config()
{
	%conf = %initconf;
}

#
# fake servers
#

sub spawn_server($&)
{
	my ($pid, $lsock, $sock, $res);
	my ($name, $helper) = @_;

	$pid = fork();
	die "spawn_server:fork($name) failed: $!\n" if ($pid < 0);
	return $pid if $pid;

	$0 = "$0 [".$name."_emu]";

	t_close_all();
	select(STDOUT); $| = 1;
	select(STDERR); $| = 1;

	# child

	eval {
		$lsock = IO::Socket::INET->new(Proto=>'tcp', Listen=>16, $reuse=>1,
			LocalAddr=>$set{"ip_$name"}, LocalPort=>$set{"port_$name"});

		die "IO::Socket::INET->new: $!\n" unless $lsock;

		for (;;) {
			$sock = $lsock->accept();

			$res = fork();
			if ($res > 0) {
				$sock->close();
				next;
			}
			die "spawn_server:child fork() failed: $!\n" if ($res < 0);

			$0 = "$0 child";

			select(STDOUT); $| = 1;
			select(STDERR); $| = 1;
			close($lsock);

			$sock->autoflush();
			&$helper($sock);

			$sock->close();
			exit(0);
		}
	};
	if ($@) {
		mdebug("! $name FAIL: $@\n");
		kill(SIGUSR2, getppid());
	}

	exit(0);
}

sub i_spawn_clamd_emu()
{
	return if ($pid_clamd_emu);

	$pid_clamd_emu = spawn_server 'clamd', sub {
		my $sock = $_[0];
		my $eicar = eicar();
		my $is_virus = 0;
	
		# SCAN filename\n
		$_ = <$sock>;

		if (!/^SCAN (.*)\n$/) {
			print $sock "none: invalid_query ERROR\n";
			return;
		}

		my $filename = $1;

		if (!open(FILE, $filename)) {
			print $sock "$filename: $! ERROR\n";
			return;
		}
		while (defined($_=<FILE>)) {
			tr%\r\n%%d;
			$is_virus = 1 if ($_ eq $eicar);
		}
		close(FILE);

		my $response = ($is_virus) ? "Clamd-Emu-Test-Signature FOUND" : "OK";

		mdebug("clamd_emu: $response [$filename]");
		print $sock "$filename: $response\n" || die "clamd_emu: can't write response!\n";
	};
}

sub i_spawn_spamd_emu()
{
	return if ($pid_spamd_emu);

	$pid_spamd_emu = spawn_server 'spamd', sub {
		my $sock = $_[0];
		my $gtube = gtube();
		my $is_spam = 0;

		# CHECK SPAMC/1.2\r\n
		$_ = <$sock>;
		unless (m%^CHECK SPAMC/(.*)\r\n%) {
			print $sock "ERROR\n";
			return;
		}
		my $version = $1;
		my ($score, $thr);
		$score = $thr = 10;

		while (defined($_=<$sock>)) {
			tr%\r\n%%d;
			$is_spam = 1 if ($_ eq $gtube);
			$score = $1 if (/^Score: (.*)$/);
			$thr = $1 if (/^Thr: (.*)$/);
		}

		$score = -2.5 unless $is_spam;

		# SPAMD/%s 0 EX_OK\r\nSpam: %*s ; %lf / %lf \r\n
		my $response = sprintf "SPAMD/%s 0 EX_OK\r\nSpam: %s ; %s / %s \r\n",
			$version, $is_spam ? "True" : "False", $score, $thr;

		mdebug("spamd_emu: " . i_printable($response));
		print $sock "$response" || die "spamd_emu: can't write response!\n";
	};
}

sub i_kill_clamd_emu()
{
	return unless defined($pid_clamd_emu);

	mdebug('i_kill_clamd_emu');
	kill(15, $pid_clamd_emu) || die "i_kill_clamd_emu: $!\n";
}

sub i_kill_spamd_emu()
{
	return unless defined($pid_spamd_emu);

	mdebug('i_kill_spamd_emu');
	kill(15, $pid_spamd_emu) || die "i_kill_spamd_emu: $!\n";
}

# process configuration


#
# network
#

sub t_listen()
{
	mdebug('listen %s:%s', $set{'ip_mta'}, $set{'mta_port'});

	return if defined($conns{'*'});

	$conns{'*'} = IO::Socket::INET->new(Proto=>'tcp', Listen=>16, $reuse=>1,
		LocalAddr=>$set{'ip_mta'}, LocalPort=>$set{'mta_port'});

	unless (defined($conns{'*'})) {
		delete $conns{'*'};
		die "t_listen: $!";
	}
}

sub t_connect($;$)
{
	my ($name, $srcip) = @_;
	$srcip = $set{'src_ip'} unless defined($srcip);

	mdebug('connect %s (%s:%s)', $name, $conf{'bind_address'}, $conf{'port'});

	$conns{$name} = IO::Socket::INET->new(Proto=>'tcp', LocalHost=>$srcip,
		PeerAddr=>$conf{'bind_address'}, PeerPort=>$conf{'port'});

	unless (defined($conns{$name})) {
		delete $conns{$name};
		die "t_connect: $!\n"
	}

	$conns{$name}->autoflush(1);
}

sub t_close($)
{
	my ($name) = @_;
	mdebug('close %s', $name);

	$conns{$name}->close();
	delete $conns{$name};
}

sub t_close_all()
{
	mdebug('close_all');
	foreach (keys %conns) {
		$conns{$_}->close();
		delete $conns{$_};
	}
#	t_sleep($set{'close_delay'});
}

sub t_accept($)
{
	my ($name) = @_;
	mdebug('t_accept [%s]', $name);

	eval {
		local $SIG{'ALRM'} = sub { die "t_accept[$name] timeout!\n"; };

		alarm($set{'timeout'});
		$conns{$name} = $conns{'*'}->accept();
		alarm(0);
	};

	die if $@;
	unless (defined($conns{$name})) {
		delete $conns{$name};
		die "t_accept: $!"
	}

	$conns{$name}->autoflush(1);
}

sub t_print($$)
{
	my ($name, $str) = @_;
	my $handle = $conns{$name};

	die "handle '$name' not open!\n" unless defined($handle);

	mdebug('print to [%s] string [%s]', $name, i_printable($str));

	print $handle $str;
	$prev = $str;
}

sub t_println($$)
{
	my ($name, $str) = @_;

	t_print($name, "$str\r\n");
}

sub t_println_push($@)
{
	my ($name, @arr) = @_;

	foreach (@arr) {
		$_ = "$_\r\n";
		t_print($name, $_);
		push @pipe, $_;
	}
}

sub i_expect($$$)
{
	my ($name, $regex, $is_regex) = @_;
	my $handle = $conns{$name};
	die "handle '$name' not open!\n" unless defined($handle);

	my $f = $is_regex ? "t_expect_regex" : "t_expect";

	my $cregex = i_printable($regex);
	mdebug('%s from [%s] expect [%s]', $f, $name, $cregex);

	my $line;
	eval {
		local $SIG{'ALRM'} = sub { die "$f($name,[$cregex]) timeout!\n"; };
		alarm($set{'timeout'});
		$line = <$handle>;
		alarm(0);
	};

	die $@ if ($@);
	die "$f($name): undefined\n" unless defined($line);

	my $cstr = i_printable($line);
	mdebug('%s from [%s] got [%s]', $f, $name, $cstr);

	my $result = ($is_regex) ? ($line =~ /$regex/) : ($line eq $regex);
	die "$f($name): expected [$cregex] but got [$cstr]!" unless $result;
}

sub t_expect($$)
{
	my ($name, $regex) = @_;
	i_expect($name, $regex, 0);
}

sub t_expect_regex($$)
{
	my ($name, $regex) = @_;
	i_expect($name, $regex, 1);
}


# $n = 'all'
sub t_expect_pop($;$)
{
	my ($name, $n) = @_;

	$n = 1 unless defined($n);
	$n = $#pipe if ($n == 0);

	for (; $n>0; $n--) {
		for (;;) {
			$_ = shift @pipe;
			last unless defined $_;

			i_expect($name, $_, 0);
		}
	}
}

sub t_expect_flush($)
{
	my ($name) = @_;
	my $handle = $conns{$name};
	mdebug('t_expect_flush [%s]', $name);

	my $line;
	eval {
		local $SIG{'ALRM'} = sub { die; };
		alarm($set{'flush_timeout'});
		for (;;) {
			$line = <$handle>;
			last unless defined($line);

			mdebug('t_expect_flush from [%s] [%s]', $name, i_printable($line));
		}
		alarm(0);
	};

	@pipe = ();
}

sub t_expect_any($)
{
	my ($name) = @_;
	my $handle = $conns{$name};
	mdebug('t_expect_any [%s]', $name);

	my $line;
	eval {
		local $SIG{'ALRM'} = sub { die "t_expect_any[$name] timeout!\n"; };
		alarm($set{'timeout'});
		for (;;) {
			$line = <$handle>;
			last unless defined($line);
		}
		alarm(0);
	};

	die $@ if ($@);
}

sub t_expect_closed($;$)
{
	my ($name, $timeout) = @_;
	my $handle = $conns{$name};
	mdebug('t_expect_closed [%s]', $name);

	$timeout = $set{'timeout'} unless defined($timeout);

	my $line;
	eval {
		local $SIG{'ALRM'} = sub { die "t_expect_closed[$name] timeout!\n"; };
		alarm($timeout);
		$line = <$handle>;
		alarm(0);
	};

	die $@ if ($@);

	if (defined($line)) {
		my $cstr = i_printable($line);
		die "t_expect_closed($name): got [$cstr]\n";
	}

	delete $conns{$name};
}

sub t_expect_nothing($;$)
{
	my ($name, $timeout) = @_;
	my $handle = $conns{$name};
	mdebug('t_expect_nothing [%s]', $name);

	$timeout = $set{'flush_timeout'} unless defined($timeout);

	my $line;
	eval {
		local $SIG{'ALRM'} = sub { die "OK\n"; };
		alarm($timeout);
		$line = <$handle>;
		alarm(0);
	};

	if ($@ ne "OK\n") {
		my $cstr = i_printable($line);
		die "t_expect_nothing($name): got [$cstr]\n";
	}
}


sub eicar()
{
	my $body;

	$body = 'X5O!P%@AP';
	$body .= '[4\\PZX54(P^)';
	$body .= '7CC)7}$EICA';
	$body .= 'R-STANDARD-A';
	$body .= 'NTIVIRUS-TES';
	$body .= 'T-FILE!$H+H*';

	return $body;
}

sub gtube()
{
	my $body;

	$body = 'XJS*C4JDBQAD';
	$body .= 'N1.NSBN3*2ID';
	$body .= 'NEN*GTUBE-ST';
	$body .= 'ANDARD-ANTI-';
	$body .= 'UBE-TEST-EMA';
	$body .= 'IL*C.34X';

	return $body;
}

sub t_eicar_body()
{
	return (
		'Subject: eicar test message',
		'MIME-Version: 1.0', 
		'Content-Type: multipart/mixed; boundary="=-=-="',
		'',
		'--=-=-=',
		'Content-Type: application/octet-stream',
		'Content-Disposition: attachment; filename=eicar.gif',
		'Content-Description: EICAR test file',
		'',
		eicar(),
		'--=-=-=--',
		'',
		'.');
}

sub t_gtube_body(;$)
{
	my $score = $_[0];
	$score = 10 unless defined($score);

	return (
		'Subject: gtube test message',
		'MIME-Version: 1.0', 
		'Content-Type: text/plain; charset=us-ascii',
		'Content-Transfer-Encoding: 7bit',
		"Score: $score",
		'',
		gtube(),
		'',
		'.');
}


#
# SMTP functions
#

sub t_change($$)
{
	($cli, $srv) = @_;
}

sub t_smtp_init(;$$;$)
{
	my ($_cli, $_srv, $srcip) = @_;

	if (defined($_cli)) {
		$cli = $_cli;
		$srv = $_srv;
	}

	mdebug('--- t_smtp_init(%s,%s)', defined($_cli) ? $_cli : '', defined($_srv) ? $_srv : '');
	t_connect($cli, $srcip);
	t_accept($srv);

	# mandatory - otherwise proxy pipeline queue gets out-of-order
	t_println($srv, '220 fake MTA says hello');
	t_expect($cli, $prev);
}

sub t_smtp_helo()
{
	mdebug('--- t_smtp_helo()');
	t_println($cli, 'HELO fake.MUA');
	t_expect($srv, $prev);
	t_println($srv, '250 Hello fake.MUA');
	t_expect($cli, $prev);
}

sub t_smtp_ehlo()
{
	mdebug('--- t_smtp_ehlo()');
	t_println($cli, 'EHLO fake.MUA');
	t_expect($srv, $prev);
	t_println_push($srv,
		'250-fake MTA greets fake.MUA',
		'250-PIPELINING',
		'250 STARTTLS'
	);
	t_expect_pop($cli);
}

sub t_smtp_mail_rcpt()
{
	t_println_push($cli,
		'MAIL FROM: <bartek@test.test>',
		'RCPT TO: <bartek@test.test>'
	);
	t_expect_pop($srv);
	t_println_push($srv, '250 OK', '250 OK');
	t_expect_pop($cli);
}


sub t_smtp_send()
{
	t_smtp_mail_rcpt();

	t_println($cli, 'DATA');
	t_expect($srv, $prev);
	t_println($srv, '354 go ahead');
	t_expect($cli, $prev);
	
	t_println_push($cli,
		'From: source@test.com',
		'To: source@test.com',
		'',
		'the one and only line',
		'.'
	);
	
	t_expect_pop($srv);
	
	t_println($srv, '250 Spool OK');
	t_expect($cli, $prev);
}

sub t_smtp_one($$)
{
	my ($verb, $response) = @_;

	t_println($cli, $verb);
	t_expect($srv, $prev);
	t_println($srv, $response);
	t_expect($cli, $prev);
}

sub t_smtp_quit()
{
	t_println($cli, 'QUIT');
	t_expect($srv, $prev);
	t_println($srv, '221 Bye bye');
	t_expect($cli, $prev);
	t_close($srv);
	t_expect_closed($cli);
}


#
# start

#
# test configuration

$set{'debug'} = 2;
$set{'path'} = cwd();
$set{'path_log'} = "$set{path}/log";
$set{'log'} = "$set{path_log}/test.log";

$set{'src_ip'} = '127.0.0.1';
$set{'src_ip_alt'} = '127.0.0.2';
$set{'test_port'} = 2110;

$set{'ip_mta'} = $set{'src_ip'};
$set{'mta_port'} = $set{'test_port'} + 1;
$set{'ip_clamd'} = $set{'ip_mta'};
$set{'port_clamd'} = $set{'test_port'} + 2;
$set{'ip_spamd'} = $set{'ip_mta'};
$set{'port_spamd'} = $set{'test_port'} + 3;

$set{'timeout'} = 5;
$set{'flush_timeout'} = 1;
$set{'process_timeout'} = 3;
$set{'close_delay'} = 0.0;

$set{'bin'} = "$set{path}/../src/smtp-gated";
$set{'args'} = '-fff';
$set{'conf'} = "$set{path}/log/test.conf";
#$set{'redir'} = "1>$set{log} 2>$set{log}";
$set{'redir'} = '';

for (;;) {
	$_ = $ARGV[0];
	last unless defined($_) and /^-/;

	if (/^-long$/) {
		shift @ARGV;
		$set{'long'} = 1;
	} else {
		print "unknown argument: $_\n";
		exit 3;
	}
}

#
# read compilation definitions

open(DEF, "$set{bin} -V|") || die "can't read defines: $!\n";
while (defined($_=<DEF>)) {
	chomp;
	next unless /^ *(.*?) *: +(.*?) *$/o;

	$defs{$1} = $2;
}
close(DEF);
#printf "%s => %s\n", $_, $defs{$_} foreach (sort keys %defs);

open(DEF, "$set{bin} -t|") || die "can't read defines: $!\n";
while (defined($_=<DEF>)) {
	chomp;
	next unless /^ *([^ ]+) *(.*?) *$/o;

	$conf{$1} = $2;
}
close(DEF);
#printf "%s => %s\n", $_, $conf{$_} foreach (sort keys %conf);


#
# daemon default configuration

$conf{'pidfile'} = "$set{path}/test.pid";
$conf{'spool_path'} = "$set{path}/msg";
$conf{'lock_path'} = "$set{path}/lock";
$conf{'proxy_name'} = 'proxy.auto.test';
$conf{'bind_address'} = $set{'ip_mta'};
$conf{'port'} = $set{'mta_port'} + 10;
$conf{'fixed_server'} = $set{'ip_mta'};
$conf{'fixed_server_port'} = $set{'mta_port'};
$conf{'clamd_path'} = "$set{ip_clamd}:$set{port_clamd}";
$conf{'spamd_path'} = "$set{ip_spamd}:$set{port_spamd}";
$conf{'log_level'} = 7;
$conf{'max_connections'} = 8;
$conf{'max_per_host'} = 4;
$conf{'ignore_errors'} = 0;
$conf{'spam_max_size'} = 0;
%initconf = %conf;

#$conf{'log_helo'} = 1;
#$conf{'log_mail_from'} = 7;
#$conf{'log_rcpt_to'} = 7;
#$conf{'nat_header'} = 0;

$set{'lock'} = "$conf{lock_path}/$set{src_ip}";

#
# setup

$| = 1;

unless (-d $conf{'lock_path'}) {
	die "lock_path[$conf{lock_path}] is not a directory!\n" if (-e $conf{'lock_path'});
	mkdir $conf{'lock_path'} || die "can't mkdir lock_path[$conf{lock_path}]: $!\n";
}

unless (-d $conf{'spool_path'}) {
	die "spool_path[$conf{spool_path}] is not a directory!\n" if (-e $conf{'spool_path'});
	mkdir $conf{'spool_path'} || die "can't mkdir spool_path[$conf{spool_path}]: $!\n";
}

eval {
	my $testconn = IO::Socket::INET->new(Proto=>'tcp', Listen=>1, $reuse=>1,
		LocalAddr=>$set{'src_ip_alt'}, LocalPort=>$set{'mta_port'}+10);

	die unless defined($testconn);
	$testconn->close();
};
$set{'alt_ip_ok'} = ($@) ? 0 : 1;
#printf "alt_ip: %s\n", $set{'alt_ip_ok'};


#
# find tests

my @tests;

if (@ARGV == 0) {
	opendir(DIR, ".") || die "can't open '.' directory: $!\n";
	@tests = sort grep { /^[0-9].*\.t$/o } readdir(DIR);
	closedir(DIR);
} else {
	@tests = @ARGV;
}

print '=' x 54,"\n";
printf "%16s found: %s test(s)\n", "", scalar @tests;
print '=' x 54,"\n";

#
# run tests

open(STDERR, '>'.$set{log}) || die "can't redirect STDERR to $set{log}: $!\n";
$| = 1;
print STDERR "*** $0 pid $$\n";
print STDERR "*** ".scalar localtime()."\n\n\n";
unlink($conf{'pidfile'});

my ($passed, $failed, $notrun);
$passed = $failed = $notrun = 0;

# signals
$SIG{'__WARN__'} = sub { print STDERR "\n--- WARNING: $_[0]" };
$SIG{'CHLD'} = 'IGNORE';
$SIG{'HUP'} = 'IGNORE';
$SIG{'USR1'} = 'IGNORE';
$SIG{'USR2'} = sub { die; };
#$SIG{'TERM'} = sub { die };

# autoflush
#select(STDOUT); $| = 1;
#select(STDERR); $| = 1;

my %results;

eval {
	i_spawn_clamd_emu();
	i_spawn_spamd_emu();
	i_save_config();
	t_spawn_proxy();

	foreach (@tests) {
		$test_name = $_;
		i_progress("");
#		printf "running %-26s ...  ", $test_name;
		print STDERR "\n\n--- $test_name ---\n";
		$set{'start'} = time();

		# cleanup
		unlink($set{'lock'});

		t_listen();
		t_init_config();
		# test script must do reload by itself
#		t_save_config();

		$prev = undef;
		@pipe = ();
		$cli = 'cli';
		$srv = 'srv';

		eval {
			require "$test_name";
		};
		printf STDERR "--- DURATION: %s\n", time() - $set{'start'};
		if ($@ =~ m%^TODO\n%io) {
			$notrun++;
			$results{$test_name} = '[todo]';
			print STDERR "\n--- $test_name: TODO\n";
		} elsif ($@ =~ m%^N/A\n%io) {
			$notrun++;
			$results{$test_name} = '[n/a]';
			print STDERR "\n--- $test_name: N/A\n";
		} elsif ($@ =~ m%^LONG\n%io) {
			$notrun++;
			$results{$test_name} = '[long]';
			print STDERR "\n--- $test_name: LONG\n";
		} elsif ($@) {
			$failed++;
			chomp $@;
			$results{$test_name} = 'FAILED';
			print STDERR "\n--- $test_name: FAILED: $@\n";
		} else {
			$passed++;
			$results{$test_name} = 'pass';
			print STDERR "\n--- $test_name: PASSED\n";
		}

		i_progress($results{$test_name});
		printf "%s\n",  " "x16;

		t_close_all();
	}
	
	#
	# quit

	t_close_all();
	i_kill_clamd_emu();
	i_kill_spamd_emu();
	eval {
		t_signal(15);
		t_wait();
	};

	printf STDERR "%s\n", "=" x 54;
	printf STDERR "%30s\n", "SUMMARY";
	printf STDERR "%s\n", "=" x 54;
	printf STDERR "%-10s %-32s\n", $results{$_}, $_ foreach (sort keys %results);
	printf STDERR "%s\n", '=' x 54;
	printf STDERR "  passed:%-3s | failed:%-3s | skipped:%-3s | total:%-3s\n", $passed, $failed, $notrun, scalar @tests;
	printf STDERR "%s\n", '=' x 54;

	printf "%s\n", '=' x 54;
	printf "  passed:%-3s | failed:%-3s | skipped:%-3s | total:%-3s\n", $passed, $failed, $notrun, scalar @tests;
	printf "%s\n", '=' x 54;
};

if ($@) {
	print "! FAIL: $@\n";
	t_signal(15);
	t_close_all();
	i_kill_clamd_emu();
	i_kill_spamd_emu();

	eval {
		t_wait();
	};
	if ($@) {
		t_signal(9);
	}
	exit(1);
}

exit($failed != 0);


