
/***************************************************************************
                          sessionmanager.cpp  -  description
                             -------------------
    begin                : lun apr 28 15:20:00 CET 2003
    copyright            : (C) 2003 by Romain Vinot
    email                : vinot@aist.enst.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef _WIN32
#pragma warning (disable : 4786)
#endif

#ifdef _WIN32
#include "zlib.h"
#else
#include <zlib.h>
#endif

#include <qdom.h>
#include <qtextstream.h>

#include "sessionmanager.h"

#include "gamemanager.h"
#include "board.h"
#include "guicommandline.h"
#include "datastoring.h"
#include "level.h"
#include "network.h"
#include "mailservice.h"

SessionManager::SessionManager(GUICommandLine *g)
  : guicmd(g)
{}

bool SessionManager::NewGameStep1(void)
{
  if (man->isInGame()) {
    if (man->getGameType()==PBEM) {
      guicmd->printPopupMessage("You're playing a PBEM game.\n"
       "You must finish your turn or your game will not be playable anymore.");
      return false;
    }
    if (!guicmd->GetYesOrNo("Are you sure you want\n"
		    "to lose your current game to begin a new one ?"))
      return false;
    board->removeAll();
    man->removeAll();
  }

  man->InitChooseGameType();
  playerTurn=MARINE_TURN;
  gameType type = man->getGameType();

  if (type==NETWORK) {
    // Waiting for the connection.
    server = new NetworkServer(guicmd);
    guicmd->Write("Waiting for the connection of your opponent...");
    return true;
  }
  return NewGameStep2();
}

bool SessionManager::NewGameStep2(bool filereceived)
{
  if (filereceived)
    ; // TODO load from content.
  else 
    if (man->InitChooseLevel()) {
      man->InitPlayerPlacement();
      return true;
    }
  return false;
}

bool SessionManager::LoadFromFilename(const char *filename)
{
  bool res = store->LoadFromFilename(filename);
  if (res) return Load();
  else {
    guicmd->Write("The file cannot be loaded.");
    return false;
  }
}

bool SessionManager::LoadFromContent(QString &content)
{
  bool res = store->LoadFromContent(content);
  if (res) return Load();
  else {
    guicmd->Write("The game cannot be loaded.");
    return false;
  }
}

bool SessionManager::Load(void)
{
  if (man->getGameType()==PBEM) {

    if (man->pbemidGenestealer==0) {
      // This is the first sending for Genestealer player.
      playerTurn = GENESTEALER_TURN;
      man->pbemidGenestealer = store->getNewPbemId();
      
      if (man->nbTurns==0) {
	// This is the initialisation phase.
	guicmd->Write("Initialisation of genestealer units");
	man->level->InitGenestealer();
	man->turn=GENESTEALER_TURN;
	store->addPbem(man->pbemidGenestealer,man->nbTurns,man->turn);
	man->nbTurns=0;
	store->SaveGameState();
	if (man->level->getFirstPlayerIsMarine()) {
	  guicmd->printPopupMessage("Your opponent will now begin the game.\n"
				    "Please save and send the file.");
	  man->SavePBEM();
	  board->removeAll();
	  man->removeAll();
	}
	else {
	  guicmd->printPopupMessage("First player is genestealer.\n"
				    "You can begin your turn.");
	  man->nbTurns=1;
	  man->InitPBEM();
	  man->beginTurn();
	}
	return true;
      }

      if (man->nbTurns==1 && man->level->isInitGenestealerNull()) {
	//There was no initialisation, so this is already first turn.
	guicmd->Write("Game successfully loaded.\n"
		      "Going to replay the last turn.");
	store->addPbem(man->pbemidGenestealer, 0, GENESTEALER_TURN);
	man->Replay();
	guicmd->Write("\nReplay has been done. You can now play your turn.");
	if (man->turn==MARINE_TURN)
	  // First turn is for genestealer : replay has not done a endTurn.
	  man->endTurn();
	man->InitPBEM();
	man->beginTurn();
	store->updatePbem(man->pbemidGenestealer, man->nbTurns, man->turn);
	return true;
      }
    }

    int pbemid;
    if (man->turn==MARINE_TURN) {
      pbemid = man->pbemidGenestealer;
      playerTurn = GENESTEALER_TURN;
    }
    else {
      pbemid = man->pbemidMarine;
      playerTurn = MARINE_TURN;
    }

    pair<int,Turn> *storedPbem = store->getPbemFromList(pbemid);
    if (man->cheatProtection < 3 &&
	(storedPbem == 0 ||
	 storedPbem->second==man->turn ||
	 (storedPbem->second==MARINE_TURN&&storedPbem->first!=man->nbTurns) ||
	 ((storedPbem->second==GENESTEALER_TURN||storedPbem->second==CAT_TURN)
	  && storedPbem->first!=man->nbTurns-1))) {
      if (man->cheatProtection==1) {
	board->removeAll();
	man->removeAll();
	man->inGame = false;
	guicmd->printPopupMessage("You are not allowed to play "
				  "this PBEM game.");
	return false;
      }
      man->hasCheated=true;
    }
    guicmd->Write("Game successfully loaded.\nGoing to replay the last turn.");
    man->Replay();
    guicmd->Write("\nReplay has been done. You can now play your turn.");
    if (man->nbTurns==0) { // Otherwise done by the Replay command.
      man->nbTurns=1;
      man->turn=MARINE_TURN;
    }
    man->InitPBEM();
    man->beginTurn();
    store->updatePbem(pbemid, man->nbTurns, man->turn);
    return true;
  }

  if (man->getGameType()==NETWORK) {
    playerTurn = GENESTEALER_TURN;
    
    if (man->nbTurns==0) {
      // This is the initialisation phase.
      man->setReplayRecordMode(true);
      guicmd->Write("Initialisation of marine units");
      man->level->InitMarine();
      man->setReplayRecordMode(false);
      guicmd->Write("Initialisation of genestealer units");
      man->level->InitGenestealer();
      
      if (man->level->getFirstPlayerIsMarine()) {
	guicmd->Write("The game can begin. First turn is for marine player.");
	man->setReplayRecordMode(true);
	man->nbTurns=1;
	man->beginTurn();
	man->Replay();
	return true;
      }
      else {
	guicmd->Write("The game can begin. "
		      "First turn is for genestealer player.");
	man->setReplayRecordMode(false);
	man->nbTurns=1;
	man->beginTurn();
      }
      return true;
    }
    else {
      cout << "TODO: Not working yet !\n";
      exit(1);
    }
  }

  guicmd->Write("Your game has been successfully loaded.");
  man->Replay();
  return true;
}

bool SessionManager::Save(const char *filename)
{
  if (!man->isInGame()) {
    guicmd->Write("You can't save when you have not begun a game.");
    return false;
  }
  else if (man->getGameType()==PBEM) {
    guicmd->printPopupMessage("You can't save during PBEM game.");
    guicmd->Write("Your PBEM game has not been saved.");
    return false;
  }
  else {
    QDomDocument doc("savegame");
    store->Save(doc);
    QString content = doc.toString();
    WriteOnFile(content,filename);
    guicmd->Write("Your game has been saved.");
    return true;
  }
}

bool SessionManager::SaveRecording(void)
{
  if (!man->isInGame()) {
    guicmd->Write("You can't save a recording without a game !");
    return false;
  }
  else {
    QDomDocument *doc = store->SaveWithRecord();
    QString content = doc->toString();
    QString filename = guicmd->GetFilename(false);
    WriteOnFile(content,filename.latin1());
    delete doc;
    return true;
  }
}

bool SessionManager::SaveRecordingToEmail(void)
{
  if (!man->isInGame()) {
    guicmd->Write("You can't save a recording without a game !");
    return false;
  }
  else {
    QDomDocument *doc = store->SaveWithRecord();
    WriteOnEmail(doc);
    delete doc;
    return true;
  }
}

bool SessionManager::EndOfTurn(void)
{
  if (man->areGenestealerInEntryZone()) {
    bool cont = guicmd->GetYesOrNo("You have some genestealer(s) in an entry "
				   "zone.\nIf you don't make them enter the "
				   "board, they will be lost.\n"
				   "Do you want to continue and lose them ?");
    man->recordCommand(QString("get yesorno ") + QString::number(cont));
    if (!cont)
      return false;
  }

  if (man->getGameType()==HOTSEAT) {
    QString command = QString("save ") + homeDir + DIR_DELIMITER + "savegame" + DIR_DELIMITER + "autosave.sav";
    guicmd->commandSave(command);
  }

  pair<bool,Turn> res = man->endTurn();
  Turn newt = res.second;
  if (res.first) { // This the end of the game.
    if (man->getGameType() == PBEM && !man->isInReplayMode()) {
      guicmd->printPopupMessage("You must now send and save the file "
				"for your opponent.");
      man->SavePBEM();
    }
    board->removeAll();
    man->removeAll();
    guicmd->ClearInfoPanel();
    return true;
  }
  if (man->getGameType()==PBEM) {
    if (!man->isInReplayMode() && playerTurn!=newt) {
      // This is the end of the player turn. Store file and quit.
      guicmd->printPopupMessage("You have played your turn for this PBEM game. Please save and send the file.");
      man->SavePBEM();
      board->removeAll();
      man->removeAll();
      guicmd->ClearInfoPanel();
      return true;
    }
  }
  if (man->getGameType()==NETWORK) {
    if (playerTurn==newt)
      man->setReplayRecordMode(false);
    else
      man->setReplayRecordMode(true);
  }
  guicmd->ClearInfoPanel();
  guicmd->DrawInterface();
  switch (man->getTurn()) {
  case MARINE_TURN:
    guicmd->Write("This is the turn of the Marine player.");
    break;
  case GENESTEALER_TURN:
    guicmd->Write("This is the turn of the Genestealer player.");
    break;
  case CAT_TURN:
    guicmd->Write("This is the turn of the CAT module.");
    if (newt==MARINE_TURN)
      guicmd->Write("Cat module is free. Marine player can control it for this turn.");
    else
      guicmd->Write("Cat module is free. Genestealer player can control it for this turn.");
    break;
  }
  if(! (man->getGameType()==PBEM && man->isInReplayMode() &&
	man->replays.size()==0) )
    // You must not run beginTurn if this is the end of the replay.
    man->beginTurn();
  if (man->getGameType()==NETWORK && man->isInReplayMode())
    man->Replay();
  return true;
}

bool SessionManager::Quit(void)

{

//  if (man->getGameType()==PBEM) {

//	 guicmd->printPopupMessage("By exiting now, you will lose the ability to continue your PBEM game.");	

//  }

  return true;

}

void SessionManager::WriteOnFile(QString &content, const char *filename)
{
  gzFile f = gzopen(filename,"wb8");
  if (!f) {
    cout << "Cannot open file " << filename << endl;
    return;
  }
  
  // Write XML doc on compressed file.
  QByteArray a;
  QTextStream ts(a, IO_WriteOnly);
  ts << content;
  char *toread = a.data();
  gzwrite(f,toread,a.size());
  gzclose(f);
}

void SessionManager::WriteOnEmail(QDomDocument *doc)
{
  // Get SMTP infos.
  SmtpInfo infos;
  infos.smtp = store->getPref("smtp");
  infos.yourEmail = store->getPref("email");
  infos.opponentEmail = man->getEmail();
  infos.port=store->getPref("smtpPort").isEmpty() 
    ? QString("25") : store->getPref("smtpPort");
  if (store->getPref("smtpUseAuth").isEmpty()) infos.useAuth=false;
  else infos.useAuth = (store->getPref("smtpUseAuth")=="yes");
  infos.login = store->getPref("smtpLogin");
  infos.password = store->getPref("smtpPassword");
  guicmd->GetSmtpInfos(&infos);
  store->addPref("smtp",infos.smtp);
  store->addPref("email",infos.yourEmail);
  store->addPref("smtpPort",infos.port);
  if (infos.useAuth) store->addPref("smtpUseAuth","yes");
  else store->addPref("smtpUseAuth","no");
  store->addPref("smtpLogin",infos.login);
  store->addPref("smtpPassword",infos.password);    
  
  // Change the email adress to your own.
  QDomElement root = doc->documentElement();
  QDomElement xmlman = root.namedItem("gamemanager").toElement();
  xmlman.setAttribute("email",infos.yourEmail);

  // We save the data to local file
  QString content = doc->toString();
  QString filename = guicmd->GetFilename(true);
  WriteOnFile(content,filename.latin1());

  // We try to send the data by email.
  if (infos.smtp=="")
    man->guicmd->printPopupMessage("Your SMTP server has not been defined.\nYou must send the file manually.");
  else {
    QFile readagain(filename);
    readagain.open(IO_ReadOnly);
    int possiblelen = (int) (content.length()*1.02)+12;
    char *zipped = (char *) malloc(possiblelen);
    int reallen = readagain.readBlock(zipped,possiblelen);
    readagain.close();
    
    Mail mail( infos.yourEmail, infos.opponentEmail);
    QFileInfo fileInfo( filename );
    QString shortname = fileInfo.fileName();
    mail.createMail(zipped, reallen, shortname, infos.playermsg);
    mail.sendMail(infos.smtp, infos.port, infos.useAuth, infos.login,
		  infos.password);
  }
}
