#!/usr/bin/perl -T
#
# Copyright  2000-01, Phil Kernick, ROTFL Enterprises
#
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE DEVELOPERS ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE DEVELOPERS BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#------------------------------------------------------------------------
# SQUEUER
#  A queueing proxy for Seti@Home
#
#  This proxy was written for two reasons:
#   1.  We got a new Mac, and it was generating work units every 5 hours,
#       and as I don't have a permanent net connection, I wanted a way
#       to be able to batch up the download and upload of work units.
#   2.  The only similar proxy is SetiQ, but the developer did it only for
#       Windows, and wouldn't release the source so that it could be ported
#       to FreeBSD.  It also only worked for a single user.  This proxy 
#       has much better handling of a multiuser disconnected site.
#
#  I dedicate the server to Monica Ciric, without whom I wouldn't have had a
#  reason to write it.  It also couldn't have been done without the help of
#  Greg Lewis, who has been beta testing it and adding cool features.  Thanks
#  also to Gary Watts for testing it on Solaris and Steve Challans for testing
#  it on WinNT and Win9x.
#
# Author
#  Phil Kernick
#  philk@rotfl.com.au
#
# History
#  $Id: squeuer.pl,v 1.3 2003/10/02 01:42:12 glewis Exp $
#
#  $Log: squeuer.pl,v $
#  Revision 1.3  2003/10/02 01:42:12  glewis
#  . Fix a bug where if all you have are existing clients with results to
#    submit, they never get submitted and no new work units are ever gotten.
#
#    Longer explanation follows:
#
#    Basically, if you have a "new" (to squeuer) user, but an existing client
#    set up then the first thing the client is likely to do is submit a result.
#    The client expects that sending a result will generate a reply of "here
#    are the stats for the user".  It will keep sending the result until it
#    gets that reply (it doesn't think its been successful until then) and
#    never request a new work unit.  Squeuer can't reply with the stats since
#    it hasn't got any for the user yet (its never sent a result).  So it
#    accepts the result, but doesn't provide the reply the client wants.
#    However, Squeuer never executes the "send results" code as it never gets
#    there (there is no get_work_unit request, so the "next;" statement in
#    the check for get_work_unit keeps Squeuer from ever trying to send
#    results), meaning it never has the stats to reply with.  So the client
#    is stuck endlessly trying to send the result and Squeuer can never provide
#    the right reply to the client since it never tries to send the result and
#    thus get the stats it needs to reply with.
#
#  Revision 1.2  2002/11/01 03:43:34  glewis
#  . Correctly terminate arguments to tr.
#
#  Revision 1.1.1.6  2002/11/01 03:38:17  glewis
#  Import Squeuer source
#
#  Revision 2.0  2001/04/18 12:10:53  philk
#  - Fixed a bug where signals weren't being correctly propagated to the transfer_loop
#  - Added reporting of total work units completed to log
#  - Added two second delay between uploading results
#  - Added result name to client log
#  - If client can't fork, force the tranfer_loop after every allowed connect
#  - Make sure the queue_depth doesn't go negative with a large number of old work units in the spool
#  - Return status information from set_proxy_password
#  - Don't wait for retry delay when proxy auth required
#
#  Revision 1.9  2001/03/21 11:20:24  philk
#  - Put under the BSD license
#  - Now require 5.005
#  - Reworked log levels to better match syslog meanings
#  - Used longer delays when queue is empty
#  - Minor fixes to startup script and config file
#  - Don't do the root check in the BEGIN
#  - Don't process actions for unspecified user_id
#  - Check for HTTP error responses
#  - Added support for Basic Auth proxies
#  - Used $LF for all protocol eol checks
#  - Added config variable: have_fork to enable single threaded mode on Win32
#  - Only process valid GET or POST attempts
#  - Reduce the time between downloads from 10 sec to 2 sec
#
#  Revision 1.8  2001/02/27 08:06:52  philk
#  - Add compatibility option: have_syslog
#  - Make all calls to syslog use full package spec
#  - Make sure every die ends in \n
#  - Accept configuration lines case insensitive
#  - Allow config file to be in same directory as binary
#  - First attempt at WinNT and Win95 compatibility
#  - Use local to set RS as not all O/S can do it per handle
#
#  Revision 1.7  2001/02/11 05:41:30  philk
#  - send data to current seti server, even if the client tells us the old one
#  - better handling of unknown return status values
#  - new return status (-34 = OBSOLETE_VERSION) handled
#  - accept write_pid = "no" to mean don't write the pid
#  - stop accepting the old Perl syntax in the config file
#  - when dumping config, do it sorted
#  - additional logging
#  - download new units before uploading existing results
#
#  Revision 1.6  2001/01/30 01:36:02  philk
#  - Check for failed return in gethostbyaddr
#  - Replaced pid_path configuration variable with write_pid
#  - Solaris can't do exclusive read locks, so change all of the reads to read/write.
#
#  Revision 1.5  2001/01/09 09:37:13  philk
#  - Squeuer now running as squeuer user.  squeuer startup script created.
#  - Dump the configuration to the log on startup
#  - Change the failure messages when uploading and downloading.
#  - Handle the case where Seti forces a client version upgrade.
#  - Added a log_level variable so that if not using syslog we can limit the debug messages.
#  - Added the option of printing out the pid so that root can write it as part of the startup.
#  - Turned on taint checking.
#  - Catch a TERM in main process and pass it onto the transfer_loop.
#  - If unit it is more than a week old, drop it, decrease the queue depth, and log a critical failure.
#  - Wake the transfer_loop on a HUP signal to the main process.
#  - Changed to new Seti server for v3.03 clients.
#
#  Revision 1.4  2000/11/25 11:38:14  philk
#  Put work unit stats into file into user's spool
#  Fixed but in handling error returns from a proxy server
#  Check for non-zero sah users_statistics status
#  Fixed several strict warnings
#  Much more strict checking of configuration file options
#  Added extra history documentation
#
#  Revision 1.3  2000/11/14 12:48:07  philk
#  when the queue is empty, stop retrying to serve a work unit after 30mins
#  check ppid in transfer loop, and exit if parent died
#  try to write pid file
#  sanity check fields used as paths from the request packet
#  changed config file to flat text
#  don't overwrite a result file if it already exists
#  added an allow_hosts option to limit connections
#  if a work unit received from the sah server is corrupt, keep trying
#
#  Revision 1.2  2000/10/18 14:36:26  philk
#  Added check for missing user directory.
#  Added alternate config file option.
#  Added new logging options.
#  Always return the oldest queued work unit.
#  Added a connection retry limit.
#
#  Revision 1.1  2000/10/18 11:54:15  philk
#  Initial revision
#
#

require 5.005;

#
# The ActiveState port of Perl to Win32 is lacking a number of standard
# functions that we use.  So, where necessary we fake them.
# This might also be true of other non-Unix-like operating systems.
#
BEGIN {
  package FakeIt;

  use Config;
  if ($Config{'osname'} eq "MSWin32") {
    if (Win32::IsWinNT()) {
      *main::getppid = sub { return 0; };
      *main::alarm = sub { return 0; };
    }
    elsif (Win32::IsWin95()) {
      *main::flock   = sub { return 1; };
      *main::getppid = sub { return 0; };
      *main::alarm = sub { return 0; };
    }
  }
}

#
# Note: all of these are part of the standard distribution so there is
# nothing to install - don't pollute it with stuff from CPAN!
#
use Config;
use Cwd;
use Fcntl ':flock';
use FileHandle;
use File::Spec;
use File::Basename;
use Socket qw(:DEFAULT :crlf);
use IO::Socket;
use Net::hostent;
use Getopt::Std;

# Check to see that we are NOT running as root
if ($Config{'osname'} ne "MSWin32" && $< == 0) {
  die "Don't run squeuer as root.\n";
}

%sah_status = ( 0    => 'SUCCESS',
		-34  => 'OBSOLETE_VERSION',
		-38  => 'NEW_VERSION',
                -66  => 'DUPLICATE_IGNORED',
                -999 => 'CONNECTION_FAILED');

# Parse command line options
getopts('f:') || die "Usage: squeuer [-f config_file]\n";
$config_file = ($main::opt_f) ? $main::opt_f : undef;

# Go to our home directory and get the configuration file
undef %auth;
undef %conf;
$home = &squeuer_config($config_file);

# Write the pid if requested
if ($conf{'write_pid'}) {
  print $$;
  STDOUT->flush();
}

# Start logging
if ($conf{'have_syslog'}) {
  require Sys::Syslog;
}
elsif ($conf{'log_method'} eq 'syslog') {
  die "We don't have syslog, but it has been set as the log_method.\n";
}

&open_log;
&write_log('debug', "-" x 40);
&write_log('debug', "Squeuer configuration:");
foreach $key (sort keys %conf) {
  &write_log('debug', "      $key\t: $conf{$key}");
}
&write_log('info', "-" x 40);
&write_log('debug', "squeuer home is $home");
&write_log('info', "starting squeuer $conf{squeuer_version} with pid $$");

local $SIG{CHLD} = IGNORE;
local $SIG{ALRM} = IGNORE;

if ($conf{'have_fork'}) {
  # fork the transfer loop
  if (!($transfer_pid = fork())) {
    &transfer_loop;
    exit(0);
  }
  &write_log('debug', "started transfer_loop with pid $transfer_pid");
  $SIG{TERM} = sub { kill(TERM, $transfer_pid); exit(0); };
  $SIG{HUP}  = sub { kill(ALRM, $transfer_pid); };
}
else {
  # run it once to start
  &transfer_loop;
}

# open the listener socket for the client to connect to
&write_log('debug', "opening local listener on port $conf{server_port}");
$queue = new IO::Socket::INET( Proto     => 'tcp',
			       LocalPort => $conf{'server_port'},
			       Listen    => SOMAXCONN,
			       Reuse     => 1)
  || die "Couldn't open socket on port $conf{server_port}\n";

# generate the allow hosts pattern match
foreach $ah (split(/\s*,\s*/, $conf{'allow_hosts'})) {
  &write_log('debug', "adding allow_host rule $ah");
  $ah =~ s/\./\\./g;   # change . into \.
  $ah =~ s/\*/.*/g;    # change * into .*
  $ah = "^$ah\$";      # anchor pattern absolutely
  push @allow_hosts, $ah;
}

# wait for a client to bang on our socket
$/ = $LF;
while ($seti = $queue->accept()) {
  $seti->autoflush(1);
  if ($hostinfo = gethostbyaddr($seti->peeraddr)) {
    $hostname = $hostinfo->name || $seti->peerhost;
  }
  else {
    $hostname = $seti->peerhost;
  }

  # see if this connection is allowed
  $allow = 0;
  foreach $ah (@allow_hosts) {
    $allow++  if ($hostinfo && $hostinfo->name =~ /$ah/)
      || ($seti->peerhost =~ /$ah/);
  }

  if ($allow) {
    # read the first line the client sends
    undef $cmd;
    eval {
      local $SIG{ALRM} = sub { die "alarm\n" };
      alarm(5);
      $cmd = <$seti>;
      alarm(0);
    };

    # check to see if this is a password set attempt
    if ($cmd =~ /^GET /s) {
      &write_log('debug', "received GET request");
      &set_proxy_password($seti, $cmd);
    }
    elsif ($cmd =~ /^POST /s) {
      # check for multi-threading capability
      if ($conf{'have_fork'}) {
	if (!($pid = fork())) {
	  &squeuer($seti, $cmd, $hostname);
	  close($seti);
	  exit(0);
	}
	else {
	  &write_log('info', "client connect from $hostname with pid $pid");
	}
      }
      else {
	&write_log('info', "client connect from $hostname");
	&squeuer($seti, $cmd, $hostname);
      }
    }
    else {
      &write_log('warning', "malformed connection attempt");
    }
  }
  else {
    &write_log('warning', "client connect from $hostname denied");
  }

  close($seti);

  if ($allow and !$conf{'have_fork'}) {
    &transfer_loop;
  }

}

close($queue);
&close_log;

exit(0);


#------------------------------------------------------------------------
# Transfer completed work units to the server, and fill the spools
#

sub transfer_loop {
  my (@users, $user_id);
  my (@files, @results, $result, $data, $units, $get, $name, $strike, $alive);
  my ($email, $status, $vermaj, $vermin, $version, $post, $nresults);

  local $/ = undef;
  local $SIG{ALRM} = IGNORE;

  batter: while (getppid() != 1) {
    &write_log('debug', "processing transfer loop");
    $strike = 0;

    opendir(USERS, "$conf{spool_path}");
    @users = grep {! /^\./ && -d "$conf{spool_path}/$_" } readdir(USERS);
    closedir(USERS);

    foreach $user_id (@users) {
      if ($user_id =~ /^([0-9]+)$/) {
	$user_id = $1;
      }
      else {
	&write_log('warning', "invalid user_id $user_id");
	next;
      }

      &write_log('debug', "checking user $user_id");
      opendir(USER, "$conf{spool_path}/$user_id");
      @files = readdir(USER);
      closedir(USER);

      $email = &get_email($user_id);

      #
      # download more units
      #
      $units = grep { /^unit-/ } @files;

      if (!open(GET, "+<$conf{spool_path}/$user_id/get_work_unit")) {
	&write_log('notice', "user $user_id must try to get a work unit before queue can be filled");
      }
      else {
        flock(GET, LOCK_EX);
        $get = <GET>;
        flock(GET, LOCK_UN);
        close(GET);

        for (; $units < $conf{'queue_depth'}; $units++) {
	  ($post, $data) = &send_data_to_seti($get, $email);
	  if ($post >= 400 || !$data) {
            if (++$strike == 3 || $post == 407) {
              next batter;
            }
            else {
              sleep($conf{'retry_delay'});
              redo;
            }
	  }

	  if ($data =~ /${LF}name=(.*?)${LF}/is) {
	    &write_log('info', "got work unit %d for $user_id", $units+1);
	    $name = $1;
	  }
	  else {
	    if ($data =~ /${LF}status=(-?\d+)${LF}/is) {
	      $status = $1;
	      &write_log('alert', "get failed for $user_id ["
		         . ($sah_status{$status} || $status) . "]");
	      if (!$sah_status{$status}
		  && $data =~ /${LF}message=(.*?)${LF}/is) {
	        &write_log('alert', "message is '$1'");
	      }

	      # see if seti is forcing a new version - don't get any more units
	      $vermaj = $1  if $data =~ /${LF}major_version=(\d+)${LF}/is;
	      $vermin = $1  if $data =~ /${LF}minor_version=(\d+)${LF}/is;
	      if ($status == -34) {
	        &write_log('alert', "upgrade all clients of %d to version "
			   . "%d.%02d immediately",
			   $user_id, $vermaj, $vermin);
	        last;
	      }
	      elsif ($status == -38) {
	        &write_log('alert', "upgrade all clients of %d to version "
			   . "%d when spool empties",
			   $user_id, $vermaj+1);
	        last;
	      }
	      next;
	    }
	    else {
	      &write_log('warning', "work unit corrupt - no name found");
	      next;
	    }
	  }

	  open(DATA, ">$conf{spool_path}/$user_id/unit-$name")
	    || die "Can't write work unit $conf{spool_path}/$user_id/unit-$name\n";
	  print DATA $data;
	  close(DATA);

	  sleep(2);
        }
      }

      #
      # upload each of the results
      #
      @results = grep { /^result-/ } @files;
      foreach $result (@results) {
	if ($result =~ /^(result-[a-z0-9.]+)$/) {
	  $result = $1;
	}
	else {
	  &write_log('warning', "invalid result name $result");
	  next;
	}

	open(RESULT, "+<$conf{spool_path}/$user_id/$result");
	flock(RESULT, LOCK_EX);
	$data = <RESULT>;
	flock(RESULT, LOCK_UN);
	close(RESULT);

	($post, $stats) = &send_data_to_seti($data, $email);
        $status = -999;
	$status = $1  if $stats =~ /${LF}status=(-?\d+)${LF}/is;

        if ($post >= 400 || !$stats || $status == -999) {
          if (++$strike == 3 || $post == 407) {
            next batter;
          }
          else {
            sleep($conf{'retry_delay'});
            redo;
          }
	}
        else {
	  $nresults = -1;
	  $nresults = $1  if $stats =~ /${LF}nresults=(\d+)${LF}/is;
        }

	unlink("$conf{spool_path}/$user_id/$result")
	  || die "Couldn't delete result $conf{spool_path}/$user_id/$result\n";

        if ($status == 0) {
	  &write_log('info', "sent result for $user_id (completed $nresults)");
          &save_stats($user_id, $stats);
        }
        else {
          &write_log('alert', "send $result failed for $user_id ["
                     . ($sah_status{$status} || $status) . "]");
        }

        sleep(2);
      }

    }
  }
  continue {
    # if we don't have fork, this was called procedurally
    if (!$conf{'have_fork'}) {
      return;
    }

    &write_log('debug', "sleeping for $conf{upload_interval} minutes");
    eval {
      local $SIG{ALRM} = sub { die "alarm\n" };
      sleep(60 * $conf{'upload_interval'});
    };
    if ($@) {
      die unless ($@ eq "alarm\n");
    }
  }

  &write_log('warning', "transfer_loop exiting");
}


#------------------------------------------------------------------------
# Send a captured form to the seti server
# return the response as a (status, text) duple
#

sub send_data_to_seti {
  my $data = shift;
  my $email = shift;
  my ($sah, $head, $body, $response, $status);

  local $/ = undef;

  $status = 504;  # Gateway Timeout
  if (!($sah = new IO::Socket::INET( Proto    => 'tcp',
				     PeerAddr => $conf{'proxy_server'},
				     PeerPort => $conf{'proxy_port'} ))) {
    return ($status, undef);
  }

  # Split into header and body
  ($head, $body) = split(/$CRLF$CRLF/, $data, 2);
  $head .= $CRLF;

  # Make sure we are sending this to the correct server
  $head =~ s/^(POST http:\/\/).*(\/ HTTP\/1.0\s*)$/$1$conf{'seti_server'}$2/m;
  $head =~ s/^(Host: ).*(\s*)$/$1$conf{'seti_server'}$2/m;

  # Add the auth header if we have it
  if (($conf{'proxy_auth'} eq 'basic') && $auth{$email}) {
    $head .= "Proxy-Authorization: Basic $auth{$email}$CRLF";
  }

  # Post the data and get the response
  $sah->autoflush(1);
  print $sah $head;
  print $sah $CRLF;
  print $sah $body;
  $response = <$sah>;
  close($sah);

  $status = $1  if $response =~ /^HTTP\/\d\.\d (\d\d\d) /s;
  if ($status == 407) {
    &write_log('alert', 'proxy authentication required');
  }

  return ($status, $response);
}


#------------------------------------------------------------------------
# Process a Seti@Home client request
#

sub squeuer {
  my $client = shift;
  my $post = shift;
  my $hostname = shift;
  my ($line, $header, $hdr, $stats, $unit, $user_id, $now, $status);

  local $/ = $LF;

  # get the rest of the post
  while (($line = <$client>) ne "$LF") {
    $post .= $line;
  }
  $post .= "$LF";


  $header = $2  if $post =~ /^.*?($CRLF){2}(.*?)${LF}end_request_header${LF}/s;
  $hdr = { map { my @h = split(/=/); $h[0] => $h[1] } split(/$LF/, $header) };

  # check to see if we've seen this user before
  if (!&check_user_dir($hdr->{'user_id'})) {
    return undef;
  }

  &write_log('info', "received %s request for %s",
	    $hdr->{'operation'}, $hdr->{'user_id'});

  $now = time();

  #
  # send_result_get_user_stats
  #
  if ($hdr->{'operation'} eq "send_result_get_user_stats") {
    return undef  if !$hdr->{'user_id'};
    if (! &save_result($hdr->{'user_id'}, $post, $hostname)) {
      return undef;
    }

    while (!($stats = &get_stats($hdr->{'user_id'}))) {
      # we haven't seen this user before, so we must push this through
      &write_log('notice', "no stats for $hdr->{user_id} forcing connect");
      if ($conf{'have_fork'}) {
	kill(ALRM, $transfer_pid);
	sleep(2 * $conf{'retry_delay'});
      }
      else {
	&transfer_loop;
      }

      return undef  if time() >= $now + 1200;
    }

    print $client $stats;
  }

  #
  # get_work_unit
  #
  elsif ($hdr->{'operation'} eq "get_work_unit") {
    return undef  if !$hdr->{'user_id'};
    &save_get($hdr->{'user_id'}, $post);

    while (!($unit = &get_work_unit($hdr->{'user_id'}))) {
      # no units left in the queue
      &write_log('notice', "no units in queue for $hdr->{user_id}");

      if ($conf{'have_fork'}) {
	kill(ALRM, $transfer_pid);
	sleep(2 * $conf{'retry_delay'});
      }
      else {
	&transfer_loop;
      }

      # don't try for more than 20 minutes as the client drops the connection
      if (time() >= $now + 1200) {
	&write_log('notice', "could not get unit for %d after %d minutes",
		   $hdr->{user_id}, int((time() - $now) / 60));
	return undef;
      }
    }

    print $client $unit;
  }

  #
  # lookup_user_name
  #
  elsif ($hdr->{'operation'} eq "lookup_user_name") {
    # find the e-mail address
    $hdr->{'email_addr'} = $1  if $post =~ /${LF}email_addr=(.*?)${LF}/s;

    $user_id = &find_user($hdr->{'email_addr'});
    if (defined $user_id) {
      $stats = &get_stats($user_id);
    }
    else {
      # we haven't seen this user before, so we must push this through
      ($status, $stats) = &send_data_to_seti($post, $hdr->{'email_addr'});
      if ($status < 400 && $stats =~ /${LF}id=(\d+)${LF}/is) {
	$user_id = $1;
      }
      else {
	# didn't look like a good return packet
	return undef;
      }
      &save_stats($user_id, $stats);
    }

    print $client $stats;
  }

  #
  # unknown operation
  #
  else {
    &write_log('alert', "unknown operation - please report it");
  }
}


#------------------------------------------------------------------------
# Fake a Basic Auth response so that we get to capture the password
#

sub set_proxy_password {
  my $client = shift;
  my $get = shift;
  my ($email, $header, $line);
  
  # the requested URL is the e-mail address
  if ($get =~ /^GET .*\/(.*?) HTTP/s) {
    $email = $1;
    if ($email !~ /^.+\@.+\..+$/) {
      &write_log('warning', "invalid e-mail address");
      return undef;
    }
  }
  else {
    &write_log('warning', "malformed GET request");
    return undef;
  }

  # get the rest of the headers
  $header = "";
  while (($line = <$client>) ne "$CRLF") {
    $header .= $line;
  }

  # see if this is an authenticated connection
  if ($header =~ /^Authorization: Basic (.*)$CRLF/m) {
    $auth{$email} = $1;

    print $client "HTTP/1.0 200 OK$CRLF";
    print $client "Content-Type: text/html$CRLF";
    print $client "Pragma: no-cache$CRLF";
    print $client "$CRLF";
    print $client "<HTML><BODY>\n";
    print $client "<P>Squeuer has set your proxy password.</P>\n";
    print $client "</BODY></HTML>\n";

    &write_log('info', "proxy password set for $email");

    return 1;
  }
  else {
    print $client "HTTP/1.0 401 Unauthorized$CRLF";
    print $client "WWW-Authenticate: Basic realm=\"SAH $email\"$CRLF";
    print $client "$CRLF";

    return 0;
  }

  return undef;
}


#------------------------------------------------------------------------
# Check a user directory exists
#
sub check_user_dir {
  my $user_id = shift;

  if ($user_id =~ /^(\d+)$/) {
    $user_id = $1;
  }
  else {
    &write_log('warning', "invalid user_id: $user_id");
    return undef;
  }

  $user_dir = "$conf{spool_path}/$user_id";
  if ( ! -d "$user_dir" && $user_id != 0 ) {
    &write_log('info', "creating new user $user_id");
    mkdir($user_dir, 0777) 
      || die "Couldn't create user directory: $user_dir\n";
  }

  return $user_dir;
}


#------------------------------------------------------------------------
# Save a work unit into the done queue
#

sub save_result {
  my $user_id = shift;
  my $result = shift;
  my $hostname = shift;
  my ($name);

  if ($result =~ /${LF}name=(.*?)${LF}/is) {
    $name = $1;
    if ($name !~ /^[a-z0-9.]+$/) {
      &write_log('warning', "result name $name is invalid");
      return undef;
    }
  }
  else {
    return undef;
  }

  if (-e "$conf{spool_path}/$user_id/result-$name") {
    &write_log('warning', "result $name already exists");
    return -1;
  }

  open(RESULT, ">$conf{spool_path}/$user_id/result-$name")
    || die "Can't write result file $conf{spool_path}/$user_id/result-$name\n";
  print RESULT $result;
  close(RESULT);

  &update_client_log($user_id, $result, $hostname);

  return 1;
}


#------------------------------------------------------------------------
# Update the client log for the processed work unit
#

sub update_client_log {
  my $user_id = shift;
  my $result = shift;
  my $hostname = shift;
  my ($cpu_secs, $cpu_time, $name);

  # extract the time taken
  if ($result =~ /${LF}cpu_time=(.*?)${LF}/is) {
    $cpu_secs = $1;
    &write_log('info', "work unit for $hostname took $cpu_secs");
    if ($cpu_secs !~ /^[0-9.]+$/) {
      &write_log('warning', "cpu_time $cpu_secs is invalid");
      return undef;
    }
    $cpu_time = sprintf("%3d:%02d:%02d", int($cpu_secs / 3600),
			int(($cpu_secs / 60) % 60), int($cpu_secs % 60));
  }
  else {
    &write_log('warning', "no cpu_time found in result");
    return undef;
  }

  # extract the result name
  if ($result =~ /${LF}name=(.*?)${LF}/is) {
    $name = $1;
  }
  else {
    &write_log('warning', "no name found in result");
    return undef;
  }

  open(CLIENT, ">>$conf{spool_path}/$user_id/client_log") || return undef;
  flock(CLIENT, LOCK_EX);
  print CLIENT localtime() . "\t" . $cpu_time
      . "\t" . $name . "\t" . $hostname . "\n";
  flock(CLIENT, LOCK_UN);
  close(CLIENT);
}


#------------------------------------------------------------------------
# Save the current user statistics
#

sub save_stats {
  my $user_id = shift;
  my $stats = shift;

  if (!&check_user_dir($user_id)) {
    return undef;
  }

  open(STATS, ">$conf{spool_path}/$user_id/current_stats")
    || die "Can't write stats file $conf{spool_path}/$user_id/current_stats\n";
  flock(STATS, LOCK_EX);
  print STATS $stats;
  flock(STATS, LOCK_UN);
  close(STATS);
}


#------------------------------------------------------------------------
# Save a get_work_unit request for replaying
#

sub save_get {
  my $user_id = shift;
  my $request = shift;

  open(GET, ">$conf{spool_path}/$user_id/get_work_unit")
    || die "Can't write get_work_unit $conf{spool_path}/$user_id/get_work_unit\n";
  flock(GET, LOCK_EX);
  print GET $request;
  flock(GET, LOCK_UN);
  close(GET);
}


#------------------------------------------------------------------------
# Get the current statistics for the user
#

sub get_stats {
  my $user_id = shift;
  my ($stats);

  local $/ = undef;

  open(STATS, "+<$conf{spool_path}/$user_id/current_stats") || return undef;
  flock(STATS, LOCK_EX);
  $stats = <STATS>;
  flock(STATS, LOCK_UN);
  close(STATS);

  return $stats;
}


#------------------------------------------------------------------------
# Get a work unit from the spool
#

sub get_work_unit {
  my $user_id = shift;
  my (@units, $unit, $data, $got);
  my $user_dir = "$conf{spool_path}/$user_id";

  local $/ = undef;

  opendir(USER, "$user_dir");
  @units = sort { (stat("$user_dir/$a"))[10] <=> (stat("$user_dir/$b"))[10] }
             (grep { /^unit-/ } readdir(USER));
  closedir(USER);

  $got = 0;
  do {
    $unit = shift @units;
    return undef  if !$unit;

    if ($unit =~ /^(unit-[0-9a-z.]+)$/) {
      $unit = $1;
    }
    else {
      &write_log('warning', "invalid work unit name $unit");
      next;
    }

    open(UNIT, "+<$user_dir/$unit");
    $got = flock(UNIT, (LOCK_EX|LOCK_NB));
    close(UNIT)  if !$got;

    # check the age of the work unit
    if (time() - (stat("$user_dir/$unit"))[10] > 86400*7) {
      &write_log('notice', "work unit $unit is too old, removed");
      flock(UNIT, LOCK_UN);
      close(UNIT);
      unlink("$user_dir/$unit")
	|| die "Couldn't delete work unit $user_dir/$unit\n";
      $conf{'queue_depth'}--  if $conf{'queue_depth'} > 1;
      &write_log('alert', "queue is too deep, setting depth to %d",
		 $conf{'queue_depth'});
      $got = 0;
    }
  } until $got;

  $data = <UNIT>;
  flock(UNIT, LOCK_UN);
  close(UNIT);
  unlink("$user_dir/$unit")
    || die "Couldn't delete work unit $user_dir/$unit\n";

  return $data;
}


#------------------------------------------------------------------------
# Find the current user
#

sub find_user {
  my $email = shift;
  my (@users, $user_id, $stats);

  opendir(USERS, $conf{'spool_path'});
  @users = grep {! /^\./ && -d "$conf{spool_path}/$_" } readdir(USERS);
  closedir(USERS);

  foreach $user_id (@users) {
    $stats = &get_stats($user_id);
    if ($stats =~ /${LF}id=(\d+)${LF}.*${LF}email_addr=$email${LF}/is) {
      return $1;
    }
  }

  return undef;
}


#------------------------------------------------------------------------
# Get the e-mail address of the user_id
#

sub get_email {
  my $user_id = shift;
  my ($stats);

  $stats = &get_stats($user_id);
  if ($stats =~ /${LF}email_addr=(.*?)${LF}/s) {
    return $1;
  }

  return undef;
}


#------------------------------------------------------------------------
# Locate our home directory and source the conf file
#

sub squeuer_config {
  my $config_file = shift;
  my ($home, @dir, $opt, $val, $err);
  local $/ = $LF;

  $home = dirname($0);
  if (!File::Spec->file_name_is_absolute($home)) {
    $home = File::Spec->catdir(getcwd, $dir);
  }

  if ($home =~ /(.+)/) {
    chdir($1);
    $home = getcwd;
  }
  else {
    print STDERR "Couldn't find squeuer home directory.\n";
    exit(1);
  }

  # hard coded config defaults - the config file can override them
  $conf{'server_port'} = 17771;
  $conf{'spool_path'} = "/var/spool/squeuer";
  $conf{'write_pid'} = 0;
  $conf{'upload_interval'} = 60;
  $conf{'retry_delay'} = 300;
  $conf{'queue_depth'} = 5;
  $conf{'log_method'} = 'stderr';
  $conf{'log_level'} = 'info,notice,warning,alert';
  $conf{'proxy_server'} = '';
  $conf{'proxy_port'} = 80;
  $conf{'proxy_auth'} = '';
  $conf{'allow_hosts'} = '*';
  $conf{'have_syslog'} = 'yes';
  $conf{'have_fork'} = 'yes';

  # load the config file
  if (!$config_file) {
    # default is in current directory
    $config_file = "squeuer.conf";
    if (! -f $config_file) {
      # maybe we are in bin and config is in etc
      $config_file = File::Spec->catfile(File::Spec->updir,
					 "etc", "squeuer.conf");
      if (! -f $config_file) {
	# can't find it anywhere
	$config_file = "";
      }
    }
  }

  if (-r $config_file) {
    open(CONF, $config_file) 
        || die "Can't open configuration file ${config_file}\n";
    $err = 0;
    while(<CONF>) {
      chomp;
      s/#.*$//;          # remove comments
      next if /^\s*$/;   # remove blank lines
      if (/^\s*(\w+)\s*=\s*[\"\']?(.*?)[\"\']?\s*$/) {
	($opt, $val) = ($1, $2);
	$opt =~ tr/A-Z/a-z/;	# lowercase the option name
        if (!exists $conf{$opt}) {
          # Our log file is not open yet
          print STDERR "Unknown squeuer configuration option: $opt on line $.\n";
	  $err++;
        } 
        $conf{$opt} = $val;
      }
      else {
	print STDERR "Bad squeuer configuration line $.: $_\n";
	$err++;
      }
    }
    close(CONF);

    if ($err) {
      die "*** Squeuer terminated due to errors in the configuration file.\n";
    }
  }
  else {
    print STDERR "Squeuer config file unreadable, using defaults.\n"; 
  }

  # Check for 'no' as a config value
  foreach $opt (keys %conf) {
    $conf{$opt} = 0  if $conf{$opt} eq 'no';
  }

  # Now some config options that can't be overridden
  $conf{'seti_server'} = "shserver2.ssl.berkeley.edu";
  $conf{'squeuer_version'} = $1 if '$Revision: 1.3 $' =~ /:\s*([0-9.]+)\s*/; #'

  # Make sure the proxy is set correctly
  $conf{'proxy_server'} = $conf{'seti_server'}  if !$conf{'proxy_server'};

  # see if the spool path is available
  $conf{'spool_path'} = $1  if $conf{'spool_path'} =~ /(.+)/;
  if (! -d $conf{'spool_path'}) {
    mkdir($conf{'spool_path'}, 0777)
      || die "Can't create spool path $conf{spool_path}\n";
  }
  if (! -w $conf{'spool_path'}) {
    chmod(0755, $conf{'spool_path'})
      || die "Can't make spool path $conf{spool_path} writable\n";
  }

  return $home;
}


#------------------------------------------------------------------------
# Open the log
#

sub open_log {
  if ($conf{'log_method'} eq 'stderr') {
    $log_stream = STDERR;
  }
  elsif ($conf{'log_method'} eq 'syslog') {
    Sys::Syslog::openlog('squeuer', 'cons,pid', 'user');
  }
  elsif ($conf{'log_method'} eq 'none') {
    undef $log_stream;
  }
  else {
    $log_stream = $conf{'log_method'};
    if (! open($log_stream, ">>$conf{log_method}")) {
      printf STDERR "Can't open log file $conf{log_method}, "
	          . "reverting to STDERR\n";
      $log_stream = STDERR;
    }
  }

  $log_stream->autoflush(1)  if defined $log_stream;
}


#------------------------------------------------------------------------
# Write log data
#

sub write_log {
  my $pri = shift;
  my $text = shift;
  my @args = @_;

  if ($conf{'log_method'} eq 'syslog') {
    Sys::Syslog::syslog($pri, $text, @args);
  }
  elsif (defined $log_stream) {
    if (index($conf{'log_level'}, $pri) >= 0) {
      printf $log_stream (localtime() . " [$pri] $text\n", @args);
    }
  }
}


#------------------------------------------------------------------------
# Close the log
#

sub close_log {
  if ($conf{'log_method'} eq 'syslog') {
    Sys::Syslog::closelog;
  }
  elsif (defined $log_stream) {
    close($log_stream);
  }
}
