/*
 * Program: Synonym
 * File: config.h
 * Author: Ionut Nistor
 * Date: 20 Feb 2003
 *
 * $Id: config.h,v 1.5 2004/01/19 09:24:31 diciu Exp $
 *
 * Licensed under the Modulo Consulting Software License
 * (see file license.txt)
 * 
 */

#ifndef _CONFIG_INCLUDED
#define _CONFIG_INCLUDED

#include <sys/types.h>
#include <regex.h>

#include "synonym.h"



/* Names for the tags in the config file */
/* These are case insensitive */

#define TAG_RULES "Rules"
#define TAG_RULE "Rule"
#define TAG_CONDITION "Condition"
#define TAG_ACTION "Action"
#define TAG_HEADER "Header"
#define TAG_FROM "EnvFrom"
#define TAG_TO "EnvTo"
#define TAG_MATCH "Match"
#define TAG_ACTIONTYPE "ActionType"
#define TAG_ACTIONADDRESS "Address"
#define TAG_HTML_DISCLAIMER "HtmlDisclaimer"
#define TAG_TEXT_DISCLAIMER "TextDisclaimer"

/* Some definitions for the matching criteria */
#define MAX_FIELD_SIZE 255
#define MAX_REGEXP_SIZE 255

/* Some definitions for the actions */
#define MAX_ACTION_CUSTOMFIELD_SIZE 255
#define MAX_DISCLAIMER_SIZE 2048


/* For some specific custom fields in actions */
#define MAX_SMTP_REPLY_SIZE 64

/* Synonym criteria - chain of criteria that synonym will use in order decide which actions to perform */
typedef struct _tag_synonym_condition {
	char header_field[MAX_FIELD_SIZE+1]; /* The header whose value will be matched against the regexp */ 
	char header_regexp[MAX_REGEXP_SIZE+1]; /* Value to be matched against the header value */
	regex_t header_compiled_regexp;
	char env_from_condition[MAX_REGEXP_SIZE+1];
	regex_t env_from_compiled_regexp;
	char env_to_condition[MAX_REGEXP_SIZE+1];
	regex_t env_to_compiled_regexp;
	
	struct _tag_synonym_condition *next;
} synonym_condition;



/* Specific actions for synonym. The first one defines the 'void' or uninitialized action */
#define ACTION_NONE 0
#define ACTION_COPY 1
#define ACTION_DELETE 2
#define ACTION_REJECT 3
#define ACTION_DISCLAIMER 4

/* Action names - for use in the configuration file */
#define ACTIONNAME_COPY "Copy"
#define ACTIONNAME_DELETE "Delete"
#define ACTIONNAME_REJECT "Reject"
#define ACTIONNAME_DISCLAIMER "Disclaimer"

/* Action structure */
typedef struct {
	int action_type; /* Defined from a fixed set of actions (integer) - see above */
	char action_custom_field[MAX_ACTION_CUSTOMFIELD_SIZE+1]; /* Depending on the action, this field may be used (i.e. for the 'copy' action, this field will contain the email address) */
	char action_text_disclaimer[MAX_DISCLAIMER_SIZE+1];
	char action_html_disclaimer[MAX_DISCLAIMER_SIZE+1];
} synonym_action;

/* Used for processing of the rules - a chain of matched_actions structures will be yielded by the parse function */
typedef struct _tag_matched_action {
	synonym_action action;
	struct _tag_matched_action *next;
} matched_action;

/* Synonym rule definition - structure will be linked-type */
typedef struct _tag_synonym_rule {
	synonym_condition *conditions;
	synonym_action action;
	struct _tag_synonym_rule *next;
} synonym_rule;


/* Synonym configuration parameters (will be read from a file)*/
/* !! All processes accessing this structure should lock it (mutex) otherwise serious problems may appear while reloading the configuration !! */
typedef struct {
	synonym_rule *rules;
} _type_synonym_config;


/* Get the parameters from the configuration file
 * Input parms:
 * - name of the configuration file
 * - structure containing the config data
 */
sresult Synonym_Get_Config(char *configfile, _type_synonym_config *config);


/* Free the configuration structure (rules and conditions, which are allocated dynamically) */
void Synonym_Free_Config(_type_synonym_config *config);

/* Reload the Synonym configuration file - probably called from a signal handler */
sresult Synonym_Reload_Config(char *configfile, _type_synonym_config *config);

#endif
