/* 

    Functions to play sound on a SGI's audio device

    Copyright 1997 Lawrence T. Hoff

*/

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/ioctl.h> 

#include <dmedia/audio.h>

#include "config.h"
#include "output.h"
#include "controls.h"

static int open_output(void); /* 0=success, 1=warning, -1=fatal error */
static void close_output(void);
static void output_data(int32 *buf, int32 count);
static void flush_output(void);
static void purge_output(void);

/* export the playback mode */

#define dpm sgi_play_mode

PlayMode dpm = {
  DEFAULT_RATE, PE_16BIT|PE_SIGNED,
  -1,
  {0,0,0,0,0}, /* no extra parameters so far */
  "Sgi audio device", 'd',
  "",
  open_output,
  close_output,
  output_data,
  flush_output,
  purge_output  
};

/*************************************************************************/
/*
   Encoding will be 16-bit linear signed, unless PE_ULAW is set, in
   which case it'll be 8-bit uLaw. I don't think it's worthwhile to
   implement any 8-bit linear modes as the sound quality is
   unrewarding. PE_MONO is honored.  */

static ALport		oport;
static ALconfig		theconfig;

static int open_output(void)
{

long	    pvbuf[2];

if (dpm.encoding & PE_ULAW)
  dpm.encoding &= ~PE_16BIT;

if (!(dpm.encoding & PE_16BIT))
  dpm.encoding &= ~PE_BYTESWAP;

/* disable Audio LIbrary error messages */
ALseterrorhandler(0);

pvbuf[0] = AL_OUTPUT_RATE;
pvbuf[1] = dpm.rate;
ALsetparams(AL_DEFAULT_DEVICE, pvbuf, 2);

theconfig = ALnewconfig();
ALsetwidth(theconfig, (dpm.encoding & PE_16BIT)?AL_SAMPLE_16:AL_SAMPLE_8);
ALsetsampfmt(theconfig,AL_SAMPFMT_TWOSCOMP); /* no ulaw support yet */
ALsetchannels(theconfig, (dpm.encoding & PE_MONO)?1:2);
ALsetqueuesize(theconfig, 100000); /* some reasonable buffer size */

oport = ALopenport("the output port", "w", theconfig);
if (!oport)
    {
      ctl->cmsg(CMSG_ERROR, VERB_NORMAL, "%s: %s",
	   dpm.name, sys_errlist[errno]);
      return -1;
    }
return 0;
}

static void output_data(int32 *buf, int32 count)
{
  if (!(dpm.encoding & PE_MONO)) count*=2; /* Stereo samples */
  
  if (dpm.encoding & PE_16BIT)
    {
      if (dpm.encoding & PE_BYTESWAP)
	{
	  if (dpm.encoding & PE_SIGNED)
	    s32tos16x(buf, count);
	  else
	    s32tou16x(buf, count);
	}
      else
	{
	  if (dpm.encoding & PE_SIGNED)
	    s32tos16(buf, count);
	  else 
	    s32tou16(buf, count);
	}

      /* write the data out */
      ALwritesamps(oport, buf, count);

    }
  else
    {
      if (dpm.encoding & PE_ULAW)
	{
	  s32toulaw(buf, count);
	}
      else
	{
	  if (dpm.encoding & PE_SIGNED)
	    s32tos8(buf, count);
	  else 
	    s32tou8(buf, count);
	}
      
      /* write the data out */
      ALwritesamps(oport, buf, count);
    }
}
#include <stdio.h>
static void close_output(void)
{

  if(!oport)return;

  /* drain the port */
  while(ALgetfilled(oport));

  /* free resources */
  ALcloseport(oport);
  ALfreeconfig(theconfig);

  oport = NULL;
}

static void flush_output(void)
{
}

static void purge_output(void)
{
ALcloseport(oport);
oport = ALopenport("the output port", "w", theconfig);
}
