/* editMode.cc
   Enables the editing of a given map.

   Copyright (C) 2000  Mathias Broxvall
   		       Yannick Perret

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "general.h"
#include "gameMode.h"
#include "editMode.h"
#include "glHelp.h"
#include "map.h"
#include "settings.h"
#include <sys/stat.h>
#include <sys/types.h>

using namespace std;

#define MENU_SAVE      0
#define MENU_HEIGHT    1
#define MENU_COLOR     2
#define MENU_FLAGS     3
#define MENU_HILLS     4
#define MENU_TRANSLATE 5
#define MENU_VELOCITY  6
#define MENU_TEXTURE   7
#define MENU_WATER     8
#define MENU_SIZE      9


#define FLAG_ICE     0
#define FLAG_ACID    1
#define FLAG_SAND    2
#define FLAG_KILL    3
#define FLAG_TRAMPOLINE 4
#define FLAG_NOGRID  5
#define FLAG_TRACK   6
#define NUM_FLAGS    7

char *menuNames[MENU_SIZE] = {"Save map","Edit height","Set color","Flags","Add feature","Move map","Track velocity","Textures","Edit water height"};
char *flagNames[NUM_FLAGS] = {"Ice","Acid","Sand","Kill","Trampoline","No grid","Track"};

#define HILL_SPIKE 0
#define HILL_SMALL 1
#define HILL_MEDIUM 2
#define HILL_LARGE 3
#define HILL_HUGE 4
#define SMOOTH_SMALL 5
#define SMOOTH_LARGE 6
#define N_HILLS 7
char *hillNames[N_HILLS] = {"Spike","Small hill","Medium hill","Large hill","Huge hill","Small smooth","Large smooth"};


int switchViewpoint=0,birdsEye=0;
int markX=-1,markY;
int doAskQuit=0;

EditMode::EditMode() {
  char mapname[256];
  useHome=0;

  snprintf(mapname,sizeof(mapname)-1,"%s/.trackballs/levels/%s.map",getenv("HOME"),Settings::settings->specialLevel);

  if(fileExists(mapname)) useHome=1;
  else snprintf(mapname,sizeof(mapname)-1,"%s/levels/%s.map",SHARE_DIR,Settings::settings->specialLevel);

  map = new Map(mapname);
  map->flags |= Map::flagFlashCenter;
  map->isTransparent=1;
  x=(int) map->startPosition[0];
  y=(int) map->startPosition[1];
  raise=0.5;
  this->name = Settings::settings->specialLevel;
  color[0] = color[1] = color[2] = 0.9; color[3] = 1.0;
  menuChoise=MENU_HEIGHT;
  doSave=0;
  
  // test that we can save the map properly
  if(!useHome) {
	FILE *fp = fopen(mapname,"ab");
	if(!fp) { 
	  useHome=1; 
	  snprintf(mapname,sizeof(mapname)-1,"%s/.trackballs/levels/%s.map",getenv("HOME"),Settings::settings->specialLevel); 
	}
	else fclose(fp);
  }

  printf("Editing map '%s'\n",mapname);
}

EditMode::~EditMode() {
  /*  char mapname[256];
  sprintf(mapname,"%s.map",name);
  printf("saving map as '%s'\n",mapname);
  map->save(mapname,x,y);
  delete map;*/
}
void EditMode::display() {
  int i;

  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(40,640.0/480.0,1.0,1e20);

  /* Setup openGL matrixes for the camera perspective */
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  Cell& c=map->cell(x,y);
  Real h=c.heights[Cell::CENTER];
  if(!switchViewpoint) 
	gluLookAt(x-7.0,y-7.0, (birdsEye?20.0:10.0) + h * 0.5,
			  x,y,h,
			  0.0, 0.0, 1.0);
  else
	gluLookAt(x+7.0,y+7.0, (birdsEye?20.0:10.0) + h * 0.5,
			  x,y,h,
			  0.0, 0.0, 1.0);


  /* Some standard GL settings needed */
  glEnable(GL_CULL_FACE);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);

  GLfloat lightDiffuse[] = { 0.9, 0.9, 0.9, 0 };
  GLfloat lightPosition[] = { -100.0, -50.0, 150.0, 0.0 };
  glLightfv(GL_LIGHT0, GL_DIFFUSE, lightDiffuse);
  glLightfv(GL_LIGHT0, GL_POSITION, lightPosition);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);

  glShadeModel(GL_SMOOTH);

  int x0,y0;
  map->draw(switchViewpoint|birdsEye,0,x,y);
  map->draw(switchViewpoint|birdsEye,1,x,y);
  if(markX >= 0) {
	for(x0=min(markX,x);x0<=max(markX,x);x0++)
	  for(y0=min(markY,y);y0<=max(markY,y);y0++)
		map->drawFootprint(x0,y0);
  }

  if(menuChoise == MENU_HILLS)
	switch(hill) {
	case HILL_SPIKE: 
	  map->drawFootprint(x,y);
	  map->drawFootprint(x-1,y);
	  map->drawFootprint(x,y-1);
	  map->drawFootprint(x-1,y-1);
	  break;
	case HILL_SMALL: for(x0=x-1;x0<=x+1;x0++) for(y0=y-1;y0<=y+1;y0++) map->drawFootprint(x0,y0); break;
	case HILL_MEDIUM: for(x0=x-2;x0<=x+2;x0++) for(y0=y-2;y0<=y+2;y0++) map->drawFootprint(x0,y0); break;
	case HILL_LARGE: for(x0=x-3;x0<=x+3;x0++) for(y0=y-3;y0<=y+3;y0++) map->drawFootprint(x0,y0); break;
	case HILL_HUGE: for(x0=x-5;x0<=x+5;x0++) for(y0=y-5;y0<=y+5;y0++) map->drawFootprint(x0,y0); break;
	case SMOOTH_SMALL: for(x0=x-2;x0<=x+2;x0++) for(y0=y-2;y0<=y+2;y0++) map->drawFootprint(x0,y0); break;
	case SMOOTH_LARGE: for(x0=x-4;x0<=x+4;x0++) for(y0=y-4;y0<=y+4;y0++) map->drawFootprint(x0,y0); break;
	}
  else
	map->drawFootprint(x,y);

  displayFrameRate();
  char str[256];
  Enter2DMode();
  glColor4f(0.0,0.0,0.0,0.5);
  glBegin(GL_POLYGON); 
  glVertex2i(0,0); glVertex2i(screenWidth,0); glVertex2i(screenWidth,160); glVertex2i(0,160);
  glEnd();
  glLineWidth(1.0);
  glColor4f(0.8,0.8,0.8,0.8);
  glBegin(GL_LINES); glVertex2i(0,161); glVertex2i(screenWidth,161); glEnd();

  sprintf(str,"Raise: %f",raise);
  glColor3f(1.0,1.0,1.0);
  draw2DString(infoFont,str,10,20+0*18,255,255,0);
  sprintf(str,"Pos: %d,%d",x,y);
  draw2DString(infoFont,str,10,20+1*18,255,255,0);
  if(menuChoise == MENU_HILLS) {
	sprintf(str,"Feature: %s",hillNames[hill]);
	draw2DString(infoFont,str,10,20+2*18,255,255,0);
  } else {
	sprintf(str,"Color %1.1f %1.1f %1.1f %1.1f",color[0],color[1],color[2],color[3]);
	draw2DString(infoFont,str,10,20+2*18,255,255,0);
	glColor4f(color[0],color[1],color[2],color[3]);
	glBegin(GL_POLYGON); 
	glVertex2i(180,18+1*18); glVertex2i(200,18+1*18); glVertex2i(200,14+2*18); glVertex2i(180,14+2*18);
	glEnd();
  }
  glColor3f(1.0,1.0,1.0);

  for(i=0;i<MENU_SIZE;i++) {
    if(menuChoise == i) draw2DString(infoFont,menuNames[i],220,20+i*18,128,255,128);
    else draw2DString(infoFont,menuNames[i],220,20+i*18,64,192,64);
  }

  for(i=0;i<NUM_FLAGS;i++) {
    sprintf(str,"%d. %s: %s",i+1,flagNames[i],map->cell(x,y).flags & (1<<i) ? "yes" : "no");
    draw2DString(infoFont,str,350,20+i*18,128,255,128);
  }

  {
	Cell& c=map->cell(x,y);
	if(menuChoise == MENU_WATER) {
	  draw2DString(infoFont,"Water height",screenWidth-280,40,255,255,0);
	  sprintf(str,"%2.1f",c.waterHeights[3]);
	  draw2DString(infoFont,str,screenWidth-70,20,255,255,0);
	  sprintf(str,"%2.1f",c.waterHeights[0]);
	  draw2DString(infoFont,str,screenWidth-70,60,255,255,0);
	  sprintf(str,"%2.1f",c.waterHeights[2]);
	  draw2DString(infoFont,str,screenWidth-30,40,255,255,0);
	  sprintf(str,"%2.1f",c.waterHeights[1]);
	  draw2DString(infoFont,str,screenWidth-110,40,255,255,0);
	  sprintf(str,"%2.1f",c.waterHeights[Cell::CENTER]);
	  draw2DString(infoFont,str,screenWidth-70,40,255,255,0);
	} else if(menuChoise == MENU_VELOCITY) {
	  sprintf(str,"Velocity: %2.1f %2.1f",c.velocity[0],c.velocity[1]);
	  draw2DString(infoFont,str,screenWidth-260,40,255,255,0);
	} else {
	  draw2DString(infoFont,"Height",screenWidth-200,40,255,255,0);
	  sprintf(str,"%2.1f",c.heights[3]);
	  draw2DString(infoFont,str,screenWidth-70,20,255,255,0);
	  sprintf(str,"%2.1f",c.heights[0]);
	  draw2DString(infoFont,str,screenWidth-70,60,255,255,0);
	  sprintf(str,"%2.1f",c.heights[2]);
	  draw2DString(infoFont,str,screenWidth-30,40,255,255,0);
	  sprintf(str,"%2.1f",c.heights[1]);
	  draw2DString(infoFont,str,screenWidth-110,40,255,255,0);
	  sprintf(str,"%2.1f",c.heights[Cell::CENTER]);
	  draw2DString(infoFont,str,screenWidth-70,40,255,255,0);
	}
  }

  if(doAskQuit) {
	message("Quit without saving?","Yes/No");
  } else if(doSave) 
	message("Save map? (y/n)","Saving...");
  else if(menuChoise == MENU_SAVE) 
	message("Save map? (y/n)"," ");

  Leave2DMode();
}
void EditMode::key(int key) {  
  Cell& c = map->cell(x,y);
  int i,j;
  int shift = SDL_GetModState() & (KMOD_LSHIFT | KMOD_RSHIFT);
  int ctrl = SDL_GetModState() & (KMOD_LCTRL | KMOD_RCTRL);
  int x1,y1;
  int xLow=x,xHigh=x,yLow=y,yHigh=y;
  int set_flag;
  int corner;
  int north,east;

  if(key >= SDLK_KP0 && key <= SDLK_KP9)
	key = '0' + key - SDLK_KP0;

  if(doAskQuit) {
	if(key == 'y') exit(0);
	else if(key == 'n') doAskQuit=0;
	return;
  }
  if(menuChoise == MENU_SAVE) {
	if(key == 'y') { doSave=2; menuChoise = MENU_HEIGHT; }
	else if(key == 'n') menuChoise = MENU_HEIGHT;
	return;
  }

  /* For functions which manipulate a large region */
  if(markX >= 0) {
	xLow = min(x,markX); xHigh=max(x,markX);
	yLow = min(y,markY); yHigh=max(y,markY);
  } else {
	xLow = xHigh = x;
	yLow = yHigh = y;
  }

  if(menuChoise == MENU_FLAGS && ( key >= '1' && key <= '9' ) ) {
	i=key - '1';
	set_flag = (map->cell(x,y).flags & (1<<i))?0:1;

	for(x1=xLow;x1<=xHigh;x1++)
	  for(y1=yLow;y1<=yHigh;y1++) {
		Cell& c2=map->cell(x1,y1);
		c2.flags = (c2.flags & ~(1<<i)) | (set_flag?(1<<i):0);
	  }
	return;
  }

  if(menuChoise == MENU_HILLS && key >= '1' && key < '1'+N_HILLS) {
	hill=key - '1';
  }

  if(key >= SDLK_F1 && key < SDLK_F1+MENU_SIZE) {
	menuChoise = key - SDLK_F1;
	hill = 0;
	return;
  }

  /*
  // YP: to handle texture specification for cells
  if (menuChoise == MENU_TEXTURE)   {
      if (ctrl &&( (key == SDLK_UP)||(key == SDLK_RIGHT) ))   {
          c.texture++;
	  if (c.texture >= numTextures)
	    c.texture = -1;
          return;
      }
      if (ctrl &&((key == SDLK_DOWN)||(key == SDLK_LEFT) ))   {
          c.texture--;
	  if (c.texture < -1)
	    c.texture = numTextures-1;
          return;
      }
	  }*/


  /* These keys preloads rotated and scaled textures */
  if(menuChoise == MENU_TEXTURE && key >= '1' && key <= '8') {
	for(x1=xLow;x1<=xHigh;x1++)
	  for(y1=yLow;y1<=yHigh;y1++) {
		Cell& c2=map->cell(x1,y1);
		for(north=0;north<2;north++)
		  for(east=0;east<2;east++) {
			double angle=(2.0*M_PI * (key-'1'))/8.0;
			c2.textureCoords[Cell::NORTH*north+Cell::EAST*east][0] = 
			  ((x1+east)*cos(angle) + (y1+north)*sin(angle))*raise/4.0 * (shift?-1.:1.);
			c2.textureCoords[Cell::NORTH*north+Cell::EAST*east][1] = 
			  (- (x1+east)*sin(angle) + (y1+north)*cos(angle))*raise/4.0;
		  }
	  }	
  }

  int mx,my;
  switch(key) {
  case 'r': 
	if(shift) { markX = x; markY = y; }
	else markX = -1; 
	break;
  case 'v': switchViewpoint = switchViewpoint ? 0 : 1; break;
  case 'w': case SDLK_UP: 
	if(menuChoise == MENU_TRANSLATE) {
	  if(!switchViewpoint)
		for(mx=map->width-1;mx>0;mx--)
		  for(my=0;my<map->height;my++)
			map->cell(mx,my) = map->cell(mx-1,my);
	  else
		for(mx=0;mx<map->width-1;mx++)
		  for(my=0;my<map->height;my++)
			map->cell(mx,my) = map->cell(mx+1,my);
	} else  
	  if (shift && ctrl)  {
	    if (c.flags & CELL_NOLINENORTH)
	      c.flags = ~((~c.flags) & CELL_NOLINENORTH);
	    else
	      c.flags |= CELL_NOLINENORTH; }
	  else
	  x+=(switchViewpoint?-1:1)*(shift?5:1); 
	break;
  case 's': case 'x': case SDLK_DOWN: 
	if(menuChoise == MENU_TRANSLATE) {
	  if(!switchViewpoint)
		for(mx=0;mx<map->width-1;mx++)
		  for(my=0;my<map->height;my++)
			map->cell(mx,my) = map->cell(mx+1,my);
	  else
		for(mx=map->width-1;mx>0;mx--)
		  for(my=0;my<map->height;my++)
			map->cell(mx,my) = map->cell(mx-1,my);
	} else	  
	  if (shift && ctrl)  {
	    if (c.flags & CELL_NOLINESOUTH)
	      c.flags = ~((~c.flags) & CELL_NOLINESOUTH);
	    else
	      c.flags |= CELL_NOLINESOUTH; }
	  else
	  x-=(switchViewpoint?-1:1)*(shift?5:1); 
	break;
  case 'a': case SDLK_LEFT: 
	if(menuChoise == MENU_TRANSLATE) {
	  if(!switchViewpoint)
		for(my=map->height;my>0;my--)
		  for(mx=0;mx<map->width;mx++)
			map->cell(mx,my) = map->cell(mx,my-1);
	  else
		for(my=0;my<map->height-1;my++)
		  for(mx=0;mx<map->width;mx++)
			map->cell(mx,my) = map->cell(mx,my+1);	  
	} else    
	  if (shift && ctrl)  {
	    if (c.flags & CELL_NOLINEWEST)
	      c.flags = ~((~c.flags) & CELL_NOLINEWEST);
	    else
	      c.flags |= CELL_NOLINEWEST; }
	  else
	  y+=(switchViewpoint?-1:1)*(shift?5:1); 
	break;
  case 'd': case SDLK_RIGHT: 
	if(menuChoise == MENU_TRANSLATE) {
	  if(!switchViewpoint)
		for(my=0;my<map->height-1;my++)
		  for(mx=0;mx<map->width;mx++)
			map->cell(mx,my) = map->cell(mx,my+1);
	  else
		for(my=map->height;my>0;my--)
		  for(mx=0;mx<map->width;mx++)
			map->cell(mx,my) = map->cell(mx,my-1);
	} else    
	  if (shift && ctrl)  {
	    if (c.flags & CELL_NOLINEEAST)
	      c.flags = ~((~c.flags) & CELL_NOLINEEAST);
	    else
	      c.flags |= CELL_NOLINEEAST; }
	  else
	  y-=(switchViewpoint?-1:1)*(shift?5:1); 
	break;
  case 'q': raise -= shift?1.0:0.1; break;
  case 'e': raise += shift?1.0:0.1; break;
  case 'b': birdsEye = birdsEye ? 0 : 1; break;
  case 't': 
    if(map->flags & Map::flagTranslucent)
      map->flags &= ~Map::flagTranslucent;
    else
      map->flags |= Map::flagTranslucent;
  break;

  case 'c': 
    if(map->flags & Map::flagShowCross)
      map->flags &= ~Map::flagShowCross;
    else
      map->flags |= Map::flagShowCross;
  break;

  case 'u': case 'h': case 'm': case 'k':
	if(menuChoise == MENU_VELOCITY) {
	  for(x1=xLow;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh;y1++) {
		  Cell& c2=map->cell(x1,y1);
		  if(key == 'u') c2.velocity[1] += shift?0.5:0.1;
		  if(key == 'm') c2.velocity[1] -= shift?0.5:0.1;
		  if(key == 'h') c2.velocity[0] -= shift?0.5:0.1;
		  if(key == 'k') c2.velocity[0] += shift?0.5:0.1;
		}
	  break;
	}

	if(menuChoise == MENU_TEXTURE) {
	  for(x1=xLow;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh;y1++) {
		  Cell& c2=map->cell(x1,y1);
		  for(i=0;i<4;i++) {
			if(key == 'u') c2.textureCoords[i][1] += shift?0.5/4.:0.1/4.;
			if(key == 'm') c2.textureCoords[i][1] -= shift?0.5/4.:0.1/4.;
			if(key == 'h') c2.textureCoords[i][0] -= shift?0.5/4.:0.1/4.;
			if(key == 'k') c2.textureCoords[i][0] += shift?0.5/4.:0.1/4.;
		  }
		}
	  break;
	}


	if(key == 'u')      corner=Cell::NORTH+Cell::EAST;
	else if(key == 'h') corner = Cell::NORTH+Cell::WEST;
	else if(key == 'm') corner = Cell::SOUTH+Cell::WEST;
	else if(key == 'k') corner = Cell::SOUTH+Cell::EAST;
	
	if(menuChoise == MENU_HEIGHT) {
	  for(x1=xLow;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh;y1++) {
		  Cell& c2=map->cell(x1,y1);

		  if(ctrl && shift)
			for(i=0;i<5;i++) c2.heights[i] = c.heights[corner]; // note the use of 'c' here!			
		  else {
			c2.heights[corner] += shift?-raise:raise; 
			c2.heights[Cell::CENTER] += (shift?-raise:raise) / 4;	
		  }
		}
	} else if(menuChoise == MENU_WATER) {
	  /* Take care of previously not initialized water heights */
	  if(c.waterHeights[corner] <= -100.0) { 
		c.waterHeights[corner] = c.heights[corner]; 
		c.waterHeights[Cell::CENTER] = c.heights[Cell::CENTER];
	  }
	  for(x1=xLow;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh;y1++) {
		  Cell& c2=map->cell(x1,y1);

		  if(ctrl && shift)
			for(i=0;i<5;i++) c2.waterHeights[i] = c.waterHeights[corner]; // note the use of 'c' here!			
		  else {
			c2.waterHeights[corner] += shift?-raise:raise; 
			c2.waterHeights[Cell::CENTER] += (shift?-raise:raise) / 4;	
		  }
		}
	}

	else if(menuChoise == MENU_COLOR) {
	  for(x1=xLow;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh;y1++) {
		  Cell& c2=map->cell(x1,y1);

		  if(shift)
			/* Pick up color */
			if(ctrl)
			  for(i=0;i<4;i++)
				color[i] = c2.wallColors[corner][i];
			else		  
			  for(i=0;i<4;i++)
				color[i] = c2.colors[corner][i];
		  else
			/* Write color */
			if(ctrl)
			  for(i=0;i<4;i++)
				c2.wallColors[corner][i] = color[i];
			else
			  for(i=0;i<4;i++)
				c2.colors[corner][i] = color[i];
		}
	}
    break;
  
  case SDLK_TAB:
  	// smooth the cells if a selection is active
	if(menuChoise == MENU_HEIGHT) {
	
	  if (ctrl) {
	    c.heights[Cell::CENTER] = 
		   (c.heights[Cell::NORTH+Cell::WEST] +
		    c.heights[Cell::SOUTH+Cell::WEST] +
		    c.heights[Cell::NORTH+Cell::EAST] +
		    c.heights[Cell::SOUTH+Cell::EAST]) / 4.;
	    c.heights[Cell::NORTH+Cell::WEST] =
	      c.heights[Cell::SOUTH+Cell::WEST] =
	      c.heights[Cell::NORTH+Cell::EAST] =
	      c.heights[Cell::SOUTH+Cell::EAST] =
	          c.heights[Cell::CENTER];
	    break;
	  }
	
	  if (markX == -1) {
	  	c.heights[Cell::CENTER] = 
		   (c.heights[Cell::NORTH+Cell::WEST] +
		    c.heights[Cell::SOUTH+Cell::WEST] +
		    c.heights[Cell::NORTH+Cell::EAST] +
		    c.heights[Cell::SOUTH+Cell::EAST]) / 4.;
	  break;
	  }
	
	  // smooth edges
	  for(x1=xLow+1;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh-1;y1++) {
		  Cell& cur=map->cell(x1,y1);
		  Cell& ca1=map->cell(x1,y1+1);
		  Cell& ca2=map->cell(x1-1,y1);
		  Cell& ca3=map->cell(x1-1,y1+1);

		  cur.heights[Cell::NORTH+Cell::WEST] =
		  ca1.heights[Cell::SOUTH+Cell::WEST] =
		  ca2.heights[Cell::NORTH+Cell::EAST] =
		  ca3.heights[Cell::SOUTH+Cell::EAST] =
		     (cur.heights[Cell::NORTH+Cell::WEST] +
		      ca1.heights[Cell::SOUTH+Cell::WEST] +
		      ca2.heights[Cell::NORTH+Cell::EAST] +
		      ca3.heights[Cell::SOUTH+Cell::EAST]) / 4.;
		}
	
	  // smooth the center of the cell
	  for(x1=xLow;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh;y1++) {
		  Cell& cur=map->cell(x1,y1);

		  cur.heights[Cell::CENTER] =
		     (cur.heights[Cell::NORTH+Cell::WEST] +
		      cur.heights[Cell::SOUTH+Cell::WEST] +
		      cur.heights[Cell::NORTH+Cell::EAST] +
		      cur.heights[Cell::SOUTH+Cell::EAST]) / 4.;
		}
	
	}
	break;
  
  case 'j': 
	if(menuChoise == MENU_VELOCITY) {
	  for(x1=xLow;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh;y1++) {
		  Cell& c2=map->cell(x1,y1);
		  c2.velocity[0] = c2.velocity[1] = 0.0;
		}
	  break;
	}

	if(menuChoise == MENU_HEIGHT) {
	  for(x1=xLow;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh;y1++) {
		  Cell& c2=map->cell(x1,y1);
		  if(ctrl)
			if(shift) {
			  for(i=0;i<5;i++) c2.heights[i] = c.heights[Cell::CENTER]; // note the use of 'c' here!
			} else
			  c2.heights[Cell::CENTER] = (c2.heights[0]+c2.heights[1]+c2.heights[2]+c2.heights[3])/4.0;
		  else
			c2.heights[Cell::CENTER] += (shift?-raise:raise); 
		}
	} else if(menuChoise == MENU_COLOR) {
	  if(shift)
		/* Pick up color */		
		for(i=0;i<4;i++)
		  color[i] = c.colors[Cell::CENTER][i];
	  else
		/* Write color */
		if(!ctrl)
		  for(i=0;i<4;i++)
			c.colors[Cell::CENTER][i] = color[i];
	} 
	
	break;
  case ' ': 
	if(menuChoise == MENU_HILLS) {
	  switch(hill) {
	  case HILL_SPIKE:
		map->cell(x,y).heights[Cell::SOUTH+Cell::WEST] += shift?-raise:raise;
		map->cell(x-1,y).heights[Cell::SOUTH+Cell::EAST] += shift?-raise:raise;
		map->cell(x,y-1).heights[Cell::NORTH+Cell::WEST] += shift?-raise:raise;
		map->cell(x-1,y-1).heights[Cell::NORTH+Cell::EAST] += shift?-raise:raise;
		break;
	  case HILL_SMALL:
		//makeHill(1);
		map->cell(x,y).heights[Cell::CENTER] += (shift?-raise:raise) * 1.2;
		for(i=0;i<4;i++) map->cell(x,y).heights[i] += (shift?-raise:raise) * 1.0;
		map->cell(x,y+1).heights[Cell::SOUTH+Cell::EAST] += (shift?-raise:raise) * 1.0;
		map->cell(x,y+1).heights[Cell::SOUTH+Cell::WEST] += (shift?-raise:raise) * 1.0;
		map->cell(x,y-1).heights[Cell::NORTH+Cell::EAST] += (shift?-raise:raise) * 1.0;
		map->cell(x,y-1).heights[Cell::NORTH+Cell::WEST] += (shift?-raise:raise) * 1.0;
		map->cell(x+1,y).heights[Cell::SOUTH+Cell::WEST] += (shift?-raise:raise) * 1.0;
		map->cell(x+1,y).heights[Cell::NORTH+Cell::WEST] += (shift?-raise:raise) * 1.0;
		map->cell(x-1,y).heights[Cell::SOUTH+Cell::EAST] += (shift?-raise:raise) * 1.0;
		map->cell(x-1,y).heights[Cell::NORTH+Cell::EAST] += (shift?-raise:raise) * 1.0;
		map->cell(x+1,y+1).heights[Cell::SOUTH+Cell::WEST] += (shift?-raise:raise) * 1.0;
		map->cell(x-1,y+1).heights[Cell::SOUTH+Cell::EAST] += (shift?-raise:raise) * 1.0;
		map->cell(x+1,y-1).heights[Cell::NORTH+Cell::WEST] += (shift?-raise:raise) * 1.0;
		map->cell(x-1,y-1).heights[Cell::NORTH+Cell::EAST] += (shift?-raise:raise) * 1.0;
		map->cell(x+1,y+1).heights[Cell::CENTER] += (shift?-raise:raise) * 0.25;
		map->cell(x-1,y+1).heights[Cell::CENTER] += (shift?-raise:raise) * 0.25;
		map->cell(x+1,y-1).heights[Cell::CENTER] += (shift?-raise:raise) * 0.25;
		map->cell(x-1,y-1).heights[Cell::CENTER] += (shift?-raise:raise) * 0.25;
		map->cell(x+1,y).heights[Cell::CENTER] += (shift?-raise:raise) * 0.50;
		map->cell(x-1,y).heights[Cell::CENTER] += (shift?-raise:raise) * 0.50;
		map->cell(x,y+1).heights[Cell::CENTER] += (shift?-raise:raise) * 0.50;
		map->cell(x,y-1).heights[Cell::CENTER] += (shift?-raise:raise) * 0.50;
		break;
	  case HILL_MEDIUM:	makeHill(2); break;
	  case HILL_LARGE:makeHill(3); break;
	  case HILL_HUGE:makeHill(5); break;
	  case SMOOTH_SMALL:doSmooth(2); break;
	  case SMOOTH_LARGE:doSmooth(4); break;
	  }
	  return;
	} else if(menuChoise == MENU_TEXTURE) {
	  int newTexture=mymod(c.texture+1+(shift?1:-1),numTextures+1)-1;
	  for(x1=xLow;x1<=xHigh;x1++)
		for(y1=yLow;y1<=yHigh;y1++) {
		  Cell& c2=map->cell(x1,y1);
		  c2.texture = newTexture;
		}
	}

	/* Pickup color works only for a specific corner of cell */
	if(menuChoise == MENU_COLOR && shift) break; 

	for(x1=xLow;x1<=xHigh;x1++)
	  for(y1=yLow;y1<=yHigh;y1++) {
		Cell& c2=map->cell(x1,y1);
		if(menuChoise == MENU_HEIGHT)
		  for(i=0;i<5;i++) c2.heights[i] += (shift?-raise:raise);
		else if(menuChoise == MENU_WATER)
		  for(i=0;i<5;i++) {
			if(c2.waterHeights[i] <= -100.0) c2.waterHeights[i]=c2.heights[i]+(shift?-raise:raise);		
			else c2.waterHeights[i] += (shift?-raise:raise);
		  }
		else if(menuChoise == MENU_COLOR) {
		  if(ctrl)
			for(i=0;i<4;i++)
			  for(j=0;j<4;j++)
				c2.wallColors[i][j] = color[j];
		  else
			for(i=0;i<5;i++)
			  for(j=0;j<4;j++)
				c2.colors[i][j] = color[j];
		}
	  }
    break;
  case '1': color[0] += 0.1; if(color[0] > 1.0) color[0] = 0.0; break;
  case '2': color[1] += 0.1; if(color[1] > 1.0) color[1] = 0.0; break;
  case '3': color[2] += 0.1; if(color[2] > 1.0) color[2] = 0.0; break;
  case '4': color[3] += 0.1; if(color[3] > 1.0) color[3] = 0.0; break;
  }

	/*
  case SDLK_DOWN: menuChoise++; if(menuChoise >= MENU_SIZE) menuChoise = 0; break;
  case SDLK_UP: menuChoise--; if(menuChoise < 0) menuChoise = MENU_SIZE-1; break;*/
}
void EditMode::special(int key,int mx,int my) {
  printf("editmode: %d\n",key);
}
void EditMode::idle(Real td) {
  if(doSave == 1) {
	char mapname[256];
	char str[256];

	snprintf(str,sizeof(str)-1,"%s/.trackballs",getenv("HOME"));
	mkdir(str,S_IXUSR|S_IRUSR|S_IWUSR|S_IXGRP|S_IRGRP|S_IWGRP);
	snprintf(str,sizeof(str)-1,"%s/.trackballs/levels",getenv("HOME"));
	mkdir(str,S_IXUSR|S_IRUSR|S_IWUSR|S_IXGRP|S_IRGRP|S_IWGRP);

	if(useHome)
	  snprintf(mapname,sizeof(mapname)-1,"%s/.trackballs/levels/%s.map",getenv("HOME"),name);
	else
	  snprintf(mapname,sizeof(mapname)-1,"%s/levels/%s.map",SHARE_DIR,name);
	printf("saving map as '%s'\n",mapname);
	map->save(mapname,x,y);
	doSave=0;
	menuChoise = MENU_HEIGHT;
  }
  if(doSave > 1) doSave--;
  time += td;
}
void EditMode::activated() {
}
void EditMode::deactivated() {
}
double steps(double v) { return ((int)(v/0.1))*0.1; }
void EditMode::makeHill(int radius) {
  int mx,my;
  int diameter=radius*2+1;
  int shift = SDL_GetModState() & (KMOD_LSHIFT | KMOD_RSHIFT);

  for(mx=-radius;mx<=radius;mx++)
	for(my=-radius;my<=radius;my++) {
	  map->cell(x+mx,y+my).heights[Cell::CENTER] += 
		((int)(sin(1.*M_PI*(mx+radius+0.5)/diameter)*sin(1.*M_PI*(my+radius+0.5)/diameter)*(shift?-raise:raise)*10.))*.1;
	  map->cell(x+mx,y+my).heights[Cell::NORTH+Cell::EAST] += 
		((int)(sin(1.*M_PI*(mx+radius+1.0)/diameter)*sin(1.*M_PI*(my+radius+1.0)/diameter)*(shift?-raise:raise)*10.))*.1;
	  map->cell(x+mx,y+my).heights[Cell::NORTH+Cell::WEST] += 
		((int)(sin(1.*M_PI*(mx+radius)/diameter)*sin(1.*M_PI*(my+radius+1.0)/diameter)*(shift?-raise:raise)*10.))*.1;
	  map->cell(x+mx,y+my).heights[Cell::SOUTH+Cell::EAST] += 
		((int)(sin(1.*M_PI*(mx+radius+1.0)/diameter)*sin(1.*M_PI*(my+radius)/diameter)*(shift?-raise:raise)*10.))*.1;
	  map->cell(x+mx,y+my).heights[Cell::SOUTH+Cell::WEST] += 
		((int)(sin(1.*M_PI*(mx+radius)/diameter)*sin(1.*M_PI*(my+radius)/diameter)*(shift?-raise:raise)*10.))*.1;
	}
}
void EditMode::doSmooth(int radius) {
  int mx,my,i;
  int diameter=radius*2+1;
  int shift = SDL_GetModState() & (KMOD_LSHIFT | KMOD_RSHIFT);
  double avgHeight=0.0;
  int north,east;

  for(mx=-radius;mx<=radius;mx++)
	for(my=-radius;my<=radius;my++) {
	  Cell& c1=map->cell(x+mx,y+my);
	  for(i=0;i<5;i++)
		avgHeight+=c1.heights[i];
	}
  avgHeight = avgHeight / (5.*diameter*diameter);
  for(mx=-radius;mx<=radius;mx++)
	for(my=-radius;my<=radius;my++) {
	  Cell& c=map->cell(x+mx,y+my);
	  double s1=sin(1.*M_PI*(mx+radius+0.5)/diameter)*sin(1.*M_PI*(my+radius+0.5)/diameter)*(shift?-raise:raise);
	  c.heights[Cell::CENTER] = steps(c.heights[Cell::CENTER] * (1.-s1) + avgHeight * s1);
	  for(north=0;north<2;north++)
		for(east=0;east<2;east++) {
		  double s=sin(1.*M_PI*(my+radius+1.0*north)/diameter)*sin(1.*M_PI*(mx+radius+1.0*east)/diameter)*(shift?-raise:raise);
		  c.heights[Cell::NORTH*north+Cell::EAST*east] = steps(c.heights[Cell::NORTH*north+Cell::EAST*east] * (1.-s) + avgHeight * s);
		}		  
	}
}

void EditMode::askQuit() { doAskQuit=1; }
