/* settingsMode.cc
   Allows the user to change game settings

   Copyright (C) 2003  Mathias Broxvall

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "general.h"
#include "gameMode.h"
#include "menuMode.h"
#include "glHelp.h"
#include "SDL/SDL_image.h"
#include "settingsMode.h"
#include "game.h"
#include "settings.h"
#include "menuMode.h"
#include "sound.h"
#include "font.h"
#include "calibrateJoystickMode.h"

#define Y_BASE (screenHeight/2-35)
#define Y_DELTA ((screenHeight==480?35:40))
#define Y_IMG_DIFF 52


using namespace std;

SDL_Surface *SettingsMode::title, *SettingsMode::labels[NUM_SETTINGS], *SettingsMode::return2menu,*SettingsMode::return2menuHigh;
SettingsMode *SettingsMode::settingsMode;

extern void changeScreenResolution();

void SettingsMode::init() {
  char str[256];
  int i;

  sprintf(str,"%s/images/gamesettings.png",SHARE_DIR);
  title=IMG_Load(str);
  if(!title) { printf("Error: Failed to load %s\n",str); exit(0); } 

  sprintf(str,"%s/images/return2menu.png",SHARE_DIR);
  return2menu=IMG_Load(str);
  if(!return2menu) { printf("Error: Failed to load %s\n",str); exit(0); } 

  sprintf(str,"%s/images/return2menuHigh.png",SHARE_DIR);
  return2menuHigh=IMG_Load(str);
  if(!return2menuHigh) { printf("Error: Failed to load %s\n",str); exit(0); } 


  for(i=0;i<NUM_SETTINGS;i++) {
	sprintf(str,"%s/images/settings%d.png",SHARE_DIR,i+1);
	labels[i]=IMG_Load(str);
	if(!labels[i]) { printf("Error: Failed to load %s\n",str); exit(0); } 
  }

  settingsMode = new SettingsMode();
}
SettingsMode::SettingsMode() {}
void SettingsMode::activated() {}
void SettingsMode::deactivated() {Settings::settings->save();}
void SettingsMode::display() {
  int w,h,i;

  glPushAttrib(GL_ENABLE_BIT);
  Enter2DMode();
  glColor4f(1.0,1.0,1.0,1.0);
  glClearColor(0.0,0.0,0.0,0.0);
  glClear(GL_COLOR_BUFFER_BIT); // | GL_DEPTH_BUFFER_BIT);
  drawSurface(title,(screenWidth-title->w)/2,64,title->w,title->h);

  char str[256];
  int opt;
  glColor3f(1.0,1.0,1.0);
  /* Mouse sensitivity */
  sprintf(str,"%1.2f", Settings::settings->mouseSensitivity);
  displayMenu(labels[SETTING_MOUSE_SENSITIVITY],str,SETTING_MOUSE_SENSITIVITY);

  /* Screen resolution */
  sprintf(str,"%dx%dx%d", screenWidth, screenHeight,Settings::settings->colorDepth);
  displayMenu(labels[SETTING_SCREEN_RESOLUTION],str,SETTING_SCREEN_RESOLUTION);

  /* gfx details */
  switch(Settings::settings->gfx_details) {
  case 0: sprintf(str,"None"); break;
  case 1: sprintf(str,"Minimalistic"); break;
  case 2: sprintf(str,"Simple"); break;
  case 3: sprintf(str,"Standard"); break;
  case 4: sprintf(str,"Extra"); break;
  case 5: sprintf(str,"Everything"); break;
  }
  displayMenu(labels[SETTING_GFX_DETAILS],str,SETTING_GFX_DETAILS);

  /* Screen windowed */
  if(Settings::settings->is_windowed) sprintf(str,"No");
  else sprintf(str,"Yes");
  displayMenu(labels[SETTING_SCREEN_WINDOWED],str,SETTING_SCREEN_WINDOWED);

  /* show fps */
  if(Settings::settings->showFPS) sprintf(str,"Yes");
  else sprintf(str,"No");
  displayMenu(labels[SETTING_SHOW_FPS],str,SETTING_SHOW_FPS);

  /* Volumes */
  sprintf(str,"%d%%",(int)(Settings::settings->musicVolume*100.0));
  displayMenu(labels[SETTING_MUSIC_VOLUME],str,SETTING_MUSIC_VOLUME);
  sprintf(str,"%d%%",(int)(Settings::settings->sfxVolume*100.0));
  displayMenu(labels[SETTING_EFFECTS_VOLUME],str,SETTING_EFFECTS_VOLUME);

  /* Joystick */
  i=Settings::settings->joystickIndex;
  if(i) snprintf(str,255,"%s",SDL_JoystickName(i-1));
  else snprintf(str,255,"no joystick");
  displayMenu(labels[SETTING_JOYSTICK],str,SETTING_JOYSTICK);

  /* return */
  if(selected == SETTING_RETURN)
	drawSurface(return2menuHigh,screenWidth/2-return2menuHigh->w/2,screenHeight-20-Y_IMG_DIFF,return2menuHigh->w,return2menuHigh->h);
  else
	drawSurface(return2menu,screenWidth/2-return2menu->w/2,screenHeight-20-Y_IMG_DIFF,return2menu->w,return2menu->h);
  glColor4f(1.0,1.0,1.0,1.0);

  glEnable(GL_BLEND);
  int mouseX, mouseY;
  drawMousePointer();
  displayFrameRate();
  Leave2DMode();
  glPopAttrib();
}
void SettingsMode::key(int key) {
  if(key == SDLK_SPACE) { int x,y; SDL_GetMouseState(&x,&y); mouseDown(1,x,y); }
  if(key == SDLK_RETURN) { int x,y; SDL_GetMouseState(&x,&y); mouseDown(3,x,y); }
  if(key == SDLK_ESCAPE) GameMode::activate(MenuMode::menuMode);
}
void SettingsMode::idle(Real td) {
  int x,y;

  tickMouse(td);

  SDL_GetMouseState(&x,&y);
  Uint8 *keystate = SDL_GetKeyState(NULL);
  if(keystate[SDLK_LEFT]) { x-=(int)(150/fps); SDL_WarpMouse(x,y); }
  if(keystate[SDLK_RIGHT]) { x+=(int)(150/fps); SDL_WarpMouse(x,y); }
  if(keystate[SDLK_UP]) { y-=(int)(150/fps); SDL_WarpMouse(x,y); }
  if(keystate[SDLK_DOWN]) { y+=(int)(150/fps); SDL_WarpMouse(x,y); }

  Uint8 mouseState=SDL_GetMouseState(&x,&y); x+=32; y+=32;
  if(y >= Y_BASE && y <= Y_BASE+Y_DELTA*NUM_SETTINGS)
	selected = (y-Y_BASE) / Y_DELTA;
  else if(y >= screenHeight - 20) selected = SETTING_RETURN;
  else selected = -1;
}
void SettingsMode::displayMenu(SDL_Surface *image,char *str,int opt) {
  glColor4f(1.0,1.0,1.0,1.0);
  drawSurface(image,screenWidth/2-310,Y_BASE+opt*Y_DELTA-Y_IMG_DIFF,image->w,image->h);

  if(selected == opt)
	//draw2DString(menuFont,str,screenWidth/2+100,Y_BASE+opt*Y_DELTA+4,220,220,220);
	Font::drawSimpleText(0,str,screenWidth/2+100,Y_BASE+opt*Y_DELTA-22, 20., 20., 220/256.,220/256.,220/256.,1.);
  else
	//draw2DString(menuFont,str,screenWidth/2+100,Y_BASE+opt*Y_DELTA+4,220,220,64);
	Font::drawSimpleText(0,str,screenWidth/2+100,Y_BASE+opt*Y_DELTA-22, 20., 20., 220/256.,220/256.,64/256.,1.);
}

extern int screenResolutions[4][2],nScreenResolutions;

void SettingsMode::mouseDown(int button,int x,int y) {
  x+=32; y+=32;
  if(y >= Y_BASE && y <= Y_BASE+Y_DELTA*NUM_SETTINGS)
	selected = (y-Y_BASE) / Y_DELTA;
  else if(y >= screenHeight - 20) selected = SETTING_RETURN;
  else selected = -1;
  
  Settings *settings=Settings::settings;
  if(selected != -1) {
	int left=button==1;

	switch(selected) {
	case SETTING_MOUSE_SENSITIVITY:
	  Settings::settings->mouseSensitivity += left?0.25:-0.25; 
	  if(Settings::settings->mouseSensitivity > 5.001) Settings::settings->mouseSensitivity = 0.0;
	  if(Settings::settings->mouseSensitivity < 0.000) Settings::settings->mouseSensitivity = 5.00;
	  break;
	case SETTING_SCREEN_RESOLUTION:
	  do {
		settings->resolution = settings->resolution + (left?1:-1);
		if(settings->resolution == nScreenResolutions) {
		  settings->resolution=0;
		  if(settings->colorDepth == 16) settings->colorDepth=32;
		  else settings->colorDepth=16;
		} else if(settings->resolution == -1) {
		  settings->resolution=nScreenResolutions-1;
		  if(settings->colorDepth == 16) settings->colorDepth=32;
		  else settings->colorDepth=16;
		}

		changeScreenResolution();
	  } while(!screen);
	  break;
	case SETTING_SCREEN_WINDOWED:
	  Settings::settings->is_windowed = Settings::settings->is_windowed ? 0 : 1;
	  screen = SDL_SetVideoMode(screenWidth,screenHeight, 16, SDL_SWSURFACE | SDL_OPENGL | 
								(Settings::settings->is_windowed?0:SDL_FULLSCREEN));
	  break;
	case SETTING_GFX_DETAILS:
	  settings->gfx_details = mymod((settings->gfx_details + (left?1:-1)),6);
	  break;
	case SETTING_SHOW_FPS:
	  Settings::settings->showFPS = Settings::settings->showFPS ? 0 : 1; 
	  break;
	case SETTING_RETURN:
	  GameMode::activate(MenuMode::menuMode);
	  break;
	case SETTING_MUSIC_VOLUME:
	  settings->musicVolume = mymod(((int) (settings->musicVolume*10.0))+(left?1:-1),11)*0.1;
	  volumeChanged();
	  break;
	case SETTING_EFFECTS_VOLUME:
	  settings->sfxVolume = mymod(((int) (settings->sfxVolume*10.0))+(left?1:-1),11)*0.1;
	  volumeChanged();
	  playEffect(SFX_GOT_FLAG);
	  break;
	case SETTING_JOYSTICK:
	  settings->joystickIndex = mymod(settings->joystickIndex+(left?1:-1),SDL_NumJoysticks()+1);
	  if(settings->joystickIndex > 0) {
		GameMode::activate(CalibrateJoystickMode::calibrateJoystickMode);
	  }
	  break;
	}
  }
}
