/* libai/ai_m_smpnl.c
 *
 *  Copyright (C) 1997  Timothy M. Vanderhoek
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Tim Vanderhoek
 *  ac199@hwcn.org
 */

#include "ai.h"
#include "ai_err.h"

#include <stdio.h>

static int nolay (ai_t, set_t, hand_t, set_t);

/*
 * See comments for ai_move().
 *
 * This will make moves for AI's of type SIMPLE and NOLAY.
 *
 * Internal use only.
 */
set_t
ai_m_smpnl (
 	ai_t ai,
 	const hand_t hand,  /* My hand... */
 	const set_t set  /* What set I have to lay on */
) {
 	int i;
 	static struct set_s ret;

 	/* Make sure we have at least one card... */
 	for (i=0; i<14; i++) if (hand->cards[i] != NAC) break;
 	ai_err (i == 14, "libai: ai_move(): You have no cards\n");

 	/* We'll use a very simple logic for now... */

 	/* Find anything that beats the currently-laid set... */
 	ret.card = set->card;
 	ret.num = set->num;
 	if (ret.card == JOKER) {
 		/* Have to check this to prevent wrap-around... */
 		ret.card = NAC;
 		ret.num = 1;
 		return &ret;
 	}
 	ret.card++;

 	/* Special case where we are laying on a NAC (note that if
 	 * we don't special-case this, we will get caught in an infinite
 	 * loop for the part where we start seeing how few cards we can
 	 * lay in the instance that we're laying wilds). */
 	if (set->card == NAC) {
 		int x;

 		/* Find our lowest single card */
 		for (i=0; hand->cards[i] == NAC; i++) ;
 		/* See if we can turn it into a set of > 1 */
 		for (x=1; i+1 != 14 && hand->cards[i] == hand->cards[i+1];
 		  i++,x++) ;

 		ret.card = hand->cards[i];
 		ret.num = x;

 		return &ret;
 	}

 	/* The second for statement is designed so that we can lay jokers
 	 * on doubles */
 	for (; ret.card <= JOKER; ret.card++) /* next for() nests */
 	for (ret.num = 1; ret.num < 5; ret.num++) {
 		if (ass_set (hand, &ret) &&
 		  ass_setlt (set, &ret)) {
 			if (! nolay (ai, &ret, hand, set)) return &ret;
 		}
 	}

 	ret.card = NAC;
 	ret.num = 1;
 	return &ret;
}

/*
 * Returns true if we don't really want to lay this set...
 */
static int
nolay (
 	ai_t ai,  /* the ai */
 	set_t ret,  /* What I plan to return */
 	hand_t hand,  /* My hand */
 	set_t set  /* What I have to lay on */
) {
 	if (set->card == NAC) return 0;

 	/*
 	 * If the card we're thinking of laying is too much higher
 	 * than what we're laying on, then don't lay it unless
 	 *
 	 * 1) We're the fourth person and the set isn't from the
 	 *    third person (if it is, they'll probably go low when they
 	 *    start, anyways).
 	 * 2) We have a chance to screw-over the person who did start.
 	 * 3) We have very few cards that are lower.
 	 */

 	/* This implements only #3 from the above list. */
 	/* NOTE that if we're the second person to lay a card, then
 	 * who cares how much higher it is, since the other two people
 	 * will have to go higher, still.  If we're third or fourth, though,
 	 * then we do care.  I'm not sure how this reasoning should be
 	 * affected if some people are out of the game, already... */
 	if (ai->l == NOLAY && ai->pgone != 1 && ai->pgone != 3) {
 		int diff, c, cc;
 		int h, l;

 		/* If the number of cards we have that are lower than the
 		 * card we're planning to lay is 2.8 times the number of
 		 * cards we have that are higher than the card we're
 		 * planning to lay, then don't lay it. */

 		/* We use >= and <= so that we count doubles that
 		 * aren't part of the set we're laying.  The fact that
 		 * we count cards we're laying is undone later */

 		/* How many cards are higher? */
 		for (h=c=0; c<14; c++)
 			if (hand->cards[c] >= ret->card) h++;

 		/* How many cards lower? */
 		for (l=c=0; c<14; c++)
 			if (hand->cards[c] <= ret->card &&
 			  hand->cards[c] != NAC) l++;

 		if ((l-ret->num) / 4 > (h-ret->num)) return 1;
 	}

 	/*
 	 * Okay, we don't want to lay it if we're breaking up a pair, unless
 	 * it's a high-numbered pair that we'll probably need to break-up,
 	 * anyways...
 	 */
 	if (ai->l == NOLAY) {
 		int i, x;

 		for (x=i=0; i<14; i++) if (hand->cards[i] == ret->card) x++;
 		if (x != ret->num) {
 			/* Okay, we're breaking-up a set, but what are the
 			 * chances that we'd need to break this set up,
 			 * anyways?  If it's high, we probably need to break
 			 * it up, regardless, but if it's low, then
 			 * forget-it... */
 			if (ret->card <= SIX) return 1;  /* don't lay it */
 		}
 	}

 	return 0;
}
