/* aprez/aprez.c
 *
 *  Copyright (C) 1997  Timothy M. Vanderhoek
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Tim Vanderhoek
 *  ac199@hwcn.org
 */


#include "colour.h"
#include <ai.h>
#include <ass.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <termios.h>
#include <unistd.h>

#define MAX_CARDN 20
#define MAX_PLACN 10

#ifdef __CYGWIN32__
int tcgetattr (int, struct termios *);
int tcsetattr (int, int, const struct termios *);
#endif

struct ai_s ai[3];  /* Our three ai players */
game_t game;  /* Our game */
char s1[50], s2[50], s3[50], s4[50];  /* What our AI's say */
int SGBplay[4];  /* when you went during this round */
int SGBtot[4][4];  /* How many times has each person had each position */

char * cardname (card_t card, const char *);
void movename (char *, set_t);
void bigmessage (const char *, ...);
void mondomessage (const char * m);
void littlemessage (const char *, ...);
void drawscreen (int);
set_t choose (void);
char * itoa (int);
void loadgame (const char *);
void savegame (const char *);

void
savegame (const char * f) {
	FILE * sg;

	sg = fopen (f, "wb");
	fwrite (ai, sizeof (*ai), 3, sg);
	fwrite (game, sizeof (*game), 1, sg);
	fwrite (s1, 50, 1, sg);
	fwrite (s2, 50, 1, sg);
	fwrite (s3, 50, 1, sg);
	fwrite (s4, 50, 1, sg);
	fwrite (SGBtot, sizeof (int), 16, sg);
	fwrite (SGBplay, sizeof (*SGBplay), 4, sg);
	fclose (sg);
}

void loadgame (const char * f) {
	FILE *sg;

	sg = fopen (f, "rb");
	fread (ai, sizeof (*ai), 3, sg);
	fread (game, sizeof (*game), 1, sg);
	fread (s1, 50, 1, sg);
	fread (s2, 50, 1, sg);
	fread (s3, 50, 1, sg);
	fread (s4, 50, 1, sg);
	fread (SGBtot, sizeof (int), 16, sg);
	fread (SGBplay, sizeof (*SGBplay), 4, sg);
	fclose (sg);
}
/*
 * This uses the same trick as SCR to avoid having to free()
 * what it returns.  However, be aware that if you exceed
 * MAX_CARDN number of calls to cardname(), cardname() will
 * start overwriting previously returned strings
 */
char *
cardname (card_t card, const char * add) {
	char * ret;
	static char strings[MAX_CARDN][50];
	static string_no = 0;

	ret = strings[string_no];

	sprintf (ret, "%s%s%s%s",
	    card == NAC ?
	      scr (RESET, RED, BLACK)
	    : card < JACK ?
	      scr (RESET, GREEN, BLACK)
	    : card < TWO ?
	      scr (RESET, BLUE, BLACK)
	    : scr (RESET, RED, BLACK),
	    card == NAC   ? "..." :
	    card == THREE ? "Three" :
	    card == FOUR  ? "Four" :
	    card == FIVE  ? "Five" :
	    card == SIX   ? "Six" :
	    card == SEVEN ? "Seven" :
	    card == EIGHT ? "Eight" :
	    card == NINE  ? "Nine" :
	    card == TEN   ? "Ten" :
	    card == JACK  ? "Jack" :
	    card == QUEEN ? "Queen" :
	    card == KING  ? "King" :
	    card == ACE   ? "Ace" :
	    card == TWO   ? "Two" :
	    card == JOKER ? "Joker" : "bad card #",
	    add,
	    scr (RESET, WHITE, BLACK));

	string_no++;
	string_no %= MAX_CARDN;

	return ret;
}

char *
placename (position_t pos) {
	char * ret;
	static char strings[MAX_PLACN][50];
	static string_no = 0;

	ret = strings[string_no];

	sprintf (ret, "%s%s%s",
	    pos == ASS ?
	      scr (RESET, YELLOW, RED)
	    : pos == VICEASS ?
	      scr (RESET, MAGENTA, RED)
	    : pos == VICE ?
	      scr (RESET, YELLOW, GREEN)
	    : scr (RESET, MAGENTA, GREEN),
	    pos == ASS       ? "Ass"
	    : pos == VICEASS ? "Vice-Ass"
	    : pos == VICE    ? "Vice-President"
	    : "President",
	    scr (RESET, WHITE, BLACK));

	string_no++;
	string_no %= MAX_PLACN;

	return ret;
}

int
main (int argc, char ** argv) {
	struct termios newterm, oldterm;
	int i, ii;

	tcgetattr (STDIN_FILENO, &oldterm);
	tcgetattr (STDIN_FILENO, &newterm);
	newterm.c_lflag &= ~(ICANON|ECHO);
	newterm.c_cc[VMIN] = 1;
	tcsetattr (fileno(stdin), TCSANOW, &newterm);

	/* Init our ais and game */
	game = malloc (sizeof (*game));
	ass_init (game);
	ai_init (&ai[0], 2, SIMPLE);
	ai_init (&ai[1], 3, NOLAY);
	ai_init (&ai[2], 4, RAND);
	for (i=0; i < 4; i++) for (ii=0; ii < 4; ii++) SGBtot[i][ii] = 0;

	for (;;) {
		int goer;

		if (!ass_turn (game)) {
			hand_t j;
			int freshgame;

			strcpy (s1, "");
			strcpy (s2, "");
			strcpy (s3, "");
			strcpy (s4, "");
			for (i=0; i<4; i++) SGBplay[i] = 0;

			ass_deal (game);
			for (i=0; i<3; i++) {
				ai_notify (&ai[i], game->ppl[0].pos,
				    game->ppl[1].pos, game->ppl[2].pos,
				    game->ppl[3].pos);
			}

			/* If this is a new game, everyone is an ass */
			if (game->ppl[0].pos != ASS || game->ppl[1].pos != ASS)
				freshgame = 0;
			else
				freshgame = 1;
			j = ass_switch (game);
			/* Increase count of positionings for players */
			/* But don't do it if this is the initial round
			 * when everyone is an ass */
			if (!freshgame)
				for (i=0; i<4; i++)
					SGBtot[i][game->ppl[i].pos]++;

			/* Tell the player that we dealt out a new round
			 * and what cards they gave away, and what
			 * cards they gave. */
			if (j->cards[0] == NAC) goto dontdothis;
			switch (game->ppl[0].pos) {
			case ASS:
				bigmessage ("You gave away a %s and a %s.\n",
				    cardname (j->cards[0], ""),
				    cardname (j->cards[1], ""));
				bigmessage ("You received a %s and a %s.\n",
				    cardname (j->cards[2], ""),
				    cardname (j->cards[3], ""));
				break;
			case VICEASS:
				bigmessage ("You gave away a %s.\n",
				    cardname (j->cards[4], ""));
				bigmessage ("You received a %s.\n",
				    cardname (j->cards[5], ""));
				break;
			case VICE:
				bigmessage ("You gave away a %s.\n",
				    cardname (j->cards[5], ""));
				bigmessage ("You recieved a %s.\n",
				    cardname (j->cards[4], ""));
				break;
			case PREZ:
				bigmessage ("You gave away a %s and a %s.\n",
				    cardname (j->cards[2], ""),
				    cardname (j->cards[3], ""));
				bigmessage ("You received a %s and a %s.\n",
				    cardname (j->cards[0], ""),
				    cardname (j->cards[1], ""));
				break;
			}
			bigmessage ("You placed Ass %d times, Vice-Ass %d "
			            "times, Vice %d times, and Pres %d "
			            "times\n"
			            "John placed Ass %d times, Vice-Ass %d "
			            "times, Vice %d times, and Pres %d "
			            "times\n"
			            "Terry placed Ass %d times, Vice-Ass %d "
			            "times, Vice %d times, and Pres %d "
			            "times\n"
			            "Joan placed Ass %d times, Vice-Ass %d "
			            "times, Vice %d times, and Pres %d "
			            "times\n",
			    SGBtot[0][0], SGBtot[0][1], SGBtot[0][2],
			    SGBtot[0][3], SGBtot[1][0], SGBtot[1][1],
			    SGBtot[1][2], SGBtot[1][3], SGBtot[2][0],
			    SGBtot[2][1], SGBtot[2][2], SGBtot[2][3],
			    SGBtot[3][0], SGBtot[3][1], SGBtot[3][2],
			    SGBtot[3][3]);
dontdothis:
		}  /* if (!ass_turn (game)) */

		/* Take someone's turn... */
		goer = ass_turn (game);
		assert (goer != 0);

		if (ass_a_clear (game)) {
			littlemessage (
			    "Press any key to begin next round. . .\n");
			getchar();
			strcpy (s1, "");
			strcpy (s2, "");
			strcpy (s3, "");
			strcpy (s4, "");
			for (i=0; i<4; i++) SGBplay[i] = 0;
		}

		if (goer == 1) {
			set_t z;

			/* It is our famed player's turn... */
			for (; ass_move (game, (z = choose ()));)
				bigmessage ("Can't make that move!\n");
			movename (s4, z);
			for (i=0; i<3; i++)
				ai_watch (&ai[i], z, 1);
			/* Find highest SGBplay */
			for (ii=i=0; i<4; i++)
				if (SGBplay[i] > ii) ii = SGBplay[i];
			SGBplay[0] = ii + 1;
		} else {
			set_t z;

			/* Make the AI go... */
			z = ai_move (&ai[goer-2], &game->ppl[goer-1].hand, &game->down);
			ass_move (game, z);
			for (i=0; i<3; i++)
				if (i != goer - 2)
					ai_watch (&ai[i], z, goer);
			for (ii=i=0; i<4; i++)
				if (SGBplay[i] > ii) ii = SGBplay[i];
			SGBplay[goer-1] = ii + 1;
			switch (goer) {
			case 2:
				movename (s1, z);
				break;
			case 3:
				movename (s2, z);
				break;
			case 4:
				movename (s3, z);
				break;
			}
		}

		drawscreen (0);
	}
tcsetattr (fileno(stdin), TCSAFLUSH, &oldterm);
return 0;
}

set_t
choose (void) {
	int n1, n2, n3, n4;
	int i, ii;
	signed int max;
	static callingmyself = 0;
	static signed c;  /* The card selected */
	static ci;  /* How many of the card selected */
	static struct set_s ret;
	static signed current;

	if (!callingmyself) current = ci = 0, c = -1;

	cls ();

	/* How many cards does each AI player have? */
	for (i=0, n1=0; i<15; i++) if (game->ppl[1].hand.cards[i] != NAC) n1++;
	for (i=0, n2=0; i<15; i++) if (game->ppl[2].hand.cards[i] != NAC) n2++;
	for (i=0, n3=0; i<15; i++) if (game->ppl[3].hand.cards[i] != NAC) n3++;
	for (i=0, n4=0; i<15; i++) if (game->ppl[0].hand.cards[i] != NAC) n4++;

	ass_sort (&(game->ppl[0].hand));
	for (max = -1, i=0; i<15; i++) {
		if (game->ppl[0].hand.cards[i] != NAC) {
		if (i >= c && i < c+ci)
			printf ("%s***%s%s", scr (RESET, BLUE, WHITE),
			                scr (RESET, WHITE, BLACK),
			                cardname (game->ppl[0].hand.cards[i], ""));
		else
			printf ("%s", cardname (game->ppl[0].hand.cards[i], ""));
		max = i;
		}
#ifdef __CYGWIN32__
		/* Win95 doesn't seem to handle BLINK well... */
		printf (scr (HIGH, WHITE, BLACK));
#else
		printf (scr (BLINK, WHITE, BLACK));
#endif
		if (i == current) printf ("  <<--\n"); else printf ("\n");
		printf (scr (RESET, WHITE, BLACK));
	}
	assert (max != -1);  /* Should always have some cards... */

	printf ("\n");
	printf ("%s John has %d cards.  %s%s %s\n",
	    placename (game->ppl[1].pos), n1,
	    SGBplay[1] ? "Moves #" : "", itoa (SGBplay[1]), s1);
	printf ("%s Terry has %d cards.  %s%s %s\n",
	    placename (game->ppl[2].pos), n2,
	    SGBplay[2] ? "Moves #" : "", itoa (SGBplay[2]), s2);
	printf ("%s Joan has %d cards.  %s%s %s\n",
	    placename (game->ppl[3].pos), n3,
	    SGBplay[3] ? "Moves #" : "", itoa (SGBplay[3]), s3);
	printf ("The human has %d cards.  %s%s %s\n",
	    n4, SGBplay[0] ? "Move #" : "", itoa (SGBplay[0]), s4);
	printf ("\n");
	if (game->down.card == NAC)
		printf ("There is no set currently laid.\n");
	else {
		printf ("The current set is %d %s.\n",
		    game->down.num,
		    cardname (game->down.card, game->down.num == 1 ? "" : "s"));
	}
	printf (scr (HIGH, WHITE, BLACK));
	printf ("Press \"u\" to move up, \"j\" to move down, ");
	if (current >= c && current < c+ci)
		printf ("\"s\" to unselect, ");
	else
		printf ("\"s\" to select, ");
	if (ci)
		printf ("and \"l\" to lay it!\n");
	else
		printf ("and \"l\" to pass.\n");
	printf ("Press \"h\" for %shelp%s.", scr (RESET, CYAN, BLACK),
	    scr (RESET, WHITE, BLACK));

	i = tolower (getchar());
	if (i == 'u') current--;
	if (i == 'j') current++;
	if (i == 's') {
		if (c != -1 &&
		  /* the additional selection must be right after the last */
		  c + ci == current &&
		  /* must be same cardtype */
		  game->ppl[0].hand.cards[c] == game->ppl[0].hand.cards[current]) {
			ci++;
		} else if (c == current) {
			c = -1;
			ci = 0;
		} else {
			c = current;
			ci = 1;
		}
		current++;
	}
	if (i == 'l') {
		if (c == -1) {
			int sure;
			cls();
			printf ("Are you %sSURE%s you want to pass (Y/N)?",
			    scr (RESET, RED, BLACK), scr (RESET, WHITE, BLACK));
			for (; (sure = tolower (getchar())); )
				if (sure == 'y' || sure == 'n') break;
			if (sure == 'y') {
				ret.card = NAC;
				ret.num = 1;
				return &ret;
			}
		} else {
			ret.card = game->ppl[0].hand.cards[c];
			ret.num = ci;
			return &ret;
		}
	}
	if (i == 'v') {
		struct termios term;
		char s[51];
		cls();

		/* Turn echo on */
		tcgetattr (fileno(stdin), &term);
		term.c_lflag |= ECHO | ICANON;
		tcsetattr (fileno(stdin), TCSANOW, &term);

		printf ("Save to what filename? ");
		fgets (s, 51, stdin);

		/* Remove possible nl */
		if (s[strlen(s)-1] == '\n') s[strlen(s)-1] = '\0';

		if (s && *s) savegame (s);

		term.c_lflag &= ~(ECHO|ICANON);
		tcsetattr (fileno(stdin), TCSANOW, &term);
	}
	if (i == 'b') {
		struct termios term;
		char s[51];
		cls();

		/* turn echo on */
		tcgetattr (fileno(stdin), &term);
		term.c_lflag |= ECHO;
		tcsetattr (fileno(stdin), TCSANOW, &term);

		printf ("Load from what file? ");
		fgets (s, 51, stdin);

		/* Remove possible nl */
		if (s[strlen(s)-1] == '\n') s[strlen(s)-1] = '\0';

		if (s && *s) loadgame(s);

		term.c_lflag &= ~(ECHO);
		tcsetattr (fileno(stdin), TCSANOW, &term);
	}
	if (i == 'h') {
		bigmessage ("
j\tdown one
u\tup one
s\ttoggle select on card
l\tlay (or pass if no cards are selected)
v\tsave game
b\tload game
h\tcommand reference
w\twarranty and license
a\tabout
C-C\tquit (Ctrl-C)
");
	}
	if (i == 'w') {
		mondomessage ("!COPYING!");
	}
	if (i == 'a') {
		mondomessage ("!ABOUT!");
	}

	if (current < 0) current = 0;
	if (current > max) current = max;

	callingmyself = 1;
	choose();
	callingmyself = 0;
	return &ret;
}

/*
card1
card2
card3
card4
card5
card6
card7
card8
card9
card10
card11
card12
card13
card14

Ass John has 10 cards left.
Vice-Ass Terry has 13 cards left.
President Joan has 5 cards left.

You must lay on 2 threes.

<<<short instruction>>>>
*/

void
drawscreen (int curcard) {
	int n1, n2, n3, n4;
	int i, ii;
	cls ();

	/* How many cards does each AI player have? */
	for (i=0, n1=0; i<15; i++) if (game->ppl[1].hand.cards[i] != NAC) n1++;
	for (i=0, n2=0; i<15; i++) if (game->ppl[2].hand.cards[i] != NAC) n2++;
	for (i=0, n3=0; i<15; i++) if (game->ppl[3].hand.cards[i] != NAC) n3++;
	for (i=0, n4=0; i<15; i++) if (game->ppl[0].hand.cards[i] != NAC) n4++;

	ass_sort (&(game->ppl[0].hand));
	for (i=0; i<15; i++) {
		if (game->ppl[0].hand.cards[i] != NAC)
			printf ("%s\n", cardname (game->ppl[0].hand.cards[i], ""));
	}

	printf ("\n");
	printf ("%s John has %d cards.  %s%s %s\n",
	    placename (game->ppl[1].pos), n1,
	    SGBplay[1] ? "Moves #" : "", itoa (SGBplay[1]), s1);
	printf ("%s Terry has %d cards.  %s%s %s\n",
	    placename (game->ppl[2].pos), n2,
	    SGBplay[2] ? "Moves #" : "", itoa (SGBplay[2]), s2);
	printf ("%s Joan has %d cards.  %s%s %s\n",
	    placename (game->ppl[3].pos), n3,
	    SGBplay[3] ? "Moves #" : "", itoa (SGBplay[3]), s3);
	printf ("The human has %d cards.  %s%s %s\n",
	    n4, SGBplay[0] ? "Move #" : "", itoa (SGBplay[0]), s4);
	printf ("\n");
	if (game->down.card == NAC)
		printf ("There is no set currently laid.\n");
	else
		printf ("The current set is %d %s.\n",
		    game->down.num,
		    cardname (game->down.card, game->down.num == 1 ? "" : "s"));

}

void
bigmessage (const char * m, ...) {
	va_list ap;

	cls ();
	va_start (ap, m);
	vprintf (m, ap);
	va_end (ap);

	printf ("\n\nPress any key to continue . . .\n");
	getchar();
}

void
mondomessage (const char * m) {
	int lines;

	while (*m) {
		cls();
		for (lines=0; lines<20 && *m; m++) {
			if (*m == '\n') lines++;
			putchar (*m);
		}
		printf ("\n\nPress any key to continue . . .\n");
		getchar();
	}
}

void
movename (char * s, set_t set) {
	if (set->card == NAC) {
		strcpy (s, "Passes...");
		return;
	}
	sprintf (s, "Lays %d %s.", set->num,
	    cardname (set->card, set->num == 1 ? "" : "s"));
}

/* This is meant to display a message at the bottom of the screen.
 * This is actually difficult to do... */
void
littlemessage (const char * m, ...) {
	va_list ap;

	va_start (ap, m);
	vprintf (m, ap);
	va_end (ap);
}

/* Sheesh... */
char *
itoa (int i) {
	static char ret[10];

	if (i) sprintf (ret, "%d", i); else strcpy (ret, "");

	return ret;
}
