#ifndef __ASS_H
#define __ASS_H

/*
 * These are only guaranteed to be ascending in value.  The first value is
 * assigned only to prevent different compilers from beginning with a
 * different starting value (ie. 1) and producing code that doesn't work
 * with the library compiled with a compiler that begins at 0.  The purpose
 * is simply store cards.
 */
typedef enum {
 	NAC = 0,  /* Not A Card */
 	THREE,
 	FOUR,
 	FIVE,
 	SIX,
 	SEVEN,
 	EIGHT,
 	NINE,
 	TEN,
 	JACK,
 	QUEEN,
 	KING,
 	ACE,
 	TWO,
 	JOKER
} card_t;

/*
 * One set.  Eg. Three 2's, One 5, etc.
 */
typedef struct set_s {
 	card_t card;  /* Which card is waiting to be laid on */
 	int num;      /* Is it a pair of 1, 2, 3, or 4? */
} * set_t;

/*
 * Guaranteed to be ascending in value.  First is assigned to avoid possible
 * compiler-interoperability problems.  The purpose is simply to denote
 * what position a player has in the game.
 */
typedef enum {
 	ASS = 0,  /* 4th place */
 	VICEASS,  /* 3rd place */
 	VICE,     /* 2nd place */
 	PREZ      /* 1st place */
} position_t;

/*
 * The set of cards that a player may have.
 */
typedef struct hand_s {
 	card_t cards[54];  /* complete list of all cards, using NAC
 	                    * for empty ones */
} * hand_t;

/*
 * A player...  in the game...
 */
typedef struct player_s {
 	position_t pos;            /* Player's current position */
 	int out;                   /* person ## out?  (1-4) 0 == newdeal/game */
 	struct hand_s hand;        /* The current hand */
} * player_t;

typedef struct game_s {
 	struct player_s ppl[4];    /* The players in the game...  It is
 	                            * guaranteed that no positions are
 	                            * duplicated in these people except in the
 	                            * case where everyone is an ASS at the
 	                            * beginning of the game */
 	struct set_s down;         /* what is waiting to be layed on */
 	int start;                 /* round-starter. -1 iff invalid. ppl[x] */
 	int cur;                   /* Who just took their turn?  ppl[x] */
 	int last;                  /* Who laid the last set (index to ppl[]). */
} * game_t;

/*
 * It is assumed that you call these only at valid points in time,
 * and that you pass them only valid arguments.  This allows the macros
 * to avoid having to do error-checking (which would be impossible for
 * them).  If you define ASS_NODEFS before include <ass.h>, you will
 * prevent the macros from being defined and you will use the functions
 * instead.  Alternatively, you can selectively #undef macros.
 */
player_t ass_GamePlayer (game_t, int);  /* Return player # */
int ass_GamePos (game_t, position_t);  /* Return # of player of position x */
set_t ass_GameDown (game_t);  /* Return currently laid set */
hand_t ass_Hand (player_t);  /* Return player's hand */
card_t ass_HandCard (hand_t, int);  /* Return card # from hand */
card_t ass_SetCard (set_t);  /* Return the card a set is made of */
int ass_SetNum (set_t);  /* Return the number of cards in a set */
position_t ass_PlayerPos (player_t);  /* Return player's position */

void ass_makeset (set_t, card_t, int);  /* set set_t to int card_t's */

#ifndef ASS_NODEFS

/*
 * The macros that can replace the functions of the same name.
 */
#define ass_GamePlayer(a,b) (&((a)->ppl[(b)-1]))
#define ass_GameDown(a) (&((a)->down))
#define ass_Hand(a) (&((a)->hand))
#define ass_HandCard(a,b) ((a)->cards[(b)-1])
#define ass_GameCard(a,b) ((a)->cards[(b)])
/* Return a set's card */
#define ass_SetCard(a) ((a)->card)
#define ass_SetNum(a) ((a)->num)
#define ass_PlayerPos(a) ((a)->pos)

#define ass_makeset(a,b,c) ((void) ((a)->card = (b), (a)->num = (c)))

#endif  /* ASS_NODEFS */

/* This ass_a_*() functions will occur automatically, but you can
 * make them occur earlier by calling them explicitly.  Yes, ok, so there
 * is only one right now...  :) */
int ass_a_clear (game_t);       /* Clear laid set if about to begin new round */
void ass_a_shuffle (game_t);    /* make winner PREZ, etc. */

void ass_init (game_t);         /* Call to initialize a game */
void ass_deal (game_t);         /* Deal some cards out! */
hand_t ass_switch (game_t);     /* give ASS's cards to PREZ, etc... */
int ass_move (game_t, set_t);   /* Lay down some cards, baby! */
/* Should provide alternate version of above, that doesn't use set_t */
int ass_turn (game_t);          /* Whose turn is it?? */
int ass_set (hand_t, set_t);    /* Does set_t exist in hand_t? */
int ass_setlt (set_t, set_t);   /* TRUE if set#1 is less than set#2 */
int ass_all (game_t);           /* Has everyone had their turn? */
void ass_sort (hand_t);         /* Sort said hand */

#endif /* __ASS_H */
