/* prez/parse.y
 *
 *  Copyright (C) 1997  Timothy M. Vanderhoek
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Tim Vanderhoek
 *  ac199@hwcn.org
 */

/* Grammar for the President CLI */

%{

#include "../rand/rand.h"
#include <ass.h>
#include <ai.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>

#define NAMELEN 25  /* Max length of a player's name */

char * getln (char *, int);
int message (const char *, ...);
void savegame (const char *);
void play_ai (void);

game_t  SGBgame;  /* The game we're playing!! :) */
int SGBplaysinround;  /* How many plays have taken place in this round */
int SGBcomerr;

/* ! COPYING ! gets replaced with the license by the Makefile */
const char * SGBlicense="!COPYING!";
const char * SGBabout="!ABOUT!";

enum ptype {
	tAI = 0,
	tHUMAN
};

enum ptype ptypes[4];  /* Wether the players are AIs or HUMANs */
struct ai_s pai[4];  /* The AI players */

struct word {
	const char * s;  /* string to recognize */
	int v;  /* value to return */
	card_t card;  /* A card */
};

char pnames[4][NAMELEN+1];  /* Player names... */

/* translate x into a proper string representing the card */
#define cardname(x) ((x) == NAC      ? "" : \
                     (x) == THREE    ? "Three" : \
                     (x) == FOUR     ? "Four" : \
                     (x) == FIVE     ? "Five" : \
                     (x) == SIX      ? "Six" : \
                     (x) == SEVEN    ? "Seven" : \
                     (x) == EIGHT    ? "Eight" : \
                     (x) == NINE     ? "Nine" : \
                     (x) == TEN      ? "Ten" : \
                     (x) == JACK     ? "Jack" : \
                     (x) == QUEEN    ? "Queen" : \
                     (x) == KING     ? "King" : \
                     (x) == ACE      ? "Ace" : \
                     (x) == TWO      ? "Two" : \
                     (x) == JOKER    ? "Joker" : \
                     (assert (0), "NONSENSICAL CARDNUMBER!"))

/* translate x into a proper string representing the position */
#define placename(x) ((x) == ASS     ? "Ass" : \
                      (x) == VICEASS ? "Vice-Ass" : \
                      (x) == VICE    ? "Vice" : \
                      (x) == PREZ    ? "President" : \
                      (assert (0), "NONSENSICAL PLACENAME!"))
%}

%union {
	char * string;
	int player;
	struct set_s set;
	card_t card;
	enum ai_level aitype;
}

%token PLAYERS
%token SHOW
%token LICENSE
%token SAVE
%token LOAD
%token ABOUT
%token CURSET
%token NAME
%token CARDS
%token AI
%token TEST
%token HUMAN
%token SIMPLE_T
%token NOLAY_T
%token DEAL
%token RAND_T
%token PLAY
%token SET
%token HELP
%token QUIT
%token PASS
%token TWO_T
%token THREE_T
%token FOUR_T
%token FIVE_T
%token SIX_T
%token SEVEN_T
%token EIGHT_T
%token NINE_T
%token TEN_T
%token JACK_T
%token QUEEN_T
%token KING_T
%token ACE_T
%token JOKER_T
/* We use comerr so that lines like

set 3\n

don't unnecessarily take us out of preinit mode. */
%token COMERR
%token <string> STRING
%token <player> PNAME

%type <player> player
%type <card> card
%type <set> set
%type <aitype> aitype

%%

/* This is the stuff that can only be done immediately after loading prez */
preinit: /* empty */
	| preinit SET player AI '\n' {
		ptypes[$3-1] = tAI;
		ai_init (&pai[$3-1], $3, NOLAY);
		message ("%s is now an AI\n", pnames[$3-1]);
	}
	| preinit SET player AI NOLAY_T '\n' {
		ptypes[$3-1] = tAI;
		ai_init (&pai[$3-1], $3, NOLAY);
		message ("%s is now an AI\n", pnames[$3-1]);
	}
	| preinit SET player AI SIMPLE_T '\n' {
		ptypes[$3-1] = tAI;
		ai_init (&pai[$3-1], $3, SIMPLE);
		message ("%s is now an AI\n", pnames[$3-1]);
	}
	| preinit SET player AI RAND_T '\n' {
		ptypes[$3-1] = tAI;
		ai_init (&pai[$3-1], $3, RAND);
		message ("%s is now an AI\n", pnames[$3-1]);
	}
	| preinit SET player HUMAN '\n' {
		ptypes[$3-1] = tHUMAN;
		message ("%s is now a HUMAN\n", pnames[$3-1]);
	}
	| preinit HELP '\n' {
		message ("\tSHOW CARDS [player]\n"
		         "\tDEAL  (do this when the prompt is \"(deal) >>\"\n"
		         "\tPLAY # [TWOS THREES FOURS ... JOKERS]\n"
		         "\tSHOW CURSET  (show what card you must play on)\n"
		         "\tSHOW PLAYERS\n"
		         "\tSHOW LICENSE (shows license and warranty)\n"
		         "\tSHOW ABOUT  (About TVP and prez!!  Important!)\n"
		         "\tSAVE filename\n"
		         "\tLOAD filename\n"
		         "\tQUIT\n");
		}
	| preinit NAME player STRING '\n' {
		message ("%s is now named %s\n", pnames[$3-1], $4);
		strcpy (pnames[$3-1], $4);
	}
	| preinit SHOW LICENSE '\n' {
		bigmessage (SGBlicense);
	}
	| preinit SHOW ABOUT '\n' {
		bigmessage (SGBabout);
	}
	| preinit error
	| preinit '\n'
	| preinit command { SGBcomerr = 1; }
;

command: /* empty */
	| command error
	| command '\n'
	| command SET player AI '\n' {
		message ("You cannot change a player's constitution at this\n");
		message ("point in the game.\n");
	}
	| command SET player AI RAND_T {
		message ("You cannot change a player's constitution at this\n");
		message ("point in the game.\n");
	}
	| command SET player AI SIMPLE_T {
		message ("You cannot change a player's constitution at this\n");
		message ("point in the game.\n");
	}
	| command SET player AI NOLAY_T {
		message ("You cannot change a player's constitution at this\n");
		message ("point in the game.\n");
	}
	| command SET player HUMAN '\n' {
		message ("You cannot change a player's constitution at this\n");
		message ("point in the game.\n");
	}
	| command SHOW LICENSE '\n' {
		bigmessage (SGBlicense);
	}
	| command SHOW CARDS player '\n' {
		int ac_f=0;  /* Do they have any cards at all? */
		int i;
		hand_t h;

		h = ass_Hand (ass_GamePlayer (SGBgame, $4));

		if (ass_turn (SGBgame) != $4) {
			/* It's not player's turn.  Can't show the cards
			 * they have!!!  We can tell how many they have,
			 * though... */
			for(i=1; i<15; i++) if(ass_HandCard(h,i) != NAC) ac_f++;
			message ("%s has %d cards.\n", pnames[$4-1], ac_f);
		} else {
			message ("%s has the following cards:\n", pnames[$4-1]);
			for (i=1; i<=14; i++) {
				if (ass_HandCard (h, i) != NAC) {
					message ("\t%s\n",
					    cardname(ass_HandCard(h,i)));
					ac_f = 1;
				}
			}
			if (!ac_f) message ("\tNo Cards!\n");
		}
	}
	| command SHOW CARDS '\n' {
		/* same as above, but assume curplayer */
		int i, ac_f=0;
		hand_t h = ass_Hand (ass_GamePlayer (SGBgame,
		    ass_turn (SGBgame)));

		/* 99% of the time it is impossible for the curplayer to
		 * have no cards, but some of the times, it is... (eg.
		 * start of game). */
		for (i=1; i<=14; i++) if (ass_HandCard (h, i) != NAC) break;
		if (i == 15) {
			message ("You have no cards!!\n");
		} else {
			message ("You have the following cards:\n");
			for (i=1; i<=14; i++) {
				if (ass_HandCard (h, i) != NAC) {
					message ("\t%s\n",
					    cardname (ass_HandCard (h,i)));
				}
			}
		}
	}
	| command SHOW PLAYERS '\n' {
		int i, x, ac_f;
		hand_t h;

		for (i=1; i<5; i++) {
			h = ass_Hand (ass_GamePlayer (SGBgame, i));
			ac_f = 0;
			for(x=1; x<15; x++) if(ass_HandCard(h,x) != NAC) ac_f++;
			message ("%s %s has %d card%s.\n",
			    placename (ass_PlayerPos (ass_GamePlayer (
			    SGBgame, i))), pnames[i-1],
			    ac_f, ac_f != 1 ? "s" : "");
		}
	}
	| command NAME player STRING '\n' {
		message ("%s is now named %s\n", pnames[$3-1], $4);
		strcpy (pnames[$3-1], $4);
	}
	| command SHOW CURSET '\n' {
		if (ass_SetCard(ass_GameDown (SGBgame)) != NAC) {
			message ("The currently-laid set is %d %s's\n",
			    ass_SetNum (ass_GameDown(SGBgame)),
			    cardname(ass_SetCard(ass_GameDown(SGBgame))));
		} else {
			message ("There is no currently-laid set!\n");
		}
	}
	| command DEAL '\n' {
 		hand_t j;
		int i;

		ass_deal (SGBgame);
		message ("The cards have been dealt!\n");
		j = ass_switch (SGBgame);
		if (ass_HandCard (j, 1) != NAC) {
			/* Okay, cards were actually given away... */
 			message ("%s the %s gave away a %s and a %s\n",
			    pnames[ass_GamePos (SGBgame, ASS)-1],
			    placename (ASS),
			    cardname (ass_HandCard (j, 1)),
			    cardname (ass_HandCard (j, 2)));
			message ("%s the %s gave away a %s and a %s\n",
			    pnames[ass_GamePos (SGBgame, PREZ)-1],
			    placename (PREZ),
			    cardname (ass_HandCard (j, 3)),
			    cardname (ass_HandCard (j, 4)));
			message ("%s the %s gave away a %s\n",
			    pnames[ass_GamePos (SGBgame, VICEASS)-1],
			    placename (VICEASS),
			    cardname (ass_HandCard (j, 5)));
			message ("%s the %s gave away a %s\n",
			    pnames[ass_GamePos (SGBgame, VICE)-1],
			    placename (VICE),
			    cardname (ass_HandCard (j, 6)));
		}

		for (i=0; i<4; i++) if (ptypes[i] == tAI) {
			ai_notify (&pai[i],
			    ass_PlayerPos (ass_GamePlayer (SGBgame, 1)),
			    ass_PlayerPos (ass_GamePlayer (SGBgame, 2)),
			    ass_PlayerPos (ass_GamePlayer (SGBgame, 3)),
			    ass_PlayerPos (ass_GamePlayer (SGBgame, 4)));
		}
	}
	| command HELP '\n' {
		message ("\tSHOW CARDS [player]\n"
		         "\tDEAL  (do this when the prompt is \"(deal) >>\"\n"
		         "\tPLAY # [TWOS THREES FOURS ... JOKERS]\n"
		         "\tSHOW CURSET  (show what card you must play on)\n"
		         "\tSHOW PLAYERS\n"
		         "\tSAVE filename\n"
		         "\tLOAD filename\n"
		         "\tQUIT\n");
		}
	| command PLAY set '\n' {
		int goer = ass_turn (SGBgame);
		hand_t h = ass_Hand (
		    ass_GamePlayer (SGBgame, ass_turn (SGBgame)));

		if (ptypes[goer-1] == tAI) {
			message ("You can't make moves for an AI.  "
			         "Try just \"PLAY\"\n");
			goto falsebreak;
		}

		if (ass_turn (SGBgame)) {
			if (!ass_setlt (ass_GameDown (SGBgame), &$3)) {
				message ("You can't lay that set!!\n");
				message ("Maybe you should do \"PLAY pass\" "
				         "instead...\n");
			} else if (!ass_set (h, &$3)) {
				message ("You don't have that set to lay!!\n");
				message ("Maybe you should do a \"PLAY pass\" "
				         "instead...\n");
			} else {
				if (ass_move (SGBgame, &$3))
					message ("Couldn't play set...\n");
				else {
					int i;

					message ("Move taken!\n");

					/* Notify AI's of move */
					for (i=0; i<4; i++) if (ptypes[i]== tAI)
						ai_watch (&pai[i], &$3, goer);
					
					SGBplaysinround++;
				}
			}

#ifdef DEBUG
			savegame ("auto_save.asg");
#endif /* not DEBUG */

 			if (ass_a_clear (SGBgame)) {
 				message ("Cleared the set!  New Round!\n");
				SGBplaysinround = 1;
			}
		} else
			message ("You must \"DEAL\" first.\n");

		play_ai ();

falsebreak:
	}
	| command PLAY '\n' {
#ifdef DEBUG
			savegame ("auto_save.asg");
#endif /* not DEBUG */

		play_ai ();
	}
	| command TEST aitype aitype aitype aitype '\n' {
		int i;
		int pp[4][4];  /* final positions */
		enum ai_level ll[4];

		ll[0] = $3;
		ll[1] = $4;
		ll[2] = $5;
		ll[3] = $6;

		for (i=0; i<4; i++) pp[0][i] = 0;
		for (i=0; i<4; i++) pp[1][i] = 0;
		for (i=0; i<4; i++) pp[2][i] = 0;
		for (i=0; i<4; i++) pp[3][i] = 0;

		ptypes[0] = tAI;
		ptypes[1] = tAI;
		ptypes[2] = tAI;
		ptypes[3] = tAI;

		message (NULL);

		for (i=0; i<350; i++) {
			int t[4];

			ai_init (&pai[0], 1, ll[t[0] = myrand()%4]);
			ai_init (&pai[1], 2, ll[t[1] = myrand()%4]);
			ai_init (&pai[2], 3, ll[t[2] = myrand()%4]);
			ai_init (&pai[3], 4, ll[t[3] = myrand()%4]);

			ass_init (SGBgame);
			ass_deal (SGBgame);
			ai_notify (&pai[0], ASS, ASS, ASS, ASS);
			ai_notify (&pai[1], ASS, ASS, ASS, ASS);
			ai_notify (&pai[2], ASS, ASS, ASS, ASS);
			ai_notify (&pai[3], ASS, ASS, ASS, ASS);

			play_ai();

			ass_a_shuffle (SGBgame);

			pp[t[0]][SGBgame->ppl[t[0]].pos]++;
			pp[t[1]][SGBgame->ppl[t[1]].pos]++;
			pp[t[2]][SGBgame->ppl[t[2]].pos]++;
			pp[t[3]][SGBgame->ppl[t[3]].pos]++;
		}

		message (NULL);

		message ("ass:%d\tvas:%d\tvp:%d\tpres:%d\n",
		    pp[0][ASS], pp[0][VICEASS], pp[0][VICE], pp[0][PREZ]);
		message ("ass:%d\tvas:%d\tvp:%d\tpres:%d\n",
		    pp[1][ASS], pp[1][VICEASS], pp[1][VICE], pp[1][PREZ]);
		message ("ass:%d\tvas:%d\tvp:%d\tpres:%d\n",
		    pp[2][ASS], pp[2][VICEASS], pp[2][VICE], pp[2][PREZ]);
		message ("ass:%d\tvas:%d\tvp:%d\tpres:%d\n",
		    pp[3][ASS], pp[3][VICEASS], pp[3][VICE], pp[3][PREZ]);
	}
	| command SAVE STRING '\n' {
		savegame ($3);
	}
	| command LOAD STRING '\n' {
		loadgame ($3);

		/* The next four lines are the four lines that are
		 * run right after the auto-save-game feature.  I don't
		 * want to save after, since ass_a_clear could change
		 * important state, but I definately definately do want
		 * them run, since not running them causes me confusion. */
		if (ass_a_clear (SGBgame)) {
			message ("Cleared the set!  New Round!\n");
			SGBplaysinround = 1;
		}
	}
	| command QUIT '\n' {
		exit(0);
	}
;

player: '1' {
		$$ = 1;
	}
	| '2' {
		$$ = 2;
	}
	| '3' {
		$$ = 3;
	}
	| '4' {
		$$ = 4;
	}
	| PNAME {
		$$ = $1;
	}
;

set: '1' card {
		ass_makeset (&$$, $2, 1);
		($$).card = $2;
	}
	| '2' card {
		ass_makeset (&$$, $2, 2);
	}
	| '3' card {
		ass_makeset (&$$, $2, 3);
	}
	| '4' card {
		ass_makeset (&$$, $2, 4);
	}
	| PASS {
		ass_makeset (&$$, NAC, 0);
	}
;

aitype: NOLAY_T {
		$$ = NOLAY;
	}
	| SIMPLE_T {
		$$ = SIMPLE;
	}
	| RAND_T {
		$$ = RAND;
	}
;

card: THREE_T { $$ = THREE; }
	| '3' { $$ = THREE; }
	| FOUR_T { $$ = FOUR; }
	| '4' { $$ = FOUR; }
	| FIVE_T { $$ = FIVE; }
	| '5' { $$ = FIVE; }
	| SIX_T { $$ = SIX; }
	| '6' { $$ = SIX; }
	| SEVEN_T { $$ = SEVEN; }
	| '7' { $$ = SEVEN; }
	| EIGHT_T { $$ = EIGHT; }
	| '8' { $$ = EIGHT; }
	| NINE_T { $$ = NINE; }
	| '9' { $$ = NINE; }
	| TEN_T { $$ = TEN; }
	| '1' '0' { $$ = TEN; }
	| JACK_T { $$ = JACK; }
	| QUEEN_T { $$ = QUEEN; }
	| KING_T { $$ = KING; }
	| ACE_T { $$ = ACE; }
	| TWO_T { $$ = TWO; }
	| JOKER_T { $$ = JOKER; }
%%

void
yyerror (const char * errmesg) {
	if (SGBcomerr == 1) SGBcomerr = 2;
	fprintf (stderr, "\nYow!  I can't make sense out of your sentence!  Try again, please!\n\n");
}

struct word lexvals [] = {
	{ "SHOW", SHOW },
	{ "CURSET", CURSET },
	{ "CARDS", CARDS },
	{ "PLAYERS", PLAYERS},
	{ "AI", AI },
	{ "HUMAN", HUMAN },
	{ "NOLAY", NOLAY_T },
	{ "RAND", RAND_T },
	{ "SIMPLE", SIMPLE_T },
	{ "PLAY", PLAY },
	{ "SET", SET },
	{ "TEST", TEST },
	{ "LOAD", LOAD },
	{ "SAVE", SAVE },
	{ "LICENSE", LICENSE },
	{ "ABOUT", ABOUT },
	{ "QUIT", QUIT },
	{ "DEAL", DEAL },
	{ "HELP", HELP },
	{ "NAME", NAME },
	{ "PASS", PASS },
	{ "TWO", TWO_T },
	{ "TWOS", TWO_T },
	{ "THREE", THREE_T },
	{ "THREES", THREE_T },
	{ "FOUR", FOUR_T },
	{ "FOURS", FOUR_T },
	{ "FIVE", FIVE_T },
	{ "FIVES", FIVE_T },
	{ "SIX", SIX_T },
	{ "SIXS", SIX_T },
	{ "SEVEN", SEVEN_T },
	{ "SEVENS", SEVEN_T },
	{ "EIGHT", EIGHT_T },
	{ "EIGHTS", EIGHT_T },
	{ "NINE", NINE_T },
	{ "NINES", NINE_T },
	{ "TEN", TEN_T },
	{ "TENS", TEN_T },
	{ "JACK", JACK_T },
	{ "JACKS", JACK_T },
	{ "QUEEN", QUEEN_T },
	{ "QUEENS", QUEEN_T },
	{ "KING", KING_T },
	{ "KINGS", KING_T },
	{ "ACE", ACE_T },
	{ "ACES", ACE_T },
	{ "JOKER", JOKER_T },
	{ "JOKERS", JOKER_T },
	{ "1", '1'},
	{ "2", '2'},
	{ "3", '3'},
	{ "4", '4'},
	{ "\n", '\n'},
	{ NULL, 0 }
};

int
yylex (void) {
	static char s[81];  /* What kind of initialization guarantees
	                     * am I given here????? */
	static return_nl_f = 0;  /* Return a \n token? */
	char * t = NULL;
	int i;

	if (return_nl_f) return return_nl_f = 0, '\n';

	while (!t) {
		t = strtok (NULL, " ");
		if (!t) {
			if (getln (s, 81) == NULL) return 0;
			t = strtok (s, " ");
			/* yyerrok; */
		}
	}
	/* This hack is necessary since we have to detect the '\n' case,
	 * but we can't use it as a delimiter in strtok(). */
	if (t[strlen(t)-1] == '\n') {
		t[strlen(t)-1] = '\0';
		return_nl_f = 1;
	}

	/* At some point, it might be nifty to add identification
	 * of partial tokens... ie. recognize SHO to mean SHOW, but
	 * for now we'll not bother.  Partly because this means messing
	 * with YACC internals...  Of course, if we never get around to
	 * adding it, then we might as well have implemented this in lex
	 * instead of straight C. */
	for (i=0; lexvals[i].s; i++)
		if (!strcasecmp (t, lexvals[i].s)) return lexvals[i].v;

	/* Couldn't find a matching token.  Perhaps it's a player's name. */
	for (i=0; i<4; i++) if (!strcasecmp (t, pnames[i])) {
		yylval.player = i + 1;
		return PNAME;
	}

	/* Okay, does it actually equal anything?  Perhaps we strtok()
	 * returned _just_ a \n and our \n checking code removed it leaving
	 * an empty string. */
	if (!strlen (t)) {
		return_nl_f = 0;
		return '\n';
	}

	/* Okay, not a player name.  Must be a string-literal. */
	yylval.string = t;
	return STRING;
}

/*
 * Returns TRUE if the char is a token otherwise recognized by the lexer.
 */
int
isToken (char * s) {
	int i;

	for (i=0; lexvals[i].s; i++)
		if (!strcasecmp (s, lexvals[i].s)) return 1;

	return 0;
}

/*
 * Outputs a little string for a little use to see!
 *
 * Call this function with a null argument (ie. message (NULL)) to
 * toggle the display of messages (this is used for, eg. the auto-
 * test undocumented function).
 */
int
message (const char * mesg, ...) {
	static on = 1;
	va_list va;
	int x;

	if (!mesg) {
		on = !on;
		return 0;
	}

	if (!on) return 0;

	va_start (va, mesg);
	x = vprintf (mesg, va);
#if 0
	printf ("\n");
#endif
	va_end (va);

	return x;
}

/*
 * This is used to print a really big string...  :)  It waits for a keypress
 * after everyline before continueing onto the next.  Make sure that the
 * newlines are embedded.
 */
void
bigmessage (const char * mesg) {
	printf ("\nPress ENTER to print the next line, and continue\n"
	        "doing so until you have read the whole following message!"
	        "\n\n");
	while (*mesg) {
		while (*mesg != '\n') {
			putchar (*mesg);
			mesg++;
		}
		getchar();
		mesg++;
	}
}

/*
 * Like fgets() except that this will print out a nice prompt
 * explaining whose turn it is first, and it will assume that you
 * wish to use the stdin stream.
 */
char *
getln (char * s, int count) {
	if (ass_turn (SGBgame))
		printf ("(%s %s #%d) >>",
		    placename (ass_PlayerPos (
		    ass_GamePlayer (SGBgame, ass_turn (SGBgame)))),
		    pnames[ass_turn(SGBgame)-1], SGBplaysinround);
	else
		printf ("(deal) >>");

	return fgets (s, count, stdin);
}

/*
 * This will save the game to file.  It's primary use is to save the
 * gamestate so that if we find an error, we can just load the game
 * as saved from the last known good turn and watch what happens using
 * the debugger.
 */
void
savegame (const char * file) {
 	FILE * sg;

	/* Look-out!  We overwrite pretty easily...  :) */
	sg = fopen (file, "wb");
	fwrite (SGBgame, sizeof(*SGBgame), 1, sg);
	fwrite (&SGBplaysinround, sizeof(SGBplaysinround), 1, sg);
	fwrite (pai, sizeof (struct ai_s), 4, sg);
	fwrite (ptypes, sizeof (enum ptype), 4, sg);
	fwrite (pnames, NAMELEN + 1, 4, sg);
	fclose (sg);
}

/*
 * This will load a game from file.  You'd better make sure that the
 * file is a legit game...  :)
 */
void
loadgame (const char * file) {
	FILE * sg;

	if (!(sg = fopen (file, "rb"))) {
		fprintf (stderr, "%s\n", strerror (errno));
		exit(1);
	}
	if (!(fread (SGBgame, sizeof(*SGBgame), 1, sg))) {
		fprintf (stderr, "%s\n", strerror (errno));
		exit(1);
	}
	if (!fread (&SGBplaysinround, sizeof(SGBplaysinround), 1, sg)) {
		fprintf (stderr, "%s\n", strerror (errno));
		exit(1);
	}
	if (!fread (pai, sizeof (struct ai_s), 4, sg)) {
		fprintf (stderr, "%s\n", strerror (errno));
		exit(1);
	}
	if (!fread (ptypes, sizeof (enum ptype), 4, sg)) {
		fprintf (stderr, "%s\n", strerror (errno));
		exit(1);
	}
	if (!fread (pnames, NAMELEN + 1, 4, sg)) {
		fprintf (stderr, "%s\n", strerror (errno));
		exit(1);
	}
	if (fclose (sg)) {
		fprintf (stderr, "%s\n", strerror (errno));
		exit(1);
	}
}

/*
 * Do the AI, thing, eh?
 *
 * The ai_notify() and ai_watch() stuff is done elsewhere.
 */
void
play_ai (void) {
	set_t s;
	int i;

	/* While the person who's going now is an AI, go them. */
	for (; ass_turn(SGBgame) && ptypes[ass_turn(SGBgame)-1] == tAI; ) {
		int goer;

		goer = ass_turn (SGBgame);

		/* Possibly clear the set */
		if (ass_a_clear (SGBgame)) {
			message ("Cleared the set!  New Round!\n");
			SGBplaysinround = 1; /* should be 0??  No, it seems to
			                      * work quite well at 1...  :) */
		}

		/* Find out what the AI wants to do */
		s = ai_move (&pai[goer-1],
		    ass_Hand (ass_GamePlayer (SGBgame, goer)),
		    ass_GameDown (SGBgame));
		assert (!ass_move (SGBgame, s));  /* Make the actual move */

		/* Record the move with any other AIs */
		for (i=0; i<4; i++) if (ptypes[i] == tAI &&
		  i != goer - 1)
			ai_watch (&pai[i], s, goer);

		/* Notify the user of the move */
		if (s->card != NAC) {
			message ("AI: %s %s laid %d %s%s\n",
			    placename(ass_PlayerPos (ass_GamePlayer (SGBgame,
			      goer))),
			    pnames[goer-1],
			    s->num,
			    cardname(s->card),
			    s->num == 1 ? "" : "s");
		} else {
			message ("AI: %s %s passed\n",
			    placename (ass_PlayerPos (ass_GamePlayer (SGBgame,
			      goer))),
			    pnames[goer-1]);
		}

		SGBplaysinround++;
	}

	if (ass_a_clear (SGBgame)) {
		message ("Cleared the set!  New Round!\n");
		SGBplaysinround = 1;
	}
}


int
main (char *c, char **v) {
	int i;

	printf ("TVP, prez CLI, Copyright (C) 1997, Timothy M. Vanderhoek\n"
	        "This is distributed under the GNU GPL.  Please type\n"
	        "``SHOW LICENSE'' for more information on the license and\n"
	        "warranty.  Type ``HELP'' to see a command summary.\n");
	SGBgame = malloc (sizeof (*SGBgame));
	SGBplaysinround = 1;
	ass_init (SGBgame);

	for (i=0; i<4; i++) ptypes[i] = tHUMAN;

	/* These will only actually get used if someone is made an AI */
	for (i=0; i<4; i++) ai_init (&pai[i], i+1, SIMPLE);

	strcpy (pnames[0], "Jane");
	strcpy (pnames[1], "John");
	strcpy (pnames[2], "Jeff");
	strcpy (pnames[3], "Joan");

	exit (yyparse());
}
