#!/bin/sh
#
#     __  ___          __      __   
#    /  |/  /___  ____/ /_  __/ /__ 
#   / /|_/ / __ \/ __  / / / / / _ \
#  / /  / / /_/ / /_/ / /_/ / /  __/
# /_/  /_/\____/\__,_/\__,_/_/\___/ 
#     ______                           __           
#    / ____/___  ____ _   _____  _____/ /____  _____
#   / /   / __ \/ __ \ | / / _ \/ ___/ __/ _ \/ ___/
#  / /___/ /_/ / / / / |/ /  __/ /  / /_/  __/ /    
#  \____/\____/_/ /_/|___/\___/_/   \__/\___/_/     
#
#                                               V 1.0
#
# 26 Aug 2004 Giulio Canevari
#
# A simple way to encode a module to a sampled audio file format
#
# This program is free software, released under GPL;
# see http://www.gnu.org to read the license (GPL)
#
# Thanks to :
#  Heikki Orsila and Michael Doering for the UADE program and for help
#  Ralf Hoffmann for his precious suggestions about shell-scripting
#  Juergen Mueller for the soxexam man page
#  Matley for help
#
# Known Problems:
#  1)the fade out/fade effect is added every time to EFFECT, so it works as intented only for the first file
#  The problem is that i have to toggle the fade part from $EFFECT and readd it every subsongs or file,
#  because it contains the time-stop value that is subsong/file dependent
#  2)Using a fixed arithmetic, something it steals some time ( less than 1 sec ) to the end of file.


COMMENT="UADE using ModuleConverter"
UADEOPTIONS="-pan 0.8 -one -sit 7"

if [ -n "$UADEEXTRAOPTIONS" ]; then
    UADEOPTIONS=" $UADEOPTIONS $UADEEXTRAOPTIONS "
    echo "Adding $UADEEXTRAOPTIONS to UADEOPTIONS"
fi

function uade_convert ()
{
    create_rawtmp
    
    if [ "$MULTI" = false ]; then
        uade $UADEOPTIONS "$MODULE" -outpipe 1 > "$TF"
    fi
    if [ "$MULTI" = true ]; then
        uade $UADEOPTIONS "$MODULE" -sub $SUBSONGNUMBER  -outpipe 1 > "$TF"
    fi
    
    create_wavetmp
    
    if [ "$FADEOUT" = true ]; then
        get_raw_length
        effect_fade_out
        FADEOUTADDED="true"
    fi
    
    if [ "$FADE" = true ]; then
        get_raw_length
        effect_fade
        FADEADDED="true"
    fi
    
    sox_convert
    rm -f "$TF"
    
    if [ -z "$NORMALIZE" ]; then
        option_norm
    fi
    if [ "$NORMALIZE" = true ]; then
        normalize "$WAVE"
    fi
}

function encode_file ()
{
    if [ "$MULTI" = false ]; then
        OUTPUT=`basename "$MODULE"`
    fi
    if [ "$MULTI" = true ]; then
        OUTPUT=`basename "$MODULE"`-sub$SUBSONGNUMBER
    fi
    
    case "$MODE" in
        flac)
            if [ ! -e "Track-$OUTPUT.flac" ]; then
                flac --best -T title="$MODULENAME" -T tracknumber="$SUBSONGNUMBER" "$WAVE" -o "Track-$OUTPUT.flac"
            else
                flac --best -T title="$MODULENAME" -T tracknumber="$SUBSONGNUMBER" "$WAVE" -o "Track-$OUTPUT-$$-$RANDOM.flac"
            fi
            ;;
        mp3)
            if [ ! -e "Track-$OUTPUT.mp3" ]; then
                lame -b 192 --tt "$MODULENAME" --tc "$COMMENT" --tn "$SUBSONGNUMBER" "$WAVE" "Track-$OUTPUT.mp3"
            else
                lame -b 192 --tt "$MODULENAME" --tc "$COMMENT" --tn "$SUBSONGNUMBER" "$WAVE" "Track-$OUTPUT-$$-$RANDOM.mp3"
            fi
            ;;
        ogg)
            if [ ! -e "Track-$OUTPUT.ogg" ]; then
                oggenc -q 6 "$WAVE" -t "$MODULENAME" -N "$SUBSONGNUMBER" -o "Track-$OUTPUT.ogg"
                # how does -c "$COMMENT" work?
            else
                oggenc -q 6 "$WAVE" -t "$MODULENAME" -N "$SUBSONGNUMBER" -o "Track-$OUTPUT-$$-$RANDOM.ogg"
            fi
            ;;
        cdr)
            if [ ! -e "Track-$OUTPUT.cdr" ]; then
                sox "$WAVE" "Track-$OUTPUT.cdr"
            else
                sox "$WAVE" "Track-$OUTPUT-$$-$RANDOM.cdr"
            fi
            ;;
        wave)
            if [ ! -e "Track-$OUTPUT.wav" ]; then
                mv "$WAVE" "Track-$OUTPUT.wav"
            else
                mv "$WAVE" "Track-$OUTPUT-$$-$RANDOM.wav"
            fi
            ;;
    esac
    
    delete_wavetmp
}

function sox_convert ()
{
    sox  -t  raw -r 44100 -c 2 -s -w "$TF" "$WAVE" $EFFECT
}

function usage ()
{
    echo "Usage: `basename $0` [OPTIONS] [FILE1] [FILE2] ..."
    echo "Options:"
    echo "--help        Display this help text and exit"
    echo "--version     Display version info and exit."
    echo
    echo "--ogg         Use ogg as file output [ default ]."
    echo "--mp3         Use mp3 as file output."
    echo "--flac        Use flac as file output."
    echo "--wave        Use wave as file output."
    echo "--cdr         Use cdr as file output."
    echo
    echo "--norm        File will be normalized [ default ]."
    echo "--no-norm     File won't be normalized."
    echo
    echo "--mountains   Adds a mountains effect."
    echo "--hall        Adds a hall effect."
    echo "--garage      Adds a garage effect."
    echo "--chorus      Adds a chorus effect."
    echo
    echo "--fade-in     Adds a fade-in effect."
    echo "--fade-out    Adds a fade-out effect.(buggy)"
    echo "--fade        Adds a fade (in&out) effect.(buggy)"
    echo
    echo "Some influential environment variables:"
    echo "UADEEXTRAOPTIONS      See man uade for uade extra options"
    echo
    echo "Examples:"
    echo "export UADEEXTRAOPTIONS=\"-P /usr/local/share/uade/players/PTK-Prowiz -st 60 -fi\""
    echo "`basename $0` --flac --mountains --no-norm Statix/p6x.trsi_statix_intro"
    echo
    echo "To reset UADEEXTRAOPTIONS : export UADEEXTRAOPTIONS=\"\""

}

function print_version ()
{
    echo "Module Converter 1.0"
    echo "26 August 2004 Giulio Canevari"
}

function check_programs ()
{
    for p in $PROGS; do
        if [ -z "`which $p 2>/dev/null`" ]; then
            echo "You need $p to run this script!"
            exit 5
        fi
    done
}

function mode_flac ()
{
    MODE="flac"
    PROGS="uade sox flac"
    check_programs
}

function mode_mp3 ()
{
    MODE="mp3"
    PROGS="uade sox lame"
    check_programs
}

function mode_wave ()
{
    MODE="wave"
    PROGS="uade sox"
    check_programs
}

function mode_ogg ()
{
    MODE="ogg"
    PROGS="uade sox oggenc"
    check_programs
}

function mode_cdr ()
{
    MODE="cdr"
    PROGS="uade sox"
    check_programs
}

function option_norm ()
{
    NORMALIZE="true"
    PROGS="normalize"
    check_programs
}

function option_nonorm ()
{
    NORMALIZE="false"
}

function effect_mountains ()
{
    EFFECT="$EFFECT echo 0.8 0.9 1000.0 0.3"
}

function effect_hall ()
{
    EFFECT="$EFFECT reverb 1.0 600.0 180.0 200.0 220.0 240.0"
}

function effect_garage ()
{
    EFFECT="$EFFECT echos 0.8 0.7 40.0 0.25 63.0 0.3"
}

function effect_chorus ()
{
    EFFECT="$EFFECT chorus 0.7 0.9 55.0 0.4 0.25 2.0 -t"
}

function effect_fade_in ()
{
    EFFECT="$EFFECT fade t 2"
}

function effect_fade_out ()
{
    get_raw_length
    EFFECT="$EFFECT fade t 0 $STOPTIME 2"
}

function effect_fade ()
{
    get_raw_length
    EFFECT="$EFFECT fade t 2 $STOPTIME 2"
}

function get_raw_length ()
{
    RAWTMPFILENAME=`basename $TF`
    STOPTIME=$(( `ls -l "$TMP" |grep $RAWTMPFILENAME |awk '{ print $5 }' ` / 176400))
}

function check_tmp ()
{
    if [ -z "$TMP" ]; then
        TMP=/tmp
    fi
}

function create_rawtmp ()
{
    TF="$TMP/ModuleConverter-$$-$RANDOM.raw"
    if [ ! -e "$TF" ]; then
        touch "$TF"
        chmod 0600 "$TF"
    fi
}

function create_wavetmp ()
{
    WAVE=$TMP/`basename $TF .raw`.wav
    touch "$WAVE"
    chmod 0600 "$WAVE"
}

function create_infotmp ()
{
    TMPINFO="$TMP/ModuleConverter-$$-$RANDOM.txt"
    if [ ! -e "$TMPINFO" ]; then
        touch "$TMPINFO"
        chmod 0600 "$TMPINFO"
    fi
}

function delete_wavetmp ()
{
    if [ -e "$WAVE" ]; then
        rm -f "$WAVE"
    fi
}

function get_module_info ()
{
    create_infotmp

    extract_player_name
    
    if [ "$PLAYERNAME" ]; then
        uade -P "$PLAYERNAME" -t 3 "$MODULE" 2> "$TMPINFO" -outpipe 1 > /dev/null
    else
        uade -t 3 "$MODULE" 2> "$TMPINFO" -outpipe 1 > /dev/null
    fi
    
    MINSUBSONG=`cat $TMPINFO |grep 'uade: subsong' |sed -e "s|uade: subsong info: minimum: \(.*\) maximum: \(.*\) current: \(.*\)|\1|"`
    MAXSUBSONG=`cat $TMPINFO |grep 'uade: subsong' |sed -e "s|uade: subsong info: minimum: \(.*\) maximum: \(.*\) current: \(.*\)|\2|"`
    MODULENAME=`cat $TMPINFO |grep "modulename"| sed 's/.*modulename: //'`
    
    rm $TMPINFO
    
    if [ -z "$MODULENAME" ]; then
        MODULENAME="`basename $MODULE`"
    fi
    
    NUMBEROFSUBSONGS=$[MAXSUBSONG - MINSUBSONG +1 ]
}

function extract_player_name ()
{
    PLAYERNAME=`echo $UADEOPTIONS |grep "\-P" |sed 's/.*-P //' |awk '{ print $1}'`
}

if [ "$#" -lt "1" ]; then
    print_version
    echo
    usage
    exit 1
fi

while [ "$1" != "" ] ; do
	case $1 in
        --help|-h)
            usage
            exit 0
        ;;
        --version|-v)
            print_version
            exit 0
        ;;
        --flac)
            mode_flac
        ;;
        --mp3)
            mode_mp3
        ;;
        --wave)
            mode_wave
        ;;
        --ogg)
            mode_ogg
        ;;
        --cdr)
            mode_cdr
        ;;
        --norm)
            option_norm
        ;;
        --no-norm)
            option_nonorm
        ;;
        --hall)
            effect_hall
        ;;
        --mountains)
            effect_mountains
        ;;
        --garage)
            effect_garage
        ;;
        --chorus)
            effect_chorus
        ;;
        --fade-in)
            effect_fade_in
        ;;
        --fade-out)
            FADEOUT="true"
        ;;
        --fade)
            FADE="true"
        ;;
        *)
        
            if [ -e "$1" ]; then
            
                if [ -z "$MODE" ]; then
                    mode_ogg
                fi  
                
                MODULE="$1"
                SUBSONGNUMBER="1"
                
                check_tmp
                get_module_info
                
                if [ "$NUMBEROFSUBSONGS" -lt "2" ]; then
                    MULTI="false"
                    uade_convert
                    encode_file
                else
                    MULTI="true"
                    for SUBSONGNUMBER in `seq "$MINSUBSONG" "$MAXSUBSONG" `
                        do
                            uade_convert
                            encode_file
                    done
                fi
            
            else
                echo $i : File not found!
            fi
        ;;
    esac
    shift
done
