#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#if (WIN32|WINNT)
#include <io.h>
#include <direct.h>
#else
#include <unistd.h>
#include <sys/wait.h>
#endif
#include <fcntl.h>
#include <sys/stat.h>

#include "udm_utils.h"
#include "udm_parser.h"


#define PARSER_COMMENT 	  '#'
#define PARSER_DELIMITERS " \t\n\r"
#define PARSER_QUOTES     "\""
#define MAXPARSERS 256
#define MAXARGV 16

typedef struct parser_struct{
        char *from_mime;
	char *to_mime;
        char *parser;
	char **argv;
	int fileposition;	
	char *charset;
} UDM_PARSER;

static UDM_PARSER parsers [MAXPARSERS];
static int nparsers=0;

int UdmAddParser(char *buffer){
	char *mimes;
	char *command;
	char *token;
	char *lt;
	char *local_buffer;
	int pos;
	UDM_PARSER *parser;

	/* Now we should process following format
	<from_mime> <to_mime>[;encoding] ["command line [$1]"] */

	parser=parsers+nparsers;
	if(!(mimes=UdmGetToken(buffer,PARSER_QUOTES,&lt))) return -1;
	for(pos=0;mimes[pos];pos++)
		if(mimes[pos]==';') mimes[pos]=' ';
	command=UdmGetToken(NULL,PARSER_QUOTES,&lt);
	if((parser->from_mime=UdmGetToken(mimes,PARSER_DELIMITERS,&lt)))
		parser->from_mime=strdup(parser->from_mime);
	if((parser->to_mime  =UdmGetToken(NULL,PARSER_DELIMITERS,&lt)))
		parser->to_mime  =strdup(parser->to_mime);
	if((parser->charset  =UdmGetToken(NULL,PARSER_DELIMITERS,&lt)))
		parser->charset  =strdup(parser->charset);
	
		
	/* Let's create argv[] for each parser */
	if(command) {
	    local_buffer=(char*)malloc(sizeof(char*)*UDMSTRSIZ);
	    strncpy(local_buffer,command,UDMSTRSIZ);
	    if((parser->parser=UdmGetToken(local_buffer,PARSER_DELIMITERS,&lt)))
		parser->parser=strdup(parser->parser);
	    parser->argv=(char**)malloc(sizeof(char*)*MAXARGV);
	    parser->fileposition=0; pos=1;
	    parser->argv[0]=parser->parser;
	    do{
		if((token=UdmGetToken(NULL,PARSER_DELIMITERS,&lt))){
			parser->argv[pos]=strdup(token);
			if(!strcmp(token,"$1")) parser->fileposition=pos;
		} else parser->argv[pos]=token;
	    }while(token && ++pos<MAXARGV);
	} else parser->parser=NULL;
	nparsers++;
	return 0;
}




#ifdef USE_PARSER

/* This function executes an external parser and sends buf to it's STDIN
   It returns STDOUT of parser. */

char *parse(char *buf, int buflen, char *parser, char *argv[], int maxlen){
    char *result = NULL;
    int wr[2];
    int rd[2];    
    int n;
    pid_t pid;    
	/*printf("'%s' '%s' '%s'\n",parser,argv[0],argv[1]);*/

    /* Create write and read pipes */
    pipe(wr);
    pipe(rd);
    
    /* Fork a clild */
    pid = fork();
    
    if (pid>0){

        char string[UDMSTRSIZ];
	result=buf;

	/* Close other pipe ends */
	close (wr[0]);
	close (rd[1]);

	/* Send string to be parsed */
	if(buflen)write(wr[1], buf, buflen);
	close(wr[1]);
	memset(result,0,buflen);
	memset(string,0,sizeof(string));
	while ((n=read(rd[0],string,sizeof(string)-1))>0){
	    strncat(result,string,maxlen-strlen(result));
	    memset(string,0,sizeof(string));
	}
	wait(NULL); /* inserted by Manfred Bathelt */
    } 
	
    if (pid==0){
	/* Close other pipe ends */
	close (wr[1]);
	close (rd[0]);
	
	/* Connect pipe to stdout */
	if (rd[1]!=STDOUT_FILENO){
	    dup2(rd[1],STDOUT_FILENO);
	    close(rd[1]);
	}

	/* Connect pipe to stdin */
	if (wr[0]!=STDIN_FILENO){
	    dup2(wr[0],STDIN_FILENO);
	    close(wr[0]);
	}
	execvp(parser,argv);
	
    }
    return result;
}


/*
 Some parsers need a file to perform a seek. So, we just create temporary file,
 parse it and return STDOUT as a result. Result should be freed
 To be binary-safe, we pass length parameter of buffer
*/

char *parse_file (char *buf, int length, int parsernum, int maxlen){
    int fd;
    char *filename="";
    char *result;
    char *buffer;

    /* Create temporary file */
    buffer=buf;
    if(parsers[parsernum].fileposition){
	filename=tmpnam(NULL);
	umask(022);
	chdir("/tmp");
	fd=open(filename,O_RDWR|O_CREAT,0600);

	/* Write to the temporary file */
	write(fd,buf,length);
	close(fd);
    
	/* Place file name into the argument list */
	parsers[parsernum].argv[parsers[parsernum].fileposition]=filename;

	*buffer=0;
	length=0; /* We should not pass anything to STDIN if file is used */
    }

    /* Call external parser with buffer as STDIN */
    result=parse(buffer,length,parsers[parsernum].parser,parsers[parsernum].argv,maxlen);
    
    /* Remove temporary file */
    if(parsers[parsernum].fileposition) unlink(filename);

    return result;
}

char *UdmExecParser(char *from_mime, int *to_mime, char *buf, int length, int maxlen){
    int i;
    
    for(i=0;i<nparsers;i++){
	if(!strcmp(parsers[i].from_mime,from_mime)){
            *to_mime=i;
	    if(parsers[i].parser)
		return parse_file(buf,length,i,maxlen);
	    else
		return buf;
	}
    }
    return NULL;
}

char * UdmParserParam(int mime, int type){
	switch(type){
		case UDM_MIME_FROM:
			return(parsers[mime].from_mime);
		case UDM_MIME_TO:
			return(parsers[mime].to_mime);
		case UDM_MIME_CHARSET:
			return(parsers[mime].charset);
		default:
			return ("");
	}
}


#endif
