#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#if (WIN32|WINNT)
#include <winsock.h>
#include <time.h>
#include <direct.h>
#else
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <netdb.h>
#ifdef HAVE_SELECT_H
#include <select.h>
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#endif
#include <fcntl.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <regex.h>

#include "udm_common.h"
#include "udm_proto.h"
#include "udm_utils.h"
#include "udm_mutex.h"

#ifndef INADDR_NONE
#define INADDR_NONE (unsigned long)(-1)
#endif

char *MirrorRoot = NULL;
char *MirrorHeadersRoot = NULL;
static int NET_BUF_SIZE=10240;


/***************  AddType stuff **********************/
static int mmimes=0;
static int nmimes=0;

typedef struct mime_struct {
	regex_t	reg;
	char	*mime_type;
} MIME;
static MIME * Mime=NULL;

int UdmAddType(char * mime_type,char *reg, char *errstr){
#define ERRSTRSIZE 100
char regerrstr[ERRSTRSIZE];
int err;
	if(nmimes>=mmimes){
		if(mmimes){
			mmimes+=16;
			Mime=(MIME *)realloc(Mime,mmimes*sizeof(MIME));
		}else{
			mmimes=16;
			Mime=(MIME *)malloc(mmimes*sizeof(MIME));
		}
	}
	Mime[nmimes].mime_type=strdup(mime_type);
	err=regcomp(&(Mime[nmimes].reg),reg,REG_EXTENDED|REG_ICASE);
	if(err){
		regerror(err,&(Mime[nmimes].reg),regerrstr,ERRSTRSIZE);
		sprintf(errstr,"Wrong regex in config file: %s: %s", reg,regerrstr);
		free(Mime[nmimes].mime_type);
		return(1);
	}
	nmimes++;
	return(0);
}
void UdmFreeTypes(){
int i;
	for(i=0;i<nmimes;i++){
		free(Mime[i].mime_type);
		regfree(&(Mime[i].reg));
	}
	nmimes=mmimes=0;
	UDM_FREE(Mime);
}
static char * ContentType(char *url){
int i;
#define NS 10
regmatch_t subs[NS];
int err;
	for(i=0;i<nmimes;i++){
		err=regexec(&(Mime[i].reg),url,NS,subs,0);
		if(!err)return(Mime[i].mime_type);
	}
	return(NULL);
}



/************************* Mirroring stuff ********************/
int UdmBuild(char *path, int omode) {
	struct stat sb;
	mode_t numask, oumask;
	int first, last, retval;
	char *p;

	p = path;
	oumask = 0;
	retval = 0;
	if (p[0] == UDMSLASH)		/* Skip leading slashes. */
		++p;
	for (first = 1, last = 0; !last ; ++p) {
		if (p[0] == '\0')
			last = 1;
		else if (p[0] != UDMSLASH)
			continue;
		*p = '\0';
		if (p[1] == '\0')
			last = 1;
		if (first) {
			/*
			 * POSIX 1003.2:
			 * For each dir operand that does not name an existing
			 * directory, effects equivalent to those cased by the
			 * following command shall occcur:
			 *
			 * mkdir -p -m $(umask -S),u+wx $(dirname dir) &&
			 *    mkdir [-m mode] dir
			 *
			 * We change the user's umask and then restore it,
			 * instead of doing chmod's.
			 */
			oumask = umask(0);
#if (WIN32||WINNT)
			numask = oumask;
#else
			numask = oumask & ~(S_IWUSR | S_IXUSR);
#endif
			(void)umask(numask);
			first = 0;
		}
		if (last)
			(void)umask(oumask);
		if (stat(path, &sb)) {
			if (errno != ENOENT ||
#if (WIN32||WINNT)
			    mkdir(path) < 0
#else
			    mkdir(path, last ? omode :
				  S_IRWXU | S_IRWXG | S_IRWXO) < 0
#endif
				 ){
				/* warn("%s", path); */
				retval = 1;
				break;
			}
		}
		else if ((sb.st_mode & S_IFMT) != S_IFDIR) {
			if (last)
				errno = EEXIST;
			else
				errno = ENOTDIR;
			/* warn("%s", path); */
			retval = 1;
			break;
		}
		if (!last)
			*p = UDMSLASH;
	}
	if (!first && !last)
		(void)umask(oumask);
	return (retval);
}


int   UdmMirrorGET(char *schema, char *hostname, char *path,
		char *filename, char *buf, int maxsize,
		int days, char *errstr) {

	int size = 0, l;
	int fbody, fheader;
	char str[UDMSTRSIZ]="";
	char *content_type = NULL;
	struct stat sb;
	time_t nowtime;
	int have_headers=0;

	nowtime = time(NULL);

	if (days <= 0)return(0);
	
	/* MirrorRoot is not specified, nothing to do */
	if(!MirrorRoot)return(0);

	strcpy(str, MirrorRoot);strcat(str,"/");
	strcat(str, schema);strcat(str,"/");
	strcat(str, hostname);
	strcat(str, path);
	strcat(str, filename && strlen(filename) ? filename : "index.html");

	if ((fbody = open(str, O_RDONLY)) == -1){
		sprintf(errstr,"Mirror file %s not found", str);
		return UDM_MIRROR_NOT_FOUND;
	}

	/* Check on file mtime > days ? return */
	if (fstat(fbody, &sb)) {
		/* error handler */
	}
	if (nowtime > sb.st_mtime + days ) {
		close(fbody);
		sprintf(errstr,"%s is older then %d secs, retrieving", str, days);
		return UDM_MIRROR_EXPIRED;
	}

	if(MirrorHeadersRoot){
		strcpy(str, MirrorHeadersRoot);strcat(str,"/");
		strcat(str, schema);strcat(str,"/");
		strcat(str, hostname);
		strcat(str, path);
		strcat(str, filename && strlen(filename) ? filename : "index.html");
		strcat(str,".header");

		if ((fheader = open(str, O_RDONLY))>=0) {
			size = read(fheader, buf, maxsize);
			close(fheader);
			strcpy(buf + size, "\r\n\r\n");
			have_headers=1;
		}
	}
	if(!have_headers){
		/* header file not found   */
		/* get body as file method */
		sprintf(buf,"HTTP/1.0 200 OK\r\n");
		if((content_type=ContentType(filename && strlen(filename) ? filename : "index.html")))
			sprintf(UDM_STREND(buf),"Content-Type: %s\r\n",content_type);
		sprintf(UDM_STREND(buf),"\r\n");
	}

	l = strlen(buf);
	size = read(fbody, buf + l, maxsize - l) + l;
	close(fbody);
	return size;
}

int   UdmMirrorPUT(char *schema, char *hostname, char *path,
		char *filename, char *header, char *content, 
		int size, char *errstr) {

	int fd;
	char str[UDMSTRSIZ]="";

	/* Put Content if MirrorRoot is specified */
	if(MirrorRoot){
		strcpy(str, MirrorRoot);strcat(str,"/");
		strcat(str, schema);strcat(str,"/");
		strcat(str, hostname);
		strcat(str, path);

		if(UdmBuild(str, 0755) != 0){
			sprintf(errstr, "Can't create dir %s", str);
			return UDM_MIRROR_CANT_BUILD;
		}

		strcat(str,"/");
		strcat(str, filename && strlen(filename) ? filename : "index.html");
		if ((fd = open(str, O_CREAT|O_WRONLY,0644)) == -1){
			sprintf(errstr,"Can't open mirror file %s\n", str);
			return UDM_MIRROR_CANT_OPEN;
		}
		size = write(fd, content, size);
		close(fd);
	}

	/* Put Headers if MirrorHeadersRoot is specified */
	if(MirrorHeadersRoot){
		strcpy(str, MirrorHeadersRoot);strcat(str,"/");
		strcat(str, schema);strcat(str,"/");
		strcat(str, hostname);
		strcat(str, path);

		if(UdmBuild(str, 0755) != 0){
			sprintf(errstr, "Can't create mirror dir %s", str);
			return UDM_MIRROR_CANT_BUILD;
		}

		strcat(str,"/");
		strcat(str, filename && strlen(filename) ? filename : "index.html");
		strcat(str,".header");

		if ((fd = open(str, O_CREAT|O_WRONLY,0644)) == -1){
			sprintf(errstr,"Can't open mirror file %s\n", str);
			return UDM_MIRROR_CANT_OPEN;
		}
		size = write(fd, header, strlen(header));
		close(fd);
	}
	return(0);
}



/************** WinSock initialization ***********************/

#if (WIN32|WINNT)
static int isWSA = 0;
int startWSA(){
WORD wVersionRequested;
WSADATA wsaData;
	if (isWSA) return(0);
	wVersionRequested = MAKEWORD(1, 1);
	/* start up winsock services */
	if (WSAStartup(wVersionRequested, &wsaData) != 0) {
		isWSA = 0;
		return(0);
	} else {
		isWSA = 1;
		return(1);
	}
}
static void stopWSA(){
	if(isWSA){
		WSACleanup();
		isWSA = 0;
	}
}
#endif



/****************** Open TCP Connect **********************/
static int open_host(char *hostname,int port, int timeout)
{
int net;
struct hostent *host;
struct sockaddr_in sin;

#if (WIN32|WINNT)
	startWSA();
#endif

	UDMMEMZERO((char*)&sin,sizeof(sin));

	if (port){
		sin.sin_port= htons((u_short)port);
	}else{
		return(UDM_NET_ERROR);
	}

	if ((sin.sin_addr.s_addr=inet_addr(hostname)) != INADDR_NONE){
		sin.sin_family=AF_INET;
	}else{
	
		if(LockProc)LockProc(UDM_LOCK,UDM_LOCK_RESOLVE);
		/* It is not reentrant, hide this under mutex */
		host=gethostbyname(hostname);
		if(LockProc)LockProc(UDM_UNLOCK,UDM_LOCK_RESOLVE);

		if (host){
			sin.sin_family=host->h_addrtype;
			memcpy(&sin.sin_addr, host->h_addr, host->h_length);
		}else{
			return(UDM_NET_CANT_RESOLVE);
		}
	}

	net=socket(AF_INET, SOCK_STREAM, 0);

	if(connect(net, (struct sockaddr *)&sin, sizeof (sin)))
		return(UDM_NET_CANT_CONNECT);

	return(net);
}



/***************** HTTP codes and messages *************************/

char *UdmHTTPErrMsg(int code){
	switch(code){
	case 0:   return("Not indexed yet");
	case 200: return("OK");
	case 301: return("Moved Permanently");
	case 302: return("Moved Temporarily");
	case 303: return("See Other");
	case 304: return("Not Modified");
	case 300: return("Multiple Choices");
	case 305: return("Use Proxy (proxy redirect)");
	case 400: return("Bad Request");
	case 401: return("Unauthorized");
	case 402: return("Payment Required");
	case 403: return("Forbidden");
	case 404: return("Not found");
	case 405: return("Method Not Allowed");
	case 406: return("Not Acceptable");
	case 407: return("Proxy Authentication Required");
	case 408: return("Request Timeout");
	case 409: return("Conflict");
	case 410: return("Gone");
	case 411: return("Length Required");
	case 412: return("Precondition Failed");
	case 413: return("Request Entity Too Large");
	case 414: return("Request-URI Too Long");
	case 415: return("Unsupported Media Type");
	case 500: return("Internal Server Error");
	case 501: return("Not Implemented");
	case 502: return("Bad Gateway");
	case 505: return("Protocol Version Not Supported");
	case 503: return("Service Unavailable");
	case 504: return("Gateway Timeout");
	default:  return("Unknown status");
	}
}

int UdmHTTPResponseType(int status){

	switch(status){
	case 1:   return(1);
	case 200: return(UDM_HTTP_STATUS_OK);

	case 301:  /* Moved Permanently */
	case 302:  /* Moved Temporarily */
	case 303:  /* See Other */ return(UDM_HTTP_STATUS_REDIRECT);

	case 304:  /* Not Modified*/ return(UDM_HTTP_STATUS_NOT_MODIFIED);

	case 300:  /* Multiple Choices              */
	case 305:  /* Use Proxy (proxy redirect)    */
	case 400:  /* Bad Request ??? We tried ...  */
	case 401:  /* Unauthorized                   */
	case 402:  /* Payment Required              */
	case 403:  /* Forbidden                     */
	case 404:  /* Not found                     */
	case 405:  /* Method Not Allowed            */
	case 406:  /* Not Acceptable                */
	case 407:  /* Proxy Authentication Required */
	case 408:  /* Request Timeout               */
	case 409:  /* Conflict                      */
	case 410:  /* Gone                          */
	case 411:  /* Length Required               */
	case 412:  /* Precondition Failed           */
	case 413:  /* Request Entity Too Large      */
	case 414:  /* Request-URI Too Long          */
	case 415:  /* Unsupported Media Type        */
	case 500:  /* Internal Server Error         */
	case 501:  /* Not Implemented               */
	case 502:  /* Bad Gateway                   */
	case 505:  /* Protocol Version Not Supported */
		return(UDM_HTTP_STATUS_DELETE);

	case 503: /* Service Unavailable */
	case 504: /* Gateway Timeout */
		return(UDM_HTTP_STATUS_RETRY);

	default: return(UDM_HTTP_STATUS_UNKNOWN);
	}
}


/***********************************************************/


#ifdef USE_HTTP
int UdmHTTPGet(char *host,int port,char *header,
    UDM_INDEXER *Indexer, int maxsize, int read_timeout)
{
    int fd;
    fd_set sfds;
    int status, nread=0;
	    
    struct timeval tv;

    /* Connect to HTTP or PROXY server */
    if( (fd=open_host(host,port,read_timeout)) < 0 )
       return(fd);

    /* Send HTTP request */
    if( send(fd,header,strlen(header),0) < 0 )
       return -1;

    /* Retrieve response */
    tv.tv_sec = (long) read_timeout;
    tv.tv_usec = 0;

    while( 1 ){
       FD_ZERO( &sfds );
       FD_SET( fd, &sfds );

       status=select(FD_SETSIZE, &sfds, NULL, NULL, &tv);
       if( status == -1 ){
               nread=UDM_NET_ERROR;
               break;
       }else if( status == 0 ){
               nread=UDM_NET_TIMEOUT;
               break;
       }else{
           if(FD_ISSET(fd,&sfds))
           {
               status = recv(fd, Indexer->buf+nread, NET_BUF_SIZE, 0);
               if( status < 0 )
               {
                   nread = status;
                   break;
               }else if( status == 0 ){
                   nread += status;
                   break;
               }else{
                   nread += status;
                   if( (nread+NET_BUF_SIZE) >= maxsize )
                       break;
               }
           }else
               break;
       }
    }
    close(fd);
    return(nread);
}
#endif
  

#ifdef USE_NEWS
int UdmNEWSGet(char *host,int port,char *header,
	char *dist,int maxsize,int read_timeout){
int fd,size,status;
char *s,*c,*e;
char str[UDMSTRSIZ];
FILE *f,*f1;
char filename[UDMSTRSIZ];
char command[32]="";
char proto[32]="";
int a=0,b=0,from=0,to=0,headers=0;
char *email=NULL;
char *msg_id=NULL;
char *subj=NULL;
int has_content=0;
int has_if_modified=0;
char *lt;

	sscanf(header,"%s%s%s",command,str,proto);
	if(str[0]=='/')strcpy(filename,str+1);
	else	strcpy(filename,str);
	s=UdmGetToken(header,"\r\n",&lt);
	while(s){
		if(!UDM_STRNCASECMP(s,"If-Modified-Since: "))
			has_if_modified=1;
		s=UdmGetToken(NULL,"\r\n",&lt);
	}

	/* Connect to NEWS server */
	fd=open_host(host,port,read_timeout);	/* Connect           */
	if(fd<0)return(fd);			/* Could not connect */


	size=0;s=dist;
	f=fdopen(fd,"r");f1=fdopen(fd,"w");
	fgets(str,sizeof(str),f); /* Read hello string */

	/* All news groups on the server */
	if((!strcmp(filename,"/"))||(!strcmp(filename,""))){
		fprintf(f1,"list\r\n");fflush(f1);
		fgets(str,sizeof(str),f);
		sprintf(dist,"HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n<HTML><BODY>\n");
		while(fgets(str,sizeof(str),f)){
			if(str[0]=='.')break;
			if((c=strchr(str,' ')))*c=0;
			sprintf(UDM_STREND(dist),"<A HREF=\"news://%s/%s\"></A>\n",host,str);
		}
		sprintf(UDM_STREND(dist),"</BODY></HTML>\n");
	}else /* List of the articles in one news group */
	if(!strchr(filename,'@')){
		fprintf(f1,"group %s\r\n",filename);fflush(f1);
		fgets(str,sizeof(str),f);
		sscanf(str,"%d%d%d%d",&a,&b,&from,&to);
		if(a != NNTP_GROUP_OK){
			/* Newsgroup does not exist */
			status=404;
			sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
		}else{
			fprintf(f1,"xover %d-%d\r\n",from,to);fflush(f1);
			fgets(str,sizeof(str),f);
			sprintf(dist,"HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n<HTML><BODY>\n");
			while(fgets(str,sizeof(str),f)){
				if(str[0]=='.')break;
				if((e=UdmGetToken(str,"\t\r\n",&lt)))
				if((subj=UdmGetToken(NULL,"\t\r\n",&lt)))
				if((email=UdmGetToken(NULL,"\t\r\n",&lt)))
				if((e=UdmGetToken(NULL,"\t\r\n",&lt)))
				if((msg_id=UdmGetToken(NULL,"\t\r\n",&lt)));
				str[0]=0;
				if(msg_id)sscanf(msg_id,"<%[^>]s>",str);
				if(str[0] && ((strlen(dist)+strlen(host)+strlen(str)) < maxsize - 20))
					sprintf(UDM_STREND(dist),"<A HREF=\"news://%s/%s\"></A>\n",host,str);
				/* in the line above, I left 20 bytes of free */
				/* space at the end of the string for the     */
				/* closing string: */
			}
		}
		sprintf(UDM_STREND(dist),"</BODY></HTML>\n");

	}else{ /* One article */
		if(has_if_modified){
			fprintf(f1,"stat <%s>\r\n",filename);fflush(f1);
			fgets(str,sizeof(str),f);sscanf(str,"%d",&a);
			if(a==223){
				status=304; /* Not modified */
				sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}else{
				status=404; /* Not found */
				sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}
		}else{
			fprintf(f1,"article <%s>\r\n",filename);fflush(f1);
			fgets(str,sizeof(str),f);sscanf(str,"%d",&a);
			if(a==220){
				headers=1;has_content=0;
				sprintf(dist,"HTTP/1.0 200 OK\r\n");
				while(fgets(str,sizeof(str),f)){
					if(str[0]=='.')break;
					if(headers){
						if(!UDM_STRNCASECMP(str,"Content-Type:"))has_content=1;
						if((!strcmp(str,"\r\n"))||(!strcmp(str,"\n"))){
							headers=0;
							if(!has_content)strcat(dist,"Content-Type: text/plain\r\n");
						}
					}
					if((int)(strlen(dist)+strlen(str))<maxsize)
						strcat(dist,str);
				}
			}else{
				status=404;
				sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}
		}
	}
	fprintf(f1,"quit\r\n");fflush(f1);
	fclose(f);/* Close NEWS connection */
	fclose(f1);
	return(strlen(dist));
}

/* FIXME !!! */
/* nntp:// schema does not work properly yet */

int UdmNNTPGet(char *host,int port,char *header,
	char *dist,int maxsize,int read_timeout){
int fd,size,status;
char *s,*c,*e;
char str[UDMSTRSIZ];
FILE *f,*f1;
char filename[UDMSTRSIZ];
char command[32]="";
char proto[32]="";
int a=0,b=0,from=0,to=0,headers=0;
int has_content=0;
int has_if_modified=0;
char *lt;

	sscanf(header,"%s%s%s",command,str,proto);
	if(str[0]=='/')strcpy(filename,str+1);
	else	strcpy(filename,str);

	s=UdmGetToken(header,"\r\n",&lt);
	while(s){
		if(!UDM_STRNCASECMP(s,"If-Modified-Since: "))has_if_modified=1;
		s=UdmGetToken(NULL,"\r\n",&lt);
	}

	/* Connect to NNTP server */
	fd=open_host(host,port,read_timeout);	/* Connect           */
	if(fd<0)return(fd);			/* Could not connect */

	size=0;s=dist;
	f=fdopen(fd,"r");f1=fdopen(fd,"w");
	fgets(str,sizeof(str),f); /* Read hello string */

	/* All news groups on the server */
	if((!strcmp(filename,"/"))||(!strcmp(filename,""))){
		fprintf(f1,"list\r\n");fflush(f1);
		fgets(str,sizeof(str),f);
		sprintf(dist,"HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n<HTML><BODY>\n");
		while(fgets(str,sizeof(str),f)){
			if(str[0]=='.')break;
			if((c=strchr(str,' ')))*c=0;
			sprintf(UDM_STREND(dist),"<A HREF=\"nntp://%s/%s\"></A>\n",host,str);
		}
		sprintf(UDM_STREND(dist),"</BODY></HTML>\n");
	}else /* List of the articles in one news group */
	if(!strchr(filename,'@')){
		fprintf(f1,"group %s\r\n",filename);fflush(f1);
		fgets(str,sizeof(str),f);
		sscanf(str,"%d%d%d%d",&a,&b,&from,&to);
		fprintf(f1,"xover %d-%d\r\n",from,to);fflush(f1);
		fgets(str,sizeof(str),f);
		sprintf(dist,"HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n<HTML><BODY>\n");
		while(fgets(str,sizeof(str),f)){
			if(str[0]=='.')break;
			if((e=UdmGetToken(str,"\t\r\n",&lt)))
				sprintf(UDM_STREND(dist),
				"<A HREF=\"nntp://%s/%s/%s\"></A>\n",
				host,filename,e);
		}
		sprintf(UDM_STREND(dist),"</BODY></HTML>\n");
	}else{ /* One article */
		if(has_if_modified){
			fprintf(f1,"stat <%s>\r\n",filename);fflush(f1);
			fgets(str,sizeof(str),f);sscanf(str,"%d",&a);
			if(a==223){
				status=304; /* Not modified */
				sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}else{
				status=404; /* Not found */
				sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}
		}else{
			fprintf(f1,"article <%s>\r\n",filename);fflush(f1);
			fgets(str,sizeof(str),f);sscanf(str,"%d",&a);
			if(a==220){
				headers=1;has_content=0;
				sprintf(dist,"HTTP/1.0 200 OK\r\n");
				while(fgets(str,sizeof(str),f)){
					if(str[0]=='.')break;
					if(headers){
						if(!UDM_STRNCASECMP(str,"Content-Type:"))has_content=1;
						if((!strcmp(str,"\r\n"))||(!strcmp(str,"\n"))){
							headers=0;
							if(!has_content)strcat(dist,"Content-Type: text/plain\r\n");
						}
					}
					if((int)(strlen(dist)+strlen(str))<maxsize)
						strcat(dist,str);
				}
			}else{
				status=404;
				sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}
		}
	}
	fprintf(f1,"quit\r\n");fflush(f1);
	fclose(f);/* Close NNTP connection */
	fclose(f1);
	return(strlen(dist));
}
#endif


#ifdef USE_FILE
int UdmFILEGet(char *header,char *dist,int maxsize){
int fd,size,l,status=0,is_dir;
struct stat sb,sb1;
DIR * dir;
struct dirent *rec;
char filename[UDMSTRSIZ];
char newfilename[UDMSTRSIZ];
char openname[UDMSTRSIZ];
char command[32]="";
char proto[32]="";
char *content_type=NULL,*s;

	sscanf(header,"%s%s%s",command,filename,proto);
	strcpy(newfilename,filename);
	UdmUnescapeCGIQuery(openname,newfilename);

	s=openname;
#if (WIN32|WINNT)
	while(*s){
		if(*s=='/')*s='\\';
		s++;
	}
	if ((strlen(openname)>3)&&(openname[strlen(openname)-1]=='\\'))
		openname[strlen(openname)-1]='\0';
#endif
	if(stat(openname,&sb)){
		switch(errno){
			case ENOENT: status=404;break; /* Not found */
			case EACCES: status=403;break; /* Forbidden*/
			default: status=500;
		}
		sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
		return(strlen(dist));
	}

	/* If directory is given without ending "/"   */
	/* we must redirect to the same URL with "/"  */
	if((sb.st_mode&S_IFDIR)&&(filename[strlen(filename)-1]!='/')){
		status=301;
		sprintf(dist,"HTTP/1.0 %d %s\r\nLocation: file:%s/\r\n\r\n",status,UdmHTTPErrMsg(status),filename);
		return(strlen(dist));
	}
	
	if(sb.st_mode&S_IFDIR){
		if((dir=opendir(openname))){
			strcpy(dist,"HTTP/1.0 200 OK\r\nContent-type: text/html\r\n\r\n<HTML><BODY>\n");
			while((rec = readdir (dir))){
				char escaped_name[UDMSTRSIZ]="";
				char *s,*e;
				/*
				This does not work on Solaris
				is_dir=(rec->d_type==DT_DIR);
				*/
				sprintf(newfilename,"%s%s",filename,rec->d_name);
				if(stat(newfilename,&sb1)){
					/* FIXME!!! Handler should be here */
				}
				is_dir=((sb1.st_mode&S_IFDIR)>0);

				e=escaped_name;
				for(s=rec->d_name;*s;s++){
					if(strchr(" %&<>+[](){}/?#'\"\\;,",*s)){
						sprintf(e,"%%%X",(int)*s);
						e+=3;
					}else{
						*e=*s;
						e++;
					}
				}
				*e=0;
				sprintf(UDM_STREND(dist),"<A HREF=\"%s%s\">%s%s</A>\n",
					escaped_name,is_dir?"/":"",
					escaped_name,is_dir?"/":"");
			}
			closedir(dir);
			strcpy(UDM_STREND(dist),"</BODY><HTML>\n");
			return(strlen(dist));
		}else{
			switch(errno){
				case ENOENT: status=404;break; /* Not found */
				case EACCES: status=403;break; /* Forbidden*/
				default: status=500;
			}
			sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			return(strlen(dist));
		}
	}else{
		if((fd=open(openname,O_RDONLY))<0){
			switch(errno){
				case ENOENT: status=404;break;
				case EACCES: status=403;break;
				default: status=1;
			}
			sprintf(dist,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			return(strlen(dist));
		}
		sprintf(dist,"HTTP/1.0 200 OK\r\n");
		if((content_type=ContentType(filename)))
			sprintf(UDM_STREND(dist),"Content-Type: %s\r\n",content_type);
		sprintf(UDM_STREND(dist),"\r\n");
		l=strlen(dist);
		size=read(fd,dist+l,maxsize-l)+l;
		close(fd);
		return(size);
	}
}
#endif
