/***************************************************************************
       introspect.cpp  --  lists information about available methods

    copyright            : (C) 2002-2006 by Ewald Arnold
    email                : ulxmlrpcpp@ewald-arnold.de

    $Id: introspect.cpp,v 1.24 2006/04/27 15:37:51 ewald-arnold Exp $

 ***************************************************************************/

/**************************************************************************
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ***************************************************************************/

#include <ulxmlrpcpp/ulxmlrpcpp.h>  // always first header

#include <ulxmlrpcpp/ulxr_tcpip_connection.h> // first, don't move: important msvc #include bug
#include <ulxmlrpcpp/ulxr_requester.h>
#include <ulxmlrpcpp/ulxr_value.h>
#include <ulxmlrpcpp/ulxr_except.h>
#include <ulxmlrpcpp/ulxr_http_protocol.h>

using namespace ulxr;

#include <iostream>


int main(int argc, char **argv)
{
  try
  {
    if (argc != 2)
    {
      ULXR_CERR << ULXR_PCHAR("usage: ") << argv[0]
                << ULXR_PCHAR(" server:port/resource\n")
                << ULXR_PCHAR("  E.g.: myhost.com:80/rpc/server.php\n")
                << ULXR_PCHAR("\n");
      return 1;
    }

    unsigned port = 80;
    ulxr::CppString resource = ULXR_PCHAR("/");
    ulxr::CppString server = ULXR_GET_STRING(argv[1]);
    std::size_t pos;
    if ((pos = server.find('/') ) != ulxr::CppString::npos)
    {
       resource = server.substr(pos);
       server.erase(pos);
    }

    if ((pos = server.find(':')) != ulxr::CppString::npos)
    {
       ulxr::CppString ps = server.substr(pos+1);
       char *endp;
       std::string s = getLatin1(ps);
       port = strtol(s.c_str(), &endp, 10);
       if (endp != 0 && *endp != '\0')
       {
         ULXR_CERR << ULXR_PCHAR( "Error in port number: ")
                   << ULXR_GET_STRING(endp) << ULXR_PCHAR("\n\n");
         return 1;
       }
       server.erase(pos);
    }

    ULXR_COUT << ULXR_PCHAR("Connecting to ")
              << server << ULXR_PCHAR(":") << port << resource
              << ULXR_PCHAR("...\n\n");

    Integer i(1);
    TcpIpConnection conn (false, server, port);
    HttpProtocol prot(&conn);
    Requester client(&prot);
    MethodResponse resp;

    MethodCall list_methods (ULXR_PCHAR("system.listMethods"));
    MethodCall method_help (ULXR_PCHAR("system.methodHelp"));
    MethodCall method_sig (ULXR_PCHAR("system.methodSignature"));
    MethodCall get_capabilities (ULXR_PCHAR("system.getCapabilities"));

    resp = client.call(list_methods, resource);
    Array meth = resp.getResult();

    bool have_capabilities = false;
    ULXR_COUT << ULXR_PCHAR("Found ") << meth.size() <<  ULXR_PCHAR(" methods:\n\n");

    for (unsigned m = 0; m < meth.size(); ++m)
    {
      RpcString name = meth.getItem(m);
      if (name.getString() == get_capabilities.getMethodName() )
        have_capabilities = true;

      ULXR_COUT << m+1 << ULXR_PCHAR(") ") << name.getString() << std::endl;

      method_sig.clear();
      method_sig.addParam(name);
      resp = client.call(method_sig, resource);
      if (resp.getResult().getType() == RpcArray)
      {
        Array meths = resp.getResult();
        for (unsigned ms = 0; ms < meths.size(); ++ms)
        {
          Array sign = meths.getItem(ms);
          ulxr::CppString s;
          for (unsigned is = 0; is < sign.size(); ++is)
          {
             if (s.length() != 0)
               s += ULXR_PCHAR(", ");
             s +=RpcString(sign.getItem(is)).getString();
          }
          ULXR_COUT << ULXR_PCHAR(" Signature: ") << s << ULXR_PCHAR("\n");
        }
        ULXR_COUT << ULXR_PCHAR("\n");
      }
      else
        ULXR_COUT << ULXR_PCHAR(" Signature: none available\n\n");

      method_help.clear();
      method_help.addParam(name);
      resp = client.call(method_help, resource);
      ulxr::CppString help =RpcString(resp.getResult()).getString();
      pos = 0;
      while ((pos = help.find('\n', pos)) != ulxr::CppString::npos)
      {
         help.insert (pos+1, ULXR_PCHAR("  "));
         pos += 3;   // 2 spaces indentation
      }
      ULXR_COUT << ULXR_PCHAR(" Documentation: \n")
               << ULXR_PCHAR("  ") << help << ULXR_PCHAR("\n\n");
    }

    ULXR_COUT << ULXR_PCHAR("Checking for well known system capabilities:\n");
    if (have_capabilities)
    {
      resp = client.call(get_capabilities, resource);
      Struct cap = resp.getResult();
      if (cap.hasMember(ULXR_PCHAR("faults_interop")))
      {
        Struct inter = cap.getMember(ULXR_PCHAR("faults_interop"));
       RpcString url = inter.getMember(ULXR_PCHAR("specUrl"));
        Integer vers = inter.getMember(ULXR_PCHAR("specVersion"));

       ULXR_COUT << ULXR_PCHAR(" Server supports interoperational fault codes as defined at:\n ")
                 << url.getString() << ULXR_PCHAR(" with version ")
                 << vers.getInteger() << std::endl;
      }
    }
    else
     ULXR_COUT << ULXR_PCHAR(" None avaliable.\n");
  }
  catch(XmlException &ex)
  {
    ULXR_COUT << ULXR_PCHAR("Xml Error occured: ") << ex.why()
              << ULXR_PCHAR(" (") << ex.getErrorString() << ULXR_PCHAR(") ")
              << ULXR_PCHAR(" in line ") << ex.getErrorLine() << std::endl;
  }
  catch(Exception &ex)
  {
    ULXR_COUT << ULXR_PCHAR("Error occured: ") << ex.why() << std::endl;
    return 1;
  }
  catch(std::exception &ex)
  {
     ULXR_COUT << ULXR_PCHAR("Error occured: ") << ULXR_GET_STRING(ex.what()) << std::endl;
     return 1;
  }
  catch(...)
  {
     ULXR_COUT << ULXR_PCHAR("unknown Error occured.\n");
     return 1;
  }
  return 0;
}
