/***************************************************************************
                  ulxr_value.h  -  values passed by xml-rpc
                             -------------------
    begin                : Sun Mar 10 2002
    copyright            : (C) 2002-2006 by Ewald Arnold
    email                : ulxmlrpcpp@ewald-arnold.de

    $Id: ulxr_value.cpp,v 1.54 2006/05/06 08:12:51 ewald-arnold Exp $

 ***************************************************************************/

/**************************************************************************
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ***************************************************************************/

// upon requst you may also use intrinsic types like bool, int, double, char*
// which will automatically generate their xmlrpc counterparts.
//
// always compile, but #include just upon demand!
//
#define ULXR_USE_INTRINSIC_VALUE_TYPES

// upon requst you may also use the following construct
//    ("second" << i2)
//
// always compile, but #include just upon demand!
//
#define ULXR_STRUCT_MEMBER_FROM_NAME_VALUE


#include <ulxmlrpcpp/ulxmlrpcpp.h>  // always first header

#include <cstdio>
#include <ctime>
#include <vector>

#include <ulxmlrpcpp/ulxr_value.h>
#include <ulxmlrpcpp/ulxr_valueparse_wb.h>
#include <ulxmlrpcpp/ulxr_except.h>
#include <ulxmlrpcpp/ulxr_htmlform_handler.h>


#define ULXR_ASSERT_RPCTYPE(x) \
  if (x != getType() )    \
  { \
    throw ParameterException(ApplicationError, (CppString) \
                           ULXR_PCHAR("Value type mismatch.\n") \
                           ULXR_PCHAR("Expected: ") + ULXR_GET_STRING(#x) + ULXR_PCHAR(".\n") \
                           ULXR_PCHAR("Actually have: ") +getTypeName() + ULXR_PCHAR(".")); \
  }


namespace ulxr {



//////////////////////////////////////////////////////


bool ULXR_EXPORT Value::isVoid() const
{
  return baseVal->isVoid();
}


bool ULXR_EXPORT Value::isBoolean() const
{
  return baseVal->isBoolean();
}


bool ULXR_EXPORT Value::isInteger() const
{
  return baseVal->isInteger();
}


bool ULXR_EXPORT Value::isDouble()   const
{
  return baseVal->isDouble();
}


bool ULXR_EXPORT Value::isArray() const
{
  return baseVal->isArray();
}


bool ULXR_EXPORT Value::isStruct() const
{
  return baseVal->isStruct();
}


bool ULXR_EXPORT Value::isString() const
{
  return baseVal->isString();
}


bool ULXR_EXPORT Value::isBase64() const
{
  return baseVal->isBase64();
}


bool ULXR_EXPORT Value::isDateTime() const
{
  return baseVal->isDateTime();
}


ValueType ULXR_EXPORT Value::getType() const
{
  return baseVal->getType();
}


CppString ULXR_EXPORT Value::getTypeName() const
{
  return baseVal->getTypeName();
}


CppString ULXR_EXPORT Value::getSignature(bool deep) const
{
  CppString s;
  if (baseVal != 0)
    s = baseVal->getSignature(deep);
  return s;
}


CppString ULXR_EXPORT Value::getXml(int indent) const
{
  CppString ret;
  if (baseVal != 0)
    ret = baseVal->getXml(indent);
  return ret;
}


std::string ULXR_EXPORT Value::getWbXml() const
{
  std::string ret;
  if (baseVal != 0)
    ret = baseVal->getWbXml();
  return ret;
}


Struct* ULXR_EXPORT Value::getStruct()
{
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  return structVal;
};


const Struct* ULXR_EXPORT Value::getStruct() const
{
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  return structVal;
};


Array* ULXR_EXPORT Value::getArray()
{
  ULXR_ASSERT_RPCTYPE(RpcArray);
  return arrayVal;
};


const Array* ULXR_EXPORT Value::getArray() const
{
  ULXR_ASSERT_RPCTYPE(RpcArray);
  return arrayVal;
};


ULXR_EXPORT Value::operator Boolean& ()
{
  ULXR_ASSERT_RPCTYPE(RpcBoolean);
  return *boolVal;
}


ULXR_EXPORT Value::operator const Boolean& () const
{
  ULXR_ASSERT_RPCTYPE(RpcBoolean);
  return *boolVal;
}


ULXR_EXPORT Value::operator Integer& ()
{
  ULXR_ASSERT_RPCTYPE(RpcInteger);
  return *intVal;
}


ULXR_EXPORT Value::operator const Integer& () const
{
  ULXR_ASSERT_RPCTYPE(RpcInteger);
  return *intVal;
}


ULXR_EXPORT Value::operator Double& ()
{
  ULXR_ASSERT_RPCTYPE(RpcDouble);
  return *doubleVal;
}


ULXR_EXPORT Value::operator const Double& () const
{
  ULXR_ASSERT_RPCTYPE(RpcDouble);
  return *doubleVal;
}


ULXR_EXPORT Value::operator Array& ()
{
  ULXR_ASSERT_RPCTYPE(RpcArray);
  return *arrayVal;
}


ULXR_EXPORT Value::operator const Array& () const
{
  ULXR_ASSERT_RPCTYPE(RpcArray);
  return *arrayVal;
}


ULXR_EXPORT Value::operator Struct& ()
{
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  return *structVal;
}


ULXR_EXPORT Value::operator const Struct& () const
{
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  return *structVal;
}


ULXR_EXPORT Value::operator RpcString& ()
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
  return *stringVal;
}


ULXR_EXPORT Value::operator const RpcString& () const
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
  return *stringVal;
}


ULXR_EXPORT Value::operator Base64& ()
{
  ULXR_ASSERT_RPCTYPE(RpcBase64);
  return *base64Val;
}


ULXR_EXPORT Value::operator const Base64& () const
{
  ULXR_ASSERT_RPCTYPE(RpcBase64);
  return *base64Val;
}


ULXR_EXPORT Value::operator DateTime& ()
{
  ULXR_ASSERT_RPCTYPE(RpcDateTime);
  return *dateVal;
}


ULXR_EXPORT Value::operator const DateTime& () const
{
  ULXR_ASSERT_RPCTYPE(RpcDateTime);
  return *dateVal;
}


ULXR_EXPORT Value::Value ()
{
   voidVal = new Void();
}


ULXR_EXPORT Value::Value (const Value &val)
{
   baseVal = val.baseVal->cloneValue();
}


ULXR_EXPORT Value::Value (const Boolean &val)
{
   boolVal = new Boolean(val);
}


ULXR_EXPORT Value::Value (const Integer &val)
{
   intVal = new Integer(val);
}


ULXR_EXPORT Value::Value (const Double &val)
{
   doubleVal = new Double(val);
}


ULXR_EXPORT Value::Value (const Array &val)
{
   arrayVal = new Array(val);
}


ULXR_EXPORT Value::Value (const Struct &val)
{
   structVal = new Struct(val);
}


ULXR_EXPORT Value::Value (const RpcString &val)
{
   stringVal = new RpcString(val);
}


ULXR_EXPORT Value::Value (const Base64 &val)
{
   base64Val = new Base64(val);
}



ULXR_EXPORT Value::Value (const DateTime &val)
{
   dateVal = new DateTime(val);
}


#ifdef ULXR_USE_INTRINSIC_VALUE_TYPES


ULXR_EXPORT Value::Value (const char *val)
{
   stringVal = new RpcString(val);
}


#ifdef ULXR_UNICODE

ULXR_EXPORT Value::Value (const wchar_t *val)
{
  Cpp16BitString wstr = val;
   stringVal = new RpcString(unicodeToUtf8(wstr));
}

#endif


ULXR_EXPORT Value::Value (const double &val)
{
   doubleVal = new Double(val);
}


ULXR_EXPORT Value::Value (const int &val)
{
   intVal = new Integer(val);
}


ULXR_EXPORT Value::Value (const bool &val)
{
   boolVal = new Boolean(val);
}


#endif


Value & ULXR_EXPORT Value::operator= (const Value &val)
{
   ValueBase *temp = val.baseVal->cloneValue();
   delete baseVal;
   baseVal = temp;
   return *this;
}


ULXR_EXPORT Value::~Value ()

{
  delete baseVal;
  baseVal = 0;
}


//////////////////////////////////////////////////////


ULXR_EXPORT ValueBase::ValueBase (ValueType t)
     : type(t)
{
}


ValueType ULXR_EXPORT ValueBase::getType() const
{
  return type;
}


ULXR_EXPORT ValueBase::~ValueBase ()
{
}


bool ULXR_EXPORT ValueBase::isVoid() const
{
  return type == RpcVoid;
}


bool ULXR_EXPORT ValueBase::isBoolean() const
{
  return type == RpcBoolean;
}


bool ULXR_EXPORT ValueBase::isInteger() const
{
  return type == RpcInteger;
}


bool ULXR_EXPORT ValueBase::isDouble() const
{
  return type == RpcDouble;
}


bool ULXR_EXPORT ValueBase::isArray() const
{
  return type == RpcArray;
}


bool ULXR_EXPORT ValueBase::isStruct() const
{
  return type == RpcStruct;
}


bool ULXR_EXPORT ValueBase::isString() const
{
  return type == RpcStrType;
}


bool ULXR_EXPORT ValueBase::isBase64() const
{
  return type == RpcBase64;
}


bool ULXR_EXPORT ValueBase::isDateTime() const
{
  return type == RpcDateTime;
}


CppString ULXR_EXPORT ValueBase::getTypeName() const
{
  CppString ret = ULXR_PCHAR("RpcValue");
  switch(getType() )
  {
     case RpcVoid:
       ret = ULXR_PCHAR("RpcVoid");
     break;

     case RpcInteger:
       ret = ULXR_PCHAR("RpcInteger");
     break;

     case RpcDouble:
       ret = ULXR_PCHAR("RpcDouble");
     break;

     case RpcBoolean:
       ret = ULXR_PCHAR("RpcBoolean");
     break;

     case RpcStrType:
       ret = ULXR_PCHAR("RpcString");
     break;

     case RpcDateTime:
       ret = ULXR_PCHAR("RpcDateTime");
     break;

     case RpcBase64:
       ret = ULXR_PCHAR("RpcBase64");
     break;

     case RpcArray:
       ret = ULXR_PCHAR("RpcArray");
     break;

     case RpcStruct:
       ret = ULXR_PCHAR("RpcStruct");
     break;
  }
  return ret;
}


//////////////////////////////////////////////////////


ULXR_EXPORT Void::Void ()
 : ValueBase(RpcVoid)
{
}


ULXR_EXPORT Void::~Void ()
{
}


ValueBase * ULXR_EXPORT Void::cloneValue() const
{
  ULXR_ASSERT_RPCTYPE(RpcVoid);
  return new Void(*this);
}


CppString ULXR_EXPORT Void::getSignature(bool /*deep*/) const
{
  ULXR_ASSERT_RPCTYPE(RpcVoid);
  return getValueName();
}


CppString ULXR_EXPORT Void::getValueName()
{
  return ULXR_PCHAR("void");
}


std::string ULXR_EXPORT Void::getWbXml() const
{
  ULXR_ASSERT_RPCTYPE(RpcVoid);
  return "";
}


CppString ULXR_EXPORT Void::getXml(int /*indent*/) const
{
  ULXR_ASSERT_RPCTYPE(RpcVoid);
  return ULXR_PCHAR("");
}


//////////////////////////////////////////////////////


ULXR_EXPORT Boolean::Boolean ()
 : ValueBase(RpcBoolean)
{
}


ULXR_EXPORT Boolean::Boolean (bool b)
 : ValueBase(RpcBoolean), val(b)
{
}


ULXR_EXPORT Boolean::~Boolean ()
{
}


ULXR_EXPORT Boolean::Boolean (const CppString &s)
  : ValueBase(RpcBoolean)
{
  if (   s == ULXR_PCHAR("true") || s == ULXR_PCHAR("TRUE")
      || s == ULXR_PCHAR("1") || s == ULXR_PCHAR("on") || s == ULXR_PCHAR("ON"))
    val = true;
  else
    val = false;
}


ValueBase * ULXR_EXPORT Boolean::cloneValue() const
{
  ULXR_ASSERT_RPCTYPE(RpcBoolean);
  return new Boolean(*this);
}


CppString ULXR_EXPORT Boolean::getSignature(bool /*deep*/) const
{
  ULXR_ASSERT_RPCTYPE(RpcBoolean);
  return getValueName();
}


CppString ULXR_EXPORT Boolean::getValueName()
{
  return ULXR_PCHAR("bool");
}


std::string ULXR_EXPORT Boolean::getWbXml() const
{
  ULXR_ASSERT_RPCTYPE(RpcBoolean);
  std::string s;

  s = ValueParserWb::wbToken_Value;
  s += ValueParserWb::wbToken_Boolean;

  if (val)
    s += getWbXmlString(ULXR_PCHAR("1"));
  else
    s += getWbXmlString(ULXR_PCHAR("0"));

  s += WbXmlParser::wbxml_END;
  s += WbXmlParser::wbxml_END;
  return s;
}


CppString ULXR_EXPORT Boolean::getXml(int indent) const
{
  ULXR_ASSERT_RPCTYPE(RpcBoolean);
  CppString s (indent, ' ');
  s += ULXR_PCHAR("<value><boolean>");

  if (val)
    s += ULXR_PCHAR("1"); // "true"
  else
    s += ULXR_PCHAR("0"); // "false"

  s += ULXR_PCHAR("</boolean></value>");
  return s;
}


bool ULXR_EXPORT Boolean::getBoolean () const
{
  ULXR_ASSERT_RPCTYPE(RpcBoolean);
  return val;
}


void ULXR_EXPORT Boolean::setBoolean(const bool newval)
{
  ULXR_ASSERT_RPCTYPE(RpcBoolean);
  val = newval;
}


//////////////////////////////////////////////////////


ULXR_EXPORT Integer::Integer ()
  : ValueBase(RpcInteger)
{
}


ULXR_EXPORT Integer::Integer (int i)
 : ValueBase(RpcInteger), val(i)
{
}


ULXR_EXPORT Integer::~Integer ()
{
}


ULXR_EXPORT Integer::Integer (const CppString &s)
  : ValueBase(RpcInteger)
{
//  val = ulxr_atoi(s.c_str() );
  char *endp;
  val = strtol(getLatin1(s).c_str(), &endp, 10 );
//  if (*endp != 0)
// FIXME
}


ValueBase * ULXR_EXPORT Integer::cloneValue() const
{
  ULXR_ASSERT_RPCTYPE(RpcInteger);
  return new Integer(*this);
}



CppString ULXR_EXPORT Integer::getSignature(bool /*deep*/) const
{
  ULXR_ASSERT_RPCTYPE(RpcInteger);
  return getValueName();
}


CppString ULXR_EXPORT Integer::getValueName()
{
  return ULXR_PCHAR("int");
}


std::string ULXR_EXPORT Integer::getWbXml() const
{
  ULXR_ASSERT_RPCTYPE(RpcInteger);
  std::string s;
  s = (char)ValueParserWb::wbToken_Value;
  s += (char)ValueParserWb::wbToken_I4;

  char buff[48];
  ulxr_sprintf(buff, "%d", val);
  s += getWbXmlString(ULXR_GET_STRING(buff));
  s += (char)WbXmlParser::wbxml_END;
  s += (char)WbXmlParser::wbxml_END;
  return s;
}


CppString ULXR_EXPORT Integer::getXml(int indent) const
{
  ULXR_ASSERT_RPCTYPE(RpcInteger);
  CppString s (indent, ' ');
  s += ULXR_PCHAR("<value><i4>");
  char buff[40];
  ulxr_sprintf(buff, "%d", val);
  s += ULXR_GET_STRING(buff);
  s += ULXR_PCHAR("</i4></value>");
  return s;
}


int  ULXR_EXPORT Integer::getInteger () const
{
  ULXR_ASSERT_RPCTYPE(RpcInteger);
  return val;
}


void ULXR_EXPORT Integer::setInteger(const int newval)
{
  ULXR_ASSERT_RPCTYPE(RpcInteger);
  val = newval;
}


//////////////////////////////////////////////////////


ULXR_EXPORT Double::Double ()
  : ValueBase(RpcDouble)
{
}


ULXR_EXPORT Double::Double (double d)
  : ValueBase(RpcDouble)
  , val(d)
{
}


ULXR_EXPORT Double::~Double ()
{
}


ULXR_EXPORT Double::Double (const CppString &s)
  : ValueBase(RpcDouble)
{
  val = atof(getLatin1(s).c_str() );
}


ValueBase * ULXR_EXPORT Double::cloneValue() const
{
  ULXR_ASSERT_RPCTYPE(RpcDouble);
  return new Double(*this);
}


CppString ULXR_EXPORT Double::getSignature(bool /*deep*/) const
{
  ULXR_ASSERT_RPCTYPE(RpcDouble);
  return getValueName();
}


CppString ULXR_EXPORT Double::getValueName()
{
  return ULXR_PCHAR("double");
}


std::string ULXR_EXPORT Double::getWbXml() const
{
  ULXR_ASSERT_RPCTYPE(RpcDouble);
  std::string s;
  char buff[48];
  ulxr_sprintf(buff, "%f", val);
  s = ValueParserWb::wbToken_Value;
  s += ValueParserWb::wbToken_Double;
  s += getWbXmlString(ULXR_GET_STRING(buff));
  s += (char)WbXmlParser::wbxml_END;
  s += (char)WbXmlParser::wbxml_END;
  return s;
}


CppString ULXR_EXPORT Double::getXml(int indent) const
{
  ULXR_ASSERT_RPCTYPE(RpcDouble);
  CppString s (indent, ' ');
  s += ULXR_PCHAR("<value><double>");
  char buff[48];
  ulxr_sprintf(buff, "%f", val);
  s += ULXR_GET_STRING(buff);
  s += ULXR_PCHAR("</double></value>");
  return s;
}


double ULXR_EXPORT Double::getDouble () const
{
  ULXR_ASSERT_RPCTYPE(RpcDouble);
  return val;
}


void ULXR_EXPORT Double::setDouble(const double newval)
{
  ULXR_ASSERT_RPCTYPE(RpcDouble);
  val = newval;
}


//////////////////////////////////////////////////////


ULXR_EXPORT RpcString::RpcString ()
  : ValueBase(RpcStrType)
{
}


ULXR_EXPORT RpcString::RpcString (const Cpp8BitString &s)
 : ValueBase(RpcStrType)
{
  setString(s);
}


ULXR_EXPORT RpcString::~RpcString ()
{
}


#ifdef ULXR_UNICODE

ULXR_EXPORT RpcString::RpcString (const Cpp16BitString &s)
  : ValueBase(RpcStrType)
{
  setString(s);
}

#endif

ValueBase * ULXR_EXPORT RpcString::cloneValue() const
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
  return new RpcString(*this);
}


CppString ULXR_EXPORT RpcString::getSignature(bool /*deep*/) const
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
  return getValueName();
}


CppString ULXR_EXPORT RpcString::getValueName()
{
  return ULXR_PCHAR("string");
}


std::string ULXR_EXPORT RpcString::getWbXml() const
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
  std::string s;
  s = ValueParserWb::wbToken_Value;
  s += ValueParserWb::wbToken_String;
  s += getWbXmlString(val);
  s += (char)WbXmlParser::wbxml_END;
  s += (char)WbXmlParser::wbxml_END;
  return s;
}


CppString ULXR_EXPORT RpcString::getXml(int indent) const
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
  CppString s (indent, ULXR_CHAR(' '));
  s += ULXR_PCHAR("<value><string>");
  s += xmlEscape(val);
  s += ULXR_PCHAR("</string></value>");
  return s;
}


CppString ULXR_EXPORT RpcString::getString () const
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
  return val;
}


#ifdef ULXR_UNICODE

Cpp16BitString ULXR_EXPORT RpcString::getUnicodeString () const
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
#ifdef ULXR_UNICODE
  return val;
#else
  return utf8ToUnicode(val);
#endif
}

#endif

#if (defined(HAVE_ICONV_H) || defined(HAVE_ICONV)) && !defined (ULXR_UNICODE)

CppString ULXR_EXPORT RpcString::getString (const char *encoding) const
{
  return utf8ToEncoding(val, encoding);
}


void ULXR_EXPORT RpcString::setString(const Cpp8BitString &newval, const char *encoding)
{
  val = encodingToUtf8(newval, encoding);
}

#endif


void ULXR_EXPORT RpcString::setString(const Cpp8BitString &newval)
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
#ifdef ULXR_UNICODE
  val = getUnicode(newval);
#else
  val = newval;
#endif
}


#ifdef ULXR_UNICODE

void ULXR_EXPORT RpcString::setString(const Cpp16BitString& newval)
{
  ULXR_ASSERT_RPCTYPE(RpcStrType);
  val = newval;
}

#endif

//////////////////////////////////////////////////////


ULXR_EXPORT Base64::Base64 ()
  : ValueBase(RpcBase64)
{
}


ULXR_EXPORT Base64::Base64 (const CppString &s)
  : ValueBase(RpcBase64)
{
  setString(s);
}


ULXR_EXPORT Base64::~Base64 ()
{
}


CppString ULXR_EXPORT Base64::getBase64() const
{
  return val;
}


void ULXR_EXPORT Base64::setBase64(const CppString s)
{
  val = s;
}


ValueBase * ULXR_EXPORT Base64::cloneValue() const
{
  ULXR_ASSERT_RPCTYPE(RpcBase64);
  return new Base64(*this);
}


CppString ULXR_EXPORT Base64::getSignature(bool /*deep*/) const
{
  ULXR_ASSERT_RPCTYPE(RpcBase64);
  return getValueName();
}


CppString ULXR_EXPORT Base64::getValueName()
{
  return ULXR_PCHAR("base64");
}


CppString ULXR_EXPORT Base64::getXml(int indent) const
{
  ULXR_ASSERT_RPCTYPE(RpcBase64);
  CppString s (indent, ' ');
  s += ULXR_PCHAR("<value><base64>");
  s += val;
  s += ULXR_PCHAR("</base64></value>");
  return s;
}


std::string ULXR_EXPORT Base64::getWbXml() const
{
  ULXR_ASSERT_RPCTYPE(RpcBase64);
  std::string s;
  s += ValueParserWb::wbToken_Value;
  s += ValueParserWb::wbToken_Base64;
  s += getWbXmlString(val);
  s += (char)WbXmlParser::wbxml_END;
  s += (char)WbXmlParser::wbxml_END;
  return s;
}


CppString ULXR_EXPORT Base64::getString () const
{
  ULXR_ASSERT_RPCTYPE(RpcBase64);
  return decodeBase64(val);
}


void ULXR_EXPORT Base64::setString(const CppString &newval)
{
  ULXR_ASSERT_RPCTYPE(RpcBase64);
  val = encodeBase64(newval);
}


//////////////////////////////////////////////////////


ULXR_EXPORT DateTime::DateTime ()
  : ValueBase(RpcDateTime)
{
}


ULXR_EXPORT DateTime::DateTime (const CppString &s)
 : ValueBase(RpcDateTime)
 , val(s)
{
}


ULXR_EXPORT DateTime::DateTime (const ulxr_time_t &tm)
  : ValueBase(RpcDateTime)
{
  setDateTime(tm);
}


ULXR_EXPORT DateTime::~DateTime ()
{
}


ValueBase * ULXR_EXPORT DateTime::cloneValue() const
{
  ULXR_ASSERT_RPCTYPE(RpcDateTime);
  return new DateTime(*this);
}


CppString ULXR_EXPORT DateTime::getSignature(bool /*deep*/) const
{
  ULXR_ASSERT_RPCTYPE(RpcDateTime);
  return getValueName();
}


CppString ULXR_EXPORT DateTime::getValueName()
{
  return ULXR_PCHAR("dateTime.iso8601");
}


std::string ULXR_EXPORT DateTime::getWbXml() const
{
  ULXR_ASSERT_RPCTYPE(RpcDateTime);
  std::string s;
  s += ValueParserWb::wbToken_Value;
  s += ValueParserWb::wbToken_Date;
  s += getWbXmlString(val);
  s += (char)WbXmlParser::wbxml_END;
  s += (char)WbXmlParser::wbxml_END;
  return s;
}


CppString ULXR_EXPORT DateTime::getXml(int indent) const
{
  ULXR_ASSERT_RPCTYPE(RpcDateTime);
  CppString s (indent, ULXR_CHAR(' '));
  s += ULXR_PCHAR("<value><dateTime.iso8601>");
  s += val;
  s += ULXR_PCHAR("</dateTime.iso8601></value>");
  return s;
}


CppString ULXR_EXPORT DateTime::getDateTime () const
{
  ULXR_ASSERT_RPCTYPE(RpcDateTime);
  return val;
}


static CppString padded(unsigned num)
{
  CppString s;
  if (num < 10)
    s += L'0';
  s += HtmlFormHandler::makeNumber(num);
  return s;
}


void ULXR_EXPORT DateTime::setDateTime (const ulxr_time_t &tmt)
{
  CppString s;
  ulxr_tm * ltm = ulxr_localtime(&tmt);
  s += padded(ltm->tm_year+1900);
  s += '-';
  s += padded(ltm->tm_mon+1);
  s += '-';
  s += padded(ltm->tm_mday);
  s += ULXR_PCHAR("T");
  s += padded(ltm->tm_hour);
  s += ':';
  s += padded(ltm->tm_min);
  s += ':';
  s += padded(ltm->tm_sec);
  setDateTime(s);
}


void ULXR_EXPORT DateTime::setDateTime(const CppString &newval)
{
  ULXR_ASSERT_RPCTYPE(RpcDateTime);
  val = newval;
}


//////////////////////////////////////////////////////


ULXR_EXPORT Array::Array ()
  : ValueBase(RpcArray)
{
}


ULXR_EXPORT Array::~Array ()
{
}


unsigned ULXR_EXPORT Array::size() const
{
  return values.size();
}


void ULXR_EXPORT Array::clear()
{
  values.clear();
}


ValueBase * ULXR_EXPORT Array::cloneValue() const
{
  ULXR_ASSERT_RPCTYPE(RpcArray);
  return new Array(*this);
}


CppString ULXR_EXPORT Array::getSignature(bool deep) const
{
  if (!deep)
    return getValueName();


  ULXR_ASSERT_RPCTYPE(RpcArray);
  CppString s;
  if (values.size() == 0)
    return ULXR_PCHAR("[]");

  s += ULXR_CHAR('[');
  bool comma = values.size() >= 1;
  for (unsigned i = 0; i < values.size(); ++i)
  {
    if (comma && i != 0)
      s += ULXR_CHAR(',');
    s += values[i].getSignature();
  }
  s += ULXR_CHAR(']');

  return s;
}


CppString ULXR_EXPORT Array::getValueName()
{
  return ULXR_PCHAR("array");
}


std::string ULXR_EXPORT Array::getWbXml() const
{
  ULXR_ASSERT_RPCTYPE(RpcArray);
  std::string s;
  s = ValueParserWb::wbToken_Value;
  s += ValueParserWb::wbToken_Array;
  s += ValueParserWb::wbToken_Data;

  for (std::vector<Value>::const_iterator
         it = values.begin(); it != values.end(); ++it)
  {
    s += (*it).getWbXml();
  }

  s += WbXmlParser::wbxml_END;
  s += WbXmlParser::wbxml_END;
  s += WbXmlParser::wbxml_END;
  return s;
}


CppString ULXR_EXPORT Array::getXml(int indent) const
{
  ULXR_ASSERT_RPCTYPE(RpcArray);
  CppString ind (indent, ' ');
  CppString s = ind + ULXR_PCHAR("<value>\n");
  s += ind + ULXR_PCHAR(" <array>\n");
  s += ind + ULXR_PCHAR("  <data>\n");

  for (std::vector<Value>::const_iterator
         it = values.begin(); it != values.end(); ++it)
  {
    s += (*it).getXml(indent+3) + ULXR_PCHAR("\n");
  }

  s += ind + ULXR_PCHAR("  </data>\n");
  s += ind + ULXR_PCHAR(" </array>\n");
  s += ind + ULXR_PCHAR("</value>");
  return s;
}


void ULXR_EXPORT Array::addItem(const Value &item)
{
  values.push_back(item);
}


void ULXR_EXPORT Array::setItem(unsigned idx, const Value &item)
{
  if (idx < values.size() )
    values[idx] = item;
}


Value ULXR_EXPORT Array::getItem(unsigned idx)
{
  if (idx < values.size() )
    return values[idx];
  else
   return Value(RpcString());   // FIXME: exception
}


//////////////////////////////////////////////////////


ULXR_EXPORT Struct::Member::Member(const CppString &str_, const Value &t_)
 : str(str_)
 , t(t_)
{
}


const CppString & Struct::Member::getName() const
{
  return str;
}


const Value & Struct::Member::getValue() const
{
  return t;
}


//////////////////////////////////////////////////////


ULXR_EXPORT Struct::Struct ()
 : ValueBase(RpcStruct)
{
}


ULXR_EXPORT Struct::~Struct ()
{
}


ValueBase * ULXR_EXPORT Struct::cloneValue() const
{
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  return new Struct(*this);
}


void ULXR_EXPORT Struct::clear()
{
  val.clear();
}


unsigned ULXR_EXPORT Struct::size() const
{
  return val.size() ;
}


const Struct::Members& ULXR_EXPORT Struct::getAllMembers() const
{
  return val;
}


CppString ULXR_EXPORT Struct::getSignature(bool deep) const
{
  if (!deep)
    return getValueName();

  ULXR_ASSERT_RPCTYPE(RpcStruct);
  CppString s;
  if (val.size() == 0)
    return ULXR_PCHAR("{}");

  if (val.size() > 1)
    s += ULXR_CHAR('{');

  for (Members::const_iterator it = val.begin(); it != val.end(); ++it)
  {
    s += ULXR_CHAR('{');
    s += (*it).first;
    s += ULXR_CHAR(',');
    s += (*it).second.getSignature();
    s += ULXR_CHAR('}');
  }

  if (val.size() > 1)
    s += ULXR_CHAR('}');

  return s;
}


CppString ULXR_EXPORT Struct::getValueName()
{
  return ULXR_PCHAR("struct");
}


CppString ULXR_EXPORT Struct::getXml(int indent) const
{
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  CppString ind (indent, ' ');
  CppString s = ind + ULXR_PCHAR("<value>\n");
  s += ind + ULXR_PCHAR(" <struct>\n");

  for (Members::const_iterator it = val.begin(); it != val.end(); ++it)
  {
    s += ind + ULXR_PCHAR("  <member>\n");
    s += ind + ULXR_PCHAR("   <name>") + (*it).first + ULXR_PCHAR("</name>\n");
    s += (*it).second.getXml(indent+3) + ULXR_PCHAR("\n");
    s += ind + ULXR_PCHAR("  </member>\n");
  }

  s += ind + ULXR_PCHAR(" </struct>\n");
  s += ind + ULXR_PCHAR("</value>");
  return s;
}


std::string ULXR_EXPORT Struct::getWbXml() const
{
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  std::string s;
  s = ValueParserWb::wbToken_Value;
  s += ValueParserWb::wbToken_Struct;

  for (Members::const_iterator it = val.begin(); it != val.end(); ++it)
  {
    s += ValueParserWb::wbToken_Member;
    s += ValueParserWb::wbToken_Name;
    s += getWbXmlString((*it).first);
    s += (char)WbXmlParser::wbxml_END;
    s += (*it).second.getWbXml();
    s += (char)WbXmlParser::wbxml_END;
  }

  s += WbXmlParser::wbxml_END;
  s += WbXmlParser::wbxml_END;
  return s;
}


void ULXR_EXPORT Struct::addMember(const CppString &name, const Value &item)
{
  ULXR_TRACE("Struct::addMember(string, Value)");
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  val.insert(Member_pair(name, item));
}


bool ULXR_EXPORT Struct::hasMember(const CppString &name) const
{
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  return val.find(name) != val.end();
}


Value ULXR_EXPORT Struct::getMember(const CppString &name) const
{
  ULXR_ASSERT_RPCTYPE(RpcStruct);
  Members::const_iterator it = val.find(name);
  if (it  == val.end())
    throw RuntimeException(ApplicationError,
                           ulxr_i18n(ULXR_PCHAR("Attempt to get unknown Struct member: ")) +name);

  return (*it).second;
}


std::vector<CppString> ULXR_EXPORT Struct::getMemberNames() const
{
  std::vector<CppString> names;
  for (Members::const_iterator it = val.begin(); it != val.end(); ++it)
     names.push_back((*it).first);
  return names;
}


#ifdef ULXR_STRUCT_MEMBER_FROM_NAME_VALUE

Struct::Member ULXR_EXPORT operator<< (const CppString &str, const Value &t)
{
  ULXR_TRACE("Struct::Member operator<< (string, T)");
  return Struct::Member(str, t);
}

#endif

Struct::Member ULXR_EXPORT make_member (const CppString &str, const Value &t)
{
  ULXR_TRACE(ULXR_PCHAR("Member operator<< (string, T)"));
  return Struct::Member(str, t);
}


Struct ULXR_EXPORT & operator<< (Struct &st, const Struct::Member &k)
{
  ULXR_TRACE(ULXR_PCHAR("operator<<(Struct, Struct::Member)"));
  st.addMember(k.getName(), k.getValue());
  return st;
}


};  // namespace ulxr

