/*
 * Copyright (C) 2000 Bang Jun-Young
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THE AUTHOR ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION,
 * AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER
 * RESULTING FROM THE USE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <string.h>
#include <uniksc.h>

void convert_to_utf16(FILE *, FILE *), convert_to_ksc5601(FILE *, FILE *);
void count_characters(FILE *);
void print_help(void), print_license(void);

int use_stdout_flag, count_flag, big_endian_flag, convert_to_utf16_flag = 1;

int
main(int argc, char **argv, char **envp)
{
	int i, file_count = 0;

	char *srcfile_path, destfile_path[255];
	FILE *infile, *outfile;
	char *destfile_extention;

	if (argc < 2) {
		print_help();
		exit(0);
	}

	/* Parse command-line arguments. */
	for (i = 1; i < argc; ++i) {
		if (argv[i][0] == '-') {
			char *s = argv[i] + 1, opt;

			while ((opt = *s++) != '\0') {
				switch (opt) {
				case 'u':
					convert_to_utf16_flag = 1;
					break;
				case 'k':
					convert_to_utf16_flag = 0;
					break;
				case 'c':
					use_stdout_flag = 1;
					break;
				case 'n':
					count_flag = 1;
					break;
				case 'b':
					big_endian_flag = 1;
					break;
				case 'l':
					big_endian_flag = 0;
					break;
				case 'h':
					print_help();
					exit(0);
				case 'V':
					print_license();
					exit(0);
				default:
					printf("Unknown option: -%c\n", opt);
					exit(0);
		    		}
			}		
		} else {
			if (!file_count) {
				if (strlen(argv[i]) > 240) {
					printf("Pathname too long: %s\n", 
						argv[i]);
					exit(0);
				}
				srcfile_path = argv[i];
				file_count++;
			} else {
				strcpy(destfile_path, argv[i]);
				file_count++;
			}
		}
	}

	if (!file_count) {
		print_help();
		exit(0);
	}

	if ((infile = fopen(srcfile_path, "r")) == NULL) {
		printf("No such file or directory: %s\n", srcfile_path);
		exit(0);
	}

	if (use_stdout_flag)
		outfile = stdout;
	else if (count_flag) {
		count_characters(infile);
		fclose(infile);
		exit(0);
	}

       	if (file_count < 2) {
       		strcpy(destfile_path, srcfile_path);
       		if (convert_to_utf16_flag)
			destfile_extention = ".utf16";
		else
			destfile_extention = ".ksc5601";
		strcat(destfile_path, destfile_extention);
       	}
       	if ((outfile = fopen(destfile_path, "w+")) == NULL) {
       		printf("Couldn't open for writing: %s\n", destfile_path);
       		exit(0);
       	}

	if (convert_to_utf16_flag)
		convert_to_utf16(infile, outfile);
	else
		convert_to_ksc5601(infile, outfile);

	fclose(infile);
	fclose(outfile);

	return 0;
}

void
convert_to_utf16(FILE *infile, FILE *outfile)
{
	int ch[2];

	/* Insert architecure-specific codes. */
	if ((ch[0] = fgetc(infile)) != EOF) {
		if (!big_endian_flag) {
	 		/* Insert a little-endian mark. */
			fputc(0xff, outfile);
			fputc(0xfe, outfile);
		} else {
			/* XXX */
		}
		ungetc(ch[0], infile);
	}

	while ((ch[0] = fgetc(infile)) != EOF) {
		if (ch[0] > 0x7f) {
			/* It's a Hangul character. */
                        ch[1] = fgetc(infile);
			ch[0] = ksc5601_to_utf16(ch[0] << 8 | ch[1]);
			fputc(ch[0] & 0xff, outfile);
			fputc(ch[1] >> 8, outfile);
		} else {
			fputc(ch[0], outfile);
			fputc('\0', outfile);
		}	
	}
}

void
convert_to_ksc5601(FILE *infile, FILE *outfile)
{
#if 0	/* Right now this function might be harmful, so... */
	int ch[0];

	while ((ch[0] = fgetc(infile)) != EOF) {
		ch[1] = fgetc(infile);
		ch[0] = utf16_to_ksc5601(ch[0] | ch[1] << 8);
		fputc(ch[0] >> 8, outfile);
		fputc(ch[0] & 0xff, outfile);
	}
#else
	printf("This feature is disabled in this version.\n");
	exit(0);
#endif
}

void
count_characters(FILE *infile)
{
	int ch;
	int ksc_count = 0, ascii_count = 0;

	while ((ch = fgetc(infile)) != EOF) {
		if (ch > 0x7f) {
			/* It's a Hangul character. */
                        ch = fgetc(infile);
			ksc_count++;
		} else {
			ascii_count++;
		}	
	}
	
	printf("\t%d\t%d\n", ksc_count, ascii_count);
}

void
print_help(void)
{
	char *help_msg =
"uniksc converts files between Korean KSC-5601 and Unicode UTF-16.\n"
"\n"
"Usage: uniksc [options] srcfile destfile\n"
"\n"
"Options:\n"
"  -u          convert to UTF-16 (default)\n"
"  -k          convert to KSC-5601\n"
"  -b          big-endian mode (doesn't work yet)\n"
"  -l          little-endian mode (default)\n"
"  -c          output to standard output\n"
"  -n          don't covert; just print the number of characters to convert\n"
"  -h          display this help\n"
"  -V          display version and license information\n"
"\n";

	printf("%s", help_msg);
}

void
print_license(void)
{
	char *license_msg =
"uniksc version 0.9, released 2000-10-09\n"
"Copyright (C) 2000 Bang Jun-Young\n"
"All rights reserved.\n"
"\n"
"Redistribution and use in source and binary forms, with or without\n"
"modification, are permitted provided that the following conditions\n"
"are met:\n"
"1. Redistributions of source code must retain the above copyright\n"
"   notice, this list of conditions and the following disclaimer.\n"
"2. Redistributions in binary form must reproduce the above copyright\n"
"   notice, this list of conditions and the following disclaimer in the\n"
"   documentation and/or other materials provided with the distribution.\n"
"3. The name of the author may not be used to endorse or promote\n"
"   products derived from this software without specific prior written\n"
"   permission.\n"
"\n"
"THE AUTHOR ALLOWS FREE USE OF THIS SOFTWARE IN ITS \"AS IS\" CONDITION,\n"
"AND DISCLAIM ANY LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER\n"
"RESULTING FROM THE USE OF THIS SOFTWARE.\n"
"\n";

	printf("%s", license_msg);
}

