/* -- updatedd: utils.h --
 *
 * Copyright (c) 2002, 2003 Philipp Benner <philipp@philippb.tk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * The GNU C Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with the GNU C Library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA.
 */

#include <config.h>
#include <error.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/ioctl.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <unistd.h>
#include <net/if.h>
#include <arpa/inet.h>

#ifdef BASE64 /* -------------------------------------------------------- */

static char table64[]=
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

inline void
  base64encode(char *intext, char *output)
{
	unsigned char ibuf[3];
	unsigned char obuf[4];
	int i;
	int inputparts;

	while(*intext) {
		for (i = inputparts = 0; i < 3; i++) {
			if(*intext) {
				inputparts++;
				ibuf[i] = *intext;
				intext++;
			} else
				ibuf[i] = 0;
		}

		obuf [0] = (ibuf [0] & 0xFC) >> 2;
		obuf [1] = ((ibuf [0] & 0x03) << 4) | ((ibuf [1] & 0xF0) >> 4);
		obuf [2] = ((ibuf [1] & 0x0F) << 2) | ((ibuf [2] & 0xC0) >> 6);
		obuf [3] = ibuf [2] & 0x3F;

		switch(inputparts) {
		case 1: /* only one byte read */
			sprintf(output, "%c%c==",
				table64[obuf[0]],
				table64[obuf[1]]);
			break;
		case 2: /* two bytes read */
			sprintf(output, "%c%c%c=",
				table64[obuf[0]],
				table64[obuf[1]],
				table64[obuf[2]]);
			break;
		default:
			sprintf(output, "%c%c%c%c",
				table64[obuf[0]],
				table64[obuf[1]],
				table64[obuf[2]],
				table64[obuf[3]] );
			break;
		}
		output += 4;
	}
	*output=0;
}

#endif /* BASE64 */

#ifndef HAVE_DPRINTF /* -------------------------------------------------------- */

# include <stdarg.h>
# include <sys/types.h>

# define MAXLEN	2048

inline int
  dprintf(int s, const char *fmt, ...)
{

	char *buffer;
	int n;
	va_list va;

	va_start(va, fmt);
	n = vsprintf(NULL, fmt, va);
	if(n > MAXLEN)
		std_err(ERROR, "dprintf() failed: string is too long");
	if((buffer = (char *)malloc((n+1) * sizeof(char))) == NULL)
		std_err(ERROR | PERR, "malloc() failed");
	(void)vsnprintf(buffer, n+1, fmt, va);
	*(buffer+n) = '\0';
	va_end(va);

	if(write(s, buffer, n) == -1)
		n = -1;
	free(buffer);

	return n;

}

#endif /* HAVE_DPRINTF */

/* --------------------------------------------------------------------------- */

#include <netdb.h>

inline char *
  get_connection(char *hostname, int port, int *s)
{

	struct	sockaddr_in addr;
	struct	hostent *host;

	if((host = gethostbyname(hostname)) == NULL) {
		return "gethostbyname() failed";
	}

	addr.sin_family	=	AF_INET;
	addr.sin_port	=	htons(port);
	addr.sin_addr	=	*(struct in_addr*)host->h_addr;

	if((*s = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
		return "socket() failed";
	}

	if(connect(*s, (struct sockaddr*)&addr, sizeof(addr)) == -1) {
		return "connect() failed";
	}

	return NULL;

}

#ifdef GET_IFADDR /* -------------------------------------------------------- */

inline unsigned long int
  get_ifaddr(char *ifname)
{

	int	s;
	struct	ifreq ifr;
	struct	sockaddr_in sa;

	(void)strncpy(ifr.ifr_name, ifname, IFNAMSIZ);
	if((s = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		return 0;

	if(s >= 0) {
		if(ioctl(s, SIOCGIFADDR, &ifr)) {
			(void)close(s);
			return 0;
		}

		(void)close(s);
		(void)memcpy(&sa, &ifr.ifr_addr, 16);
		return(sa.sin_addr.s_addr);
	} else {
		(void)close(s);
		return 0;
	}

}

#endif /* GET_IFADDR */

#ifndef HAVE_GETOPT_LONG /* -------------------------------------------------------- */

int __getopt_initialized;
char *__getopt_nonoption_flags;
static int nonoption_flags_len;
static char *nextchar;

char *optarg;
int optbind = 1;
int opterr = 1;
int optopt = '?';

static int first_nonopt;
static int last_nonopt;
static char *posixly_correct;

#define SWAP_FLAGS(ch1, ch2)\
if (nonoption_flags_len > 0)                                            \
{                                                                       \
char __tmp = __getopt_nonoption_flags[ch1];                             \
__getopt_nonoption_flags[ch1] = __getopt_nonoption_flags[ch2];          \
__getopt_nonoption_flags[ch2] = __tmp;                                  \
}

static enum
{
	REQUIRE_ORDER, PERMUTE, RETURN_IN_ORDER
} ordering;

struct option
{
# if (defined __STDC__ && __STDC__) || defined __cplusplus
	const char *name;
# else
	char *name;
# endif
	/* has_arg can't be an enum because some compilers complain about
	   type mismatches in all the code that assumes it is an int.  */
	int has_arg;
	int *flag;
	int val;
};

static char *
my_index (str, chr)
const char *str;
int chr;
{
        while (*str)
        {
                if (*str == chr)
                        return (char *) str;
                str++;
        }
        return 0;
}

static void
exchange (argv)
char **argv;
{
	int bottom = first_nonopt;
	int middle = last_nonopt;
	int top = optind;
	char *tem;

	/* Exchange the shorter segment with the far end of the longer segment.
     That puts the shorter segment into the right place.
     It leaves the longer segment in the right place overall,
     but it consists of two parts that need to be swapped next.  */

#if defined _LIBC && defined USE_NONOPTION_FLAGS
	/* First make sure the handling of the `__getopt_nonoption_flags'
     string can work normally.  Our top argument must be in the range
     of the string.  */
	if (nonoption_flags_len > 0 && top >= nonoption_flags_max_len)
	{
		/* We must extend the array.  The user plays games with us and
		   presents new arguments.  */
		char *new_str = malloc (top + 1);
		if (new_str == NULL)
			nonoption_flags_len = nonoption_flags_max_len = 0;
		else
		{
			memset (__mempcpy (new_str, __getopt_nonoption_flags,
					   nonoption_flags_max_len),
				'\0', top + 1 - nonoption_flags_max_len);
			nonoption_flags_max_len = top + 1;
			__getopt_nonoption_flags = new_str;
		}
	}
#endif

	while (top > middle && middle > bottom)
	{
		if (top - middle > middle - bottom)
		{
			/* Bottom segment is the short one.  */
			int len = middle - bottom;
			register int i;

			/* Swap it with the top part of the top segment.  */
			for (i = 0; i < len; i++)
			{
				tem = argv[bottom + i];
				argv[bottom + i] = argv[top - (middle - bottom) + i];
				argv[top - (middle - bottom) + i] = tem;
				SWAP_FLAGS (bottom + i, top - (middle - bottom) + i);
			}
			/* Exclude the moved bottom segment from further swapping.  */
			top -= len;
		}
		else
		{
			/* Top segment is the short one.  */
			int len = top - middle;
			register int i;

			/* Swap it with the bottom part of the bottom segment.  */
			for (i = 0; i < len; i++)
			{
				tem = argv[bottom + i];
				argv[bottom + i] = argv[middle + i];
				argv[middle + i] = tem;
				SWAP_FLAGS (bottom + i, middle + i);
			}
			/* Exclude the moved top segment from further swapping.  */
			bottom += len;
		}
	}

	/* Update records for the slots the non-options now occupy.  */

	first_nonopt += (optind - last_nonopt);
	last_nonopt = optind;
}

static const char *
_getopt_initialize (argc, argv, optstring)
int argc;
char *const *argv;
const char *optstring;
{
	/* Start processing options with ARGV-element 1 (since ARGV-element 0
     is the program name); the sequence of previously skipped
     non-option ARGV-elements is empty.  */

	first_nonopt = last_nonopt = optind;

	nextchar = NULL;

	posixly_correct = getenv ("POSIXLY_CORRECT");

	/* Determine how to handle the ordering of options and nonoptions.  */

	if (optstring[0] == '-')
	{
		ordering = RETURN_IN_ORDER;
		++optstring;
	}
	else if (optstring[0] == '+')
	{
		ordering = REQUIRE_ORDER;
		++optstring;
	}
	else if (posixly_correct != NULL)
		ordering = REQUIRE_ORDER;
	else
		ordering = PERMUTE;

#if defined _LIBC && defined USE_NONOPTION_FLAGS
	if (posixly_correct == NULL
	    && argc == __libc_argc && argv == __libc_argv)
	{
		if (nonoption_flags_max_len == 0)
		{
			if (__getopt_nonoption_flags == NULL
			    || __getopt_nonoption_flags[0] == '\0')
				nonoption_flags_max_len = -1;
			else
			{
				const char *orig_str = __getopt_nonoption_flags;
				int len = nonoption_flags_max_len = strlen (orig_str);
				if (nonoption_flags_max_len < argc)
					nonoption_flags_max_len = argc;
              __getopt_nonoption_flags =
		      (char *) malloc (nonoption_flags_max_len);
              if (__getopt_nonoption_flags == NULL)
		      nonoption_flags_max_len = -1;
              else
		      memset (__mempcpy (__getopt_nonoption_flags, orig_str, len),
			      '\0', nonoption_flags_max_len - len);
			}
		}
		nonoption_flags_len = nonoption_flags_max_len;
	}
	else
		nonoption_flags_len = 0;
#endif

	return optstring;
}


int
_getopt_internal (argc, argv, optstring, longopts, longind, long_only)
int argc;
char *const *argv;
const char *optstring;
const struct option *longopts;
int *longind;
int long_only;
{
	int print_errors = opterr;
	if (optstring[0] == ':')
		print_errors = 0;

	if (argc < 1)
		return -1;

	optarg = NULL;

	if (optind == 0 || !__getopt_initialized)
	{
		if (optind == 0)
			optind = 1;     /* Don't scan ARGV[0], the program name.  */
		optstring = _getopt_initialize (argc, argv, optstring);
		__getopt_initialized = 1;
	}

	/* Test whether ARGV[optind] points to a non-option argument.
     Either it does not have option syntax, or there is an environment flag
     from the shell indicating it is not an option.  The later information
     is only used when the used in the GNU libc.  */
#if defined _LIBC && defined USE_NONOPTION_FLAGS
# define NONOPTION_P (argv[optind][0] != '-' || argv[optind][1] == '\0'       \
                      || (optind < nonoption_flags_len                        \
                          && __getopt_nonoption_flags[optind] == '1'))
#else
# define NONOPTION_P (argv[optind][0] != '-' || argv[optind][1] == '\0')
#endif

	if (nextchar == NULL || *nextchar == '\0')
	{
		/* Advance to the next ARGV-element.  */

		/* Give FIRST_NONOPT & LAST_NONOPT rational values if OPTIND has been
		   moved back by the user (who may also have changed the arguments).  */
		if (last_nonopt > optind)
			last_nonopt = optind;
		if (first_nonopt > optind)
			first_nonopt = optind;

		if (ordering == PERMUTE)
		{
			/* If we have just processed some options following some non-options,
			   exchange them so that the options come first.  */

			if (first_nonopt != last_nonopt && last_nonopt != optind)
				exchange ((char **) argv);
			else if (last_nonopt != optind)
				first_nonopt = optind;

			/* Skip any additional non-options
			   and extend the range of non-options previously skipped.  */

			while (optind < argc && NONOPTION_P)
				optind++;
			last_nonopt = optind;
		}

		/* The special ARGV-element `--' means premature end of options.
         Skip it like a null option,
         then exchange with previous non-options as if it were an option,
         then skip everything else like a non-option.  */

		if (optind != argc && !strcmp (argv[optind], "--"))
		{
			optind++;

			if (first_nonopt != last_nonopt && last_nonopt != optind)
				exchange ((char **) argv);
			else if (first_nonopt == last_nonopt)
				first_nonopt = optind;
			last_nonopt = argc;
			optind = argc;
		}

		/* If we have done all the ARGV-elements, stop the scan
		   and back over any non-options that we skipped and permuted.  */

		if (optind == argc)
		{
			/* Set the next-arg-index to point at the non-options
			   that we previously skipped, so the caller will digest them.  */
			if (first_nonopt != last_nonopt)
				optind = first_nonopt;
			return -1;
		}

		/* If we have come to a non-option and did not permute it,
		   either stop the scan or describe it to the caller and pass it by.  */

		if (NONOPTION_P)
		{
			if (ordering == REQUIRE_ORDER)
				return -1;
			optarg = argv[optind++];
			return 1;
		}

		/* We have found another option-ARGV-element.
		   Skip the initial punctuation.  */

		nextchar = (argv[optind] + 1
			    + (longopts != NULL && argv[optind][1] == '-'));
	}

	/* Decode the current option-ARGV-element.  */

	/* Check whether the ARGV-element is a long option.

     If long_only and the ARGV-element has the form "-f", where f is
     a valid short option, don't consider it an abbreviated form of
     a long option that starts with f.  Otherwise there would be no
     way to give the -f short option.

     On the other hand, if there's a long option "fubar" and
     the ARGV-element is "-fu", do consider that an abbreviation of
     the long option, just like "--fu", and not "-f" with arg "u".

     This distinction seems to be the most useful approach.  */

	if (longopts != NULL
	    && (argv[optind][1] == '-'
		|| (long_only && (argv[optind][2] || !my_index (optstring, argv[optind][1])))))
	{
		char *nameend;
		const struct option *p;
		const struct option *pfound = NULL;
		int exact = 0;
		int ambig = 0;
		int indfound = -1;
		int option_index;

		for (nameend = nextchar; *nameend && *nameend != '='; nameend++)
			/* Do nothing.  */ ;

		/* Test all long options for either exact match
		   or abbreviated matches.  */
		for (p = longopts, option_index = 0; p->name; p++, option_index++)
			if (!strncmp (p->name, nextchar, nameend - nextchar))
			{
				if ((unsigned int) (nameend - nextchar)
				    == (unsigned int) strlen (p->name))
				{
					/* Exact match found.  */
					pfound = p;
					indfound = option_index;
					exact = 1;
					break;
				}
				else if (pfound == NULL)
				{
					/* First nonexact match found.  */
					pfound = p;
					indfound = option_index;
				}
				else if (long_only
					 || pfound->has_arg != p->has_arg
					 || pfound->flag != p->flag
					 || pfound->val != p->val)
					/* Second or later nonexact match found.  */
					ambig = 1;
			}

		if (ambig && !exact)
		{
			if (print_errors)
			{
#if defined _LIBC && defined USE_IN_LIBIO
				char *buf;

				__asprintf (&buf, ("%s: option `%s' is ambiguous\n"),
					    argv[0], argv[optind]);

				if (_IO_fwide (stderr, 0) > 0)
					__fwprintf (stderr, L"%s", buf);
				else
					fputs (buf, stderr);

				free (buf);
#else
				fprintf (stderr, ("%s: option `%s' is ambiguous\n"),
					 argv[0], argv[optind]);
#endif
			}
			nextchar += strlen (nextchar);
			optind++;
			optopt = 0;
			return '?';
		}

		if (pfound != NULL)
		{
			option_index = indfound;
			optind++;
			if (*nameend)
			{
				/* Don't test has_arg with >, because some C compilers don't
				   allow it to be used on enums.  */
				if (pfound->has_arg)
					optarg = nameend + 1;
				else
				{
					if (print_errors)
					{
#if defined _LIBC && defined USE_IN_LIBIO
						char *buf;
#endif

						if (argv[optind - 1][1] == '-')
						{
							/* --option */
#if defined _LIBC && defined USE_IN_LIBIO
							__asprintf (&buf, ("%s: option `--%s' doesn't allow an argument\n"),
								    argv[0], pfound->name);
#else
							fprintf (stderr, ("%s: option `--%s' doesn't allow an argument\n"),
								 argv[0], pfound->name);
#endif
						}
						else
						{
							/* +option or -option */
#if defined _LIBC && defined USE_IN_LIBIO
							__asprintf (&buf, ("%s: option `%c%s' doesn't allow an argument\n"),
								    argv[0], argv[optind - 1][0],
								    pfound->name);
#else
							fprintf (stderr, ("%s: option `%c%s' doesn't allow an argument\n"),
								 argv[0], argv[optind - 1][0], pfound->name);
#endif
						}

#if defined _LIBC && defined USE_IN_LIBIO
						if (_IO_fwide (stderr, 0) > 0)
							__fwprintf (stderr, L"%s", buf);
						else
							fputs (buf, stderr);

						free (buf);
#endif
					}

					nextchar += strlen (nextchar);

					optopt = pfound->val;
					return '?';
				}
			}
			else if (pfound->has_arg == 1)
			{
				if (optind < argc)
					optarg = argv[optind++];
				else
				{
					if (print_errors)
					{
#if defined _LIBC && defined USE_IN_LIBIO
						char *buf;

						__asprintf (&buf,
							    ("%s: option `%s' requires an argument\n"),
							    argv[0], argv[optind - 1]);

						if (_IO_fwide (stderr, 0) > 0)
							__fwprintf (stderr, L"%s", buf);
						else
							fputs (buf, stderr);

						free (buf);
#else
						fprintf (stderr,
							 ("%s: option `%s' requires an argument\n"),
							 argv[0], argv[optind - 1]);
#endif
					}
					nextchar += strlen (nextchar);
					optopt = pfound->val;
					return optstring[0] == ':' ? ':' : '?';
				}
			}
			nextchar += strlen (nextchar);
			if (longind != NULL)
				*longind = option_index;
			if (pfound->flag)
			{
				*(pfound->flag) = pfound->val;
				return 0;
			}
			return pfound->val;
		}
		/* Can't find it as a long option.  If this is not getopt_long_only,
         or the option starts with '--' or is not a valid short
         option, then it's an error.
         Otherwise interpret it as a short option.  */
		if (!long_only || argv[optind][1] == '-'
		    || my_index (optstring, *nextchar) == NULL)
		{
			if (print_errors)
			{
#if defined _LIBC && defined USE_IN_LIBIO
				char *buf;
#endif

				if (argv[optind][1] == '-')
				{
					/* --option */
#if defined _LIBC && defined USE_IN_LIBIO
					__asprintf (&buf, ("%s: unrecognized option `--%s'\n"),
						    argv[0], nextchar);
#else
					fprintf (stderr, ("%s: unrecognized option `--%s'\n"),
						 argv[0], nextchar);
#endif
				}
				else
				{
					/* +option or -option */
#if defined _LIBC && defined USE_IN_LIBIO
					__asprintf (&buf, ("%s: unrecognized option `%c%s'\n"),
						    argv[0], argv[optind][0], nextchar);
#else
					fprintf (stderr, ("%s: unrecognized option `%c%s'\n"),
						 argv[0], argv[optind][0], nextchar);
#endif
				}

#if defined _LIBC && defined USE_IN_LIBIO
				if (_IO_fwide (stderr, 0) > 0)
					__fwprintf (stderr, L"%s", buf);
				else
					fputs (buf, stderr);

				free (buf);
#endif
			}
			nextchar = (char *) "";
			optind++;
			optopt = 0;
			return '?';
		}
	}

	/* Look at and handle the next short option-character.  */

	{
		char c = *nextchar++;
		char *temp = my_index (optstring, c);

		/* Increment `optind' when we start to process its last character.  */
		if (*nextchar == '\0')
			++optind;

		if (temp == NULL || c == ':')
		{
			if (print_errors)
			{
#if defined _LIBC && defined USE_IN_LIBIO
				char *buf;
#endif

				if (posixly_correct)
				{
					/* 1003.2 specifies the format of this message.  */
#if defined _LIBC && defined USE_IN_LIBIO
					__asprintf (&buf, ("%s: illegal option -- %c\n"),
						    argv[0], c);
#else
					fprintf (stderr, ("%s: illegal option -- %c\n"), argv[0], c);
#endif
				}
				else
				{
#if defined _LIBC && defined USE_IN_LIBIO
					__asprintf (&buf, ("%s: invalid option -- %c\n"),
						    argv[0], c);
#else
					fprintf (stderr, ("%s: invalid option -- %c\n"), argv[0], c);
#endif
				}

#if defined _LIBC && defined USE_IN_LIBIO
				if (_IO_fwide (stderr, 0) > 0)
					__fwprintf (stderr, "%s", buf);
				else
					fputs (buf, stderr);

				free (buf);
#endif
			}
			optopt = c;
			return '?';
		}
		/* Convenience. Treat POSIX -W foo same as long option --foo */
		if (temp[0] == 'W' && temp[1] == ';')
		{
			char *nameend;
			const struct option *p;
			const struct option *pfound = NULL;
			int exact = 0;
			int ambig = 0;
			int indfound = 0;
			int option_index;

			/* This is an option that requires an argument.  */
			if (*nextchar != '\0')
			{
				optarg = nextchar;
				/* If we end this ARGV-element by taking the rest as an arg,
				   we must advance to the next element now.  */
				optind++;
			}
			else if (optind == argc)
			{
				if (print_errors)
				{
					/* 1003.2 specifies the format of this message.  */
#if defined _LIBC && defined USE_IN_LIBIO
					char *buf;

					__asprintf (&buf, ("%s: option requires an argument -- %c\n"),
						    argv[0], c);

					if (_IO_fwide (stderr, 0) > 0)
						__fwprintf (stderr, L"%s", buf);
					else
						fputs (buf, stderr);

					free (buf);
#else
					fprintf (stderr, ("%s: option requires an argument -- %c\n"),
						 argv[0], c);
#endif
				}
				optopt = c;
				if (optstring[0] == ':')
					c = ':';
				else
					c = '?';
				return c;
			}
			else
				/* We already incremented `optind' once;
				   increment it again when taking next ARGV-elt as argument.  */
				optarg = argv[optind++];

			/* optarg is now the argument, see if it's in the
			   table of longopts.  */

			for (nextchar = nameend = optarg; *nameend && *nameend != '='; nameend++)
				/* Do nothing.  */ ;

			/* Test all long options for either exact match
			   or abbreviated matches.  */
			for (p = longopts, option_index = 0; p->name; p++, option_index++)
				if (!strncmp (p->name, nextchar, nameend - nextchar))
				{
					if ((unsigned int) (nameend - nextchar) == strlen (p->name))
					{
						/* Exact match found.  */
						pfound = p;
						indfound = option_index;
						exact = 1;
						break;
					}
					else if (pfound == NULL)
					{
						/* First nonexact match found.  */
						pfound = p;
						indfound = option_index;
					}
					else
						/* Second or later nonexact match found.  */
						ambig = 1;
				}
			if (ambig && !exact)
			{
				if (print_errors)
				{
#if defined _LIBC && defined USE_IN_LIBIO
					char *buf;

					__asprintf (&buf, ("%s: option `-W %s' is ambiguous\n"),
						    argv[0], argv[optind]);

					if (_IO_fwide (stderr, 0) > 0)
						__fwprintf (stderr, L"%s", buf);
					else
						fputs (buf, stderr);

					free (buf);
#else
					fprintf (stderr, ("%s: option `-W %s' is ambiguous\n"),
						 argv[0], argv[optind]);
#endif
				}
				nextchar += strlen (nextchar);
				optind++;
				return '?';
			}
			if (pfound != NULL)
			{
				option_index = indfound;
				if (*nameend)
				{
					/* Don't test has_arg with >, because some C compilers don't
					   allow it to be used on enums.  */
					if (pfound->has_arg)
						optarg = nameend + 1;
					else
					{
						if (print_errors)
						{
#if defined _LIBC && defined USE_IN_LIBIO
							char *buf;

							__asprintf (&buf, ("%s: option `-W %s' doesn't allow an argument\n"),
								    argv[0], pfound->name);

							if (_IO_fwide (stderr, 0) > 0)
								__fwprintf (stderr, L"%s", buf);
							else
								fputs (buf, stderr);

							free (buf);
#else
							fprintf (stderr, ("%s: option `-W %s' doesn't allow an argument\n"),
								 argv[0], pfound->name);
#endif
						}

						nextchar += strlen (nextchar);
						return '?';
					}
				}
				else if (pfound->has_arg == 1)
				{
					if (optind < argc)
						optarg = argv[optind++];
					else
					{
						if (print_errors)
						{
#if defined _LIBC && defined USE_IN_LIBIO
							char *buf;

							__asprintf (&buf, ("%s: option `%s' requires an argument\n"),
								    argv[0], argv[optind - 1]);

							if (_IO_fwide (stderr, 0) > 0)
								__fwprintf (stderr, L"%s", buf);
							else
								fputs (buf, stderr);

							free (buf);
#else
							fprintf (stderr,
								 ("%s: option `%s' requires an argument\n"),
								 argv[0], argv[optind - 1]);
#endif
						}
						nextchar += strlen (nextchar);
						return optstring[0] == ':' ? ':' : '?';
					}
				}
				nextchar += strlen (nextchar);
				if (longind != NULL)
					*longind = option_index;
				if (pfound->flag)
				{
					*(pfound->flag) = pfound->val;
					return 0;
				}
				return pfound->val;
			}
			nextchar = NULL;
			return 'W';   /* Let the application handle it.   */
		}
		if (temp[1] == ':')
		{
			if (temp[2] == ':')
			{
				/* This is an option that accepts an argument optionally.  */
				if (*nextchar != '\0')
				{
					optarg = nextchar;
					optind++;
				}
				else
					optarg = NULL;
				nextchar = NULL;
			}
			else
			{
				/* This is an option that requires an argument.  */
				if (*nextchar != '\0')
				{
					optarg = nextchar;
					/* If we end this ARGV-element by taking the rest as an arg,
					   we must advance to the next element now.  */
					optind++;
				}
				else if (optind == argc)
				{
					if (print_errors)
					{
						/* 1003.2 specifies the format of this message.  */
#if defined _LIBC && defined USE_IN_LIBIO
						char *buf;

						__asprintf (&buf,
							    ("%s: option requires an argument -- %c\n"),
							    argv[0], c);

						if (_IO_fwide (stderr, 0) > 0)
							__fwprintf (stderr, L"%s", buf);
						else
							fputs (buf, stderr);

						free (buf);
#else
						fprintf (stderr,
							 ("%s: option requires an argument -- %c\n"),
							 argv[0], c);
#endif
					}
					optopt = c;
					if (optstring[0] == ':')
						c = ':';
					else
						c = '?';
				}
				else
					/* We already incremented `optind' once;
					   increment it again when taking next ARGV-elt as argument.  */
					optarg = argv[optind++];
				nextchar = NULL;
			}
		}
		return c;
	}
}



int
getopt_long (argc, argv, options, long_options, opt_index)
int argc;
char *const *argv;
const char *options;
const struct option *long_options;
int *opt_index;
{
	return _getopt_internal (argc, argv, options, long_options, opt_index, 0);
}

#endif /* HAVE_GETOPT_LONG */
