/* -- error.c --
 *
 * Copyright (c) 2002, 2003 Philipp Benner <philipp@philippb.tk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * The GNU C Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with the GNU C Library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <syslog.h>
#include <netdb.h>
#include <errno.h>

#include <config.h>

#define GETMODE(mode, doexit, err_type, priority)	\
	doexit   = mode&0x20;	/* 0x20: 100000 */	\
	err_type = mode&0x18;	/* 0x18: 011000 */	\
	priority = mode&0x07;	/* 0x07: 000111 */


#define STANDARD	(0<<3)
#define PERROR		(1<<3)
#define HERROR		(2<<3)
#define DEBUGINFO	7

char *
  create_buffer(int err_type, const char *msg)
{
	const char *ptr;
	char *buffer;
	int buflen = strlen(msg)+1;

	switch(err_type) {
	case PERROR:
		ptr = strerror(errno);
		break;
	case HERROR:
		ptr = hstrerror(h_errno);
		break;
	default:
		exit(EXIT_FAILURE);
	}

	buflen += strlen(ptr)+2;
	if((buffer = (char *)malloc(buflen)) == NULL) {
		perror("malloc() failed");
		exit(EXIT_FAILURE);
	}
	(void)sprintf(buffer, "%s: %s", msg, ptr);
	*(buffer+buflen-1) = '\0';

	return buffer;
}

int
  vstd_err(int mode, const char *msg, va_list az)
{
	int err_type, priority, doexit;
	int i;

	GETMODE(mode, doexit, err_type, priority);
#ifndef DEBUG
	if(priority == DEBUGINFO)
	    return 0;
#endif
	if(err_type) {
		char *buffer;
		buffer = create_buffer(err_type, msg);
		i = vfprintf(stderr, buffer, az);
		free(buffer);
	} else {
		i = vfprintf(stderr, msg, az);
	}
	(void)fprintf(stderr, "\n");

	if(doexit)
	    exit(EXIT_FAILURE);

	return ++i;
}

int
  std_err(int mode, const char *msg, ...)
{
	va_list az;
	int i;

	va_start(az, msg);
	i = vstd_err(mode, msg, az);
	va_end(az);
	return i;
}

void
  vlog_err(int mode, const char *msg, va_list az)
{
	int err_type, priority, doexit;

	GETMODE(mode, doexit, err_type, priority);
#ifndef DEBUG
	if(priority == DEBUGINFO)
	    return;
#endif
	if(err_type) {
		char *buffer;
		buffer = create_buffer(err_type, msg);
		vsyslog(priority, buffer, az);
		free(buffer);
	} else {
		vsyslog(priority, msg, az);
	}

	if(doexit)
	    exit(EXIT_FAILURE);

	return;
}

void
  log_err(int mode, const char *msg, ...)
{
	va_list az;

	va_start(az, msg);
	vlog_err(mode, msg, az);
	va_end(az);
	return;
}
