/* -- updatedd: dyndns.c --
 *
 * Copyright (c) 2002, 2003 Philipp Benner <philipp@philippb.tk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * The GNU C Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with the GNU C Library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA.
 *
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_LONG
# include <getopt.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <base64encode.h>
#include <get_connection.h>
#include <error.h>
#include <version.h>
#ifndef HAVE_DPRINTF
# include <dprintf.h>
#endif
#include <utils.h>

#define DYNDNSHOST	"members.dyndns.org"
#define CHECKIPHOST	"checkip.dyndns.org"
#define PORT		80

#define MAXDATA 	512
#define	BUFSIZE		256

void
  print_usage(char *argv, FILE *fp)
{
	(void)fprintf(fp,
		"\n                                Updatedd - \033[0;31;1mdyndns\033[0m\n\n"
		"Usage: %s [--with-backmx] [--wildcard <on|off>] [--with-syslog] [--offline] [--mx <mxhost>]\n"
		"	[--service <service>] [--force] [--ip <ip address>]\n"
		"	<--hostnames <h1,h2,h3...>> <--system <system>> <--user <user:pass>>\n\n"
		"	-b	--with-backmx		enable backmx\n"
		"	-f	--force			force update\n"
		"	-h	--help			print this help\n"
		"	-i	--ip <ip>		ip address\n"
		"	-L	--list-services		list supported services\n", argv);
	(void)fprintf(fp,
		"	-m	--mx <mxhost>		hostname of your mail exchange\n"
		"	-n	--hostnames <h1,h2...>	your hostnames, seperate each with a comma\n"
		"	-o	--offline		set host to offline mode\n"
		"	-S	--service <service>	select the type of service you are using\n"
		"	-s	--system <system>	specify the system (dyndns, statdns, custom)\n"
		"	-u	--user <user:pass>	user and password to login with\n"
		"	-v	--version		print version\n"
		"	-w	--wildcard <on|off>	enable wildcard\n"
		"	-y	--with-syslog		print error messages to syslog\n\n");

	return;
}

struct arguments {
	char *backmx;
	char *hostnames;
	char *ip;
	char *mx;
	char *offline;
	char *system;
	char *user;
	char *wildcard;
	int   force;
};

int i_syslog = 0;

static struct yesno {
	char *yes;
	char *no;
	char nothing;
} yn = { "yes", "no", '\0' };

static struct systems {
	char *dyndns;
	char *statdns;
	char *custom;
} s_system = { "dyndns", "statdns", "custom" };

struct ip_addresses {
	struct in_addr real_ip;
	struct in_addr cached;
};

static struct dyndns_return_codes {
	char *code;
	char *message;
	int  error;
}
return_codes[] = {
	{ "badauth",	"Bad authorization (username or password).",		1 },
	{ "badsys",	"The system parameter given was not valid.",		1 },
	{ "badagent",	"The useragent your client sent has been blocked "
			"at the access level.", 				1
	},
	{ "good",	"Update good and successful, IP updated.",		0 },
	{ "nochg",	"No changes, update considered abusive.",		0 },
	{ "notfqdn",	"A Fully-Qualified Domain Name was not provided.",	1 },
	{ "nohost",	"The hostname specified does not exist.",		1 },
	{ "!donator",	"The offline setting was set, when the user is "
			"not a donator.", 					1
	},
	{ "!yours",	"The hostname specified exists, but not under "
			"the username currently being used.",			1
	},
	{ "!active",	"The hostname specified is in a Custom DNS "
			"domain which has not yet been activated.",		1
	},
	{ "abuse",	"The hostname specified is blocked for abuse",		1 },
	{ "notfqdn",	"No hosts are given.",					1 },
	{ "numhost",	"Too many or too few hosts found.",			1 },
	{ "dnserr",	"DNS error encountered.",				1 },
	{ NULL,		NULL,							0 }
};

int get_flags(struct arguments *args, int argc, char *argv[]);
int check_dyndns(struct ip_addresses *ip, char *cachefile);
void update_dyndns(int s, struct arguments *args, struct ip_addresses *ip);
void update_cache(struct ip_addresses *ip, char *cachefile);
int get_ip(int s, struct ip_addresses *ip);
int get_local_cached_ip(struct ip_addresses *ip, char *cachefile);
int check_server_msg(int s, char *hostnames);
void copy_line(char *buffer, char *string);

inline char *
  cache_file_name(struct arguments *args, char cachefile[])
{

	char user[BUFSIZE];

	bzero(cachefile, BUFSIZE);
	bzero(user, BUFSIZE);

	strncpy(user, args->user, BUFSIZE);
	snprintf(cachefile, BUFSIZE, "%s%s%s%s%s",
		"/tmp/updatedd-dyndns-", strtok(user, ":"),
		"-", args->system, ".ip");

	return cachefile;

}

void
  print_error(int mode, const char *msg, ...)
{

	va_list az;

	va_start(az, msg);
	if(i_syslog == 1)
		vlog_err(mode, msg, az);
	else
		vstd_err(mode, msg, az);
	va_end(az);

	return;

}

void
  dyndns(int argc, char *argv[])
{

	struct arguments args = {
		NULL, NULL, NULL, NULL,
			NULL, NULL, NULL, NULL, 0
	};
	struct ip_addresses ip;
	int s = 0, i, result;
	char *ptr, cachefile[BUFSIZE];

	(void)memset(&ip, 0, sizeof(ip));

	if(get_flags(&args, argc, argv)) {
		print_usage(argv[0], stderr);
		exit(EXIT_FAILURE);
	}

	if(args.ip == NULL) {
		result = 1;
		while(result) {
			if((ptr = get_connection(CHECKIPHOST, PORT, &s)))
			    print_error(ERROR | EXIT, "%s: %s", ptr, CHECKIPHOST);
			result = get_ip(s, &ip);
			(void)close(s);
		}
	} else {
		if(!inet_aton(args.ip, &ip.real_ip))
			print_error(ERROR | EXIT, "invalid ip address: %s", args.ip);
	}

	
	if(check_dyndns(&ip, cache_file_name(&args, cachefile)) || args.force) {
		if((ptr = get_connection(DYNDNSHOST, PORT, &s)))
			print_error(ERROR | EXIT, "%s: %s", ptr, DYNDNSHOST);
		update_dyndns(s, &args, &ip);
		if((i = check_server_msg(s, args.hostnames)))
			print_error(ERROR | EXIT, "%i error(s) occured", i);
		else update_cache(&ip, cachefile);
		(void)close(s);
	}
	else print_error(INFO, "update is not necessary");

	return;

}

int
  get_flags(struct arguments *args, int argc, char *argv[])
{

	int c;

	while(1) {

		int option_index = 0;
		static struct option long_options[] = {
			{ "with-backmx",	0, 0, 'b' },
			{ "with-syslog",	0, 0, 'y' },
			{ "wildcard",		1, 0, 'w' },
			{ "force",		0, 0, 'f' },
			{ "offline",		0, 0, 'o' },
			{ "help",		0, 0, 'h' },
			{ "hostnames",		1, 0, 'n' },
			{ "ip",			1, 0, 'i' },
			{ "service",		1, 0, 'S' },
			{ "system",		1, 0, 's' },
			{ "mx",			1, 0, 'm' },
			{ "version",		0, 0, 'v' },
			{ "user",		1, 0, 'u' },
			{ NULL,			0, 0, 0   }
		};

		c = getopt_long(argc, argv, "bfi:yohw:n:S:s:m:vu:",
				long_options, &option_index);

		if(c == -1)
			break;

		switch(c) {
		case 'b':
			args->backmx = yn.yes;
			break;
		case 'f':
			args->force = 1;
			break;
		case 'i':
			args->ip = optarg;
			break;
		case 'y':
			i_syslog = 1;
			break;
		case 'h':
			print_usage(argv[0], stdout);
			exit(EXIT_SUCCESS);
		case 'm':
			args->mx = optarg;
			break;
		case 'n':
			args->hostnames = optarg;
			break;
		case 'o':
			args->offline = yn.yes;
			break;
		case 'S':
			break;
		case 's':
			if(strstr(optarg, "custom"))
				args->system = s_system.custom;
			else if(strstr(optarg, "statdns"))
				args->system = s_system.statdns;
			else
				args->system = s_system.dyndns;
			break;
		case 'w':
			if(strstr(optarg, "ON") || strstr(optarg, "on") || strstr(optarg, "On"))
				args->wildcard = "ON";
			else if(strstr(optarg, "OFF") || strstr(optarg, "off") || strstr(optarg, "Off"))
				args->wildcard = "OFF";
			else
				return 1;
			break;
		case 'v':
			(void)printf(VOUTPUT);
			exit(EXIT_SUCCESS);
		case 'u':
			if((args->user = (char *)malloc(strlen(optarg) + 1)) == NULL)
				print_error(ERROR | EXIT | PERR, "malloc() failed");
			strcpy(args->user, optarg);
			args->user[strlen(optarg)] = '\0';
			bzero(optarg, strlen(optarg));
			    /* prevent that anybody can see your pass with ps */
		}
	}

	if(args->backmx == NULL)
		args->backmx = yn.no;
	if(args->wildcard == NULL)
		args->wildcard = "NOCHG";
	if(args->mx == NULL)
		args->mx = &yn.nothing;
	if(args->hostnames == NULL || args->system == NULL || args->user == NULL)
		return 1;
	if(args->offline == NULL)
		args->offline = yn.no;

	return 0;

}

int
  check_dyndns(struct ip_addresses *ip, char *cachefile)
{

	if(get_local_cached_ip(ip, cachefile))
		return 1;
	if(ip->real_ip.s_addr != ip->cached.s_addr)
		return 1;
	return 0;

}

int
  get_ip(int s, struct ip_addresses *ip)
{

	char server_msg[MAXDATA];
	char *ptr;

	(void)dprintf(s,	
			"GET / HTTP/1.1\r\n"
			"Host: %s\r\n"
			"User-Agent: %s %s - %s\r\n"
			"Connection: close\r\n"
			"Pragma: no-cache\r\n\r\n",
			CHECKIPHOST, PNAME, VERSION, HOMEPAGE);

	(void)memset(server_msg, 0, sizeof(server_msg));
	if(read(s, server_msg, sizeof(server_msg) - 1) <= 0)
		print_error(ERROR | EXIT | PERR, "read() failed");

	if(strstr(server_msg, "HTTP/1.1 200 OK") ||
	   strstr(server_msg, "HTTP/1.0 200 OK")) {

		if( !((ptr = strstr(server_msg, "Address: ")) &&
		      (ptr = strtok(ptr + sizeof("Address: ") - 1, "\t\n <")) &&
		      (0 != inet_aton(ptr, &ip->real_ip)))) {

			print_error(DEBUGINFO, "received invalid server message");
			print_error(DEBUGINFO, "%s", server_msg);
			return 1;
		}

	} else {
		print_error(ERROR | EXIT, "Internal Server Error");
	}

	return 0;

}

int
  get_local_cached_ip(struct ip_addresses *ip, char *cachefile)
{

	FILE *fp;
	char ip_buffer[18] = { '\0' };

	if((fp = fopen(cachefile, "r")) == NULL)
		return 1;
	if(!(fgets(ip_buffer, sizeof(ip_buffer) - 1, fp) &&
	     inet_aton(ip_buffer, &ip->cached) != 0)) {
		(void)fclose(fp);
		print_error(ERROR | EXIT, "%s: invalid ip address", cachefile);
	}
	(void)fclose(fp);

	return 0;

}

void
  update_dyndns(int s, struct arguments *args, struct ip_addresses *ip)
{

	char *b64user;

	if(strlen(args->user) > 128)
		print_error(ERROR | EXIT, "username is too long");
	if((b64user = (char *)malloc((2 * strlen(args->user) + 1))) == NULL)
		print_error(ERROR | EXIT | PERR, "malloc() failed");
	(void)memset(b64user, 0, 2 * strlen(args->user) + 1);

	base64encode(args->user, b64user);
	(void)dprintf(s,
		"GET /nic/update?system=%s&hostname=%s&myip=%s&wildcard=%s"
		"&backmx=%s&offline=%s&mx=%s HTTP/1.1\r\n"
		"Host: %s\r\n"
		"Authorization: basic %s\r\n"
		"User-Agent: %s %s - %s\r\n"
		"Connection: close\r\n"
		"Pragma: no-cache\r\n\r\n",
		args->system, args->hostnames, inet_ntoa(ip->real_ip),
		args->wildcard,	args->backmx, args->offline, args->mx,
		DYNDNSHOST, b64user, PNAME, VERSION, HOMEPAGE);
	free(b64user);
	return;

}

void
  update_cache(struct ip_addresses *ip, char *cachefile)
{

	FILE *fp;

	(void)unlink(cachefile);
	if((fp = fopen(cachefile, "a")) == NULL)
		print_error(ERROR | EXIT | PERR, "fopen() failed");
	(void)fprintf(fp, "%s", inet_ntoa(ip->real_ip));
	fclose(fp);

	return;

}

int
  check_server_msg(int s, char *hostnames)
{

	int n, err = 0, host;
	char server_msg[MAXDATA], *ptr;
	char **host_buffer;

	/* count hostnames */
	for(n=0, host = 0; hostnames[n]; n++) {
		if(hostnames[n] == ',')
			host++;
	}

	/* copy hostnames into host_buffer */
	host_buffer = (char **)malloc(host+1);
	host_buffer[0] = strtok(hostnames, ",");
	for(n = 1; n < host+1; n++)
		host_buffer[n] = strtok(NULL, ",");

	/* get server_msg */
	(void)memset(server_msg, 0, sizeof(server_msg));
	if(read(s, server_msg, sizeof(server_msg) - 1) < 0)
		print_error(ERROR | EXIT | PERR, "read() failed");

	print_error(DEBUGINFO, "\n\nServer message:"
		     "\n--------------------------------------\n"
		     "%s--------------------------------------\n\n",
		     server_msg);
	host=0;
	if(strstr(server_msg, "HTTP/1.1 200 OK") ||
	   strstr(server_msg, "HTTP/1.0 200 OK")) {

		(void)strtok(server_msg, "\n");
		while((ptr = strtok(NULL, "\n")) != NULL) {
			for(n=0; return_codes[n].code != NULL; n++) {
				if(strstr(ptr, return_codes[n].code)) {
					if(return_codes[n].error == 1)
						err++;
					print_error(WARNING, "%s: %s",
						    host_buffer[host], return_codes[n].message);
					++host;
				}
			}
		}
	} else if(strstr(server_msg, "401 Authorization Required")) {
		print_error(ERROR | EXIT, "wrong username or password");
	} else {
		print_error(ERROR | EXIT, "Internal Server Error");
	}
	free(host_buffer);

	return err;
}
