/* -- updatedd: hn.c --
 *
 * Copyright (c) 2002, 2003 Philipp Benner <philipp@philippb.tk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * The GNU C Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with the GNU C Library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA.
 *
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_LONG
# include <getopt.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <base64encode.h>
#include <get_connection.h>
#include <get_ifaddr.h>
#include <error.h>
#include <version.h>
#ifndef HAVE_DPRINTF
# include <dprintf.h>
#endif
#include <utils.h>

#define DYNDNSHOST	"dup.hn.org"
#define PORT		80

#define CACHEFILE	"/tmp/updatedd-hn.ip"
#define MAXDATA 	512

void
  print_usage(char *argv, FILE *fp)
{
	(void)fprintf(fp,
		"\n                                Updatedd - \033[0;34;1mhn\033[0m\n\n"
		"Usage: %s [--force] [--ip <ip>] [--with-syslog] [--service <service>] <--interface <interface>>\n"
		"	<--user <user:pass>>\n\n"
		"	-h	--help			print this help\n"
		"	-f	--force			force update\n"
		"	-i	--interface <interface> interface where to grab the ip from\n"
		"	-L	--list-services		list supported services\n"
		"	-p	--ip <ip>		ip address\n", argv);
	(void)fprintf(fp,
		"	-S	--service <service>	select the type of service you are using\n"
		"	-u	--user <user:pass>	user and password to login with\n"
		"	-v	--version		print version\n"
		"	-y	--with-syslog		print error messages to syslog\n\n");

	return;
}

struct arguments
{
	char *interface;
	char *ip;
	char *user;
	int   force;
};

int i_syslog = 0;

struct ip_addresses
{
	struct in_addr cached;
	struct in_addr real_ip;
};

static struct dyndns_return_codes
{
	char *code;
	char *message;
	int  error;
}

return_codes[] =
{
	{ "101", "Successfully Updated",						0 },
	{ "201", "Failure because previous update occured less than 300 seconds ago",	1 },
	{ "202", "Failure because of server error",					1 },
	{ "203", "Failure because account is frozen (by admin)",			1 },
	{ "204", "Failure because account is locked (by user)",				1 },
	{ NULL,	NULL, 0 }
};

int get_flags(struct arguments *args, int argc, char *argv[]);
int check_dyndns(struct ip_addresses *ip, struct arguments *args);
void update_dyndns(int s, struct ip_addresses *ip, struct arguments *args);
void update_cache(struct ip_addresses *ip);
int get_local_cached_ip(struct ip_addresses *ip);
int check_server_msg(int s);

void
  print_error(int mode, const char *msg, ...)
{

	va_list az;

	va_start(az, msg);
	if(i_syslog == 1)
		vlog_err(mode, msg, az);
	else
		vstd_err(mode, msg, az);
	va_end(az);

	return;

}

void
  dyndns(int argc, char *argv[])
{

	struct arguments args = { NULL, NULL, NULL,0 };
	struct ip_addresses ip;
	char *ptr;
	int s = 0;

	(void)memset(&ip, 0, sizeof(ip));

	if(get_flags(&args, argc, argv)) {
		print_usage(argv[0], stderr);
		exit(EXIT_FAILURE);
	}

	if(check_dyndns(&ip, &args) || args.force)
	{
		if((ptr = get_connection(DYNDNSHOST, PORT, &s)))
			print_error(ERROR | EXIT, "%s: %s", ptr, DYNDNSHOST);
		update_dyndns(s, &ip, &args);
		if(check_server_msg(s))
			print_error(ERROR | EXIT, "unknown server message");
		else update_cache(&ip);
		(void)close(s);
	}
	else print_error(INFO, "update is not necessary");

	return;

}

int
  get_flags(struct arguments *args, int argc, char *argv[])
{

	int c;

	while(1)
	{
		int option_index = 0;
		static struct option long_options[] =
		{
			{ "with-syslog",	0, 0, 'y' },
			{ "help",		0, 0, 'h' },
			{ "force",		0, 0, 'f' },
			{ "interface",		1, 0, 'i' },
			{ "ip",			1, 0, 'p' },
			{ "service",		1, 0, 'S' },
			{ "version",		0, 0, 'v' },
			{ "user",		1, 0, 'u' },
			{ NULL,			0, 0, 0   }
		};

		c = getopt_long(argc, argv, "yhfi:p:S:vu:",
				long_options, &option_index);

		if(c == -1)
			break;

		switch(c) {

		case 'y':
			i_syslog = 1;
			break;
		case 'h':
			print_usage(argv[0], stdout);
			exit(EXIT_SUCCESS);
		case 'f':
			args->force = 0;
			break;
		case 'i':
			args->interface = optarg;
			break;
		case 'p':
			args->ip = optarg;
			break;
		case 'S':
			break;
		case 'v':
			(void)printf(VOUTPUT);
			exit(EXIT_SUCCESS);
		case 'u':
			if((args->user = (char *)malloc(strlen(optarg) + 1)) == NULL)
				print_error(ERROR | EXIT |  PERR, "malloc() failed");
			strcpy(args->user, optarg);
			args->user[strlen(optarg)] = '\0';
			bzero(optarg, strlen(optarg));
			    /* prevent that anybody can see your pass with ps */
		}
	}

	if(args->user == NULL || args->interface == NULL)
		return 1;

	return 0;

}

int
  check_dyndns(struct ip_addresses *ip, struct arguments *args)
{

	if(args->ip) {
		if(!inet_aton(args->ip, &ip->real_ip))
			print_error(ERROR | EXIT, "invalid ip address: %s", args->ip);
	} else {
		if((ip->real_ip.s_addr = get_ifaddr(args->interface)) == 0)
			print_error(ERROR | EXIT, "invalid interface: %s", args->interface);
	}

	if(get_local_cached_ip(ip))
		return 1;
	if(ip->real_ip.s_addr != ip->cached.s_addr)
		return 1;
	return 0;

}

int
  get_local_cached_ip(struct ip_addresses *ip)
{

	FILE *fp;
	char ip_buffer[18] = { '\0' };

	if((fp = fopen(CACHEFILE, "r")) == NULL)
		return 1;
	if(!(fgets(ip_buffer, sizeof(ip_buffer) - 1, fp) &&
	     inet_aton(ip_buffer, &ip->cached) != 0)) {
		(void)fclose(fp);
		print_error(ERROR | EXIT, "%s: invalid ip address", CACHEFILE);
	}
	(void)fclose(fp);

	return 0;

}

void
  update_dyndns(int s, struct ip_addresses *ip, struct arguments *args)
{

	char *b64user;

	if(strlen(args->user) > 128)
		print_error(ERROR | EXIT, "username is too long");
	if((b64user = (char *)malloc((2 * strlen(args->user) + 1) * sizeof(char))) == NULL)
		print_error(ERROR | EXIT | PERR, "malloc() failed");
	(void)memset(b64user, 0, 2 * strlen(args->user) + 1);

	base64encode(args->user, b64user);
	(void)dprintf(s,
		      "GET /vanity/update/?VER=1&IP=%s HTTP/1.1\r\n"
		      "Host: %s\r\n"
		      "Authorization: basic %s\r\n"
		      "User-Agent: %s %s - %s\r\n"
		      "Connection: close\r\n"
		      "Pragma: no-cache\r\n\r\n",
		      inet_ntoa(ip->real_ip), DYNDNSHOST,
		      b64user, PNAME, VERSION, HOMEPAGE);
	free(b64user);
	return;

}

void
  update_cache(struct ip_addresses *ip)
{

	FILE *fp;

	(void)unlink(CACHEFILE);
	if((fp = fopen(CACHEFILE, "a")) == NULL)
		print_error(ERROR | EXIT | PERR, "fopen() failed");
	(void)fprintf(fp, "%s", inet_ntoa(ip->real_ip));
	fclose(fp);

	return;

}

int
  check_server_msg(int s)
{

	int n;
	char server_msg[MAXDATA];

	(void)memset(server_msg, 0, sizeof(server_msg));
	if(read(s, server_msg, sizeof(server_msg) - 1) < 0)
		print_error(ERROR | EXIT | PERR, "read() failed");

	else if(strstr(server_msg, "HTTP/1.1 200 OK") ||
		strstr(server_msg, "HTTP/1.0 200 OK")) {
		for(n=0; return_codes[n].code != NULL; n++) {
			if(strstr(server_msg, return_codes[n].code)) {
				if(return_codes[n].error == 1) {
					print_error(ERROR | EXIT, return_codes[n].message);
				} else {
					print_error(NOTICE, return_codes[n].message);
					return 0;
				}
			}
		}
	} else if(strstr(server_msg, "HTTP/1.1 401 Authorization Required")) {
		print_error(ERROR | EXIT, "bad username or passwd");
	}
	else print_error(ERROR | EXIT, "Internal Server Error");

	return 1;
}
