/* -- updatedd: ods.c --
 *
 * Copyright (c) 2002, 2003 Philipp Benner <philipp@philippb.tk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * The GNU C Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with the GNU C Library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA.
 *
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_LONG
# include <getopt.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <base64encode.h>
#include <get_connection.h>
#include <get_ifaddr.h>
#include <error.h>
#include <version.h>
#ifndef HAVE_DPRINTF
# include <dprintf.h>
#endif
#include <utils.h>

#define DYNDNSHOST	"ods.org"
#define PORT		7070

#define CACHEFILE	"/tmp/updatedd-ods.ip"
#define MAXDATA 	128

void
print_usage(char *argv, FILE *fp)
{
	(void)fprintf(fp,
		"\n                                Updatedd - \033[0;32;1mods\033[0m\n\n"
		"Usage: %s [--with-syslog] [--mx <mxhost>] [--service <service>] [--interface <interface>]\n"
		"	<--hostnames <h1,h2,h3...>> <--user <user:pass>>\n\n"
		"	-h	--help			print this help\n"
		"	-i	--interface <interface>	iterface where to grab the ip from\n"
		"	-L	--list-services		list supported services\n", argv);
	(void)fprintf(fp,
		"	-m	--mx <mxhost>		hostname of your mail exchange\n"
		"	-n	--hostname <hostname>	your hostname\n"
		"	-S	--service <service>	select the type of service you are using\n"
		"	-u	--user <user:pass>	user and password to login with\n"
		"	-v	--version		print version\n"
		"	-y	--with-syslog		print error messages to syslog\n\n");

	return;
}

struct arguments {
	char *hostname;
	char *interface;
	char *mx;
	char *user;
	int   force;
};

int i_syslog = 0;

struct ip_addresses {
	struct in_addr cached;
	struct in_addr real_ip;
};

int get_flags(struct arguments *args, int argc, char *argv[]);
int check_dyndns(char *interface, struct ip_addresses *ip);
void update_dyndns(int s, struct ip_addresses *ip, struct arguments *args);
int get_local_cached_ip(struct ip_addresses *ip);
int check_server_msg(int s);
char *server_msg(int s, char *msg, int err_code);

void
  print_error(int mode, const char *msg, ...)
{

	va_list az;

	va_start(az, msg);
	if(i_syslog == 1)
		vlog_err(mode, msg, az);
	else
		vstd_err(mode, msg, az);
	va_end(az);

	return;

}

void
  dyndns(int argc, char *argv[])
{

	struct arguments args = { NULL, NULL, NULL, NULL, 0 };
	struct ip_addresses ip;
	char *ptr;
	int s = 0;

	(void)memset(&ip, 0, sizeof(ip));

	if(get_flags(&args, argc, argv)) {
		print_usage(argv[0], stderr);
		exit(EXIT_FAILURE);
	}

	if((args.interface != NULL && check_dyndns(args.interface, &ip))
	   || args.interface == NULL) {
		if((ptr = get_connection(DYNDNSHOST, PORT, &s)))
			print_error(ERROR | EXIT, "%s: %s", ptr, DYNDNSHOST);
		update_dyndns(s, &ip, &args);
		(void)close(s);
	}

	return;

}

int
  get_flags(struct arguments *args, int argc, char *argv[])
{

	int c;

	while(1) {

		int option_index = 0;
		static struct option long_options[] = {
			{ "with-syslog",		0, 0, 'y' },
			{ "help",		0, 0, 'h' },
			{ "hostnames",		1, 0, 'n' },
			{ "interface",		1, 0, 'i' },
			{ "service",		1, 0, 'S' },
			{ "mx",			1, 0, 'm' },
			{ "version",		0, 0, 'v' },
			{ "user",		1, 0, 'u' },
			{ NULL,			0, 0, 0   }
		};

		c = getopt_long(argc, argv, "yhi:n:S:m:vu:",
				long_options, &option_index);

		if(c == -1)
			break;

		switch(c) {

		case 'y':
			i_syslog = 1;
			break;
		case 'h':
			print_usage(argv[0], stdout);
			exit(EXIT_SUCCESS);
		case 'i':
			args->interface = optarg;
		case 'm':
			args->mx = optarg;
			break;
		case 'n':
			args->hostname = optarg;
			break;
		case 'S':
			break;
		case 'v':
			(void)printf(VOUTPUT);
			exit(EXIT_SUCCESS);
		case 'u':
			if((args->user = (char *)malloc(strlen(optarg) + 1)) == NULL)
				print_error(ERROR | EXIT | PERR, "malloc() failed");
			strcpy(args->user, optarg);
			args->user[strlen(optarg)] = '\0';
			bzero(optarg, strlen(optarg));
			    /* prevent that anybody can see your pass with ps */
		}
	}

	if(args->hostname == NULL || args->user == NULL)
		return 1;

	return 0;

}

int
  check_dyndns(char *interface, struct ip_addresses *ip)
{

	if((ip->real_ip.s_addr = get_ifaddr(interface)) == 0)
		print_error(ERROR | EXIT, "%s: invalid interface", interface);
	if(get_local_cached_ip(ip))
		return 1;
	if(ip->real_ip.s_addr != ip->cached.s_addr)
		return 1;
	return 0;

}

int
  get_local_cached_ip(struct ip_addresses *ip)
{

	FILE *fp;
	char ip_buffer[18] = { '\0' };

	if((fp = fopen(CACHEFILE, "r")) == NULL)
		return 1;
	if(!(fgets(ip_buffer, sizeof(ip_buffer) - 1, fp) &&
	     inet_aton(ip_buffer, &ip->cached) != 0)) {
		(void)fclose(fp);
		print_error(ERROR | EXIT, "%s: invalid ip address", CACHEFILE);
	}
	(void)fclose(fp);

	return 0;

}

void
  update_dyndns(int s, struct ip_addresses *ip, struct arguments *args)
{

	char *ip_addr, msg[MAXDATA], *ptr;
	char *user, *pass;
	FILE *fp;

	user = strtok(args->user, ":");
	pass = strtok(NULL, ":");

	if(args->interface != NULL) {
		ip_addr = inet_ntoa(ip->real_ip);

		(void)unlink(CACHEFILE);
		if((fp = fopen(CACHEFILE, "a")) == NULL)
			print_error(ERROR | EXIT | PERR, "fopen() failed");
		(void)fprintf(fp, "%s", ip_addr);
		fclose(fp);
	}
	else ip_addr = "CONNIP";

	if((ptr = server_msg(s, msg, 100)) != NULL)
		print_error(ERROR | EXIT, "%s", ptr);
	else {
		(void)dprintf(s, "LOGIN %s %s\n", user, pass);
		if((ptr = server_msg(s, msg, 225)) != NULL)
			print_error(ERROR | EXIT, "%s", ptr);
		else {
			(void)dprintf(s, "DELRR %s A\n", args->hostname);
			if((ptr = server_msg(s, msg, 901)) != NULL)
				print_error(ERROR | EXIT, "%s", ptr);
			else {
				(void)dprintf(s, "ADDRR %s A %s\n", args->hostname, ip_addr);
				if((ptr = server_msg(s, msg, 795)) != NULL)
					print_error(ERROR | EXIT, "%s", ptr);
				else print_error(NOTICE, "update successful");
			}
		}
	}

	return;

}

char *
  server_msg(int s, char *msg, int err_code)
{

	(void)memset(msg, 0, MAXDATA);
	(void)read(s, msg, MAXDATA-1);

	*(msg+3) = '\0';
	if(err_code != atoi(msg))
		return (msg+4);
	return NULL;

}
