
/*
 * Mesa 3-D graphics library
 * Version:  3.0
 * Copyright (C) 1998 Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * BRIAN PAUL, TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY 
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT 
 * OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR  
 * THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "glxlib.h"
#include "glxcommon.h"
#include "glx_varray.h"
#include "buffer_macros.h"

/** set to 1 for *lots* of debugging output and recompile **/
#define DEBUG_VARRAYS 0


#define __GLX_GET_VARRAY_BUFFER(buf, size)				\
   {									\
      if ( GLCurrent->SGIhost || ( (16 + size) > GLCurrent->bufferSize ) ) 	\
      {									\
         /*printf("Large Vertex Array or SGI server!\n");*/		\
         __GLX_GET_RENDER_BUFFER(buf, 4116, 16, size);			\
      } else {								\
         __GLX_GET_RENDER_BUFFER(buf, 193, 16, size);			\
      }									\
   }

/* 
   __GLX_PUT_element
   put a single array's element into the buffer. Uses memcpy for some speed.
   Ie, if vertex3f, copy 12 bytes from the source array to the dest array
*/

#if DEBUG_VARRAYS
#warning Debugging VertexArrays
#warning expect lots of printfs and slowness
#define __GLX_PUT_element(buf, array, index, stride, _size)		\
   {									\
         GLfloat *aptr = (GLfloat *) (array + (index * stride));	\
         GLuint  *iptr = (GLuint *) (array + (index * stride));		\
         int k;								\
         for (k = 0; k < _size/4; k++) {				\
            printf("%1.2f ", *aptr);					\
            printf("(%d) ", *iptr);					\
            aptr++; iptr++;						\
         }								\
         printf("\n");							\
         memcpy(buf, array + (index * stride), _size);			\
         buf += _size;							\
   }
#else
#define __GLX_PUT_element(buf, array, index, stride, _size)		\
   {									\
         memcpy(buf, array + (index * stride), _size);			\
         buf += _size;							\
   }
#endif


/*
   __GLX_PUT_indexed_element
   very slow implementation of putting a single index from the source array
   to the dest array. This works for all cases, and as a result, must check
   which arrays are enabled for each array type, every index. This is why
   it is very slow, but correct. Next will be a sample of a more optimized
   and direct version
*/

#define __GLX_PUT_indexed_element(buf, attribs, index)			\
      if (attribs->EdgeFlagEnabled)					\
         __GLX_PUT_element( (char *) buf, (char *) attribs->EdgeFlagPtr,\
                            index, attribs->EdgeFlagStrideB, 		\
                            GLX_pad(1));				\
      if (attribs->TexCoordEnabled[0])					\
         __GLX_PUT_element( (char *)buf,(char *)attribs->TexCoordPtr[0],\
                            index, attribs->TexCoordStrideB[0], 	\
                            GLX_pad(attribs->TexCoordSize[0] *		\
                            GLX_data_size(attribs->TexCoordType[0])));	\
      if (attribs->ColorEnabled)					\
         __GLX_PUT_element( (char *) buf, (char *) attribs->ColorPtr,	\
                            index, attribs->ColorStrideB, 		\
                            GLX_pad(attribs->ColorSize * 		\
                            GLX_data_size(attribs->ColorType)));	\
      if (attribs->IndexEnabled)					\
         __GLX_PUT_element( (char *) buf, (char *) attribs->IndexPtr,	\
                            index, attribs->IndexStrideB, 		\
                            GLX_pad(GLX_data_size(attribs->IndexType)));\
      if (attribs->NormalEnabled)					\
         __GLX_PUT_element( (char *) buf, (char *) attribs->NormalPtr,	\
                            index, attribs->NormalStrideB, GLX_pad(3 * 	\
                            GLX_data_size(attribs->NormalType)));	\
      if (attribs->VertexEnabled)					\
         __GLX_PUT_element( (char *) buf, (char *) attribs->VertexPtr,	\
                            index, attribs->VertexStrideB, 		\
                            GLX_pad(attribs->VertexSize * 		\
                            GLX_data_size(attribs->VertexType)));

/* 
   Example of a more optimized way to put Vertex Array data into the buffer
   This one is specifically optimized for q3test, and sends textures, colors,
   and vertices
*/

#define TCV_ELEMENT   (VARRAY_TEX | VARRAY_COLORS | VARRAY_VERTS)
#define __GLX_PUT_tcv_element(buf, attribs, index)			\
         __GLX_PUT_element( (char *)buf,(char *)attribs->TexCoordPtr[0],\
                            index, attribs->TexCoordStrideB[0], 	\
                            GLX_pad(attribs->TexCoordSize[0] *		\
                            GLX_data_size(attribs->TexCoordType[0])));	\
         __GLX_PUT_element( (char *) buf, (char *) attribs->ColorPtr,	\
                            index, attribs->ColorStrideB, 		\
                            GLX_pad(attribs->ColorSize * 		\
                            GLX_data_size(attribs->ColorType)));	\
         __GLX_PUT_element( (char *) buf, (char *) attribs->VertexPtr,	\
                            index, attribs->VertexStrideB, 		\
                            GLX_pad(attribs->VertexSize * 		\
                            GLX_data_size(attribs->VertexType)));

/* some other optimized paths */

#define CNV_ELEMENT   (VARRAY_COLORS | VARRAY_NORMS | VARRAY_VERTS)
#define __GLX_PUT_cnv_element(buf, attribs, index)			\
         __GLX_PUT_element( (char *) buf, (char *) attribs->ColorPtr,	\
                            index, attribs->ColorStrideB, color_size);	\
         __GLX_PUT_element( (char *) buf, (char *) attribs->NormalPtr,	\
                            index, attribs->NormalStrideB, norm_size);	\
         __GLX_PUT_element( (char *) buf, (char *) attribs->VertexPtr,	\
                            index, attribs->VertexStrideB, vert_size); 

#if 0

#define __GLX_PUT_array_info(buf, attribs)				\
   if (attribs->EdgeFlagEnabled) {					\
      __GLX_PUT_enum(buf, GL_BYTE);					\
      __GLX_PUT_int(buf, 1);						\
      __GLX_PUT_enum(buf, GL_EDGE_FLAG_ARRAY);				\
   }									\
   if (attribs->TexCoordEnabled[0]) {					\
      __GLX_PUT_enum(buf, attribs->TexCoordType[0]);			\
      __GLX_PUT_int(buf, attribs->TexCoordSize[0]);			\
      __GLX_PUT_enum(buf, GL_TEXTURE_COORD_ARRAY);			\
   }									\
   if (attribs->ColorEnabled) {						\
      __GLX_PUT_enum(buf, attribs->ColorType);				\
      __GLX_PUT_int(buf, attribs->ColorSize);				\
      __GLX_PUT_enum(buf, GL_COLOR_ARRAY);				\
   }									\
   if (attribs->IndexEnabled) {						\
      __GLX_PUT_enum(buf,  attribs->IndexType);				\
      __GLX_PUT_int(buf, 1);						\
      __GLX_PUT_enum(buf, GL_INDEX_ARRAY); 				\
   }									\
   if (attribs->NormalEnabled) {					\
      __GLX_PUT_enum(buf, attribs->NormalType);				\
      __GLX_PUT_int(buf, 3);						\
      __GLX_PUT_enum(buf, GL_NORMAL_ARRAY);				\
   }									\
   if (attribs->VertexEnabled) {					\
      __GLX_PUT_enum(buf, attribs->VertexType);				\
      __GLX_PUT_int(buf, attribs->VertexSize); 				\
      __GLX_PUT_enum(buf, GL_VERTEX_ARRAY);				\
   }
#else
char *
__GLX_PUT_array_info( char *buf, const struct gl_array_attrib *attribs)	{
   if (attribs->EdgeFlagEnabled) {				
      __GLX_PUT_enum(buf, GL_BYTE);					
      __GLX_PUT_int(buf, 1);						
      __GLX_PUT_enum(buf, GL_EDGE_FLAG_ARRAY);				
   }									
   if (attribs->TexCoordEnabled[0]) {					
      __GLX_PUT_enum(buf, attribs->TexCoordType[0]);			
      __GLX_PUT_int(buf, attribs->TexCoordSize[0]);			
      __GLX_PUT_enum(buf, GL_TEXTURE_COORD_ARRAY);			
   }									
   if (attribs->ColorEnabled) {						
      __GLX_PUT_enum(buf, attribs->ColorType);				
      __GLX_PUT_int(buf, attribs->ColorSize);				
      __GLX_PUT_enum(buf, GL_COLOR_ARRAY);				
   }									
   if (attribs->IndexEnabled) {						
      __GLX_PUT_enum(buf,  attribs->IndexType);				
      __GLX_PUT_int(buf, 1);						
      __GLX_PUT_enum(buf, GL_INDEX_ARRAY); 				
   }									
   if (attribs->NormalEnabled) {					
      __GLX_PUT_enum(buf, attribs->NormalType);				
      __GLX_PUT_int(buf, 3);						
      __GLX_PUT_enum(buf, GL_NORMAL_ARRAY);				
   }									
   if (attribs->VertexEnabled) {					
      __GLX_PUT_enum(buf, attribs->VertexType);			
      __GLX_PUT_int(buf, attribs->VertexSize); 				
      __GLX_PUT_enum(buf, GL_VERTEX_ARRAY);				
   }
   return buf;
}
#endif

int 
GLX_enabled_arrays(const struct gl_array_attrib *attribs) {
   int count = 0;

   if (attribs->EdgeFlagEnabled) count += 1;
   if (attribs->TexCoordEnabled[0]) count += 1;
   if (attribs->ColorEnabled) count += 1;
   if (attribs->IndexEnabled) count += 1;
   if (attribs->NormalEnabled) count += 1;
   if (attribs->VertexEnabled) count += 1;

   return count;
}

int 
GLX_varray_size(int num_vertices, const struct gl_array_attrib *attribs)
{
   int count = 0;
   int size = 0;
   int num_arrays = 0;

   if (attribs->EdgeFlagEnabled) {
      size = num_vertices;
      count += GLX_pad(size);
      num_arrays++;
   }
   if (attribs->TexCoordEnabled[0]) {
      size = attribs->TexCoordSize[0] * GLX_data_size(attribs->TexCoordType[0]);
      count += num_vertices * GLX_pad(size);
      num_arrays++;
   }
   if (attribs->ColorEnabled) {
      size = attribs->ColorSize * GLX_data_size(attribs->ColorType);
      count += num_vertices * GLX_pad(size);
      num_arrays++;
   }
   if (attribs->IndexEnabled) {
      size = GLX_data_size(attribs->IndexType);
      count += num_vertices * GLX_pad(size);
      num_arrays++;
   }
   if (attribs->NormalEnabled) {
      size = 3 * GLX_data_size(attribs->NormalType);
      count += num_vertices * GLX_pad(size);
      num_arrays++;
   }
   if (attribs->VertexEnabled) {
      size = attribs->VertexSize * GLX_data_size(attribs->VertexType);
      count += num_vertices * GLX_pad(size); 
      num_arrays++;
   }

   /* count /= 2; */
   count += num_arrays * sizeof(struct array_info);

   return count;
}


void
__glx_draw_arrays(GLenum mode, GLint first, GLint count)
{
   int i;
   char *dest = NULL;
#if DEBUG_VARRAYS
   char *bptr = NULL;
#endif
   const struct gl_array_attrib *attribs = &GLCurrent->Array;
   int num_vertices, enabled, size, array_info_size, dest_stride;
   __GLX_DECLARE_LARGE_VARIABLES;
  
   /* sanity check: */
   if ((count < 1) || (first<0))  {
       fprintf(stderr, "Invalid data: __glx_draw_arrays\n");
       return;
   }
  
   num_vertices = count;
   enabled = GLX_enabled_arrays(attribs);
   size = GLX_varray_size(num_vertices, attribs);
   array_info_size = (enabled * sizeof(struct array_info));
   dest_stride = (size - array_info_size) / count;


#if DEBUG_VARRAYS
   printf("Calling new DrawArrays!\n");
   printf("\tsize is %d\n", size);
   printf("\tenabled is %d\n", enabled);
   printf("\tcount is %d\n", count);
#endif
   if (size == 0) return;
   
   __GLX_GET_VARRAY_BUFFER(dest, size); 
   __GLX_PUT_int(dest, num_vertices);
   __GLX_PUT_int(dest, enabled);
   __GLX_PUT_enum(dest, mode);
   __GLX_START_LARGE_RENDER(dest_stride);
   dest = __GLX_PUT_array_info(dest, attribs);

   bytes_to_send = array_info_size;

#if DEBUG_VARRAYS
   bptr = dest;
   printf("Size is %d\n", size);
   printf("Stride is %d\n", dest_stride);
   printf("Starting with %d\n", bytes_to_send);
   printf("ArrayInfo: %d\n", array_info_size);
   printf("largeReqMaxDataLen: %d\n", GLCurrent->largeReqMaxDataLen);
#endif

   switch (GLCurrent->enabled_varrays) {
      case TCV_ELEMENT:
         for (i = first; i < first + count; i++) {
            __GLX_PUT_tcv_element(dest, attribs, i);
            __GLX_CHECK_LARGE_RENDER;
         }
         break;
      case CNV_ELEMENT:
         {
            GLuint color_size = GLX_pad(attribs->ColorSize * GLX_data_size(attribs->ColorType));
            GLuint norm_size = GLX_pad(3 * GLX_data_size(attribs->NormalType));
            GLuint vert_size = GLX_pad(attribs->VertexSize * GLX_data_size(attribs->VertexType));
         for (i = first; i < first + count; i++) {
            __GLX_PUT_cnv_element(dest, attribs, i);
            __GLX_CHECK_LARGE_RENDER;
         }
         }
         break;
      default:
         for (i = first; i < first + count; i++) {
            __GLX_PUT_indexed_element(dest, attribs, i);
            __GLX_CHECK_LARGE_RENDER;
         }
         break;
   }
   __GLX_END_LARGE_RENDER;

#if DEBUG_VARRAYS
   {
      int f;
      GLfloat *fptr = (GLfloat *) bptr;
      GLuint *iptr = (GLuint *) bptr;
      printf("Final arrays:\n");
      for (f = 0; f < dest_stride/4; f++) {
         printf("%1.2f ", *fptr); 
         printf("(%u) ", *iptr); 
         if ((f-1)%3 == 0) printf("\n");
         fptr++; iptr++;
      }
   }
#endif
}

void
__glx_draw_elements(GLenum mode, GLint count, GLenum type, const GLvoid *indices)
{
   int i;
   char *dest = NULL;
#if DEBUG_VARRAYS
   char *bptr = NULL;
#endif
   const struct gl_array_attrib *attribs = &GLCurrent->Array;
   int num_vertices, enabled, size ,array_info_size,dest_stride;
   __GLX_DECLARE_LARGE_VARIABLES;
  
   /* sanity check: */
   if ((count < 1) || !indices)  {
       fprintf(stderr, "Invalid data: __glx_draw_elements\n");
       return;
   }
  
   num_vertices = count;
   enabled = GLX_enabled_arrays(attribs);
   size = GLX_varray_size(num_vertices, attribs);
   array_info_size = (enabled * sizeof(struct array_info));
   dest_stride = (size - array_info_size) / count;
  
#if DEBUG_VARRAYS
   printf("Calling new DrawElements!\n");
   printf("\tsize is %d\n", size);
#endif

   __GLX_GET_VARRAY_BUFFER(dest, size);
   __GLX_PUT_int(dest, num_vertices);
   __GLX_PUT_int(dest, enabled);
   __GLX_PUT_enum(dest, mode);
   __GLX_START_LARGE_RENDER(dest_stride);
  
   dest = __GLX_PUT_array_info(dest, attribs);
   bytes_to_send = array_info_size;
  
  switch (type) {

      case GL_UNSIGNED_BYTE:
         {
            GLubyte *cast_indices = (GLubyte *) indices;
            for (i = 0; i < count; i++) {
               __GLX_PUT_indexed_element(dest, attribs, cast_indices[i]);
               __GLX_CHECK_LARGE_RENDER;
            }
         }
         break;

      case GL_UNSIGNED_SHORT:
         {
            GLushort *cast_indices = (GLushort *) indices;
            for (i = 0; i < count; i++) {
               __GLX_PUT_indexed_element(dest, attribs, cast_indices[i]);
               __GLX_CHECK_LARGE_RENDER;
            }
         }
         break;

      case GL_UNSIGNED_INT:
         {
            GLuint *cast_indices = (GLuint *) indices;
            for (i = 0; i < count; i++) {
               __GLX_PUT_indexed_element(dest, attribs, cast_indices[i]);
               __GLX_CHECK_LARGE_RENDER;
            }
         }
         break;

      default:
         break;
   }
   __GLX_END_LARGE_RENDER;

}

#include "varray.h"
#include "non_render.h"
#include "render.h"
void __glx_enable_hook(GLenum cap)
{
  switch(cap)
  {
    case GL_VERTEX_ARRAY_EXT : 
    case GL_NORMAL_ARRAY_EXT : 
    case GL_COLOR_ARRAY_EXT  : 
    case GL_INDEX_ARRAY_EXT  :
    case GL_TEXTURE_COORD_ARRAY_EXT:
    case GL_EDGE_FLAG_ARRAY_EXT:
    	__glx_EnableClientState(cap); 
    	break;
    default:
    	__glx_Enable(cap);
  }
}

void __glx_disable_hook(GLenum cap)
{
  switch(cap)
  {
    case GL_VERTEX_ARRAY_EXT : 
    case GL_NORMAL_ARRAY_EXT : 
    case GL_COLOR_ARRAY_EXT  : 
    case GL_INDEX_ARRAY_EXT  :
    case GL_TEXTURE_COORD_ARRAY_EXT:
    case GL_EDGE_FLAG_ARRAY_EXT:
    	__glx_DisableClientState(cap); 
    	break;
    default:
    	__glx_Disable(cap);
  }
}

GLboolean __glx_isenabled_hook(GLenum cap)
{
  switch(cap)
  {
    case GL_VERTEX_ARRAY_EXT : return GLCurrent->Array.VertexEnabled;
    case GL_NORMAL_ARRAY_EXT : return GLCurrent->Array.NormalEnabled;
    case GL_COLOR_ARRAY_EXT  : return GLCurrent->Array.ColorEnabled;
    case GL_INDEX_ARRAY_EXT  : return GLCurrent->Array.IndexEnabled;
    case GL_TEXTURE_COORD_ARRAY_EXT: return GLCurrent->Array.TexCoordEnabled[0];
    case GL_EDGE_FLAG_ARRAY_EXT: return GLCurrent->Array.EdgeFlagEnabled;
    default:
    	return __glx_IsEnabled(cap);
  }
}
