/* $Id: mach64state.c,v 1.33 2000/03/26 06:01:20 gareth Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on MGA driver: mgastate.c ???
 *
 *    Gareth Hughes <gareth@precisioninsight.com>
 */

#include <stdio.h>

#include "mach64glx.h"
#include "pb.h"
#include "dd.h"

#include "glx_symbols.h"

/*

NOTES:


*/

/*
 * mach64IsTexturingEnabled
 * FIXME: i'm sure there is an easier way of checking this
 */
struct gl_texture_object *mach64IsTexturingEnabled( const GLcontext *ctx ) {
	struct gl_texture_object	*tObj;

	if ( !( ctx->Texture.Enabled & (TEXTURE0_1D|TEXTURE0_2D|TEXTURE1_1D|TEXTURE1_2D) ) ) {
		return 0;
	}

	tObj = ctx->Texture.Unit[mach64Ctx->tmu_source[0]].Current;
	if ( !tObj ) {
		return 0;
	}

	/* check for weird mesa state where it looks like we
	are enabled but the texture object's image is an invalid pointer.
	This should probably be fixed in mesa someplace */
	if ( ( tObj != ctx->Texture.Unit[mach64Ctx->tmu_source[0]].CurrentD[2] ) &&
	     ( tObj != ctx->Texture.Unit[mach64Ctx->tmu_source[0]].CurrentD[1] ) ) {
		hwMsg( 5, "mach64IsTexturingEnabled: wtf???\n" );
		return 0;
	}

	return tObj;
}

static int mach64CalcTEX_CNTL( const GLcontext *ctx ) {
	struct gl_texture_object	*tObj;
	int		*source = mach64Ctx->tmu_source;
	int		clampS[2], clampT[2];
	int		reg;
	int		multitex;
	int		ccf;
	int		cblend;
	int		cfilt;
	int		calpha;
	int		agp;

	if ( !mach64IsTexturingEnabled( ctx ) ) {
		return 0;
	}

	/* agp texture setting */
	agp = ( mach64glx.dmaDriver >= 3 );

	/* composite texture functions */
	if ( mach64Ctx->multitex ) {
		multitex = 1;	/* multitexturing */
		tObj = ctx->Texture.Unit[source[1]].Current;
		calpha = ((mach64TextureObject_t *)(tObj->DriverData))->hasAlpha;

		switch ( ctx->Texture.Unit[mach64Ctx->tmu_source[1]].EnvMode ) {
		case GL_BLEND:
			ccf = 0;
			break;
		case GL_MODULATE:
			ccf = 1;
			break;
		case GL_REPLACE:
		case GL_DECAL:
		default:
			ccf = 1;	/* not supported by RagePro! */
			break;
		}

		if ( tObj->MinFilter == GL_LINEAR ||
		     tObj->MinFilter == GL_LINEAR_MIPMAP_NEAREST ||
		     tObj->MinFilter == GL_LINEAR_MIPMAP_LINEAR ) {
			cblend = 1;
		} else {
			cblend = 0;
		}

		if ( tObj->MagFilter == GL_LINEAR ) {
			cfilt = 1;
		} else {
			cfilt = 0;
		}
	} else {
		multitex = 0;	/* single texturing only */

		calpha = 1;
		ccf = 1;	/* multiply */
		cblend = 1;
		cfilt = 1;
	}

	/* primary texture settings */
	tObj = ctx->Texture.Unit[source[0]].Current;

	clampS[0] = ( tObj->WrapS == GL_CLAMP ) ? 1 : 0;
	clampT[0] = ( tObj->WrapT == GL_CLAMP ) ? 1 : 0;

	if ( multitex ) {
		/* secondary texture settings */
		tObj = ctx->Texture.Unit[source[1]].Current;

		clampS[1] = ( tObj->WrapS == GL_CLAMP ) ? 1 : 0;
		clampT[1] = ( tObj->WrapT == GL_CLAMP ) ? 1 : 0;
	} else {
		clampS[1] = 0;
		clampT[1] = 0;
	}

	reg =
	0			/* LOD bias */
	| ( 0 << 4 )		/* composite factor */
	| ( multitex << 8 )	/* 1 = enable texture composite (secondary) */
	| ( ccf << 9 )		/* composite combine factor */
	| ( cblend << 11 )	/* 1 = bilinear on composite minify */
	| ( cfilt << 12 )	/* 1 = bilinear on composite magnify */
	| ( calpha << 13 ) 	/* 1 = composite factor from texture alpha */
	| ( 0 << 14 )		/* 1 = textures in block tiled memory format */
	| ( 0 << 15 )		/* 1 = secondary texture in block tiled format */
	| ( 0 << 16 )		/* 1 = disable texel cache stalls */
	| ( clampS[0] << 17 )	/* texture clamping */
	| ( clampT[0] << 18 )	/* texture clamping */
	| ( 1 << 19 )		/* 1 = disable setup engine ST multiply by W - GTH: broken? */
	| ( agp << 20 )		/* 1 = textures are in AGP memory */
	| ( 0 << 21 )		/* 1 = texture is in VQ compressed format */
	| ( 0 << 22 )		/* 1 = composite texture is in VQ format */
	| ( 1 << 23 )		/* 1 = flush texel cache */
	| ( clampS[1] << 24 )	/* secondary texture S clamp */
	| ( clampT[1] << 25 )	/* secondary texture T clamp */
	| ( 0 << 28 )		/* 1 = "wrap" texture S */
	| ( 0 << 29 )		/* 1 = "wrap" texture T */
	| ( 0 << 30 )		/* 1 = cut texture cache in half */
	| ( multitex << 31 )	/* 1 = use secondary ST for secondary texture */
	;

	return reg;
}

static int mach64CalcSCALE_3D_CNTL( const GLcontext *ctx ) {
	struct gl_texture_object	*tObj;
	int		reg;
	int		s3df;
	int		afn;
	int		asat;
	int		asrc, adst;
	int		tlf;
	int		tfilt;
	int		tblend;
	int		texalpha;
	int		split;

	if ( mach64IsTexturingEnabled( ctx ) ) {
		s3df = 2;		/* texturing */
		tObj = ctx->Texture.Unit[mach64Ctx->tmu_source[0]].Current;
		texalpha = ((mach64TextureObject_t *)(tObj->DriverData))->hasAlpha;

		if ( mach64Ctx->multitex ) {
			tblend = 3;	/* trilinear blend for multitexturing */
			split = 1;	/* split texel cache for performance boost */
		} else if ( tObj->MinFilter == GL_LINEAR ||
			    tObj->MinFilter == GL_LINEAR_MIPMAP_NEAREST ||
			    tObj->MinFilter == GL_LINEAR_MIPMAP_LINEAR ) {
			tblend = 2; split = 0;
		} else {
			tblend = 0; split = 0;
		}

		if ( tObj->MagFilter == GL_LINEAR ) {
			tfilt = 1;
		} else {
			tfilt = 0;
		}

		switch ( ctx->Texture.Unit[mach64Ctx->tmu_source[0]].EnvMode ) {
		case GL_REPLACE:
			tlf = 0;
			break;
		case GL_MODULATE:
			tlf = 1;
			break;
		case GL_DECAL:
			if ( texalpha ) {
				tlf = 2;
			} else {
				tlf = 1;
			}
			break;
		case GL_BLEND:
		default:
			tlf = 1;	/* not supported by RagePro! */
			break;
		}
	} else {
		s3df = 3;	// shading

		texalpha = 0;
		tlf = 1;	// 0 = replace, 1 = modulate, 2 = alpha decal (or is it blend?)
		tfilt = 1;
		tblend = 2;

		split = 0;
	}


	/* blending control */
	if ( ctx->Color.BlendEnabled ) {
 	   	afn = 1;		// blending on, fog off

 	   	// FIXME: check for fog enabled here, which can't be done by hardware

		switch ( ctx->Color.BlendSrcRGB ) {
		case GL_ZERO:
			asrc = 0; asat = 0; break;
		case GL_ONE:
			asrc = 1; asat = 0; break;
		case GL_DST_COLOR:
			asrc = 2; asat = 0; break;
		case GL_ONE_MINUS_DST_COLOR:
			asrc = 3; asat = 0; break;
		case GL_SRC_ALPHA:
			asrc = 4; asat = 0; break;
		case GL_ONE_MINUS_SRC_ALPHA:
			asrc = 5; asat = 0; break;
		case GL_DST_ALPHA:
			if ( mach64glx.bytesPerPixel == 4 ) {
				asrc = 6; asat = 0; break;
			} else {
				/* we don't have an alpha buffer */
				asrc = 0; asat = 0; break;
			}
		case GL_ONE_MINUS_DST_ALPHA:
			if ( mach64glx.bytesPerPixel == 4 ) {
				asrc = 7; asat = 0; break;
			} else {
				/* we don't have an alpha buffer */
				asrc = 0; asat = 0; break;
			}
		case GL_SRC_ALPHA_SATURATE:
			asrc = 0; asat = 1; break;
		default:		/* never happens */
		}

		switch ( ctx->Color.BlendDstRGB ) {
		case GL_ZERO:
			adst = 0; break;
		case GL_ONE:
			adst = 1; break;
		case GL_SRC_COLOR:
			adst = 2; break;
		case GL_ONE_MINUS_SRC_COLOR:
			adst = 3; break;
		case GL_SRC_ALPHA:
			adst = 4; break;
		case GL_ONE_MINUS_SRC_ALPHA:
			adst = 5; break;
		case GL_DST_ALPHA:
			if ( mach64glx.bytesPerPixel == 4 ) {
				adst = 6; break;
			} else {
				/* we don't have an alpha buffer */
				adst = 0; break;
			}

		case GL_ONE_MINUS_DST_ALPHA:
			if ( mach64glx.bytesPerPixel == 4 ) {
				adst = 7; break;
			} else {
				/* we don't have an alpha buffer */
				adst = 0; break;
			}

		default:		/* never happens */
		}
	} else {
 	   	/* blending disabled */

		asat = 0;	// set if alpha saturate blending
		if ( !( ctx->RasterMask & FOG_BIT ) ) {
			afn = 0;	// blending off, fog off
			asrc = 1;
			adst = 0;
		} else {
			afn = 2;	// blending off, fog on
			asrc = 4;	// fog color is now dest and fog factor is alpha, so
			adst = 5;	// use GL_SRC_ALPHA GL_ONE_MINUS_SRC_ALPHA
			texalpha = 0;	// can't use texture alpha when fogging
		}
	}

	reg =
	( 1 << 0 )		// 0 = zero extend, 1 = dynamic range extend pixels to 32 bit
	| ( 0 << 1 )		// 1 = 2D dither, 0 = error diffusion dither
	| ( 0 << 2 )		// 1 = enable dither
	| ( 0 << 3 )		// 1 = reset error diffusion each line
	| ( 1 << 4 )		// 1 = round instead of dither
	| ( 0 << 5 )		// 1 = disable texture cache
	| ( s3df << 6 )
	| ( 0 << 8 )		// 1 = edge anti-alias
	| ( split << 9 )	// 1 = split texture cache
	| ( 0 << 10 )		// 1 = apple YUV mode
	| ( afn << 11 )		// alpha / fog control
	| ( asat << 13 )	// alpha saturate blending
	| ( 0 << 14 )		// 1 = limit red dither range (what for???)
	| ( 0 << 15 )		// 1 = signed dst blend clamp for mpeg
	| ( asrc << 16 )	// blend src
	| ( adst << 19 )	// blend dst
	| ( tlf << 22 )		// texture environment
	| ( 1 << 24 )		// 1 = disable mip-mapping (its broke on all ragepros!)
	| ( tfilt << 25 )	// 1 = bilinear filter texture on mag
	| ( tblend << 26 )	// minification filtering
	| ( 0 << 28 )		// 1 = LSB of alpha for texture masking
	| ( 0 << 29 )		// alpha masking mode
	| ( texalpha << 30 )	// 1 = texture has alpha
	| ( 0 << 31 )		// 1 = source pixel from host register
	;

	return reg;
}



static int mach64CalcZ_CNTL( const GLcontext *ctx ) {
	int		zcntl;

	if ( !ctx->Depth.Test ) {
		return 0;
	}

	zcntl = 1;

	switch( ctx->Depth.Func )
	{
	case GL_NEVER:
	    zcntl |= 0 << 4; break;
	case GL_LESS:
	    zcntl |= 1 << 4; break;
	case GL_LEQUAL:
	    zcntl |= 2 << 4; break;
	case GL_EQUAL:
	    zcntl |= 3 << 4; break;
	case GL_GEQUAL:
	    zcntl |= 4 << 4; break;
	case GL_GREATER:
	    zcntl |= 5 << 4; break;
	case GL_NOTEQUAL:
	    zcntl |= 6 << 4; break;
	case GL_ALWAYS:
	    zcntl |= 7 << 4; break;
	default:
	    break;
	}
	if ( ctx->Depth.Mask ) {
		zcntl |= 256;	/* enable writes to depth buffer */
	}

	return zcntl;
}


static int mach64CalcALPHA_TST_CNTL( GLcontext *ctx ) {
	int		r;
	int		enable;
	int		mode;
	int		adest;
	int		asrc;
	int		ref;
	int		specen;

	if ( ctx->Color.AlphaEnabled ) {
		enable = 1;
		ref =  ctx->Color.AlphaRef;

		switch ( ctx->Color.AlphaFunc ) {
		default:
		case GL_NEVER:
		    mode = 0; break;
		case GL_LESS:
		    mode = 1; break;
		case GL_LEQUAL:
		    mode = 2; break;
		case GL_EQUAL:
		    mode = 3; break;
		case GL_GEQUAL:
		    mode = 4; break;
		case GL_GREATER:
		    mode = 5; break;
		case GL_NOTEQUAL:
		    mode = 6; break;
		case GL_ALWAYS:
		    mode = 7; break;
		}

		/* we are going to always take the texel alpha, because
		we can't get modulated values (rage pro limitation),
		and alpha testing from vertex shading doesn't make much sense */
		asrc = 0;
	} else {
		enable = 0;
		mode = 0;
		asrc = 0;
		ref = 0;
	}

	adest = 4;		/* destination alpha is source alpha */
	specen = 0;		/* FIXME; how do we properly set specular enable? */

	r =
	( enable << 0 )
	| ( mode << 4 )
	| ( 0 << 7 )		/* ALPHA_MOD_MSB weird stuff */
	| ( adest << 8 )	/* select value to write for alpha */
	| ( asrc << 12 )	/* source alpha for comparison */
	| ( ref << 16 )		/* alpha reference value */
	| ( 0 << 24 )		/* 4 bit color index weird stuff */
	| ( 0 << 28 )		/* 4 bit color index weird stuff */
	| ( 0 << 29 )		/* 4 bit color index weird stuff */
	| ( 0 << 30 )		/* composite shadow */
	| ( specen << 31 );	/* specular lighting enable */

	return r;
}


/* =============================================================
 */


/*
 * mach64UpdateTextureState
 */
void mach64UpdateTextureState( GLcontext *ctx )
{
	struct gl_texture_object	*tObj;
	mach64TextureObject_t		*t[2];
	DMALOCALS;

	/* init the first texture */
	tObj = ctx->Texture.Unit[mach64Ctx->tmu_source[0]].Current;

	/* move a texture to the card if needed */
	if ( !tObj->DriverData ) {
		/* this will program registers, so it can't be inside a DMAGETPTR() block */
		mach64CreateTexObj( mach64Ctx, ctx->Texture.Unit[mach64Ctx->tmu_source[0]].Current );
	}
	t[0] = (mach64TextureObject_t *)tObj->DriverData;

	/* init the second texture */
	if ( mach64Ctx->multitex ) {
		tObj = ctx->Texture.Unit[mach64Ctx->tmu_source[1]].Current;

		/* move a texture to the card if needed */
		if ( !tObj->DriverData ) {
			/* again, this can't be inside a DMAGETPTR() block */
			mach64CreateTexObj( mach64Ctx, ctx->Texture.Unit[mach64Ctx->tmu_source[1]].Current );
		}
		t[1] = (mach64TextureObject_t *)tObj->DriverData;
	} else {
		/* use a single texture */
		t[1] = t[0];
	}

	DMAGETPTR( 40 );

	DMAOUTREG( MACH64_TEX_0_OFF + (t[0]->maxLog2<<2), t[0]->memBlock->ofs );
	if ( mach64Ctx->multitex ) {
		DMAOUTREG( MACH64_SECONDARY_TEX_OFF, t[1]->memBlock->ofs );
	}

	DMAOUTREG( MACH64_TEX_SIZE_PITCH,
		   ( t[0]->widthLog2 << 0 )
		   | ( t[0]->maxLog2 << 4 )
		   | ( t[0]->heightLog2 << 8 )
		   | ( t[0]->maxLog2 << 12 )		/* don't care about this */
		   | ( t[1]->widthLog2 << 16 )
		   | ( t[1]->maxLog2 << 20 )
		   | ( t[1]->heightLog2 << 24 )
		   );

	DMAOUTREG( MACH64_TEX_CNTL, mach64CalcTEX_CNTL( ctx ) );

	DMAADVANCE();

	/* note it as recently used to keep from swapping out */
	t[0]->age = t[1]->age = mach64glx.swapBuffersCount;
}


/* =============================================================
 */


/*
 * mach64DDUpdateState
 * This will be called before any beginrender if state has changed
 */
void mach64DDUpdateState( GLcontext *ctx ) {
	int		textureFormat[2];
	int		screenFormat;
	DMALOCALS;

	/* if we aren't using the accelerated triangle function,
		don't bother doing any register writes */
	if ( ctx->Driver.TriangleFunc != mach64Triangle ) {
		return;
	}

	mach64glx.c_setupPointers++;

	/* flush the vertex reuse buffer */
	mach64glx.setupVertices[0] =
	mach64glx.setupVertices[1] =
	mach64glx.setupVertices[2] = (GLuint)-1;

	/* calculate the maximum vertex buffer size */
	mach64glx.setupMax = ( ctx->CompileCVAFlag ) ? ctx->VB->FirstFree : VB_MAX;

	/* set up the tmu mappings for multitexturing */
	mach64Ctx->multitex = 0;
	mach64Ctx->tmu_source[0] = 0;
	mach64Ctx->tmu_source[1] = 1;

	if ( ctx->Texture.Enabled & ENABLE_TEX1 ) {
		if ( ctx->Texture.Enabled & ENABLE_TEX0 ) {
			mach64Ctx->multitex = 1;
		} else {
			/* just a funny way of doing single texturing */
			mach64Ctx->tmu_source[0] = 1;
		}
	}

	/* for now, we are going to set every register we care about */

	/* check for uploading a texture */
	if ( mach64IsTexturingEnabled( ctx ) ) {
		struct gl_texture_object	*tObj;
		mach64TextureObject_t		*t;

		/* do this separately for now */
		mach64UpdateTextureState( ctx );

		/* primary texture format */
		tObj = ctx->Texture.Unit[mach64Ctx->tmu_source[0]].Current;
		t = (mach64TextureObject_t *)tObj->DriverData;

		textureFormat[0] = t->textureFormat;

		if ( mach64Ctx->multitex ) {
			/* secondary texture format */
			tObj = ctx->Texture.Unit[mach64Ctx->tmu_source[1]].Current;
			t = (mach64TextureObject_t *)tObj->DriverData;

			textureFormat[1] = t->textureFormat;
		} else {
			textureFormat[1] = t->textureFormat;
		}
	} else {
		textureFormat[0] = textureFormat[1] = 4;	/* RGB 565 (FIXME: others) */
	}

	DMAGETPTR( 40 );

	/* draw to the back buffer */
	DMAOUTREG( MACH64_DST_OFF_PITCH,
		   ( (mach64DB->pitch/8) << 22 ) | (mach64DB->backBufferBlock->ofs>>3) ) ;

	/* use our depth buffer */
	if ( !mach64DB->depthBufferBlock ) {
		DMAOUTREG( MACH64_Z_CNTL, 0 );
	} else {
		DMAOUTREG( MACH64_Z_OFF_PITCH,
			   ( (mach64DB->pitch/8) << 22 ) | (mach64DB->depthBufferBlock->ofs>>3) ) ;

		DMAOUTREG( MACH64_Z_CNTL, mach64CalcZ_CNTL( ctx ) );
	}

	DMAOUTREG( MACH64_DP_MIX, BKGD_MIX_S | FRGD_MIX_S );	/* only use src */
	DMAOUTREG( MACH64_DP_SRC, BKGD_SRC_3D | FRGD_SRC_3D | MONO_SRC_ONE );

	DMAOUTREG( MACH64_GUI_TRAJ_CNTL, DST_X_LEFT_TO_RIGHT | DST_Y_TOP_TO_BOTTOM );

	DMAOUTREG( MACH64_SETUP_CNTL, DONT_START_ANY | FLAT_SHADE_OFF | SOLID_MODE_OFF | LOG_MAX_INC_ADJ );

	DMAOUTREG( MACH64_CLR_CMP_CNTL, 0 );		/* disable color compare */

	/* lots of blending / texturing bits */
	DMAOUTREG( MACH64_SCALE_3D_CNTL, mach64CalcSCALE_3D_CNTL( ctx ) );

	/* alpha test control */
	DMAOUTREG( MACH64_ALPHA_TST_CNTL, mach64CalcALPHA_TST_CNTL( ctx ) );

	/* scissor rect */
	if ( ctx->Scissor.Enabled ) {
		int		x1, x2, y1, y2;

		/* mesa does not rage check scissor for us */
		x1 = ctx->Scissor.X;
		x2 = ctx->Scissor.X + ctx->Scissor.Width - 1;
		y1 = mach64DB->height - ctx->Scissor.Y - ctx->Scissor.Height;
		y2 = mach64DB->height - ctx->Scissor.Y - 1;
		if ( x1 < 0 ) x1 = 0;
		if ( y1 < 0 ) y1 = 0;
		if ( x2 >= mach64DB->width ) x2 = mach64DB->width - 1;
		if ( y2 >= mach64DB->height ) y2 = mach64DB->height - 1;

		if ( ( x1 > x2 ) || ( y1 > y2 ) ) {
			x1 = 0; x2 = 0;
			y2 = 0; y1 = 1;
		}

		DMAOUTREG( MACH64_SC_LEFT_RIGHT, x1 | ( x2 << 16 ) );
		DMAOUTREG( MACH64_SC_TOP_BOTTOM, y1 | ( y2 << 16 ) );
	} else {
		DMAOUTREG( MACH64_SC_LEFT_RIGHT, 0 | ( ( mach64DB->width - 1 ) << 16 ) );
		DMAOUTREG( MACH64_SC_TOP_BOTTOM, 0 | ( ( mach64DB->height - 1 ) << 16 )  );
	}

	/* fog color */
	if ( ctx->Fog.Enabled ) {
		int		r, g, b;
		int		color;

		r = 255 * ctx->Fog.Color[0];
		g = 255 * ctx->Fog.Color[1];
		b = 255 * ctx->Fog.Color[2];

		/* this seems to always need to be a 24 bit, no matter the
		   graphics mode, contrary to documentation */
		color = MACH64PACKCOLOR888( r, g, b );

		/* aliased on top of DP_FRGRD_COLOR */
		DMAOUTREG( MACH64_DP_FOG_CLR, color );
	}

	/* color plane write mask */
	if ( !ctx->Color.BlendEnabled ) {
		int		color;

		color = mach64PackColor( ctx->Color.ColorMask[0], ctx->Color.ColorMask[1],
					 ctx->Color.ColorMask[2], ctx->Color.ColorMask[3] );

		DMAOUTREG( MACH64_DP_WRITE_MASK, color );
	} else {
		/* Rage Pro can't color mask and alpha blend at the same time */
		DMAOUTREG( MACH64_DP_WRITE_MASK, 0xffffffff );
	}


	/* pixel widths */
	/* is this correct for fog color with alpha texture formats? */
	switch( mach64glx.depth ) {
	case 15: screenFormat = 3; break;
	case 16: screenFormat = 4; break;
	case 32: screenFormat = 6; break;
	}

	DMAOUTREG( MACH64_DP_PIX_WIDTH,
		   ( screenFormat << 0 )		/* dst pix width */
		   | ( textureFormat[1] << 4 )		/* composite pix width */
		   | ( screenFormat << 8 )		/* src pix width */
		   | ( screenFormat << 16 )		/* host data pix width */
		   | ( textureFormat[0] << 28 )		/* scaler/3D pix width */
		   );

	DMAADVANCE();
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
