/*
 * GLX Hardware Device Driver for S3 ViRGE DMA
 * Copyright (C) 1999 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64dma.c
 *
 *    Jim Duchek <jimduchek@ou.edu>
 */

#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>

#include "s3virgeglx.h"
#include "s3virgedma.h"
#include "pb.h"

/* public vars */
s3virgeDma_buffer	*dma_buffer;	/* dmaBuffers[ activeDmaBuffer ] */


/* This will be overwritten from the default values when glx.so is
 * loaded on the client.
 */
void s3virgeServerDmaFlush( int wait );
void    (*s3virgeDoDmaFlush)( int ) = s3virgeServerDmaFlush;
s3virgeUI32	s3virgeActiveDmaBuffer = 0;

void s3virgeDmaResetBuffer( void );
static void s3virgeFlushPseudoDma( void );


static unsigned _SWAP( unsigned a ) {
	return ( ( a & 255 ) << 24 )
		| ( ( a >> 8 ) & 255 ) << 16
		| ( ( a >> 16 ) & 255 ) << 8
 		| ( a >> 24 );
}

static void delay( void )
{ }

static int cmdset = -1, rwh, fgc, bgc, cliplr, cliptb, monop0, monop1;

void s3virgeSaveState( void )
{
	if (cmdset != -1)
		return;
	cmdset = INREG((S3VIRGE_BITBLT_REG | S3VIRGE_CMDSET));
	rwh = INREG((S3VIRGE_BITBLT_REG | S3VIRGE_BITBLT_WIDTH_HEIGHT));
	fgc = INREG((S3VIRGE_BITBLT_REG | S3VIRGE_PAT_FG_COLOR));
	bgc = INREG((S3VIRGE_BITBLT_REG | S3VIRGE_PAT_BG_COLOR));
	cliplr = INREG((S3VIRGE_BITBLT_REG | S3VIRGE_CLIP_L_R));
	cliptb = INREG((S3VIRGE_BITBLT_REG | S3VIRGE_CLIP_T_B));
	monop0 = INREG((S3VIRGE_BITBLT_REG | S3VIRGE_MONO_PAT0));
	monop1 = INREG((S3VIRGE_BITBLT_REG | S3VIRGE_MONO_PAT1));
}

void s3virgeRestoreState ( void )
{
	if (cmdset == -1)
		return;

	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_CMDSET), (0xF << 27));

	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_SRC_BASE), 0);
	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_DEST_BASE), 0);
	
	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_DEST_SRC_STRIDE), (s3virgeglx.displayWidth * 2) | ((s3virgeglx.displayWidth * 2) << 16) );

	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_CLIP_L_R), cliplr);
	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_CLIP_T_B), cliptb);

	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_MONO_PAT0), monop0);
	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_MONO_PAT1), monop1);

	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_BITBLT_WIDTH_HEIGHT), rwh);
	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_BITBLT_SRC_X_Y), 0);
	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_BITBLT_DEST_X_Y), 0);
	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_PAT_FG_COLOR), fgc);
	OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_PAT_BG_COLOR), bgc);

	if ((cmdset & 0x01)) {
		OUTREG((S3VIRGE_BITBLT_REG | S3VIRGE_CMDSET), cmdset);
		OUTREG((S3VIRGE_LINE_REG | S3VIRGE_CMDSET), cmdset);
		OUTREG((S3VIRGE_POLY_REG | S3VIRGE_CMDSET), cmdset);	
	}

	
	cmdset = -1;
}


void s3virgeDmaStart( void )
{	
	if (s3virgeglx.dmaDriver > 0) {
		s3virgeMsg(1, "Starting DMA at %08x\n", dma_buffer->physAddr);
		OUTREG(S3VIRGE_CMD_DMA_BASEADDR_REG, (((long)dma_buffer->physAddr) & 0xFFFFF000) |
							((dma_buffer->size == 1024) ? S3VIRGE_CMD_DMA_BUFFERSIZE_4K
							: S3VIRGE_CMD_DMA_BUFFERSIZE_64K));
		OUTREG(S3VIRGE_CMD_DMA_WRITEP_REG, 0);
		OUTREG(S3VIRGE_CMD_DMA_READP_REG, 0);
		OUTREG(S3VIRGE_CMD_DMA_ENABLE_REG, S3VIRGE_CMD_DMA_ENABLE);
	}
}

/*
 * s3virgeWaitForDmaCompletion
 *
 */
#define	TIMEOUT_USEC		1000000

int s3virgeWaitForDmaCompletion( void )
{
    int		startTime;
    int		curTime;
    int		i;
    int		wp;

    if ( s3virgeglx.dmaDriver == 0)
    	return 0;

    if ( s3virgeglx.skipDma ) {
	return 0;
    }
    
    startTime = 0;
    curTime = 0;
    
    s3virgeMsg(1, "Regs: %08x %08x %08x %08x\n", INREG(S3VIRGE_CMD_DMA_BASEADDR_REG),
    						 INREG(S3VIRGE_CMD_DMA_WRITEP_REG),
    						 INREG(S3VIRGE_CMD_DMA_READP_REG),
    						 INREG(S3VIRGE_CMD_DMA_ENABLE_REG));
    
    wp = INREG( S3VIRGE_CMD_DMA_WRITEP_REG);
    wp &= 0xFFFF;

    while ( 1 )
    {
    	int rp;

    	rp = INREG( S3VIRGE_CMD_DMA_READP_REG );
    	rp &= 0xFFFF;
    	
    	s3virgeMsg(1, "Writep %08x, readp %08x\n", wp, rp);
    	
    	if (rp == wp) {
    		break;
    	}

	curTime = usec();
	if ( ( startTime == 0 ) || ( curTime < startTime /*wrap case*/ ) ) {
	    startTime = curTime;
	} else if ( curTime - startTime > TIMEOUT_USEC ) {
	    s3virgeMsg( 1, "waitForDmaCompletion timed out\n" );
	    break;
	}

	/* spin in place a bit so we aren't hammering the register */
	for ( i = 0 ; i < 10000 ; i++ ) {
	    delay();
	}
    }

    s3virgeMsg( 10, "waitForDmaCompletion, usec: %d\n", curTime - startTime );
    if ( S3VIRGE_ISBUSY() ) {
	s3virgeMsg( 1, "waitForDmaCompletion: still going!\n" );
    }

    return curTime - startTime;
 }

/*
 * s3virgeDmaResetBuffer
 */
void s3virgeDmaResetBuffer()
{
    return;
    dma_buffer->numDwords = 0;

    s3virgeMsg(1, "DMA reset\n");

    OUTREG(S3VIRGE_CMD_DMA_READP_REG, 0);
    OUTREG(S3VIRGE_CMD_DMA_WRITEP_REG, 0);

    /* This is required because each dma buffer is finished by a
     * return to 2d state, as expected by the X server.
     */
    if ( s3virgeDB && s3virgeCtx ) {
//	if ( MESA_VERBOSE & VERBOSE_DRIVER ) {
//	    fprintf( stderr, "needEnter3D and ENTER3D in s3virgeDmaResetBuffer\n" );
//	}
	s3virgeCtx->new_state |= S3VIRGE_NEW_CONTEXT;
    }
}


/*
 * s3virgeFlushRealDma
 */
void s3virgeFlushRealDma( void )
{
    s3virgeUI32	*table_ptr;
    int		i = 0;
	int	chunks;
	
    if ( s3virgeglx.skipDma ) {
	return;
    }

    s3virgeWaitForDmaCompletion();

//    OUTREG( S3VIRGE_CMD_DMA_ENABLE_REG, 0x00000000);
}

/*
 * s3virgeDmaFlush
 * Send all pending commands off to the hardware.
 * If we are running async, the hardware will be drawing
 * while we return to do other things.
 */
void s3virgeServerDmaFlush( int wait )
{
    int		start, end;

    if ( !__glx_is_server ) {
        s3virgeDirectClientDMAFlush( wait );
        return;
    }

    if (s3virgeglx.dmaDriver == 0)
    	return;

    s3virgeglx.c_dmaFlush++;

    if (wait || s3virgeglx.dmaDriver < 3)
    	s3virgeWaitForDmaCompletion();


    s3virgeDmaResetBuffer();
    
}

extern void S3VGEReset(int, int, char *);

/*
 * s3virgeDmaFlush
 */
void s3virgeDmaFlush( void )
{
	unsigned char tmp;
	s3virgeDoDmaFlush( 0 );

	if (s3virgeglx.dmaDriver < 3)
		OUTREG(S3VIRGE_CMD_DMA_ENABLE_REG, 0);
	
//	s3vOff();
//      s3virgeMsg(1, "Framedone\n");
//	WAITACCELIDLE();
//	s3virgeRestoreState();
//	outb(0x3d4, 0x66);
//	tmp = inb(0x3d5);
//	outb(0x3d5, tmp | 0x02);
//	usleep(10000);
//	outb(0x3d5, tmp & ~(0x02));	
}

/*
 * s3virgeDmaFinish
 */
void s3virgeDmaFinish( void )
{
	/* get out fast if we know dma isn't running, because this gets
	called for every software rendered scanline... */
	if ( !s3virgeglx.dmaActive)
		return;
	
	if (!dma_buffer->numDwords ) {
		return;
	}
	
	if (s3virgeglx.dmaDriver == 0)
    		return;

	
	s3virgeMsg(1, "Dma finished\n");

    /* note this for the performance block display */
    s3virgeglx.c_drawWaits++;

    s3virgeDoDmaFlush( 1 );
}

/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * End:
 */
