/*
 * Copyright (C) 1999 Jim Duchek <jimduchek@ou.edu>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * Copied significantly from John Carmack's mach64tex.c
 * 
 */

#include <stdlib.h>
#include <stdio.h>
#include <GL/gl.h>

#include "xsmesaP.h"

#include "s3virgeglx.h"
#include "glx_symbols.h"

/*
 * s3virgeDestroyTexObj
 * Free all memory associated with a texture and NULL any pointers
 * to it.
 */
static void s3virgeDestroyTexObj( s3virgeContextPtr ctx, s3virgeTextureObjectPtr t ) {
	s3virgeTextureObjectPtr	p,prev;

	s3virgeMsg( 10,"s3virgeDestroyTexObj( %p )\n", t->tObj );
	
 	if ( !t ) {
  		return;
  	}
  	
 	if ( t->magic != S3VIRGE_TEXTURE_OBJECT_MAGIC ) {
 		s3virgeError( "s3virgeDestroyTexObj: t->magic != S3VIRGE_TEXTURE_OBJECT_MAGIC\n" );
		return;
	}

	/* free the texture memory */
	mmFreeMem( t->memBlock );
 
 	/* free mesa's link */   
	t->tObj->DriverData = NULL;

	/* remove from the driver texobj list */
	p = s3virgeglx.textureList;
	prev = NULL;
	while( p ) {
		if (p == t) {
			if ( prev ) {
				prev->next = t->next;
			} else {
				s3virgeglx.textureList = t->next;
			}
    			break;
    		}
		prev = p;
		p = p->next;
	}

	/* clear magic to catch any bad future references */
	t->magic = 0;
	
	/* free the structure */
	free( t );
}


/*
 * s3virgeDestroyOldestTexObj
 * Throw out a texture to try to make room for a new texture
 */
static int s3virgeDestroyOldestTexObj( void ) 
{
	s3virgeTextureObjectPtr t, oldest;
	s3virgeUI32 old;
 
 	/* find the best texture to toss */
	old = 0x7fffffff;
	oldest = NULL;
	for ( t = s3virgeglx.textureList; t ; t = t->next ) {
		if ( t->age <= old ) {
			old = t->age;
			oldest = t;
		}
	}

#if 0	/* can't enable this until I get textures loading in the command stream */

	/* if the oldest texture was in use on the previous frame, then
	we are in a texture thrashing state.  Note that we can't just
	test for "in THIS frame", because textures from the same working
	set may be used in different order, and it could register as not
	thrashing.  The solution is to pick the MOST recently used texture
	that isn't currently needed for multitexture.  This will allow the
	other textures to stay resident for the next frame, rather than
	swapping everything out in order. */
	
	if ( old >= s3virgeglx.swapBuffersCount - 1 ) {
		/* newly created texture objects are always added to the
		front of the list, so just find the first one that isn't
		used for multitexture */
	        s3virgeMsg( 10, "s3virgeDestroyOldestTexObj: thrashing\n" );
		oldest = s3virgeglx.textureList;

	} else {
		s3virgeMsg( 10, "s3virgeDestroyOldestTexObj\n" );
	}
#endif
	
	if ( !oldest ) {
		/* This shouldn't happen unless the 2D resolution is high enough that
		a single texture can't be allocated in the remaining memory */
		s3virgeError("  No Texture to swap out -> Out of Memory!\n"); 
		mmDumpMemInfo( cardHeap ); 
		return -1;
	}

	/* just destroy the texture, because it can always
	be recreated directly from the mesa texture */
	
	s3virgeMsg(0, "Swapping out %08x\n", oldest->memBlock->ofs);
	
	s3virgeDestroyTexObj( s3virgeCtx, oldest );

	return 0;
}

/*
 * s3virgeConvertTexture
 * Converts a block of mesa format texture to the apropriate hardware format
 */
static void s3virgeConvertTexture( s3virgeUI32 *destPtr, int texelBytes, 
	struct gl_texture_image *image, int x, int y, int width, int height ) {
	int		i, j;
	s3virgeUI8		*src;

	s3virgeMsg(1, "Uploading tex of %dx%d, %dbpp\n", width, height, texelBytes);
	
	if ( texelBytes == 1 && ( image->Format == GL_COLOR_INDEX || image->Format == GL_INTENSITY || image->Format == GL_LUMINANCE || image->Format == GL_ALPHA ) ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 2  ; j ; j-- ) {
				int	pix;
			
				pix = src[0] | ( src[1] << 8 ) | ( src[2] << 16 ) | ( src[3] << 24 );
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_RGB ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 3;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				// FIXME
				pix = S3VIRGEPACKCOLOR555(src[0],src[1],src[2], 255) |  
					( S3VIRGEPACKCOLOR555(src[3],src[4],src[5], 255) << 16 );
				*destPtr++ = pix;
				src += 6;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_RGBA ) {
//		fprintf(stderr, "Outputting to %08x\n", destPtr);
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 4;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
				
				pix = S3VIRGEPACKCOLOR4444(src[0],src[1],src[2],src[3]) |  
					( S3VIRGEPACKCOLOR4444(src[4],src[5],src[6],src[7]) << 16 );
//				fprintf(stderr, "%04x ", pix);
				*destPtr++ = pix;
				src += 8;
			}
//			fprintf(stderr, "\n");
		}
	} else if ( texelBytes == 2 && image->Format == GL_LUMINANCE ) {
	
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
				/* FIXME: This is a, uh, really bad hack to get luminances working */
				pix = S3VIRGEPACKCOLOR4444(src[0],src[0],src[0], src[0]) |
					( S3VIRGEPACKCOLOR4444(src[1],src[1],src[1], src[1]) << 16 );
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_INTENSITY ) {

		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = S3VIRGEPACKCOLOR4444(src[0],src[0],src[0],src[0]) |  
					( S3VIRGEPACKCOLOR4444(src[1],src[1],src[1],src[1]) << 16 );
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_ALPHA ) {

		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = S3VIRGEPACKCOLOR4444(255,255,255,src[0]) |  
					( S3VIRGEPACKCOLOR4444(255,255,255,src[1]) << 16 );
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_LUMINANCE_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 2;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = S3VIRGEPACKCOLOR4444(src[0],src[0],src[0],src[1]) |  
					( S3VIRGEPACKCOLOR4444(src[2],src[2],src[2],src[3]) << 16 );
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_RGB ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 3;
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = (255<<24) | S3VIRGEPACKCOLOR888(src[0],src[1],src[2]);
				*destPtr++ = pix;
				src += 3;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_RGBA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 4;
			for ( j = width  ; j ; j-- ) {
				int	pix;
			
				pix = S3VIRGEPACKCOLOR8888(src[0],src[1],src[2],src[3]);
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_LUMINANCE ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = (255<<24) | S3VIRGEPACKCOLOR888(src[0],src[0],src[0]);
				*destPtr++ = pix;
				src += 1;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_INTENSITY ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = S3VIRGEPACKCOLOR8888(src[0],src[0],src[0],src[0]);
				*destPtr++ = pix;
				src += 1;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = S3VIRGEPACKCOLOR8888(255,255,255,src[0]);
				*destPtr++ = pix;
				src += 1;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_LUMINANCE_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (s3virgeUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 2;
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = S3VIRGEPACKCOLOR8888(src[0],src[0],src[0],src[1]);
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else {
		s3virgeError( "Unsupported texelBytes %i, image->Format %i\n", texelBytes, image->Format );
	}
}


/*
 * s3virgeUploadSubImage
 * Perform an iload based update of a resident buffer.  This is used for
 * both initial loading of the entire image, and texSubImage updates.
 */
static void s3virgeUploadSubImage( s3virgeTextureObjectPtr t, int level,
		     int x, int y, int width, int height ) 
{
	int		x2;
	int		dwords;
        s3virgeUI32		*dest;
	struct gl_texture_image *image;

	image = t->tObj->Image[level];

	if ( !image ) {
		s3virgeError( "s3virgeUploadSubImage: NULL image\n" );
		return;
	}

	s3virgeMsg(10, "s3virgeUploadSubImage: %i,%i of %i,%i at %i,%i\n", width, height,
		image->Width, image->Height, x, y );

	/* bump the performance counter */
	s3virgeglx.c_textureSwaps += ( dwords << 2 );

	/* FIXME: do this with DMA instead of writing into the framebuffer! */	
	dest = (s3virgeUI32 *)( s3virgeglx.linearBase + t->memBlock->ofs + t->offsets[level] );
	
	/* fill in the secondary buffer with properly converted texels from the
	mesa buffer */
	s3virgeConvertTexture( dest, t->texelBytes, image, x, y, width, height );
	
}

static int Log2( unsigned a ) {
	unsigned	i;
	
	for ( i = 0 ; i < 32 ; i++ ) {
		if ( ( 1<<i ) >= a ) {
			return i;
		}	
	}
	return 31;
}

/*
 * s3virgeCreateTexObj
 * Allocate space for and load the mesa images into the texture memory block.
 * This will happen before drawing with a new texture, or drawing with a
 * texture after it was swapped out or teximaged again.
 */
void s3virgeCreateTexObj( s3virgeContextPtr ctx, struct gl_texture_object *tObj ) 
{
	s3virgeTextureObjectPtr		t;
	int					ofs, size, lastlevel, i;
	PMemBlock				mem;
	struct gl_texture_image *image;
	
	s3virgeMsg( 10,"s3virgeCreateTexObj( %p )\n", tObj );

	image = tObj->Image[ 0 ];

	if ( !image ) {
		return;
	}

	t = malloc( sizeof( *t ) );
	if ( !t ) {
		FatalError( "s3virgeCreateTexObj: Failed to malloc textureObject\n" );
	}
	memset( t, 0, sizeof( *t ) );
	
	/* texture format options */
	switch (image->Format) {
		case GL_RGB:
			t->hasAlpha = 0;
			t->texelBytes = 2;
			break;
		case GL_LUMINANCE:
			// Yeah, I know, but we've gotta store the alpha
			// it bites...
		case GL_ALPHA:
		case GL_LUMINANCE_ALPHA:
		case GL_INTENSITY:
		case GL_RGBA:
			t->hasAlpha = 1;		
			t->texelBytes = 2; // Support 24 bit textures sometime, boy...
			break;
		case GL_COLOR_INDEX:
			t->hasAlpha = 0;
			t->texelBytes = 1;
			break;
		default:
			return;
	}

	ofs = 0;
	lastlevel = S3VIRGE_TEX_MAXLEVELS - 1;

	t->widthLog2 = Log2( image->Width );
	t->heightLog2 = Log2( image->Height );
	t->maxLog2 = t->widthLog2 > t->heightLog2 ? t->widthLog2 : t->heightLog2;


	for (i=0; i<=lastlevel; i++) {
		int lWidth, lHeight;
		
		t->offsets[i] = ofs;
		
		image = tObj->Image[i];
		
		if (!image) {
			lastlevel = i - 1;
			s3virgeMsg(10, "Missing levels after %d\n", lastlevel);
			break;
		}
			
		size = image->Width * image->Height * t->texelBytes;
		size = ( size + 31 ) & ~31;	/* 32 byte aligned */
		ofs += size;
	}

	t->totalSize = ofs;
	
	/* allocate a buffer for all levels, swapping out stuff if needed */
        while ( ( mem = mmAllocMem( cardHeap, t->totalSize, 6, 0 ) ) == 0 ) {
        	if ( s3virgeDestroyOldestTexObj() ) {
        		/* can't hold this texture at all */
			s3virgeMsg( 10,"s3virgeCreateTexObj: Couldn't allocate buffer\n" );
        		free( t );
        		return;
        	}
        }

	/* dump the heap contents if loglevel is high enough */
	if ( s3virgeglx.logLevel >= 15 ) {
		mmDumpMemInfo( cardHeap );
	}

	/* fill in our texture object */
	t->magic = S3VIRGE_TEXTURE_OBJECT_MAGIC;
	t->tObj = tObj;
	t->ctx = ctx;
	t->next = s3virgeglx.textureList;
	t->age = s3virgeglx.swapBuffersCount;
	s3virgeglx.textureList = t;
	
	t->memBlock = mem;

	/* base image */
	image = tObj->Image[ 0 ];

  	tObj->DriverData = t;
  
	/* load the texels */
	for (i=0; i<=lastlevel; i++) {
		image = tObj->Image[i];
		s3virgeUploadSubImage( t, i, 0, 0, image->Width, image->Height );
	}
	
	s3virgeMsg(0, "Uploaded to %08x\n", mem->ofs);
}


/*
============================================================================

Driver functions called directly from mesa

============================================================================
*/


/*
 * s3virgeTexImage
 */ 
void s3virgeTexImage( GLcontext *ctx, GLenum target,
		  struct gl_texture_object *tObj, GLint level,
		  GLint internalFormat,
		  const struct gl_texture_image *image ) {
	s3virgeTextureObject_t *t;

	s3virgeMsg( 10, "s3virgeTexImage( %p, level %i )\n", tObj, level );
	
  	/* free the driver texture if it exists */
	t = (s3virgeTextureObjectPtr) tObj->DriverData;
	if ( t ) {
 	 	s3virgeDestroyTexObj( s3virgeCtx, t );
	}
	
	/* create it */
	s3virgeCreateTexObj( s3virgeCtx, tObj );
}

/*
 * s3virgeTexSubImage
 */      
void s3virgeTexSubImage( GLcontext *ctx, GLenum target,
		  struct gl_texture_object *tObj, GLint level,
		  GLint xoffset, GLint yoffset,
		  GLsizei width, GLsizei height,
		  GLint internalFormat,
		  const struct gl_texture_image *image ) 
{
	s3virgeTextureObject_t *t;

	s3virgeMsg(10,"s3virgeTexSubImage() Size: %d,%d of %d,%d; Level %d\n",
		width, height, image->Width,image->Height, level);

	/* immediately upload it if it is resident */
	t = (s3virgeTextureObject_t *) tObj->DriverData;
	if ( t ) {
		s3virgeUploadSubImage( t, level, xoffset, yoffset, width, height );
	} else {
		s3virgeMsg(0, "Texture not resident!!\n");
	}
}

/*
 * s3virgeLoadTexturePalette
 */
void
s3virgeLoadTexturePalette( s3virgeUI8 *pal )
{
	int i;
	int n = 0;
	unsigned char tmp;

#if 0
	outb(0x3c4, 0x08);
	outb(0x3c5, 0x06);
	
	outb(0x3c4, 0x47);
	tmp = inb(0x3c5);
	outb(0x3c5, 0x0);
	

	outb(0x3c6, 0xFF);	

	outb(0x3c8, 0x00);
	
	for (i=0; i<768; i++) {
		s3virgeMsg(10, "doing %d to %d\n", i, pal[i]);
//		outb(0x3c9, (pal[i] >> 2));
		outb(0x3c9, 0x3F);
		s3virgeglx.currentTexturePalette[i] = pal[i];
	}

	outb(0x3c6, 0xFF);	

	outb(0x3c7, 0x00);
	
	for (i=0; i<768; i++) {
		tmp = inb(0x3c9);
		if (tmp != (pal[i] >> 2)) {
			s3virgeMsg(10, "%d is wrong! %d %d\n", i, tmp, (pal[i] >> 2));
		} else {
			s3virgeMsg(10, "%d is correct! %d %d\n", i, tmp, (pal[i] >> 2));
		}
	}

	outb(0x3c4, 0x08);
	outb(0x3c5, 0x00);
#endif

	for (i=0; i<512; i+=4) {
		OUTREG( (0xA000 + i), 0xFFFFFFFF);
		s3virgeMsg(10, "doing %d to %d\n", i, pal[i]);
//		outb(0x3c9, (pal[i] >> 2));
//		outb(0x3c9, 0x3F);
		s3virgeglx.currentTexturePalette[i] = pal[i];
	}

	
	s3virgeglx.updatedPalette = 0;

}

/*
 * s3virgeUpdateTexturePalette
 */
int
s3virgeUpdateTexturePalette( GLcontext *ctx, struct gl_texture_object *tObj ) 
{
	s3virgeTextureObject_t *t;
	int i, size, next, format;
	s3virgeUI8 *dst, *dst2;
	s3virgeUI8 *src;
	
	s3virgeMsg( 10, "s3virgeUpdateTexturePalette( %p )\n", tObj );
	
	if (tObj) {
		/* private palette texture */
		t = (s3virgeTextureObject_t *)tObj->DriverData;
		if (!t) {
			s3virgeError("No driver data!\n");
			return GL_FALSE;
		}
		size = tObj->PaletteSize;
		format = tObj->PaletteFormat;
		dst = t->Palette;
		src = (s3virgeUI8 *)tObj->Palette;
	} else {
		size = ctx->Texture.PaletteSize;
		format = ctx->Texture.PaletteFormat;
		dst = s3virgeCtx->GlobalPalette;
		src = (s3virgeUI8 *)ctx->Texture.Palette;
	}

	dst2 = s3virgeglx.currentTexturePalette;
	
	if (size > 256) {
		s3virgeError("Size too big, %d\n", size);
		return GL_FALSE;
	}

	switch(format) {
		case GL_RGB:
			next = 3;
			break;
		case GL_RGBA:
			next = 4;
			break;
		default:
			s3virgeMsg(10, "Format is %04x\n", format);
			s3virgeError("Unsupported index format\n");
			return GL_FALSE;
	}
	
	for (i=0; i<size; i++) {
		dst2[0] = dst[0] = src[0];
		dst2[1] = dst[1] = src[1];
		dst2[2] = dst[2] = src[2];
		dst+=3;
		dst2+=3;
		src+=next;		
	}

	s3virgeglx.updatedPalette = 1;
	s3virgeDmaFlush();
	s3virgeMsg(10, "Palette loaded.\n");
	return GL_TRUE;
}


/*
 * s3virgeDeleteTexture
 */      
void s3virgeDeleteTexture( GLcontext *ctx, struct gl_texture_object *tObj ) 
{
	s3virgeMsg( 10, "s3virgeDeleteTexture( %p )\n", tObj );
	
	/* delete our driver data */
	if ( tObj->DriverData ) {
		s3virgeDestroyTexObj( s3virgeCtx, (s3virgeTextureObject_t *)(tObj->DriverData) );
	}
}

/*
 * s3virgeIsTextureResident
 */      
GLboolean s3virgeIsTextureResident( GLcontext *ctx, struct gl_texture_object *tObj ) 
{
	GLboolean			is;
	
   	is = (tObj->DriverData != NULL);

	s3virgeMsg( 10, "s3virgeIsTextureResident( %p ) == %i\n", tObj, is );
   	
	return is;
}
