/*
 * FieldViewItem.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include "stdafx.h"

#include "FieldViewItem.h"
#include "swt.h"

#include "EulerAngles.h"
#include "Field.h"
#include "FieldView.h"
#include "FieldValue.h"
#include "FieldCommand.h"
#include "Node.h"
#include "Path.h"
#include "Proto.h"
#include "SFMFTypes.h"
#include "Scene.h"
#include "URL.h"
#include "DuneApp.h"

FieldViewItem::FieldViewItem(FieldView *view)
{
    _view = view;
    _value = NULL;
    _parent = NULL;
    _state = FVIS_COLLAPSED;
    _field = NULL;
    _index = -1;
}

FieldViewItem::~FieldViewItem()
{
    if (_value) _value->unref();
}

void
FieldViewItem::SetValue(FieldValue *value)
{
    if (_value) _value->unref();
    if (value==NULL) return;
    _value = value;
    _value->ref();
    UpdateControl();
}

FieldViewItem       *FieldViewItem::CreateItem(const Field *field, FieldView *view)
{
    FieldViewItem	*r;

    switch(field->getType()) {
      case SFBOOL:
	r = new SFBoolItem(view); break;
      case SFCOLOR:
	r = new SFColorItem(view); break;
      case SFINT32:
	r = new SFInt32Item(view); break;
      case SFFLOAT:
	r = new SFFloatItem(view); break;
      case SFNODE:
	r = new SFNodeItem(view); break;
      case SFROTATION:
	r = new SFRotationItem(view); break;
      case SFSTRING:
	r = new SFStringItem(view); break;
      case SFTIME:
	r = new SFTimeItem(view); break;
      case SFVEC2F:
	r = new SFVec2fItem(view); break;
      case SFVEC3F:
	r = new SFVec3fItem(view); break;
      case MFCOLOR:
	r = new MFColorItem(view); break;
      case MFFLOAT:
	r = new MFFloatItem(view); break;
      case MFINT32:
	r = new MFInt32Item(view); break;
      case MFNODE:
	r = new MFNodeItem(view); break;
      case MFROTATION:
	r = new MFRotationItem(view); break;
      case MFSTRING:
	r = new MFStringItem(view); break;
      case MFTIME:
	r = new MFTimeItem(view); break;
      case MFVEC2F:
	r = new MFVec2fItem(view); break;
      case MFVEC3F:
	r = new MFVec3fItem(view); break;
      default:
	r = new FieldViewItem(view); break;
    }
    r->SetField(field);
    return r;
}

MFieldViewItem::MFieldViewItem(FieldView *view) : FieldViewItem(view)
{
}

void
MFieldViewItem::InitIndexValue(int index, FieldValue *value)
{
    MFieldValue *v = (MFieldValue *) value;
    _children[index] = CreateSFItem();
    _children[index]->SetParent(this);
    _children[index]->SetField(_field);
    _children[index]->SetIndex(index);
    _children[index]->SetValue(v->getSFValue(index));
}

void
MFieldViewItem::SetValue(FieldValue *value)
{
    MFieldValue *v = (MFieldValue *) value;
    int		i;
    int		n = _children.size();

    for (i = 0; i < n; i++) {
	delete _children[i];
    }

    _children.resize(v->getSFSize());

    for (i = 0; i < v->getSFSize(); i++)
        InitIndexValue(i, value);

    FieldViewItem::SetValue(value);
}

MFieldViewItem::~MFieldViewItem()
{
    int		n = _children.size();
    for (int i = 0; i < n; i++) {
	delete _children[i];
    }
}

static void DrawFloats(const float *floats, int len, const char *labels, SDC dc,
		       int x, int y, int width)
{
    for (int i = 0; i < len; i++) {
	char		buf[30];
	sprintf(buf, "%c: %5.2f", labels[i], floats[i]);
	swDrawText(dc, x + i * width, y, buf);
    }
}


SFBoolItem::SFBoolItem(FieldView *view) : FieldViewItem(view)
{
    _button = NULL;
}

SFBoolItem::~SFBoolItem()
{
    if (_button) swDestroyWindow(_button);
}

static void
boolButtonCallback(void *data, int id)
{
    ((SFBoolItem *) data)->OnClicked();
}

void
SFBoolItem::CreateControl(const Rect &rect)
{
    _button = swCreateCheckBox("", rect.left, rect.top,
				   rect.Height()-1, rect.Height()-1,
			       _view->GetWindow());
    swSetCommandCallback(_button, boolButtonCallback);
    swSetClientData(_button, this);
}

void
SFBoolItem::MoveControl(int x, int y)
{
    if (_button) swSetPosition(_button, x, y);
}

void
SFBoolItem::UpdateControl()
{
    if (_button) swSetCheck(_button, ((SFBool *) _value)->getValue());
}

void
SFBoolItem::OnClicked()
{
    int	    check = swGetCheck(_button);
    Node   *node = _view->GetScene()->getSelection()->getNode();
    FieldValue	    *value = new SFBool(check != 0);
    _view->GetScene()->execute(new FieldCommand(node, _index, value));
}

SFColorItem::SFColorItem(FieldView *view) : FieldViewItem(view)
{
}

FieldValue *
SFColorItem::OnMouseDown(int x, int y, int modifiers)
{
    if (x < _view->GetItemHeight() - 1) {
//	CColorDialog    dlg;
//	const float	   *c = ((SFColor *) _value)->getValue();
//	COLORREF	    rgb = RGB(
//	    (BYTE) (c[0] * 255.0f), 
//	    (BYTE) (c[1] * 255.0f), 
//	    (BYTE) (c[2] * 255.0f));

//	dlg.m_cc.rgbResult = rgb;
//	dlg.m_cc.Flags |= CC_RGBINIT | CC_FULLOPEN;
//	if (dlg.DoModal() == IDOK) {
//	    COLORREF color = dlg.GetColor();
//	    return new SFColor(GetRValue(color) / 255.0f, GetGValue(color) / 255.0f, GetBValue(color) / 255.0f);
//	} else {
	    return NULL;
//	}
    } else {
	return NULL;
    }
}

SFColorItem::~SFColorItem()
{
}

void
SFColorItem::Draw(SDC dc, int x, int y)
{
    const float	   *c = ((SFColor *) _value)->getValue();
    int		    size = _view->GetItemHeight() - 1;

    DrawFloats(((SFColor *) _value)->getValue(), 3, "RGB", dc,
    	       x+size+2, y+size-2, _view->GetFloatWidth());
    swDraw3DRect(dc, _view->GetWindow(), x, y, size, size);
    swDraw3DRect(dc, _view->GetWindow(), x+1, y+1, size-2, size-2);
    swSetFGColor(dc, SW_RGB((int) (c[0] * 255.0f),
    		     (int) (c[1] * 255.0f),
		     (int) (c[2] * 255.0f)));
    swFillRect(dc, x+2, y+2, size-4, size-4);
}

void
SFColorItem::UpdateControl()
{
}

FieldValue *
SFColorItem::OnMouseMove(FieldValue *value, int index, int delta) 
{
    SFColor *newValue = new SFColor(((SFColor *) value)->getValue());
    float f = ((SFColor *) value)->getValue()[index] + delta * 0.005f;
    f = CLAMP(f, 0.0f, 1.0f);
    newValue->setValue(index, f);
    return newValue;
}

int
SFColorItem::GetFieldOffset(int xpos) const
{
    int	    offset = (xpos - _view->GetItemHeight() - 1) / _view->GetFloatWidth();
    if (offset > 2) offset = 2;
    return offset;
}

void
SFColorItem::StartEditing(MyString &buf, int offset)
{
    assert( offset >= 0 && offset < 3);

    char	tmp[20];
    sprintf(tmp, "%g", ((SFColor *) _value)->getValue()[offset]);
    buf = tmp;
}

FieldValue *
SFColorItem::StopEditing(const char *str, int offset)
{
    SFColor	*newValue = new SFColor(((SFColor *) _value)->getValue());
    newValue->setValue(offset, (float) atof(str));
    return newValue;
}

SFFloatItem::SFFloatItem(FieldView *view)
  : FieldViewItem(view)
{
}

FieldValue *
SFFloatItem::OnMouseMove(FieldValue *value, int /* index */, int delta)
{
    return new SFFloat(((SFFloat *) value)->getValue() + delta * 0.02f);
}

void
SFFloatItem::Draw(SDC dc, int x, int y)
{
    char	buf[30];

    sprintf(buf, "%8.2f", ((SFFloat *) _value)->getValue());
    swDrawText(dc, x, y+_view->GetItemHeight() - 3, buf);
}


void
SFFloatItem::StartEditing(MyString &str, int /* offset */)
{
    char	buf[30];

    sprintf(buf, "%g", ((SFFloat *) _value)->getValue());
    str = buf;
}

FieldValue *
SFFloatItem::StopEditing(const char *str, int /* offset */)
{
    return new SFFloat((float) atof(str));
}

SFInt32Item::SFInt32Item(FieldView *view)
  : FieldViewItem(view)
{
}

void
SFInt32Item::Draw(SDC dc, int x, int y)
{
    char	buf[30];

    sprintf(buf, "%d", ((SFInt32 *) _value)->getValue());
    swDrawText(dc, x, y+_view->GetItemHeight() - 3, buf);
}

FieldValue *
SFInt32Item::OnMouseMove(FieldValue *value, int /* index */, int delta)
{
    return new SFInt32(((SFInt32 *) value)->getValue() + delta);
}

void
SFInt32Item::StartEditing(MyString &str, int /* offset */)
{
    char	buf[30];

    sprintf(buf, "%d", ((SFInt32 *) _value)->getValue());
    str = buf;
}

FieldValue *
SFInt32Item::StopEditing(const char *str, int /* offset */)
{
    return new SFInt32(atoi(str));
}

SFNodeItem::SFNodeItem(FieldView *view)
  : FieldViewItem(view)
{
}

void
SFNodeItem::Draw(SDC dc, int x, int y)
{
    const char		*str;

    Node *node = ((SFNode *) _value)->getValue();
    if (node) {
	const char	*name = node->getName();
	if (name && name[0]) {
	    str = name;
	} else {
	    str = (const char *) node->getProto()->getName();
	}
	swSetFont(dc, _view->GetUFont());
	swSetFGColor(dc, 0xff0000);
	swDrawText(dc, x, y+_view->GetItemHeight() - 3, str);
	swSetFont(dc, swGetDefaultFont());
    }
}

SFRotationItem::SFRotationItem(FieldView *view)
  : FieldViewItem(view)
{
}

void
SFRotationItem::Draw(SDC dc, int x, int y)
{
    int		rotationOrder = TheApp->GetRotationOrder();
    const char *title = TheApp->GetRotationTitle();

#ifdef HAVE_EULER_ROT
    Vec3f   v = ((SFRotation *) _value)->getEulerAngles(rotationOrder);
    float   f[3] = {v.x, v.y, v.z};
    DrawFloats(f, 3, title, dc, x, y+_view->GetItemHeight() - 3,
               _view->GetFloatWidth());
#else
   DrawFloats(((SFRotation *) _value)->getValue(),4,title, dc, 
               x, y+_view->GetItemHeight() - 3,_view->GetFloatWidth());        
#endif
}

FieldValue *
SFRotationItem::OnMouseMove(FieldValue *value, int index, int delta)
{
    SFRotation *oldValue = (SFRotation *) value;
    SFRotation *newValue = new SFRotation(*oldValue);
    if (index<3) {
        int		rotationOrder = TheApp->GetRotationOrder();
        Vec3f	v;
        float	v2[3];

        v = oldValue->getEulerAngles(rotationOrder);
        v2[0] = v.x; v2[1] = v.y; v2[2] = v.z;
        v2[index] += delta;
        newValue->setEulerAngles(v2, rotationOrder);
    }
    return newValue;
}

int
SFRotationItem::GetFieldOffset(int xpos) const
{
    int offset = xpos / _view->GetFloatWidth();
#ifdef HAVE_EULER_ROT
    if (offset > 2) offset = 2;
#else 
    if (offset > 3) offset = 3;
#endif
    return offset;
}

void
SFRotationItem::StartEditing(MyString &str, int offset)
{
    int		order = TheApp->GetRotationOrder();
    char	buf[30];
#ifdef HAVE_EULER_ROT
    Vec3f     v = ((SFRotation *) _value)->getEulerAngles(order);
    sprintf(buf, "%g", ((const float *) &v)[offset]);
#else
    float v[4];
    for (int i=0;i<4;i++)
       v[i]=((SFRotation *) _value)->getValue()[i];
    sprintf(buf, "%g", ((const float *) v)[offset]);
#endif
    str = buf;
}

FieldValue *
SFRotationItem::StopEditing(const char *str, int offset)
{
    int		order = TheApp->GetRotationOrder();

#ifdef HAVE_EULER_ROT
    Vec3f	v;
    float	v2[3];
    v = ((SFRotation *) _value)->getEulerAngles(order);
    v2[0] = v.x; v2[1] = v.y; v2[2] = v.z;
    v2[offset] = (float) atof(str);
    return new SFRotation(Vec3f(v2), order);
#else
    float  v2[4];
    for (int i=0;i<4;i++)
       v2[i]=((SFRotation *) _value)->getValue()[i];
    v2[offset] = (float) atof(str);
    return new SFRotation(v2);
#endif
}

SFStringItem::SFStringItem(FieldView *view) : FieldViewItem(view)
{
    _comboBox = NULL;
}

SFStringItem::~SFStringItem()
{
    if (_comboBox != NULL) swDestroyWindow(_comboBox);
}

void
SFStringItem::Draw(SDC dc, int x, int y)
{
    int		size = _view->GetItemHeight() - 1;
    const char *str = ((SFString *) _value)->getValue();
    swDrawText(dc, x, y + _view->GetItemHeight() - 3, str);
    if (_field->getFlags() & FF_URL) {
	int	    bx = x + _view->GetItemWidth() - size;
	swDraw3DRect(dc, _view->GetWindow(), bx, y, size, size);
	swSetFGColor(dc, swGetWindowColor(_view->GetWindow(), SW_COLOR_TEXT));
	swDrawText(dc, bx + 2, y + _view->GetItemHeight() - 3, "...");
    } else if (_field->getStrings()) {
/*
	int	    x = _view->GetItemWidth() - size;
	dc->DrawFrameControl(CRect(x, 0, x + size, size), 
			     DFC_BUTTON, DFCS_BUTTONPUSH);
	CBitmap	    bmp;
	SDC	    bdc;
	bdc.CreateCompatibleDC(dc);
	bmp.LoadBitmap(OBM_COMBO);
	CBitmap *old = (CBitmap *) bdc.SelectObject(bmp);
	dc->BitBlt(x + 2, 2, 12, 12, &bdc, 0, 0, SRCCOPY);
	bdc.SelectObject(old);
*/
    }
}

FieldValue *
SFStringItem::OnMouseDown(int x, int y, int modifiers)
{
    int		    height = _view->GetItemHeight();
    int		    width = _view->GetItemWidth();
    const char    **strings = _field->getStrings();

    if (x >= width - height && x < width) {
	if (_field->getFlags() & FF_URL) {
/*
	    CFileDialog		dlg(TRUE, NULL, NULL, OFN_HIDEREADONLY, 
	"Image Files (*.gif, *.jpg)|*.gif;*.jpg|All Files (*.*)|*.*||",
				    _view);

	    if (dlg.DoModal() == IDOK) {
		URL	    parent, child;
		const char *url = _view->GetDocument()->getScene()->getURL();

		if (url[0]) {
		    parent = url;
		} else {
		    char	    cdir[_MAX_DIR];
		    GetCurrentDirectory(_MAX_DIR, cdir);
		    parent.FromDOSPath(cdir);
		}
		child.FromDOSPath(dlg.GetPathName());
		return new SFString(child.RelativeTo(parent));
	    }
*/
	} else if (strings) {
/*
	    // create popup
	    const char	    **s;
	    for (s = strings; *s != NULL; s++)
		;
	    int n = s - strings;
	    CPoint	p(_view->GetColumnWidth(0), n * _view->GetItemHeight());
	    _view->ClientToScreen(&p);
	    CWnd	 popup;
	    popup.CreateEx(0, "FunkyComboBoxClass", "", WS_POPUP | WS_BORDER | WS_VISIBLE,
			   CRect(p.x, p.y, _view->GetColumnWidth(1),
			       n * _view->GetItemHeight()), NULL,
			   0, NULL);
	    return NULL;
*/
	}
    }
    return NULL;
}

void
SFStringItem::MoveControl(int x, int y)
{
    if (_comboBox) swSetPosition(_comboBox, x, y);
}

void
SFStringItem::CreateControl(const Rect &rect)
{	
    const char **strings = _field->getStrings();

    if (strings) {
/*
	// create a combobox
	int	    style;
	_comboBox = swCreateComboBox();

	const char **s;
	HFONT	font = (HFONT) GetStockObject(DEFAULT_GUI_FONT);

	int bsize = rect.Height();
	if (_field->getFlags() & FF_FIXEDSTRINGS) {
	    style = CBS_DROPDOWNLIST;
	} else {
	    style = CBS_DROPDOWN;
	}
	_comboBox->Create(style | WS_CHILD | WS_VISIBLE, rect, _view, id);
	_comboBox->SetFont(CFont::FromHandle(font));
	int h = _comboBox->GetItemHeight(0);
	CRect	crect(rect), w;

	_comboBox->GetWindowRect(&w);
	crect.bottom += 6;

	for (s = strings; *s != NULL; s++) {
	    _comboBox->AddString(*s);
	    crect.bottom += h;
	}
	_comboBox->MoveControl(crect);
*/
    }
}

void
SFStringItem::UpdateControl()
{
    if (_comboBox) {
/*
	int index = swComboBoxFindString(_comboBox, 0, ((SFString *) _value)->getValue());
	if (index >= 0) swComboBoxSetCurSel(index);
*/
    }
}

void
SFStringItem::StartEditing(MyString &str, int /* offset */)
{
    str = ((SFString *) _value)->getValue();
}

FieldValue *
SFStringItem::StopEditing(const char *str, int /* offset */)
{
    return new SFString((const char *) str);
}

FieldValue *
SFStringItem::OnCommand(int id)
{
//    if (_comboBox && id == CBN_SELCHANGE) {
//	return new SFString(_field->getStrings()[_comboBox->GetCurSel()]);
//    } else {
	return NULL;
//    }
}

SFTimeItem::SFTimeItem(FieldView *view) : FieldViewItem(view)
{
}

void
SFTimeItem::Draw(SDC dc, int x, int y)
{
    char	buf[30];

    sprintf(buf, "%8.2f", ((SFTime *) _value)->getValue());
    swDrawText(dc, x, y + _view->GetItemHeight() - 3, buf);
}

FieldValue *
SFTimeItem::OnMouseMove(FieldValue *value, int index, int delta)
{
    return new SFTime(((SFTime *) value)->getValue() + delta * 0.01);
}

void
SFTimeItem::StartEditing(MyString &str, int /* offset */)
{
    char	buf[30];
    
    sprintf(buf, "%g", ((SFTime *) _value)->getValue());
    str = buf;
}

FieldValue *
SFTimeItem::StopEditing(const char *str, int /* offset */)
{
    return new SFTime(atof(str));
}

SFVec2fItem::SFVec2fItem(FieldView *view)
  : FieldViewItem(view)
{
}

void
SFVec2fItem::Draw(SDC dc, int x, int y)
{
    DrawFloats(((SFVec2f *) _value)->getValue(), 2, "XY", dc, x, y+_view->GetItemHeight()-3,
    	       _view->GetFloatWidth());
}

FieldValue *
SFVec2fItem::OnMouseMove(FieldValue *value, int index, int delta)
{
    SFVec2f	*v = (SFVec2f *) value;
    SFVec2f	*newValue = new SFVec2f(v->getValue());
    newValue->setValue(index, v->getValue(index) + delta * 0.02f);
    return newValue;
}

int
SFVec2fItem::GetFieldOffset(int xpos) const
{
    int offset = xpos / _view->GetFloatWidth();
    if (offset > 1) offset = 1;
    return offset;
}

void
SFVec2fItem::StartEditing(MyString &str, int offset)
{
    char	buf[30];

    sprintf(buf, "%g", ((SFVec2f *) _value)->getValue(offset));
    str = buf;
}

FieldValue *
SFVec2fItem::StopEditing(const char *str, int offset)
{
    SFVec2f     *newValue = new SFVec2f(((SFVec2f *) _value)->getValue());
    newValue->setValue(offset, (float) atof(str));
    return newValue;
}

SFVec3fItem::SFVec3fItem(FieldView *view)
  : FieldViewItem(view)
{
}

void
SFVec3fItem::Draw(SDC dc, int x, int y)
{
    DrawFloats(((SFVec3f *) _value)->getValue(), 3, "XYZ", dc, x, y+_view->GetItemHeight()-3, _view->GetFloatWidth());
}

FieldValue *
SFVec3fItem::OnMouseMove(FieldValue *value, int index, int delta)
{
    SFVec3f	*v = (SFVec3f *) value;
    SFVec3f	*newValue = new SFVec3f(v->getValue());
    newValue->setValue(index, v->getValue(index) + delta * 0.02f);
    return newValue;
}

int
SFVec3fItem::GetFieldOffset(int xpos) const
{
    int offset = xpos / _view->GetFloatWidth();
    if (offset > 2) offset = 2;
    return offset;
}

void
SFVec3fItem::StartEditing(MyString &str, int offset)
{
    char	buf[30];

    sprintf(buf, "%g", ((SFVec3f *) _value)->getValue(offset));
    str = buf;
}

FieldValue *
SFVec3fItem::StopEditing(const char *str, int offset)
{
    SFVec3f     *newValue = new SFVec3f(((SFVec3f *) _value)->getValue());
    newValue->setValue(offset, (float) atof(str));
    return newValue;
}

MFColorItem::MFColorItem(FieldView *view) : MFieldViewItem(view)
{
}

FieldViewItem *
MFColorItem::CreateSFItem()
{
    return new SFColorItem(_view);
}

MFFloatItem::MFFloatItem(FieldView *view) : MFieldViewItem(view)
{
}

FieldViewItem *
MFFloatItem::CreateSFItem()
{
    return new SFFloatItem(_view);
}

MFInt32Item::MFInt32Item(FieldView *view) : MFieldViewItem(view)
{
}

FieldViewItem *
MFInt32Item::CreateSFItem()
{
    return new SFInt32Item(_view);
}

MFNodeItem::MFNodeItem(FieldView *view) : MFieldViewItem(view)
{
}

FieldViewItem *
MFNodeItem::CreateSFItem()
{
    return new SFNodeItem(_view);
}

MFRotationItem::MFRotationItem(FieldView *view) : MFieldViewItem(view)
{
}

FieldViewItem *
MFRotationItem::CreateSFItem()
{
    return new SFRotationItem(_view);
}

MFStringItem::MFStringItem(FieldView *view) : MFieldViewItem(view)
{
}

FieldViewItem *
MFStringItem::CreateSFItem()
{
    return new SFStringItem(_view);
}

//
// when an MFString item is collapsed, draw the 1st element
//

void
MFStringItem::Draw(SDC dc, int x, int y)
{
    int		size = _view->GetItemHeight() - 1;
    if (IsCollapsed()) {
	if (((MFString *) _value)->getSize()) {
	    _children[0]->Draw(dc, x, y);
	} else if (_field->getFlags() & FF_URL) {
	    int	    bx = x + _view->GetItemWidth() - size;
	    swDraw3DRect(dc, _view->GetWindow(), bx, y, size, size);
	    swSetFGColor(dc, swGetWindowColor(_view->GetWindow(), SW_COLOR_TEXT));
	    swDrawText(dc, bx + 2, y + _view->GetItemHeight() - 3, "...");
	}
    }
}

//
// allow the 1st element to be edited, when the item is collapsed
//

bool
MFStringItem::IsEditable() const
{
//    return IsCollapsed();
    return true;
}

void
MFStringItem::StartEditing(MyString &str, int /* offset */)
{
    MFString	    *v = (MFString *) _value;

    if (IsCollapsed())
	str = "";
    else
        InsertSFValue(0);
    _children.insert(new FieldViewItem(_view), 0); 
    InitIndexValue(0, _value);
    str = v->getValue(0);
}

void
MFStringItem::InsertSFValue(int index)
{
    ((MFString *)_value)->insertSFValue(index, (FieldValue *)new SFString(""));
}

void                
MFStringItem::RemoveSFValue(int index) 
{
    ((MFString *)_value)->removeSFValue(index);
}

FieldValue *
MFStringItem::StopEditing(const char *str, int offset)
{
    if (IsCollapsed())
        return new MFString(str);
    else
        {
        ((MFString *)_value)->setSFValue(offset, new SFString(str));
        InitIndexValue(offset, _value);
        return _value;
        }
}

FieldValue *
MFStringItem::OnMouseDown(int x, int y, int modifiers)
{
    int	    width = _view->GetItemWidth();
    int	    buttonSize = _view->GetItemHeight() - 1;

    if (x >= width - buttonSize && x < width) {
	if (_field->getFlags() & FF_URL) {
/*
	    CFileDialog		dlg(TRUE, NULL, NULL, OFN_HIDEREADONLY, "Image Files (*.gif, *.jpg)|*.gif;*.jpg|All Files (*.*)|*.*||",				_view);

	    if (dlg.DoModal() == IDOK) {
		URL	    parent, child;
		const char *url = _view->GetDocument()->getScene()->getURL();

		if (url[0]) {
		    parent = url;
		} else {
		    char	    cdir[_MAX_DIR];
		    GetCurrentDirectory(_MAX_DIR, cdir);
		    parent.FromDOSPath(cdir);
		}
		child.FromDOSPath(dlg.GetPathName());
		return new MFString(child.RelativeTo(parent));
	    }
*/
	} else if (_field->getStrings()) {
	    // create popup
	}
    }
    return NULL;
}


bool
MFFloatItem::IsEditable() const
{
    return IsCollapsed();
}

void
MFFloatItem::StartEditing(MyString &str, int /* offset */)
{
    char	buf[30];
    MFFloat	    *v = (MFFloat *) _value;

    if (v->getSize() >= 1) {
        sprintf(buf, "%g", (float) (v->getValue(0)));
        str = buf;
    } else {
	str = "";
    }
}

FieldValue *
MFFloatItem::StopEditing(const char *str, int offset)
{
    return new MFFloat((float) atof(str));
}

FieldValue *
MFFloatItem::OnMouseDown(int x, int y, int modifiers)
{
    int	    width = _view->GetItemWidth();
    int	    buttonSize = _view->GetItemHeight() - 1;

    if (x >= width - buttonSize && x < width) {
	if (_field->getFlags() & FF_URL) {
/*
	    CFileDialog		dlg(TRUE, NULL, NULL, OFN_HIDEREADONLY, "Image Files (*.gif, *.jpg)|*.gif;*.jpg|All Files (*.*)|*.*||",				_view);

	    if (dlg.DoModal() == IDOK) {
		URL	    parent, child;
		const char *url = _view->GetDocument()->getScene()->getURL();

		if (url[0]) {
		    parent = url;
		} else {
		    char	    cdir[_MAX_DIR];
		    GetCurrentDirectory(_MAX_DIR, cdir);
		    parent.FromDOSPath(cdir);
		}
		child.FromDOSPath(dlg.GetPathName());
		return new MFFloat(child.RelativeTo(parent));
	    }
*/
	} else if (_field->getStrings()) {
	    // create popup
	}
    }
    return NULL;
}

void
MFFloatItem::Draw(SDC dc, int x, int y)
{
    int		size = _view->GetItemHeight() - 1;
    if (IsCollapsed()) {
	if (((MFFloat *) _value)->getSize()) {
	    _children[0]->Draw(dc, x, y);
	} else if (_field->getFlags() & FF_URL) {
	    int	    bx = x + _view->GetItemWidth() - size;
	    swDraw3DRect(dc, _view->GetWindow(), bx, y, size, size);
	    swSetFGColor(dc, swGetWindowColor(_view->GetWindow(), SW_COLOR_TEXT));
	    swDrawText(dc, bx + 2, y + _view->GetItemHeight() - 3, "...");
	}
    }
}

MFTimeItem::MFTimeItem(FieldView *view) : MFieldViewItem(view)
{
}

FieldViewItem *
MFTimeItem::CreateSFItem()
{
    return new SFTimeItem(_view);
}

MFVec2fItem::MFVec2fItem(FieldView *view) : MFieldViewItem(view)
{
}

FieldViewItem *
MFVec2fItem::CreateSFItem()
{
    return new SFVec2fItem(_view);
}

MFVec3fItem::MFVec3fItem(FieldView *view) : MFieldViewItem(view)
{
}

FieldViewItem *
MFVec3fItem::CreateSFItem()
{
    return new SFVec3fItem(_view);
}

