/*
 * InputDevice.h
 *
 * Copyright (C) 2001 J. "MUFTI" Scheurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#ifndef _INPUTDEVICE_H
#define _INPUTDEVICE_H

#include <stdio.h>
#include <stdlib.h>
#include "stdafx.h"
#include "TransformMode.h"
#include "EulerAngles.h"
#include "Quaternion.h"

class InputDeviceApp;
// class to access several joystick like devices via a unified interface

// Example for such devices are spaceballs, magellan-devices, 
// thumbsticks of gamepads and of course joysticks 8-)   

// This can also be used for Xinput devices, that not deliver zero, 
// if you take your hand away from the device like dialbox, mouse or 
// graphics tablett.

class InputDevice
   {
public:
   // true if there is a new event from the joystick
   virtual bool readInputDevice(void) = 0;
   
   Quaternion& get_quaternion(TransformMode* tm, bool local, bool check_only);

   virtual Quaternion& get_quaternion(TransformMode* tm=NULL,
                                      bool check_only=false);
   virtual Quaternion& get_localQuaternion(TransformMode* tm=NULL,
                                           bool check_only=false);
   EulerAngles& get_eulerAngles(TransformMode* tm,bool check_only,bool current);

   Vec3f& get_vector(TransformMode* tm);

   int get_number_axes(void) { return number_max_axis - number_ignored_axes; }
   int get_number_max_axis(void) { return number_max_axis; } 

   int get_number_buttons(void) { return number_buttons; }

   bool isWand(void) { return wandflag; }
   bool isHead(void) { return headflag; }
   virtual bool sendalways(void) { return alwaysflag; }
 
   virtual bool allzero(void);

   virtual bool isTracker(void) { return false; }

   virtual bool hasReadDelay(void) { return false; }

   virtual void prepareRead(void) { return; }

   virtual float maxvalue(int) const=0;

   virtual bool useCurrent(void) { return false; }

   virtual void setHeadNavigation(void) { }
   virtual bool getHeadNavigation(void) { return true; }

   virtual void setIgnoreSize(const char* ignoreSize_string) {}

   float getxyzfactor(void) { return xyzfactor; }
   float getrotfactor(void) { return rotfactor; }

   void setxyzfactor(float f) { xyzfactor = f; }
   void setrotfactor(float f) { rotfactor = f; }

   void set_firstflag(void);

   InputDevice(void) 
      {
      int i;
      static char* axesinfodata[6]= { 
         "xrot", 
         "yrot", 
         "zrot", 
         "x", 
         "y", 
         "z" 
      };
      // initialise joystick class default data
      button_pressed=-1;
      button_released=-1;
      number_buttons=0;
      number_max_axis=0;
      number_max_axes=7;
      number_ignored_axes=0;
      for (i=0;i<number_max_axes;i++)
         value[i]=0;
      headflag=false;
      wandflag=false;
      alwaysflag=false;
      num_axis_x=0;
      num_axis_y=1;
      num_axis_z=2;
      num_axis_xrot=3;
      num_axis_yrot=4;
      num_axis_zrot=5;
      num_axis_angle=6;
      zero_on_release=NULL;
      sign_swap=NULL;
      factor=NULL;
      acceleration=NULL;
      zero_size_fraction=NULL;
      for (i=0;i<6;i++)
         axesinfo[i]=axesinfodata[i];
      for (i=0;i<7;i++)
         firstflag[i]=true;
      for (i=0;i<7;i++)
         ignore[i]=false;
      alwaysflag=false;
      xyzfactor=1;
      rotfactor=1;
      }
   ~InputDevice() 
      {
      if (zero_on_release!=NULL)
         delete zero_on_release;
      if (acceleration!=NULL)
         delete acceleration;
      }
   void SetAxisInformation(char* info_string);
   void SetNumberAxes(char*);
   void SetSendAlways(bool flag) {alwaysflag=flag;}

protected:

   float get_x(TransformMode* tm=NULL,bool check_only=false,bool curr=false);
   float get_y(TransformMode* tm=NULL,bool check_only=false,bool curr=false);
   float get_z(TransformMode* tm=NULL,bool check_only=false,bool curr=false);
   float get_xrot(TransformMode* tm=NULL,bool check_only=false,bool curr=false);
   float get_yrot(TransformMode* tm=NULL,bool check_only=false,bool curr=false);
   float get_zrot(TransformMode* tm=NULL,bool check_only=false,bool curr=false);
   float get_angle(TransformMode* tm=NULL,bool check_only=false,bool curr=false);

   // is device a headtracker ?
   bool headflag;
   // is device a wand ?
   bool wandflag;
   // do device send (amost) always values ?
   bool alwaysflag;

   // axes of joystick
   int number_max_axis;
   // number of axes used in account (7 (including angle axis))
   int number_max_axes;
   // number of ignored axes of joystick
   int number_ignored_axes;

   // 0-2 offset and 3-7 rotation values from joystick (if available)
   float value[7];

   // buttons of joystick
   int number_buttons;
   bool *button;
   // last pressed/released button
   int button_pressed;
   int button_released;

   // number of axis for values
   int num_axis_x;
   int num_axis_y;
   int num_axis_z;
   int num_axis_xrot;
   int num_axis_yrot;
   int num_axis_zrot;
   int num_axis_angle;   

   // true if axis number deliver zero when device released
   bool get_zero_on_release(int axis_number);
   void set_zero_on_release(int axis_number,bool value);
   void set_zero_on_release(char* axis_string,bool value);

   // formula to accelerate values from axis
   float accelerate(float value,int num_axis);

   // signswap of value from axis
   int get_sign(int axis_number);
   void set_sign_swap(int axis_number,bool value);

   // factor: multiplcated to value from axis
   float get_factor(int axis_number);
   void set_factor(int axis_number,float value);
   void set_factor(char* axis_string,float value);

   // acceleration: additional acceleration parameter
   float get_acceleration(int axis_number);
   void set_acceleration(int axis_number,float value);
   void set_acceleration(char* axis_string,float value);

   // percent (relative to max) of value which will be ignored
   // "insensitivity of device"
   float get_zero_size_fraction(int axis_number);  
   void set_zero_size_fraction(int axis_number,float value);
   void set_zero_size_fraction(char* string,float value);

   // function to read/set configuration data like
   //    axisnumbers, zero_on_release, acceleration
   int getAxisFromInformation(char* axesinfo);
   void setAxisOfInformation(char* axesinfo,int axis_number);

protected:

   bool max_value(int index,TransformMode* tm);
   float get_value(int index,bool check_only,int num_axis);
   float get_value(TransformMode* tm, 
                   int index, bool check_only, int num_axis, bool current);

   float get_current_value(int index,int num_axis);

   // array of flags for first get from device
   bool firstflag[7];

   // flags if the value of a axis should be ignored
   bool ignore[7];

   // array of old values from device
   float oldvalue[7];

   // array of nullsize in percent
   float* zero_size_fraction;

   // array of flags if axis number arrayindex deliver zero when device is 
   // released
   bool* zero_on_release;

   // array of flags, if sign of result is swapped
   bool* sign_swap;
  
   // array of factor to multiply with value of axis
   float* factor;

   // array of acceleration per axis
   float* acceleration;

   // helper functions for SetAxisInformation
   bool scan_argument(char *string,char* argument);
   bool argument_is_axis(char* string);
   char* axesinfo[6]; 

   float xyzfactor;
   float rotfactor;

   Quaternion old_quat;
   Quaternion old_local_quat;
   };

# ifdef LINUX_JOYSTICK

// using Linux Joystick interface

#  define NAME_LENGTH 128
#  include <linux/joystick.h>

class linux_joystick : public InputDevice 
   {
public: 
   linux_joystick() {}
   ~linux_joystick();
   linux_joystick(char* device);
   bool readInputDevice(void);
private:
   int fd;
   int version;
   char name[NAME_LENGTH];
   struct js_event js;
   float maxvalue(int i) const {return(32767.0);}
   float factor;
   };

# endif

# ifdef WINDOWS_JOYSTICK

// using M$Windows Joystick interface

#  include <Windows.h>

class windows_joystick : public InputDevice 
   {
public: 
   windows_joystick() {}
   ~windows_joystick();
   windows_joystick(char* device);
   bool readInputDevice(void);
private:
   int number_of_joystick;
   int version;
   char name[MAXPNAMELEN];
   JOYCAPS joycapabilities;
   JOYINFOEX joyinfo_ex;
   int *max_value;
   int *null_value;
   float maxvalue(int i) const {return max_value[i];}
   float factor;
   };

# endif

# ifdef HAVE_SDL_JOYSTICK

// using modified Simple DirectMedia Layer joystick interface
# include "SDLjoystick/SDL_joystick.h"

class SDL_joystick : public InputDevice 
   {
public: 
//   SDL_joystick() {}
   ~SDL_joystick();
   SDL_joystick(char* device);
   bool readInputDevice(void);
private:
   SDL_Joystick* js;
   int number_of_joystick;
   int *max_value;
   int *null_value;
   float maxvalue(int i) const {return max_value[i];}
   float factor;
   };

# endif

# ifdef HAVE_XINPUT

// Xi X11 XInput devices (e.g. spaceball, magellan) 

#  include <X11/extensions/XInput.h>
extern "C" 
   {
#  include "swtxinput.h"
   }

extern Display* swGetDisplay(void);
extern Window* swGetWindow(void);

class xinput : public InputDevice 
   {
public: 
   xinput() {}
   ~xinput();
   xinput(char* device);
   bool readInputDevice(void);
private:
   char* name;
   swXinput* swxinput;
   float factor;
   float *max_value;
   float maxvalue(int i) const {return max_value[i];}
   int nullsize;
   };

# endif

# ifdef HAVE_LIBSBALL

// using libsball library for SpaceTec/LabTec Spaceball devices

#  include <sball.h>

class spaceball : public InputDevice 
   {
public: 
   spaceball() {}
   ~spaceball();
   spaceball(char* device);
   bool readInputDevice(void);
private:
   char* name;
   SBallHandle sball;
   float maxvalue(int i) const {return max_value;}
   float max_value;
   float factor;
   int nullsize;
   };

# endif

# ifdef HAVE_WINDOWS_SPACEBALL

// using libsball library for SpaceTec/LabTec Spaceball devices on M$Windows

//REPLACE
//#  include <sball.h>

class windowsspaceball : public InputDevice 
   {
public: 
   windowsspaceball() {}
   ~windowsspaceball();
   windowsspaceball(char* device);
   bool readInputDevice(void);
private:
   char* name;
// REPLACE
//   SBallHandle sball;
   float maxvalue(int i) const {return max_value;}
   float max_value;
   float factor;
   int nullsize;
   };

# endif

# ifdef HAVE_AFLOCK

// using aFlock program from VR Juggler library for 
// Ascention Flock of birds devices

#  include "Aflock.h"

class AflockDevice
   {
public: 
   AflockDevice(char* device);
   ~AflockDevice();
   Aflock* AflockDevice::getAflock(void);
   void close();
   // parameters
   void setBaud(char* baudrate_string);
   void setSync(char* sync_string);
   void setBlock(char* block_string);
   void setNumBrds(char* numBrds_string);
   void setTransmit(char* transmit_string);
   void setHemi(char* hemi_string);
   void setFilt(char* filt_string);
   void setSuddenChangeLock(char* sudden_string);
   void setReport(char* report_string);
   void setCalfile(char* calfile_string);
private:
   char* name;
   Aflock* flock;
   bool opened;   

   char* device;
   int baudrate;
   int sync;
   int block;
   int numBrds;
   int transmit;
   BIRD_HEMI hemi;
   BIRD_FILT filt;
   bool sudden_change_lock;
   char report;
   char* calfile;
   };

class aflock : public InputDevice 
   {
public: 
   aflock(AflockDevice* device,char* receiver_string,bool headflag);
   ~aflock();
   bool readInputDevice(void);
   bool sendalways(void) { return true; }
   bool hasReadDelay(void) { return true; }
   void prepareRead(void);         
   void setHeadNavigation(void) { headNavigation=true; }
   bool getHeadNavigation(void) { return isTracker() && headNavigation; }
   bool useCurrent(void) { return true; }
   void setIgnoreSize(char* ignoresize_string);

private:
   int receiver;   
   char* name;
   bool head;
   bool wand;
   bool headNavigation;
   bool isTracker(void) { return head; }
   bool isWand(void) { return wand; }
   Aflock* flock;
   float maxvalue(int i) const {return max_value;}
   float max_value;
   float factor;
   float degreefactor;
   bool aflockFirstFlag;
   float ignoreSize;
   float old_value[3];
   };
# endif

#endif
