/*
 * NodeTimeSensor.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include "stdafx.h"

#include "NodeTimeSensor.h"
#include "Proto.h"
#include "Scene.h"
#include "FieldValue.h"
#include "SFTime.h"
#include "SFBool.h"
#include "SFFloat.h"


ProtoTimeSensor::ProtoTimeSensor(Scene *scene)
  : Proto(scene, "TimeSensor")
{
    cycleInterval.set(
          addExposedField(SFTIME, "cycleInterval", new SFTime(1.0), 
		          new SFTime(0.0)));
    enabled.set(
          addExposedField(SFBOOL, "enabled", new SFBool(true)));
    loop.set(
          addExposedField(SFBOOL, "loop", new SFBool(false)));
    startTime.set(
          addExposedField(SFTIME, "startTime", new SFTime(0.0)));
    stopTime.set(
          addExposedField(SFTIME, "stopTime", new SFTime(0.0)));
    addEventOut(SFTIME, "cycleTime");
    // magic numbers still to delete, concept for "sendEvent" is needed
    fraction_changed.set(6);
    addEventOut(SFFLOAT, "fraction_changed");
    addEventOut(SFBOOL, "isActive");
    time.set(8);
    addEventOut(SFTIME, "time");
}

Node *
ProtoTimeSensor::create(Scene *scene)
{ 
    return new NodeTimeSensor(scene, this); 
}

NodeTimeSensor::NodeTimeSensor(Scene *scene, Proto *def)
  : Node(scene, def)
{
}

void
NodeTimeSensor::preDraw()
{
    if (enabled()->getValue()) _scene->addTimeSensor(this);
}

void
NodeTimeSensor::setTime(double t)
{
    double	dstopTime = stopTime()->getValue();

    if (enabled()->getValue()) {
	if (t > _startTime && t < dstopTime) {
	    _active = true;
	}
	if (_active) {
	    if (t >= dstopTime && 
                dstopTime > _startTime && !loop()->getValue()) {
		_active = false;
	    }
	    double temp = (t - _startTime) / cycleInterval()->getValue();
	    double fraction = temp - floor(temp);
	    if (fraction == 0.0 && t > _startTime) fraction = 1.0;
	    sendEvent(fraction_changed_Index(), t, 
                      new SFFloat((float) fraction));
	    sendEvent(time_Index(), t, new SFTime(t));
	}
    }
}

void
NodeTimeSensor::checkStart(bool loop, double startTime, double stopTime, 
                           double t)
{
    if (stopTime <= startTime && loop) {
	_active = true;
	_startTime = t;
    } else {
	_active = false;
	_startTime = startTime;
    }
}

void
NodeTimeSensor::start(double t)
{
    checkStart(loop()->getValue(), startTime()->getValue(), 
               stopTime()->getValue(), t);
}

void 	    
NodeTimeSensor::updateStart(int field, FieldValue *value, double t)
{
    bool statusChanged = false;
    FieldValue* fieldValue;
   
    fieldValue = getField(loop_Index());
    if (field == loop_Index()) {
        fieldValue = value;
        statusChanged = true;
    }
    
    fieldValue = getField(startTime_Index());
    if (field == startTime_Index()) {
        fieldValue = value;
        statusChanged = true;
    }

    fieldValue = getField(stopTime_Index());
    if (field == stopTime_Index()) {
        fieldValue = value;
        statusChanged = true;
    }

    checkStart(loop()->getValue(), startTime()->getValue(), 
               stopTime()->getValue(), t);
}
